
#include <math.h>

#include "external.h"
#include "openglstuff.h"
#include "textures.h"
#include "terrain.h"
#include "game.h"
#include "drawstuff.h"
#include "particles.h"

void RemovePitFromTerrain(const Pit &pit);

extern Camera			camera, debugCamera;
extern GLTexture		*textures;
extern GLUquadricObj	*quadratic;
extern int				terrainLength, terrainSize;

extern HillPiece		**terrain;

extern int				targetFPS;
extern int				xDrawOffset, zDrawOffset;
extern double			spacing;
extern float debug1, debug2;


const double pi = 3.141592653589;
const double sqrtTwo = sqrt(2.0);
const double piOver180 = pi/180.0;
const double oneEightyOverPi = 180.0/pi;

bool flight = false;
bool invincible = false;

double	maxVSpeed;
double	minSpeed;
int		airHoldTime, wallDealy;
double	gravity;

double		normalSpeed;
bool		initializeCamera;

UnicycleGuy	guy;



int			maxWalls;
int			currentWallNum;
double		wallsPerUnitArea;
BrickWall	*walls = NULL;
bool withinWall = false;
bool withinPit = false;

double		maxWallLength, minWallLength;
double		maxWallHeight, minWallHeight;
double		maxWallWidth, minWallWidth;

int			maxPits;
int			currentPitNum;
double		pitsPerUnitArea;
Pit			*pits = NULL;
Pit			*lastPit = NULL, *firstPit = NULL;
double		maxPitLength = 10;
double		minPitLength = 10;
double		maxPitWidth = 3;
double		minPitWidth = 0.5;

int			maxSmilies;
int			currentSmileyNum;
double		smiliesPerUnitArea;
double		bigSmileyRatio;
SmileyFace	*smilies = NULL;
SmileyFace	*lastSmiley = NULL, *firstSmiley = NULL;
double		minSmileyHeight = 1;
double		maxSmileyHeight = 3;
double		smileyRotation = 0;
double		smileyRotationSpeed;



void PlayGame()
{
	
	MoveUnicycleGuy();
	CreateMoreTerrain(guy.x, guy.z);
	DetectStuff();
	AdjustCamera();
}



void DetectStuff()
{
	int topLeftX, topLeftZ, oppositeX, oppositeZ;
	Pit *pit, *prevPit, *nextPit;
	SmileyFace *smiley, *prevSmiley, *nextSmiley;
	int pitsToProcess = 0, smiliesToProcess;
	
	topLeftX = terrain[xDrawOffset][zDrawOffset].x*spacing;
	topLeftZ = terrain[xDrawOffset][zDrawOffset].z*spacing;
	
	wallDealy = 0;
	for(int i = 0; i < maxWalls; ++i) {
		if (walls[i].exists) {
			if (walls[i].x < topLeftX || walls[i].x > topLeftX + terrainLength ||
				walls[i].z < topLeftZ || walls[i].z > topLeftZ + terrainLength) {
					
					oppositeX = walls[i].x + walls[i].length*cos(walls[i].rotation*piOver180);
					oppositeZ = walls[i].z + walls[i].length*sin(walls[i].rotation*piOver180);
					
					if (	(walls[i].x < topLeftX && oppositeX < topLeftX) ||
							(walls[i].x > topLeftX+terrainLength && oppositeX > topLeftX+terrainLength) ||
							(walls[i].z < topLeftZ && oppositeZ < topLeftZ) ||
							(walls[i].z > topLeftZ+terrainLength && oppositeZ > topLeftZ+terrainLength)) {
						
						
						walls[i].exists = false;
						--currentWallNum;
					}
			}
			
			walls[i].CheckForDeath();
		}
	}
	
	prevPit = NULL;
	pit = firstPit;
	
	pitsToProcess = currentPitNum;
	
	for(int i = 0; i < pitsToProcess && pit; ++i) {
		
		nextPit = pit->next;
		
		if (	pit->x < topLeftX || pit->x > topLeftX + terrainLength ||
				pit->z < topLeftZ || pit->z > topLeftZ + terrainLength) {
			
			pit->exists = false;
			--currentPitNum;
			
			if (pit == firstPit && pit == lastPit) {
				firstPit = lastPit = NULL;
				
			} else if (pit == firstPit) {
				firstPit = nextPit;
				pit->next = lastPit->next;
				lastPit->next = pit;
				
				pit = nextPit;
				
			} else if (pit == lastPit) {
				lastPit = prevPit;
				
				pit = nextPit; // unneccesary?
				
			} else {
				prevPit->next = nextPit;
				pit->next = lastPit->next;
				lastPit->next = pit;
				
				pit = nextPit;
			}
		} else {
			pit->CheckForDeath();
			
			prevPit = pit;
			pit = pit->next;
		}
	}
	
	
	
	prevSmiley = NULL;
	smiley = firstSmiley;
	
	smiliesToProcess = currentSmileyNum;
	
	for(int i = 0; i < smiliesToProcess && smiley; ++i) {
		
		nextSmiley = smiley->next;
		
		if (	smiley->x < topLeftX-2 || smiley->x > topLeftX + terrainLength ||
				smiley->z < topLeftZ-2 || smiley->z > topLeftZ + terrainLength ||
				smiley->CheckForGotten()) {
			
			smiley->exists = false;
			--currentSmileyNum;
			
			if (smiley == firstSmiley && smiley == lastSmiley) {
				firstSmiley = lastSmiley = NULL;
				
			} else if (smiley == firstSmiley) {
				firstSmiley = nextSmiley;
				smiley->next = lastSmiley->next;
				lastSmiley->next = smiley ;
				
				smiley = nextSmiley;
				
			} else if (smiley == lastSmiley) {
				lastSmiley = prevSmiley;
				
				smiley = nextSmiley; // unneccesary?
				
			} else {
				prevSmiley->next = nextSmiley;
				smiley->next = lastSmiley->next;
				lastSmiley->next = smiley ;
				
				smiley = nextSmiley;
			}
		} else {
			
			prevSmiley = smiley ;
			smiley = smiley->next;
		}
	}
}


void CreatePit(int x, int z, double rotationMin)
{
	Pit *pit;
	if (currentPitNum >= maxPits) return;
	
	if (lastPit == NULL || firstPit == NULL || currentPitNum == 0) {
		pit = &pits[0];
		lastPit = pit;
		firstPit = pit;
	} else {
		
		if (lastPit->next == NULL)
			return;
		pit = lastPit->next;
		lastPit = pit;
	}
	
	pit->ActuallyCreatePit(x, z, rotationMin);
	
	++currentPitNum;
	/*
	for(int i = 0; i < maxPits; ++i) {
		if (!pits[i].exists) {
			
			pits[i].ActuallyCreatePit(x, z, rotationMin);
			
			return;
		}
	}*/
}

void Pit::ActuallyCreatePit(int xa, int za, double rotationMin)
{
	exists = true;
	x = xa;
	z = za;
	
	rotation = RandomNumber(rotationMin+2, rotationMin + 88);//, (z*terrainLength + x));
	
	//rotation = rotationMin+47;
	
	length = (int)RandomNumber(minPitLength, maxPitLength);
	width = (int)RandomNumber(minPitWidth, maxPitWidth);
	
	if (rotation == 0)
		rotation = rotation;
	
	corners.points[kTopleft].x = x;
	corners.points[kTopleft].z = z;
	// note: we don't care about the y value
	
	corners.points[kTopRight].x = x + length * cos(rotation * piOver180);
	corners.points[kTopRight].z = z + length * sin(rotation * piOver180);
	
	corners.points[kBottomRight].x = x + length * cos(rotation * piOver180) + width * -sin(rotation * piOver180);
	corners.points[kBottomRight].z = z + length * sin(rotation * piOver180) + width * cos(rotation * piOver180);
	
	corners.points[kBottomLeft].x = x + width * -sin(rotation * piOver180);
	corners.points[kBottomLeft].z = z + width * cos(rotation * piOver180);
	
	minX = min(min(min(corners.points[0].x, corners.points[1].x), corners.points[2].x), corners.points[3].x);
	minZ = min(min(min(corners.points[0].z, corners.points[1].z), corners.points[2].z), corners.points[3].z);
	maxX = max(max(max(corners.points[0].x, corners.points[1].x), corners.points[2].x), corners.points[3].x);
	maxZ = max(max(max(corners.points[0].z, corners.points[1].z), corners.points[2].z), corners.points[3].z);
}

void Pit::CheckForDeath()
{
	double theta1, theta2, r, dx, dz, newX, newZ;
	double buffer = 0;//0.1;
	
	if (!WithinRange(guy.x, guy.z, x, z, max(length,width)) || flight || invincible) return;
	
	dx = guy.x - x;
	dz = guy.z - z;
	
	r = sqrt(dx*dx + dz*dz);
	
	if (dx > 0)
		theta1 = asin(dz/r);
	else if (dz > 0)
		theta1 = acos(dx/r);
	else
		theta1 = acos(-dx/r) + pi;
	theta2 = theta1 - (rotation * piOver180);
	
	newX = cos(theta2) * r + x;
	newZ = sin(theta2) * r + z;
	
	if (WithinRect( newX, newZ, x - buffer, z - buffer, x + length + buffer, z + width + buffer)) {
		withinPit = true;
		if (!guy.inAir) {
			FallFallyMcFallsalotDeathySplayFall();
		}
	}
}

void FallFallyMcFallsalotDeathySplayFall()
{
	if (guy.falling > -1) return;
	
	guy.falling = 0;
	
	guy.ay = gravity;
	if (guy.justLanded) {
		
		guy.vx = guy.vz = 0;
		
		
	} else {
		
		guy.vy = -1;
		guy.vz = cos(guy.ry * piOver180) * guy.speed;
		guy.vx = sin(guy.ry * piOver180) * guy.speed;
		
	}
}


void CreateWall(int x, int z, double rotationMin)
{
	if (currentWallNum >= maxWalls) return;
	
	for(int i = 0; i < maxWalls; ++i) {
		if (!walls[i].exists) {
			
			walls[i].exists = true;
			walls[i].x = x;
			walls[i].z = z;
			walls[i].height = RandomNumber(minWallHeight,maxWallHeight);
			walls[i].rotation = RandomNumber(rotationMin, rotationMin + 180);//, (z*terrainLength + x));
			walls[i].length = (int)RandomNumber(minWallLength,maxWallLength);
			walls[i].width = RandomNumber(minWallWidth,maxWallWidth);
			
			++currentWallNum;
			return;
		}
	}
}

void DrawWalls()
{
	
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, textures[kBrickWall].GetTextureID());
	
	glEnable(GL_LIGHTING);
	
	wallDealy = 0;
	for(int i = 0; i < maxWalls; ++i) {
		if (walls[i].exists)
			walls[i].Draw();
	}
}


void BrickWall::CheckForDeath()
{
	double theta1, theta2, r, dx, dz, newX, newZ, wallY;
	double buffer = 0.1;
	
	if (invincible)
		return;
	//return;
	
	if (!WithinRange(guy.x, guy.z, x, z, length) || flight) return;
	++wallDealy;
	
	dx = guy.x - x;
	dz = guy.z - z;
	
	r = sqrt(dx*dx + dz*dz);
	
	if (dx > 0)
		theta1 = asin(dz/r);
	else if (dz > 0)
		theta1 = acos(dx/r);
	else
		theta1 = acos(-dx/r) + pi;
	theta2 = theta1 - (rotation * piOver180);
	
	newX = cos(theta2) * r + x;
	newZ = sin(theta2) * r + z;
	
	if (WithinRect( newX, newZ, x - buffer, z-width/2 - buffer, x + length + buffer, z+width/2 + buffer)) {
		wallY = minYValue(guy.x, guy.z) + height-0.2;
		if (guy.y < wallY)
			KillMurderKillStabityStabStabStabKill();
		withinWall = true;
	}
}

void DrawSmilies()
{
	SmileyFace *smiley;
	
	smileyRotation += smileyRotationSpeed;
	
	glEnable(GL_LIGHT1);
	glDisable(GL_LIGHT0);
	
	smiley = firstSmiley;
	
	for(int i = 0; i < currentSmileyNum && smiley; ++i) {
		smiley->Draw(smileyRotation);
		
		smiley = smiley->next;
	}	
	
	glDisable(GL_LIGHT1);
	glEnable(GL_LIGHT0);
}

bool WithinRange(double x, double z, double cx, double cz, double r)
{
	return (x >= cx-r && x <= cx+r && z >= cz-r && z <= cz+r);
}

bool WithinRect(double x, double z, double left, double top, double right, double bottom)
{
	return (x >= left && x <= right && z >= top && z <= bottom);
}

void CreateSmiley(double x, double z)
{
	SmileyFace *smiley;
	
	x = floor(x/spacing)*spacing - 1;
	z = floor(z/spacing)*spacing - 1;
	
	if (currentSmileyNum >= maxSmilies) return;
	
	if (lastSmiley == NULL || firstSmiley == NULL || currentSmileyNum == 0) {
		smiley = &smilies[0];
		lastSmiley = smiley;
		firstSmiley = smiley;
	} else {
		
		if (lastSmiley->next == NULL)
			return;
		smiley = lastSmiley->next;
		lastSmiley = smiley;
	}
	
	smiley->x = x;
	smiley->z = z;
	smiley->y = RandomNumber(minSmileyHeight, maxSmileyHeight);
	smiley->r = RandomNumber(0,180);
	smiley->minYCalculated = false;
	smiley->shadowCalculated = false;
	smiley->exists = true;
	
	smiley->big = (RandomNumber(0,1) < bigSmileyRatio);
	
	++currentSmileyNum;
}

bool SmileyFace::CheckForGotten()
{
	double detectionRange = 0.6;
	
	if (	WithinRect(guy.x, guy.z, x-detectionRange, z-detectionRange, x+detectionRange, z+detectionRange) &&
			guy.y+2.5 >= y && guy.y-0.5 <= y) {
		// got happy face!
		
		CreateParticles(20, kPSparkle,
							x,y,z, // center
							-20, 200, -20,200,  // angle
							0,0.05,0, // initial velocity
							0, -15, 0,   // accel
							10,  // rotation
							3, 6,  // min/max speed
							0.1, 0.25,  //min/max size
							2, 2); // life
		
		return true;
	} else
		return false;
}


void DoDeviousThings(int xOffset, int zOffset)
{
	int topLeftX, topLeftZ, x, z, newSpaces;

	//topLeftX = terrain[xDrawOffset][zDrawOffset].x*spacing;
	//topLeftZ = terrain[xDrawOffset][zDrawOffset].z*spacing;
	
	topLeftX = (terrain[xDrawOffset][zDrawOffset].x+xOffset)*spacing;
	topLeftZ = (terrain[xDrawOffset][zDrawOffset].z+zOffset)*spacing;

	if (xOffset != 0) {
		newSpaces = abs(xOffset);
		x = (xOffset < 0) ? topLeftX + newSpaces: topLeftX + terrainLength - newSpaces;
		for(int i = 0; i < newSpaces; ++i) {
			for(z = topLeftZ; z < terrainLength+topLeftZ; ++z) {
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < wallsPerUnitArea)
					CreateWall(x,z, ((xOffset < 0) ? 90 : 270));
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < pitsPerUnitArea)
					CreatePit(x,z, ((xOffset < 0) ? 90 : 270));
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < smiliesPerUnitArea)
					CreateSmiley(x,z);
			}
			++x;
		}
	}
	if (zOffset != 0) {
		newSpaces = abs(zOffset);
		z = (zOffset < 0) ? topLeftZ + newSpaces: topLeftZ + terrainLength - newSpaces;
		for(int i = 0; i < newSpaces; ++i) {
			for(x = topLeftX; x < terrainLength+topLeftX; ++x) {
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < wallsPerUnitArea)
					CreateWall(x,z, ((zOffset < 0) ? 180 : 0));
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < pitsPerUnitArea)
					CreatePit(x,z, ((zOffset < 0) ? 180 : 0));
				
				if (RandomNumber(0.0, 1.0, z * terrainLength + x) < smiliesPerUnitArea)
					CreateSmiley(x,z);
			}
			++z;
		}
	}
}



void MoveUnicycleGuy()
{
	double minY;
	double airControl = 10;
	double targetVZ, targetVX;
	bool *keys = GetKeys();
	
	guy.justLanded = false;
	
	if (flight) {
		double rotationSpeed = ConvertSpeed(60), rx;
		double targetSpeed;
		bool blah = false;
		
		if (GetKey('a')) {
			guy.rz -= rotationSpeed*3.0;
			guy.ry += rotationSpeed;
			blah = true;
			if (guy.rz < -90)
				guy.rz = -90;
		} else if (keys['d']) {
			guy.rz += rotationSpeed*3.0;
			guy.ry -= rotationSpeed;
			blah = true;
			if (guy.rz > 90)
				guy.rz = 90;
		}
		if (keys['w']) {
			guy.rx += rotationSpeed;
		} else if (keys['s'])
			guy.rx -= rotationSpeed;


			if (!blah && guy.rz != 0) {
				if (guy.rz > 0) {
					guy.rz -= rotationSpeed*3;
					if (guy.rz < 0)
						guy.rz = 0;
				} else {
					guy.rz += rotationSpeed*3;
					if (guy.rz > 0)
						guy.rz = 0;
				}
			}
			
			
		if (keys[' ']) {
			
			targetSpeed = normalSpeed*2;
		} else if (keys[SDLK_RSHIFT] || keys[SDLK_LSHIFT]) {
			targetSpeed = 0;
		} else {
			targetSpeed = normalSpeed;
			
		}
	
		if (guy.ry > 360)
			guy.ry -= 360;
		if (guy.ry < 0)
			guy.ry += 360;
			
		if (guy.rx > 180)
			guy.rx = 180;
		else if (guy.rx < 0)
			guy.rx = 0;
		
		rx = guy.rx-90;
		
		
		guy.vz = cos(guy.ry*piOver180)*cos(rx*piOver180) * targetSpeed;
		guy.vx = sin(guy.ry*piOver180)*cos(rx*piOver180) * targetSpeed;
		guy.vy = -sin(rx*piOver180) * targetSpeed;
		
		//guy.vx = guy.vy = 0;
		
		guy.x += guy.vx;
		guy.y += guy.vy;
		guy.z += guy.vz;
		
		minY = minYValue(guy.x, guy.z);
		if (guy.y < minY) {
			guy.y = minY;
			guy.rx = 90;
		}
			
		return;
	}
	
	if (!guy.reallyDead) {
		if (guy.dead)
			guy.rx += ConvertSpeed(1800);
		if (guy.falling > -1) {
			
			guy.vx /= 2.0;
			guy.vz /= 2.0;
			guy.vy += guy.ay;
			
			guy.x += guy.vx;
			guy.y += guy.vy;
			guy.z += guy.vz;
			
			
			minY = minYValue(guy.x, guy.z);
			if (minY-guy.y > 200) guy.y = minY-200;
			
			if (minY-guy.y > 50)
				guy.falling = 150 - ((minY-guy.y) - 50);
			else
				guy.falling = 150;
			
			//guy.rx += ConvertSpeed(400);
			
		} else {
			if (guy.inAir) {
					
				targetVZ = cos(guy.ry * piOver180) * guy.speed;
				targetVX = sin(guy.ry * piOver180) * guy.speed;
				
				guy.vx = (targetVX + guy.vx * airControl) / (airControl + 1.0);
				guy.vz = (targetVZ + guy.vz * airControl) / (airControl + 1.0);
				
				guy.vx += guy.ax;
				guy.vz += guy.az;
				/*
				if (guy.vy > 0) {
					if (guy.airTime >= 0)
						guy.airTime -= 1;
					else {
						guy.vy += guy.ay;
						if (guy.vy < 0) guy.airTime = airHoldTime;
					}
				} else {
					if (guy.vy > -maxVSpeed)
						guy.vy += guy.ay;
					else {
						if (guy.airTime > 0) {
							guy.airTime -= 1;
							guy.vy = -maxVSpeed;
						} else
							guy.vy += guy.ay;
					}
				}*/
				guy.vy += guy.ay;
				
				guy.x += guy.vx;
				guy.y += guy.vy;
				guy.z += guy.vz;
				
				minY = minYValue(guy.x, guy.z);
				if (guy.y < minY) {
					guy.y = minY;
					guy.justLanded = true;
					guy.inAir = false;
					if (guy.dead && !withinPit)
						GuyGoSplat();
					
				}
			} else {
				
				guy.vz = cos(guy.ry * piOver180) * guy.speed;
				guy.vx = sin(guy.ry * piOver180) * guy.speed;
				guy.vy = 0;
				guy.z += guy.vz;
				guy.x += guy.vx;
				
				minY = minYValue(guy.x, guy.z);
				if (fabs(double(minY-guy.y)) > 0.2 && keys['q']) {
					minY = minYValue(guy.x, guy.z);
				}
				guy.y = minY;
				
			}
			UnicycleControls();
		}
	}
	withinPit = false;
	withinWall = false;
}


bool sparkleKey = false;
void UnicycleControls()
{
	double rotationSpeed = ConvertSpeed(90);
	double frameFraction = (targetFPS / 60.0);
	double leaningWeight = 10 * frameFraction;
	double speedWeight = 1.5 * frameFraction;
	double targetSpeed;
	bool jumpHigher = false;
	bool *keys = GetKeys();
	
	if (keys['g']) {
		if (!sparkleKey) {
			
			CreateParticles(20, kPDirt,
								guy.x, guy.y + 1, guy.z, // center
								-20, 200, guy.ry+90, guy.ry+90,  // angle
								guy.vx, guy.vy, guy.vz, // initial velocity
								0, 0, 0,   // accel
								10,  // rotation
								1, 1,  // min/max speed
								0.3, 0.3,  //min/max size
								1, 1); // life
			
			
		}
		sparkleKey = true;
	} else
		sparkleKey = false;
	
	if (keys[SDLK_RSHIFT] || keys[SDLK_LSHIFT])
		jumpHigher = true;
	
	targetSpeed = normalSpeed;
	if (!guy.dead) {
		/*
		guy.inAir = true;
		guy.ay = 0;
		guy.az = 0;
		guy.ax = 0;
		guy.airTime = airHoldTime;
		guy.vy = 0;
		if (keys['r'])
			guy.vy = normalSpeed;
		if (keys['f'])
			guy.vy = -normalSpeed;
		*/
		
		guy.targetXLean = guy.targetZLean = 0;
		
		if (GetKey('a')) {
			guy.targetZLean = -rotationSpeed * guy.speed * 50 * frameFraction * frameFraction;
			guy.ry += rotationSpeed;
		} else if (keys['d']) {
			guy.targetZLean = rotationSpeed * guy.speed * 50 * frameFraction * frameFraction;
			guy.ry -= rotationSpeed;
		}
		
		if (keys[' ']) {
			if (!guy.jumpKey && !guy.inAir) {
				guy.ay = gravity;
				guy.inAir = true;
				guy.vy = maxVSpeed + ConvertSpeed(jumpHigher * 2);
				guy.airTime = airHoldTime;
				guy.vz = cos(guy.ry * piOver180) * guy.speed;
				guy.vx = sin(guy.ry * piOver180) * guy.speed;
			
			}
			guy.jumpKey = true;
		} else {
			if (guy.vy > 0) {
				guy.vy /= 1.5;
				guy.airTime = 0;
			}
			guy.jumpKey = false;
		}
		
		if (keys['w']) {
			guy.targetXLean = 20;
			targetSpeed = normalSpeed * 2;
		} else if (keys['s'])
			targetSpeed = normalSpeed / 1000000.0;
		
		
		guy.xLean = (guy.targetXLean + guy.xLean * leaningWeight) / (leaningWeight + 1.0);
		guy.zLean = (guy.targetZLean + guy.zLean * leaningWeight) / (leaningWeight + 1.0);
		
		if (guy.zLean > 35) guy.zLean = 35;
		if (guy.zLean < -35) guy.zLean = -35;
	}
	// accelerates and deccelerates Unicycle guy:
	guy.speed = (targetSpeed + guy.speed * speedWeight) / (speedWeight + 1.0);
	
	if (guy.speed < minSpeed) guy.speed = minSpeed;
	
	
	
	
	if (guy.ry > 360)
		guy.ry -= 360;
	if (guy.ry < 0)
		guy.ry += 360;
	
	
}

double lastD = 0;

void AdjustCamera()
{
	double rotationWeightX = 9 * (targetFPS / 60.0);
	double rotationWeightY = 1 * (targetFPS / 60.0);
	double translationWeight = 8 * (targetFPS / 60.0), yBuffer = 6.0;
	double minY;
	double dx,dy,dz;
	int xi, zi;
	Camera targetCamera;
	bool *keys = GetKeys();
	// moves camera around

	if (flight)
		translationWeight = 1 * (targetFPS / 60.0);
	
	if (guy.falling > -1) {
		
		minY = minYValue(guy.x, guy.z);
		dy = minY-guy.y;
		
		targetCamera.x = guy.x - sin(guy.ry * piOver180) * 0.1;
		targetCamera.z = guy.z - cos(guy.ry * piOver180) * 0.1;
		
		targetCamera.y = minY + 1 + ( 12 - 12*(1 / (dy/5.0+1)) );
		
	} else if (keys[SDLK_LEFT]) {
		targetCamera.x = guy.x + cos(guy.ry * piOver180) * 5;
		targetCamera.y = guy.y+1;
		targetCamera.z = guy.z - sin(guy.ry * piOver180) * 5;
	} else if (keys[SDLK_RIGHT]) {
		targetCamera.x = guy.x - cos(guy.ry * piOver180) * 5;
		targetCamera.y = guy.y+1;
		targetCamera.z = guy.z + sin(guy.ry * piOver180) * 5;
	} else if (keys[SDLK_UP]) {
		targetCamera.x = guy.x;
		targetCamera.y = guy.y+12;
		targetCamera.z = guy.z;
	} else if (keys[SDLK_DOWN]) {
		targetCamera.x = guy.x;
		targetCamera.y = guy.y+200;
		targetCamera.z = guy.z;
	} else {
		
		GetTerrainPiece(guy.x, guy.z, xi,zi);
		
		if (xi < 0 || zi < 0) {
			
			GetTerrainPiece(guy.x, guy.z, xi,zi);
			
		}
		targetCamera.y = guy.y+2;
		targetCamera.x = guy.x - sin(guy.ry * piOver180) * 5;
		targetCamera.z = guy.z - cos(guy.ry * piOver180) * 5;
		
		//targetCamera.y = guy.y+2 + terrain[xi][zi].normal.phi * 3;
		
	}
	
	debugCamera = targetCamera;
	
	if (initializeCamera) {
		initializeCamera = false;
		camera = targetCamera;
	} else {/*
		dx = (targetCamera.x - camera.x);
		dy = (targetCamera.y - camera.y);
		dz = (targetCamera.z - camera.z);
		d = sqrt(dx*dx+dy*dy+dz*dz);
		
		
		camera.x = camera.x + (dx/d)*(d/7);
		camera.y = camera.y + (dy/d)*(d/7);
		camera.z = camera.z + (dz/d)*(d/7);*/
		camera.x = (camera.x * translationWeight + targetCamera.x) / (translationWeight + 1.0);
		camera.y = (camera.y * (translationWeight * yBuffer) + targetCamera.y) / ((translationWeight * yBuffer) + 1.0);
		camera.z = (camera.z * translationWeight + targetCamera.z) / (translationWeight + 1.0);
	}
	
	// Makes camera always looking at Unicycle guy:
	dx = (camera.x - guy.x);
	dy = (camera.y - guy.y);
	dz = (camera.z - guy.z);
	
	if (dz > 0)
		targetCamera.ry = atan(dx/dz) * oneEightyOverPi;
	else if (dz == 0) {
		if (dx > 0)
			targetCamera.ry = 90;
		else
			targetCamera.ry = 270;
	} else
		targetCamera.ry = atan(dx/dz) * oneEightyOverPi + 180;
	
	targetCamera.rx = -atan(dy/sqrt( dx*dx + dz*dz)) * oneEightyOverPi + 5;
	
	if (fabs(camera.ry-targetCamera.ry) > fabs(camera.ry-(targetCamera.ry+360)))
		targetCamera.ry += 360;
	
	camera.rx = (camera.rx * rotationWeightX + targetCamera.rx) / (rotationWeightX + 1.0);
	camera.ry = (camera.ry * rotationWeightY + targetCamera.ry) / (rotationWeightY + 1.0);
	
	if (camera.ry > 360) camera.ry -= 360;
	if (camera.ry < 0) camera.ry += 360;
	
	// Keeps camera above the ground:
	minY = minYValue(camera.x, camera.z) + 0.2;
	if (camera.y < minY) camera.y = minY;
	
	
	// ooga booga!
	  /*
	camera.x = guy.x;
	camera.y = guy.y+180;
	camera.z = guy.z;
	
	camera.ry = camera.rz = 0;
	camera.rx = -90;
	
	glFogf(GL_FOG_START, 10000);// Fog Start Depth
	glFogf(GL_FOG_END, 10001);// Fog End Depth
	glDisable(GL_FOG);
	   */
}

void InitSpeeds()
{
	normalSpeed		= ConvertSpeed(10);
	minSpeed		= 0;//normalSpeed/3.0;// ConvertSpeed(0.0);//ConvertSpeed(0.5);
	
	airHoldTime		= 0 * (targetFPS/60.0);
	gravity			= ConvertAcceleration(-30);
	maxVSpeed		= ConvertSpeed(15);
	/*
	// moon gravity:
	gravity = ConvertAcceleration(-10);
	maxVSpeed = ConvertSpeed(10);
	airHoldTime = 9 * (targetFPS/60.0);
	*/
}


void InitGame()
{
	camera.x = camera.y = camera.z = camera.rx = camera.ry = camera.rx = 0;
	
	InitSpeeds();
	
	guy.x = guy.z = terrainLength/2.0;
	guy.y = 0;
	
	guy.vx = guy.vy = guy.vz = 0;
	guy.ax = guy.az = guy.rx = guy.ry = guy.rz = 0;
	guy.inAir = false;
	guy.ay = gravity;
	guy.speed = normalSpeed;
	
	guy.ry = 45;
	
	guy.dead = guy.reallyDead = 0;
	guy.falling = -1;
	
	if (flight) {
		guy.ry = 0;
		guy.rx = 90;
		guy.rz = 0;
	}
	
	initializeCamera = true;
	
	CreateUnicycleGuy();
	
	maxWalls = 500;
	
	walls = new BrickWall [maxWalls];
	for(int i = 0; i < maxWalls; ++i)
		walls[i].exists = false;
	
	currentWallNum = 0;
	wallsPerUnitArea = 0.001;
	
	minWallHeight = 1;
	maxWallHeight = 2.5;
	minWallLength = 20;
	maxWallLength = 35;
	minWallWidth = 0.7;
	maxWallWidth = 0.8;
	
	
	
	maxPits = 1000;
	
	pits = new Pit [maxPits];
	for(int i = 0; i < maxPits-1; ++i) {
		pits[i].exists = false;
		pits[i].next = &(pits[i+1]);
	}
	pits[maxPits-1].exists = false;
	pits[maxPits-1].next = NULL;
	
	lastPit = NULL;
	firstPit = NULL;

	currentPitNum = 0;
	pitsPerUnitArea = 0.001;
	minPitLength = 5.1*spacing;  // CANNOT BE LESS THAN 2.1!!!!
	maxPitLength = 12.0*spacing;
	minPitWidth  = 2.1*spacing;
	maxPitWidth  = 4.0*spacing;
	
	
	
	
	maxSmilies = 5000;
	smilies = new SmileyFace [maxSmilies];
	for(int i = 0; i < maxSmilies-1; ++i) {
		smilies[i].exists = false;
		smilies[i].next = &(smilies[i+1]);
	}
	smilies[maxSmilies-1].exists = false;
	smilies[maxSmilies-1].next = NULL;
	
	currentSmileyNum = 0;
	smiliesPerUnitArea = 0.02;
	bigSmileyRatio = 0.05;
	
	minSmileyHeight = 0.6;
	maxSmileyHeight = 5;
	smileyRotation = 0;
	smileyRotationSpeed = (360.0 / targetFPS) * 1.2;
}

void KillGame()
{
	if (walls)
		delete[] walls;
	if (pits)
		delete[] pits;
	if (smilies)
		delete[] smilies;
}

double ConvertSpeed(double pps)
{
	return pps / targetFPS;
}

double ConvertAcceleration(double ppsps)
{
	return ppsps / (targetFPS*targetFPS);
}

void DrawUnicycleGuyShadow()
{
	double s = 0.4, y, alpha, xPortion, zPortion;
	int x, z, nx, nz, sx, sz;
	double shadowPortion = 0.125, shadowSize = 0.75/spacing, guyTerX, guyTerZ;
	shadowPortion /= shadowSize;
	
	glColorMask(1,1,1,0);
	glPushMatrix();
	
	glTranslatef(0,0.01,0);
	
	guyTerX = guy.x/spacing;
	guyTerZ = guy.z/spacing;
	
	GetTerrainPiece(guy.x, guy.z, sx, sz);
	
	if (guy.x < 0)
		xPortion = 1 + (guyTerX - int(guyTerX));
	else
		xPortion = guyTerX - int(guyTerX);
	if (guy.z < 0)
		zPortion = 1 + (guyTerZ - int(guyTerZ));
	else
		zPortion = guyTerZ - int(guyTerZ);
	
	xPortion *= -1;
	zPortion *= -1;
	
	y = minYValue(guy.x, guy.z);
	alpha = 1.0 / (fabs(guy.y - y) + 1);
	
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, textures[kShadow].GetTextureID());
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glDisable(GL_LIGHTING);
	
	glColor4f(1.0,1.0,1.0, s * alpha);
	
	
	glBegin(GL_TRIANGLES);
		
		z = sz - 1;
		if (z < 0) z = terrainSize-1;
		for(int zi = -1; zi < 2; ++zi) {
			x = sx - 1;
			if (x < 0) x = terrainSize-1;
			nz = z + 1;
			if (nz > terrainSize-1) nz = 0;
			
			for(int xi = -1; xi < 2; ++xi) {
				
				nx = x + 1;
				if (nx > terrainSize-1) nx = 0;
				// (0,0)
				glTexCoord2f( (xPortion + xi) * shadowPortion, (zPortion + zi) * shadowPortion);
				glVertex3f(terrain[x][z].x * spacing, terrain[x][z].y * spacing, terrain[x][z].z * spacing);
				
				// (0,1)
				glTexCoord2f( (xPortion + xi) * shadowPortion, (zPortion + zi + 1) * shadowPortion);
				glVertex3f(terrain[x][nz].x * spacing, terrain[x][nz].y * spacing, terrain[x][nz].z * spacing);
				
				// (1,1)
				glTexCoord2f( (xPortion + xi + 1) * shadowPortion, (zPortion + zi + 1) * shadowPortion);
				glVertex3f(terrain[nx][nz].x * spacing, terrain[nx][nz].y * spacing, terrain[nx][nz].z * spacing);
				
				
				// (1,1)
				glTexCoord2f( (xPortion + xi + 1) * shadowPortion, (zPortion + zi + 1) * shadowPortion);
				glVertex3f(terrain[nx][nz].x * spacing, terrain[nx][nz].y * spacing, terrain[nx][nz].z * spacing);
				
				// (1,0)
				glTexCoord2f( (xPortion + xi + 1) * shadowPortion, (zPortion + zi) * shadowPortion);
				glVertex3f(terrain[nx][z].x * spacing, terrain[nx][z].y * spacing, terrain[nx][z].z * spacing);
				
				// (0,0)
				glTexCoord2f( (xPortion + xi) * shadowPortion, (zPortion + zi) * shadowPortion);
				glVertex3f(terrain[x][z].x * spacing, terrain[x][z].y * spacing, terrain[x][z].z * spacing);
				++x;
				if (x > terrainSize-1) x = 0;
			}
			++z;
			if (z > terrainSize-1) z = 0;
		}
	glEnd();
	glPopMatrix();
	glDisable(GL_BLEND);
	glColorMask(1,1,1,1);
}

void KillMurderKillStabityStabStabStabKill()
{
	if (guy.dead && guy.vy > 0) return;
	guy.dead = true;
	
	CreateParticles(20, kPStar,
						guy.x, guy.y + 1, guy.z, // center
						-20, 200, guy.ry+90, guy.ry+90,  // angle
						guy.vx, 0, guy.vz, // initial velocity
						0, 0, 0,   // accel
						10,  // rotation
						6, 6,  // min/max speed
						0.3, 0.3,  //min/max size
						0.4, 0.4); // life
	
	if (!guy.inAir) {
		guy.inAir = true;
		
		guy.ay = gravity;
		guy.vy = maxVSpeed;
		guy.airTime = airHoldTime;
		guy.vz = cos(guy.ry * piOver180) * guy.speed;
		guy.vx = sin(guy.ry* piOver180) * guy.speed;
		
	} else {
		guy.vx += RandomNumber(-ConvertSpeed(6),ConvertSpeed(6));
		guy.vz += RandomNumber(-ConvertSpeed(6),ConvertSpeed(6));
		if (guy.vy < 0)
			guy.vy *= -1;
		
		if (guy.vy > 0 && guy.vy < ConvertSpeed(2)) guy.vy = ConvertSpeed(2);
		if (guy.vy < 0 && guy.vy > ConvertSpeed(-2)) guy.vy = ConvertSpeed(-2);
		
	}
}

void GuyGoSplat()
{
	guy.reallyDead = true;
	
	CreateParticles(15, kPDirt,
						guy.x, guy.y, guy.z, // center
						65, 90, 0,360,  // angle
						0,0,0, // initial velocity
						0, -35, 0,   // accel
						10,  // rotation
						6, 7,  // min/max speed
						0.7, 0.9,  //min/max size
						0.8, 0.8); // life
}


