/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <math.h>
#import "Vector2.h"


void vector_add(Vector2 *v, Vector2 *w, Vector2 *dest) {
    dest->x = v->x + w->x;
    dest->y = v->y + w->y;
}

void vector_sub(Vector2 *v, Vector2 *w, Vector2 *dest) {
    dest->x = v->x - w->x;
    dest->y = v->y - w->y;
}

void vector_scale(float c, Vector2 *v, Vector2 *dest) {
    dest->x = v->x * c;
    dest->y = v->y * c;
}

float vector_dot(Vector2 *v, Vector2 *w) {
    return v->x*w->x + v->y*w->y;
}

float vector_length(Vector2 *v) {
    return sqrt(v->x*v->x + v->y*v->y);
}

void vector_normalize(Vector2 *v, Vector2 *dest) {
    float length = vector_length(v);
    dest->x = v->x / length;
    dest->y = v->y / length;
}

void vector_rotate(Vector2 *v, float angle, Vector2 *dest) {
    float c = cos(angle);
    float s = sin(angle);
    float x = v->x;
    float y = v->y;
    dest->x = c*x - s*y;
    dest->y = s*x + c*y;
}

float vector_angle(Vector2 *v) {
    if (v->x > 0) {
	return atan(v->y / v->x);
    } else if (v->x < 0) {
	return M_PI + atan(v->y / v->x);
    } else if (v->y > 0) {
	return 0.5*M_PI;
    } else {
	return -0.5*M_PI;
    }
}

void edge_rotate(Edge *e, float angle, Edge *dest) {
    vector_rotate(&e->v1, angle, &dest->v1);
    vector_rotate(&e->v2, angle, &dest->v2);
}

void edge_translate(Edge *e, Vector2 *delta, Edge *dest) {
    vector_add(&e->v1, delta, &dest->v1);
    vector_add(&e->v2, delta, &dest->v2);
}

void edge_normal(Edge *e, Vector2 *normal) {
    vector_sub(&e->v2, &e->v1, normal);
    vector_rotate(normal, 0.5*M_PI, normal);
    vector_normalize(normal, normal);
}

// p is the center of the circle
// radius is the radius of the circle
// x1 is the first endpoint of the line segment
// x2 is the second endpoint of the line segment
int circle_intersects_segment(Vector2 *p, float radius, Vector2 *x1, Vector2 *x2) {

    // First check circle for exclusion from the
    // line segment's bounding box.
    if (p->x - x1->x >= radius && p->x - x2->x >= radius) return 0;
    if (x1->x - p->x >= radius && x2->x - p->x >= radius) return 0;
    if (p->y - x1->y >= radius && p->y - x2->y >= radius) return 0;
    if (x1->y - p->y >= radius && x2->y - p->y >= radius) return 0;
    
    Vector2 dir;   // direction vector for line segment
    Vector2 r;     // vector from x1 to p.
    vector_sub(x2, x1, &dir);
    vector_sub(p, x1, &r);
    // The image of t gives us the closest point on the line to the circle.
    float t = vector_dot(&dir, &r) / vector_dot(&dir, &dir);

    Vector2 normal;
    if (t < 0) {
	if (vector_dot(&r, &r) >= radius*radius)
	    return 0;
	normal = r;
	return 1;
    } else if ( t > 1) {
	Vector2 temp;
	vector_sub(p, x2, &temp);
	if (vector_dot(&temp, &temp) >= radius*radius)
	    return 0;
	normal = temp;
	return 1;
    } else { // 0 <= t <= 1
	// Calculate perpendicular distance from line to p.
	Vector2 temp;
	vector_scale(t, &dir, &temp);
	vector_sub(&temp, &r, &temp);
	if (vector_dot(&temp, &temp) >= radius*radius) 
	    return 0;
	normal.x = dir.y;
	normal.y = -dir.x;
    }

    // If the circle does in fact intersect the line,
    // then we back away from the line in a direction
    // that is both perpendicular to the line and 
    // contrary to the direction of the vector r (this assumes
    // that the velocity is never large enough to catapult
    // the ship across the line in a single time step).
    vector_normalize(&normal, &normal);
    vector_scale(radius, &normal, &normal);
    vector_scale(t, &dir, p);
    vector_add(x1, p, p);
    if (vector_dot(&normal, &r) < 0) {
	vector_sub(p, &normal, p);
    } else {
	vector_add(p, &normal, p);
    }

    return 1;
}

// p is the center of the circle
// delta is the change in position of the center
// radius is the radius of the circle
// e is the edge (line segment) that we test against
int check_moving_circle_against_edge(Vector2 *p, Vector2 *delta, float radius, Edge *e) {

    Vector2 q;  // new location of circle
    vector_add(p, delta, &q);

    // First check circle for exclusion from the
    // line segment's bounding box.
    float x1 = e->v1.x, y1 = e->v1.y;
    float x2 = e->v2.x, y2 = e->v2.y;
    if (q.x - x1 >= radius && q.x - x2 >= radius) return 0;
    if (x1 - q.x >= radius && x2 - q.x >= radius) return 0;
    if (q.y - y1 >= radius && q.y - y2 >= radius) return 0;
    if (y1 - q.y >= radius && y2 - q.y >= radius) return 0;
    
    Vector2 dir;   // direction vector for line segment
    Vector2 r;     // vector from x1 to q.
    vector_sub(&e->v2, &e->v1, &dir);
    vector_sub(&q, &e->v1, &r);
    // The image of t gives us the closest point on the line to the circle.
    float t = vector_dot(&dir, &r) / vector_dot(&dir, &dir);

    Vector2 normal;
    if (t < 0) {
	if (vector_dot(&r, &r) >= radius*radius)
	    return 0;
	t = 0;
	normal = r;
	return 1;
    } else if ( t > 1) {
	Vector2 temp;
	vector_sub(&q, &e->v2, &temp);
	if (vector_dot(&temp, &temp) >= radius*radius)
	    return 0;
	t = 1;
	normal = temp;
	return 1;
    } else { // 0 <= t <= 1
	// Calculate perpendicular distance from line to q.
	Vector2 temp;
	vector_scale(t, &dir, &temp);
	vector_sub(&temp, &r, &temp);
	if (vector_dot(&temp, &temp) >= radius*radius)
	    return 0;
	normal.x = dir.y;
	normal.y = -dir.x;
    }

    // If the circle intersects the line, then we scale the passed
    // in delta vector so that the circle moves as far as it can
    // in the delta direction without intersection.
    vector_normalize(&normal, &normal);
    // Make sure that the normal is pointing in the right direction
    // (i.e. from the edge to the ship's old location).
    Vector2 r0; // vector from x1 to p.
    vector_sub(p, &e->v1, &r0);
    if (vector_dot(&r0, &normal) < 0) {
	vector_scale(-1.0, &normal, &normal);
    }

    Vector2 perp;  // a unit vector perpendicular to normal
    vector_normalize(&dir, &perp);
    
    // We find the components of delta in the
    // (normal, perp) coordinate frame.
    float normal_component = vector_dot(delta, &normal);
    float perp_component = vector_dot(delta, &perp);
    
    // Calculate perpendicular distance from line to p.
    Vector2 temp;
    vector_scale(t, &dir, &temp);
    vector_sub(&temp, &r0, &temp);
    float distance = vector_length(&temp);

    // We calculate the new normal component of delta.
    normal_component = -(distance - radius);
    
    // Then we convert delta from the (normal, perp) coordinate
    // system back into the regular (x, y) coordinate system.
    float x_component = normal_component*normal.x + perp_component*perp.x;
    float y_component = normal_component*normal.y + perp_component*perp.y;

    delta->x = x_component;
    delta->y = y_component;

    return 1;

}

// p is the original position of the point.
// delta is the change in position.
// edge is the edge we check against for intersection.
int check_moving_point_against_edge(Vector2 *p, Vector2 *delta, Edge *edge)
{
    Vector2 dir;  // dir is the direction vector of the edge from v1 to v2.
    vector_sub(&edge->v2, &edge->v1, &dir);
    Vector2 r;  // r is the vector pointing from v1 to p.
    vector_sub(p, &edge->v1, &r);

    // We parameterize two lines:
    //    l1(t) = p + t*delta
    //    l2(s) = v1 + s*dir
    // and then find their intersection by setting them equal to each
    // other and solving the system of equations obtained by looking
    // at each component separately.  Thus,
    
    float denom = dir.y*delta->x - dir.x*delta->y;
    
    // If the denominator is zero, then the two lines are parallel.
    // (note that the denominator is the z-component of the cross product).
    // Parallel lines do not intersect.
    if (denom == 0) return 0;
    
    float t = (dir.x*r.y - r.x*dir.y) / denom;
    if (t < 0 || t > 1) return 0;

    float s = (r.y*delta->x - r.x*delta->y) / denom;
    if (s < 0 || s > 1) return 0;

    // Otherwise the two line segments intersect.  We scale delta
    // by t so that the point moves right up to the edge.
    vector_scale(t, delta, delta);
    return 1;
}

int check_moving_point_against_circle(Vector2 *p, Vector2 *delta, Vector2 *center, float radius) {
    
    Vector2 q;  // q is new location of point
    vector_add(p, delta, &q);

    if (q.x - center->x >= radius || center->x - q.x >= radius) return 0;
    if (q.y - center->y >= radius || center->y - q.y >= radius) return 0;

    Vector2 d;  // d is vector from center of circle to q
    vector_sub(center, &q, &d);
    float dd = vector_dot(&d,&d);  // dd is d dot d = |d|^2
    if (dd >= radius*radius) return 0;

    float distance = sqrt(dd); // distance is the length of d
    vector_scale( (distance - radius) / distance, delta, delta);

    return 1;
}

// p is the vertex of the ray.
// delta is the vector direction of the ray.
// e is the edge we're testing against.
int ray_intersects_edge(Vector2 *p, Vector2* delta, Edge *edge, float *collisionTime) {
    Vector2 dir;  // dir is the direction vector of the edge from v1 to v2.
    vector_sub(&edge->v2, &edge->v1, &dir);
    Vector2 r;  // r is the vector pointing from v1 to p.
    vector_sub(p, &edge->v1, &r);

    // We parameterize two lines:
    //    l1(t) = p + t*delta  <-- is the ray
    //    l2(s) = v1 + s*dir   <-- is the edge line segment
    // and then find their intersection by setting them equal to each
    // other and solving the system of equations obtained by looking
    // at each component separately.  Thus,
    
    float denom = dir.y*delta->x - dir.x*delta->y;
    
    // If the denominator is zero, then the two lines are parallel.
    // (note that the denominator is the z-component of the cross product).
    // Parallel lines do not intersect.
    if (denom == 0) return 0;
    
    float s = (r.y*delta->x - r.x*delta->y) / denom;
    if (s < 0 || s > 1) return 0;

    float t = (dir.x*r.y - r.x*dir.y) / denom;
    if (t < 0) return 0;

    // Otherwise, the ray will intersect the edge.
    *collisionTime = t;
    return 1;    
}
