/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <Cocoa/Cocoa.h>
#import "Utility.h"

@interface NamedTexture : NSObject
{
    @public
    GLuint texture;    // texture reference
    IntegerRect rect;  // dimensions of texture
    int refCount;      // number of times used
}
@end
@implementation NamedTexture
@end

static NSMutableDictionary *textureDictionary = nil;

void deleteLoadedTextures()
{
    if (textureDictionary == nil) return;
    NSEnumerator *enumerator = [textureDictionary objectEnumerator];
    NamedTexture *t;
    while ( (t = [enumerator nextObject]) ) {
	glDeleteTextures( 1, &(t->texture) );
    }
    [textureDictionary release];
    textureDictionary = nil;
}

GLuint loadTextureNamed(NSString *name, IntegerRect *rect)
{
    if (textureDictionary == nil) {
	textureDictionary = [[NSMutableDictionary alloc] initWithCapacity:10];
    }

    NamedTexture *t;

    t = [textureDictionary objectForKey:name];
    if (t != nil) {
	t->refCount++;
	if (rect) {
	    rect->width = t->rect.width;
	    rect->height = t->rect.height;
	}
	return t->texture;
    }

    GLenum format;
    GLuint texture;

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glGenTextures(1, &texture);

    NSImage *image = [NSImage imageNamed:name];
    if (!image) [NSException raise:@"InvalidNameException" 
			     format:@"Could not load image %@", name];
    NSBitmapImageRep *bitmap = [[NSBitmapImageRep alloc] initWithData: [image TIFFRepresentation]];
    [image release];  // don't need image anymore
    if (!bitmap) [NSException raise:@"InvalidNameException" 
			      format:@"Could not load bitmap"];

    glBindTexture(GL_TEXTURE_2D, texture);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    // Determine if the pixel data has an alpha component.
    if ([bitmap samplesPerPixel] == 3) format = GL_RGB;
    else if ([bitmap samplesPerPixel] == 4) format = GL_RGBA;
    else NSCAssert(0, @"loadTextureNamed: unknown image format");

    glTexImage2D(
	GL_TEXTURE_2D,         // target texture
	0,                     // mipmap level 
	format,                // internal format
	[bitmap pixelsWide],   // width
	[bitmap pixelsHigh],   // height
	0,                     // border    
	format,                // format of pixel data
	GL_UNSIGNED_BYTE,      // type of pixel data
	[bitmap bitmapData]    // pixel data
	);

    // create and store texture info for future calls
    t = [[NamedTexture alloc] init];
    t->texture = texture;
    t->rect.width = [bitmap pixelsWide];
    t->rect.height = [bitmap pixelsHigh];

    if (rect) {
	rect->width = t->rect.width;
	rect->height = t->rect.height;
    }

    t->refCount = 1;
    [textureDictionary setObject:t forKey:name];
    [t release];

    [bitmap release];

    return texture;
}


void drawStringAtPoint(NSString *string, NSPoint point, BOOL center)
{
    IntegerRect rect;
    GLuint texture = loadTextureNamed(@"courier_font_small", &rect);
    glBindTexture(GL_TEXTURE_2D, texture);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    if (center) { // this only makes sense if the string is a single line
	rect.x = point.x - [string length] * 7.5;
	rect.y = point.y - 12;
    } else {
	rect.x = point.x;
	rect.y = point.y;
    }
    rect.width = 16;
    rect.height = 32;
    int length = [string length];
    int i;
    for (i = 0; i < length; i++) {
	unichar c = [string characterAtIndex:i];
	int fontIndex = 0;
	const GLfloat wx = 1.0/16.0;
	const GLfloat wy = 1.0/8.0;

	if ('!' <= c && c <= '~') fontIndex = c;
	else if (c == '\n') { rect.x = point.x; rect.y -= 24; continue; }
	else if (c == ' ') { rect.x += 14; continue; }
	else { fontIndex = 127; }

	GLfloat x = (float)(fontIndex % 16)/16.0;
	GLfloat y = (float)(fontIndex / 16)/8.0;

	glBegin(GL_QUADS);
	glTexCoord2f(x,y+wy); glVertex2i(rect.x, rect.y);
	glTexCoord2f(x+wx,y+wy); glVertex2i(rect.x + rect.width, rect.y);
	glTexCoord2f(x+wx,y); glVertex2i(rect.x + rect.width, rect.y + rect.height);
	glTexCoord2f(x,y); glVertex2i(rect.x, rect.y + rect.height);
	glEnd();
	rect.x += 15;
    }

}

extern void drawLEDStringAtPoint(NSString *string, NSPoint point, float size, int position)
{
    int i, length = [string length];

    const float w = size;
    const float l = 2*size;
    const float sq2 = sqrt(2);
    const float x0 = 0, x1 = w, x2 = w+l, x3 = w+l+w;
    const float y0 = 0, y1 = w, y2 = w+l, y3 = w+l+w, y4 = w+l+w+l, y5 = w+l+w+l+w;
    const GLfloat vertices[24][2] = { {x1,y0}, {x2,y0}, {x2,y1}, {x3,y1}, 
				      {x3,y2}, {x2,y2}, {x2,y3}, {x3,y3},
				      {x3,y4}, {x2,y4}, {x2,y5}, {x1,y5},
				      {x1,y4}, {x0,y4}, {x0,y3}, {x1,y3},
				      {x1,y2}, {x0,y2}, {x0,y1}, {x1,y1},
				      {w*sq2/4+w, w*sq2/4+w+l},    // Last 4 vertices are for
				      {-w*sq2/4+w, -w*sq2/4+w+l},  // the bottom diagonal
				      {-w*sq2/4+w+l, -w*sq2/4+w},  // segment
				      {w*sq2/4+w+l, w*sq2/4+w} };

    const GLubyte segments[8][4] = { {9, 10, 11, 12}, {6, 7, 8, 9}, {2, 3, 4, 5}, {0, 1, 2, 19}, 
				     {16, 17, 18, 19}, {12, 13, 14, 15}, {5, 6, 15, 16}, 
				     {20, 21, 22, 23} };
    const char active[36][8] = { { 0, 1, 2, 3, 4, 5,-1,-1 },   // zero
				 { 1, 2,-1,-1,-1,-1,-1,-1 },   // one
				 { 0, 1, 3, 4, 6,-1,-1,-1 },   // two
				 { 0, 1, 2, 3, 6,-1,-1,-1 },   // three
				 { 1, 2, 5, 6,-1,-1,-1,-1 },   // four
				 { 0, 2, 3, 5, 6,-1,-1,-1 },   // five
				 { 0, 2, 3, 4, 5, 6,-1,-1 },   // six
				 { 0, 1, 2,-1,-1,-1,-1,-1 },   // seven
				 { 0, 1, 2, 3, 4, 5, 6,-1 },   // eight
				 { 0, 1, 2, 3, 5, 6,-1,-1 },   // nine
				 { 0, 1, 2, 4, 5, 6,-1,-1 },   // A
				 { 2, 3, 4, 5, 6,-1,-1,-1 },   // b
				 { 0, 3, 4, 5,-1,-1,-1,-1 },   // C
				 { 1, 2, 3, 4, 6,-1,-1,-1 },   // d
				 { 0, 3, 4, 5, 6,-1,-1,-1 },   // E
				 { 0, 4, 5, 6,-1,-1,-1,-1 },   // F
				 { 0, 1, 2, 3, 5, 6,-1,-1 },   // g
				 { 1, 2, 4, 5, 6,-1,-1,-1 },   // H
				 { 4, 5,-1,-1,-1,-1,-1,-1 },   // I
				 { 1, 2, 3, 4,-1,-1,-1,-1 },   // J
				 { 4, 5, 6, 7,-1,-1,-1,-1 },   // k
				 { 3, 4, 5,-1,-1,-1,-1,-1 },   // L
				 { 0, 1, 2, 4, 5,-1,-1,-1 },   // M
				 { 2, 4, 6,-1,-1,-1,-1,-1 },   // n
				 { 2, 3, 4, 6,-1,-1,-1,-1 },   // o
				 { 0, 1, 4, 5, 6,-1,-1,-1 },   // P
				 { 0, 1, 2, 5, 6,-1,-1,-1 },   // q
				 { 0, 1, 4, 5, 6, 7,-1,-1 },   // R
				 { 0, 5, 6, 2, 3,-1,-1,-1 },   // S
				 { 0, 4, 5,-1,-1,-1,-1,-1 },   // T
				 { 2, 3, 4,-1,-1,-1,-1,-1 },   // u
				 { 1, 2, 5, 7,-1,-1,-1,-1 },   // V
				 { 1, 2, 3, 4, 5,-1,-1,-1 },   // W
				 { 1, 2, 3, 5, 6,-1,-1,-1 },   // y
				 { 0, 1, 3, 4, 6,-1,-1,-1 } }; // Z
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(2, GL_FLOAT, 0, vertices);

    glPushMatrix();
    if (position == 0) { // bottom left justified
	glTranslatef(point.x, point.y, 0);
    } else if (position == 1) { // bottom right justified
	glTranslatef(point.x - (3*w+l)*length, point.y, 0);
    } else if (position == 2) { // top left justified
	glTranslatef(point.x, point.y - (3*w+2*l), 0);
    } else if (position == 3) { // top right justified
	glTranslatef(point.x - (3*w+l)*length, point.y - (3*w+2*l), 0);
    } else { // centered
	glTranslatef(point.x - 0.5*(3*w+l)*length, point.y - 0.5*(3*w+2*l), 0);
    }

    for (i = 0; i < length; i++) {
	int j, index;
	char ch = [string characterAtIndex:i];
	if ('0' <= ch && ch <= '9') index = ch - '0';
	else if ('a' <= ch && ch <= 'z') index = 10 + ch - 'a';
	else if ('A' <= ch && ch <= 'Z') index = 10 + ch - 'A';
	else {
	    glTranslatef(3*w+l, 0, 0);
	    continue;
	}
	for (j = 0; active[index][j] != -1; j++) {
	    glDrawRangeElements(GL_LINE_LOOP, 0, 19, 4, GL_UNSIGNED_BYTE, segments[active[index][j]]);
	}
	glTranslatef(3*w+l, 0, 0);
    }

    glDisableClientState(GL_VERTEX_ARRAY);
    glPopMatrix();
}
