/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import "HexMaze.h"
#import "Utility.h"  // for randomFloatBetween()


@implementation HexMazeCoord
+ (HexMazeCoord *)coord:(int)initX :(int)initY
{
    HexMazeCoord *coord = [[HexMazeCoord alloc] init];
    coord->x = initX;
    coord->y = initY;
    return [coord autorelease];
}
@end

@implementation HexMaze
- (id)initWithSize:(NSSize)size
{
    self = [super init];
    _width = size.width;
    _height = size.height;
    _size = _width * _height;
    _data = (HexMazeCell *)malloc(_size * sizeof(HexMazeCell));
    return self;
}

- (void)dealloc
{
    free(_data);
    [super dealloc];
}

// This starts us off with a maze in which no cell has been
// visited and none of the exits of any cell are open.
- (void)clearMaze
{
    int i, j;
    for (i = 0; i < _size; i++) {
	_data[i].condemned = NO;
	_data[i].visited = NO;
	for (j = 0; j < 6; j++) {
	    _data[i].exits[j] = NO;
	}
    }
}

- (void)createRandomMaze
{
    // path keeps track of which cells we have visited
    // and in what order we visited them.
    NSMutableArray *path = [NSMutableArray arrayWithCapacity:10];

    [self clearMaze];

    // Start from a random cell location.
    [path addObject:[HexMazeCoord coord:random()%_width : random()%_height]];

    // This is the list of possible exits from the cell.
    int exitIndices[] = { 0, 1, 2, 3, 4, 5 };
    // oppExit is used to mark the corresponding exit of
    // the adjacent cell as being open.  In the adjacent
    // cell, this exit is on the opposite side of the one
    // in the current cell.
    int oppExit[] = { 3, 4, 5, 0, 1, 2 };
    
    // Usually, the last cell in the path is one that we
    // just added at the end of the previous loop and haven't
    // actually visited yet.  However, whenever we reach a dead-end
    // or carve back into a piece of the maze that we've already
    // explored, then the last cell will be one that we've visited
    // recently and we will begin backtracking to find a new cell
    // to tunnel into unexplored area from.
    while ([path count]) {
	HexMazeCoord *coord = [path lastObject];
	HexMazeCell *cell = [self cellAtCoord:coord];
	if (cell->visited) {
	    // Remove visited cells while backtracking.
	    [path removeLastObject];
	} else {
	    // Mark new cells as visited.
	    cell->visited = YES;
	}

	// We want to randomize the order in which we search for
	// exits to take from a cell.  So we keep a list of the 
	// exit indices which we permute a little more each iteration.
	int count = 20;
	while (count--) {
	    int i = random()%6, j = random()%6;
	    int temp = exitIndices[i];
	    exitIndices[i] = exitIndices[j];
	    exitIndices[j] = temp;
	}

	// Create a list of neighboring coordinates.  The relative
	// neighboring coordinates depend on whether or not the
	// x-coordinate of the current cell is odd or even.
	int x = coord->x, y = coord->y;
	int odd = x % 2;
	int even = 1 - odd;
	HexMazeCoord *neighbor[] = { [HexMazeCoord coord:x-1:y+even],
				     [HexMazeCoord coord:x:y+1],
				     [HexMazeCoord coord:x+1:y+even],
				     [HexMazeCoord coord:x+1:y-odd],
				     [HexMazeCoord coord:x:y-1],
				     [HexMazeCoord coord:x-1:y-odd] };

	// Find an exit out of the cell.  If there isn't one that we
	// can take, then we will start backtracking to previous cells.
	for (count = 0; count < 6; count++) {
	    int i = exitIndices[count];
	    if (cell->exits[i] == NO) {
		HexMazeCell *adjCell = [self cellAtCoord:neighbor[i]];
		if (adjCell && adjCell->condemned == NO) {
		    if (adjCell->visited == NO) {
			cell->exits[i] = YES;
			adjCell->exits[ oppExit[i] ] = YES;
			// Start from this cell in the next
			// iteration of the while loop.
			[path addObject:neighbor[i]];
			break;
		    }
		    // We will randomly break through into already
		    // explored areas so that we can create loops.
		    // However after we do this, we have to start
		    // backtracking, so we don't add the adjacent
		    // cell to the path.
		    else if (random()%20 == 0) {
			cell->exits[i] = YES;
			adjCell->exits[ oppExit[i] ] = YES;
		    }
		}
	    }
	}
    }
}

- (HexMazeCell *)cellAtCoord:(HexMazeCoord *)coord
{
    int x = coord->x, y = coord->y;
    if (x < 0 || x >= _width || y < 0 || y >= _height)
	return nil;
    return &(_data[x + y*_width]);
}

// This creates a list of edges which define the
// maze, so that the game controller can draw and
// do collision detection with the maze walls.  The
// caller passes in a reference to an Edge * variable
// and is is later responsible for freeing up the memory
// allocated for the edges variable.  The number of
// edges stored in the edges variable is returned.
// The cellRadius arguments tells us how big the 
// constructed hexagonal cells will be.
- (int)edgeList:(Edge **)edgeArray withCellRadius:(const float)cellRadius
{
    const float hexHeight = 1.73205080757*cellRadius;  // = cellRadius*sqrt(3)
    const float yOffset = -(float)(_height/2)*hexHeight;
    const float xOffset = -(float)(_width/2)*1.5*cellRadius;
    const float wallJitter = 30;  // used to break up monotony of walls
    
    // We precalculate the vertices of a reference hexagon.
    const Vector2 vertices[] = { { -cellRadius, 0 },
				 { cellRadius*cos(2*M_PI/3), cellRadius*sin(2*M_PI/3) },
				 { cellRadius*cos(M_PI/3), cellRadius*sin(M_PI/3) },
				 { cellRadius, 0 },
				 { cellRadius*cos(-M_PI/3), cellRadius*sin(-M_PI/3) },
				 { cellRadius*cos(-2*M_PI/3), cellRadius*sin(-2*M_PI/3) },
				 { -cellRadius, 0 } };
    
    if (*edgeArray) free(*edgeArray);
    *edgeArray = NULL;
    int count = 0;  // current capacity of edgeArray
    int index = 0;  // index into edgeArray of current edge

    int x, y, i;
    for (y = 0; y < _height; y++) {
	float yCenter = y*hexHeight + yOffset;
	for (x = 0; x < _width; x++) {
	    int odd = x % 2;
	    float xCenter = 1.5*x*cellRadius + xOffset;
	    HexMazeCell *cell = &(_data[x + y*_width]);
	    for (i = 0; i < 6; i++) {
		if (cell->exits[i] == NO) {
		    // We only add the top three sides unless the cell
		    // happens to lie on a border of the maze.
		    if (i == 3 && x != _width-1 && !(y == 0 && odd == 1)) continue;
		    if (i == 4 && y != 0) continue;
		    if (i == 5 && x != 0 && !(y == 0 && odd == 1)) continue;

		    // Whenever the edgeArray runs out of space for the next
		    // two edges, we make edgeArray larger by using realloc.
		    if (index + 2 >= count) {
			count += 1024;
			*edgeArray = (Edge *)realloc(*edgeArray, count*sizeof(Edge));
		    }
		    Edge *edge1 = (*edgeArray) + index;
		    index += 2;

		    edge1->v1.x = vertices[i].x + xCenter;
		    edge1->v1.y = vertices[i].y + yCenter - odd*hexHeight/2.0;
		    edge1->v2.x = 0.5*(vertices[i].x + vertices[i+1].x) + xCenter + randomFloatBetween(-wallJitter, wallJitter);
		    edge1->v2.y = 0.5*(vertices[i].y + vertices[i+1].y) + yCenter - odd*hexHeight/2.0 + randomFloatBetween(-wallJitter, wallJitter);

		    Edge *edge2 = edge1 + 1;
		    edge2->v1.x = edge1->v2.x;
		    edge2->v1.y = edge1->v2.y;
		    edge2->v2.x = vertices[i+1].x + xCenter;
		    edge2->v2.y = vertices[i+1].y + yCenter - odd*hexHeight/2.0;
		    

		}
	    }
	}
    }

    return index;
}

@end
