/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <OpenGL/gl.h>
#import <OpenGL/glu.h>
#import "GameView.h"
#import "Utility.h"
#import "GameController.h"

@implementation GameView

+ (id)sharedWindow
{
    static NSWindow *sharedWindow = nil;

    if (!sharedWindow) {
	sharedWindow = [[NSWindow alloc] initWithContentRect:NSZeroRect
					 styleMask:NSTitledWindowMask 
					 | NSClosableWindowMask
					 | NSMiniaturizableWindowMask
					 // For some reason, if we try to use 
					 // NSBackingStoreRetained here, the corners
					 // of the window will not be drawn correctly.
					 backing:NSBackingStoreBuffered
					 defer:NO];
    }

    return sharedWindow;
}

- (id)initWithFrame:(NSRect)rect controller:(GameController *)controller
{
    [super initWithFrame:rect];
    game = controller;
    fullScreen = NO;

    // Register for the applicationDidFinishLaunching notification
    // so that we can start up the custom run loop.
    [[NSNotificationCenter defaultCenter] 
	addObserver:self
	selector:@selector(applicationDidFinishLaunching:)
	name:NSApplicationDidFinishLaunchingNotification
	object:NSApp];

    // Register for hide/unhide notifications so that we
    // can suspend the game execution when hidden.
    [[NSNotificationCenter defaultCenter] 
	addObserver:self
	selector:@selector(applicationDidHide:)
	name:NSApplicationDidHideNotification
	object:NSApp];
    [[NSNotificationCenter defaultCenter] 
	addObserver:self
	selector:@selector(applicationDidUnhide:)
	name:NSApplicationDidUnhideNotification
	object:NSApp];

    return self;
}

- (BOOL)fullScreen { return fullScreen; }
- (BOOL)showFPS { return showFPS; }
- (void)setShowFPS:(BOOL)flag { showFPS = flag; }

- (void)dealloc
{
    [openGLContext release];
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [super dealloc];
}

- (void)createNewOpenGLContext {
    [self createNewOpenGLContext:fullScreen];
}

- (void)createNewOpenGLContext:(BOOL)fullScreenFlag
{
    NSOpenGLContext *oldContext = openGLContext;

    // Since we don't share contexts, any texture objects loaded
    // in the old context will be invalid in the new context.
    // Therefore we have to make sure that they all get reloaded.
    deleteLoadedTextures();

    if (oldContext && fullScreenFlag == NO && fullScreen == YES) {
	// We need to revert back to the original display mode.
	[openGLContext clearDrawable];
	CGDisplaySwitchToMode(kCGDirectMainDisplay, originalDisplayMode);
	CGDisplayRelease(kCGDirectMainDisplay);
    }

    if (fullScreenFlag == YES && fullScreen == NO) {
	// If the window is miniaturized when we change to full
	// screen, then we should deminiaturize it first so that 
	// the game will be in the correct state.
	if (_window) [_window deminiaturize:self];

	// Remember the original display mode.
	originalDisplayMode = CGDisplayCurrentMode(kCGDirectMainDisplay);
    
	CGDisplayCapture(kCGDirectMainDisplay);

	// Switch resolutions.
	CGDisplayErr err;
	CFDictionaryRef mode;

	// Get the best display mode with the parameters we want.
	mode = CGDisplayBestModeForParameters(kCGDirectMainDisplay, 24, 640, 480, 0);
	
	NSAssert(mode, @"couldn't find display mode");
    
	// Switch to selected resolution.
	err = CGDisplaySwitchToMode(kCGDirectMainDisplay, mode);

	NSAssert(err == CGDisplayNoErr, @"error switching resolution");

	// Remove any previous duplicate notifications
	[[NSNotificationCenter defaultCenter] 
	    removeObserver:self
	    name:NSApplicationWillTerminateNotification
	    object:NSApp];
	// Register for the applicationWillTerminate notification
	// so that we can reset the display mode before quitting.
	[[NSNotificationCenter defaultCenter] 
	    addObserver:self
	    selector:@selector(applicationWillTerminate:)
	    name:NSApplicationWillTerminateNotification
	    object:NSApp];
    }

    fullScreen = fullScreenFlag;

    screenWidth = CGDisplayPixelsWide(kCGDirectMainDisplay);
    screenHeight = CGDisplayPixelsHigh(kCGDirectMainDisplay);

    #define ADD_ATTR(attr) attrs[attrCount++] = attr;
    NSOpenGLPixelFormatAttribute attrs[32];
    int attrCount = 0;

    if (fullScreen) ADD_ATTR(NSOpenGLPFAFullScreen);
    ADD_ATTR(NSOpenGLPFADoubleBuffer);
    ADD_ATTR(NSOpenGLPFADepthSize);
    ADD_ATTR(16);
    ADD_ATTR(NSOpenGLPFAScreenMask);
    ADD_ATTR(CGDisplayIDToOpenGLDisplayMask(kCGDirectMainDisplay));
    ADD_ATTR(0);
    
    NSOpenGLPixelFormat *pixelFormat;
    pixelFormat = [[NSOpenGLPixelFormat alloc] initWithAttributes:attrs];
    if (pixelFormat == nil) [NSApp terminate:self];
    openGLContext = [[NSOpenGLContext alloc] initWithFormat:pixelFormat shareContext:nil];
    if (openGLContext == nil) [NSApp terminate:self];
    [pixelFormat release];

    if (fullScreen) {
	[openGLContext setFullScreen];
	[NSCursor hide];
    } else {
	if (!_window) {
	    NSRect rect = NSMakeRect(screenWidth/2-320, screenHeight/2-240, 640, 480);
	    _window = [GameView sharedWindow];
	    [_window setFrame:[NSWindow frameRectForContentRect:rect 
					styleMask:NSTitledWindowMask 
					| NSClosableWindowMask
					| NSMiniaturizableWindowMask] 
		     display:YES];
	    [_window setContentView:self];
	    [_window setDelegate:self];
	}
	if (oldContext) [NSOpenGLContext clearCurrentContext];
	[oldContext release];

	[openGLContext setView:self]; // has to happen after view is attached to window.
	[NSCursor unhide];
    }
    
    // Swap buffers only during vertical retrace.
    long swapInterval[] = { 1 };
    [openGLContext setValues:swapInterval forParameter:NSOpenGLCPSwapInterval];

    [openGLContext makeCurrentContext];

}

- (void)applicationDidFinishLaunching:(NSNotification *)notification
{
    if (!fullScreen) [_window makeKeyAndOrderFront:nil];

    [self startCustomRunLoop];
}

- (void)applicationWillTerminate:(NSNotification *)notification
{
    [NSOpenGLContext clearCurrentContext];
    if (fullScreen) {
	[openGLContext clearDrawable];
	CGDisplaySwitchToMode(kCGDirectMainDisplay, originalDisplayMode);
	CGDisplayRelease(kCGDirectMainDisplay);
    }
}

- (void)applicationDidHide:(NSNotification *)notification {
    if (fullScreen) [game toggleFullScreen:self];
    suspended = YES; 
}
- (void)applicationDidUnhide:(NSNotification *)notification { 
    suspended = NO;
    [game tabulaRasa];
    [self startCustomRunLoop];
}
- (void)windowDidMiniaturize:(NSNotification *)notification { 
    if (fullScreen) [game toggleFullScreen:self];
    suspended = YES; 
}
- (void)windowDidDeminiaturize:(NSNotification *)notification { 
    suspended = NO; 
    [game tabulaRasa];
    [self startCustomRunLoop];
}

- (void)windowWillMiniaturize:(NSNotification *)notification {
    // This is needed so that the mini window image isn't blank.
    [self copyOpenGLToQuartz];
}

- (void)windowWillClose:(NSNotification *)notification
{
    [NSApp terminate:self];
}

- (void)startCustomRunLoop
{
    SEL theSelector = @selector(customRunLoop);
    NSMethodSignature *signature = [self methodSignatureForSelector:theSelector];
    NSInvocation *invocation = [NSInvocation invocationWithMethodSignature:signature];
    [invocation setSelector:theSelector];
    [invocation setTarget:self];
    [NSTimer scheduledTimerWithTimeInterval:0.0
	     invocation:invocation
	     repeats:NO];
}

// We roll our own run loop because it is currently the only
// way that I know of to ensure that we get all of the key events.
// If we try to subclass NSResponder's keyDown: and keyUp: methods,
// we will not receive certain keyUp events when a modifier
// key is being held down.
- (void)customRunLoop
{
    NSEvent *event;
    while (!suspended) {
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

	[game run];
	// Draw the view.
	[self drawRect:NSZeroRect];

	event = [NSApp nextEventMatchingMask:NSAnyEventMask
		       untilDate:[NSDate distantPast]
		       inMode:NSEventTrackingRunLoopMode
		       dequeue:YES];

	// We observe all of the events that come through before
	// transformations like key-equivalents and mnemonics 
	// take place.  We pass the events that we're interested 
	// in on to the game controller instance.
	if (event) {
	    switch ([event type]) {
	    case NSKeyDown:      [game keyDown:event];      break;
	    case NSKeyUp:        [game keyUp:event];        break;
	    case NSFlagsChanged: [game flagsChanged:event]; break;
	    default: break;
	    }
	} 

	// Then we send the event to where ever it's supposed to go.
	[NSApp sendEvent:event];

	[pool release];
    }
}

- (void)drawRect:(NSRect)rect
{
#define FRAMES_PER_COUNT 5
    static int frameCounter = FRAMES_PER_COUNT - 1;
    static UInt64 time;
    static float fps = 0;

    [game draw];

    if (showFPS) {
	frameCounter++;
	if (frameCounter == FRAMES_PER_COUNT) {
	    frameCounter = 0;
	    UnsignedWide currTime = {0,0};
	    Microseconds(&currTime);
	    float deltaTime = (float)(UnsignedWideToUInt64(currTime) - time);
	    deltaTime /= 1000000.0;  // convert deltaTime to seconds.
	    time = UnsignedWideToUInt64(currTime);  // reset for next interval.
	    
	    fps = FRAMES_PER_COUNT/deltaTime;
	}
        glMatrixMode(GL_PROJECTION);
	glPushMatrix();
        glLoadIdentity();
        gluOrtho2D(0, 640, 0, 480);
	glMatrixMode(GL_MODELVIEW);

	glPushMatrix();
	glLoadIdentity();

	glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glPolygonMode(GL_FRONT, GL_FILL);

	glColor3ub(0xff,0xff,0xff);
	drawStringAtPoint([NSString stringWithFormat:@"%.1f fps", fps], NSMakePoint(0,0), NO);

	glPopAttrib();
	glPopMatrix();

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
    }

    [openGLContext flushBuffer];
}

- (BOOL)isFlipped { return YES; }
- (BOOL)acceptsFirstResponder { return YES; }
- (void)keyDown:(NSEvent *)event 
{ 
    // Capture the keyDown events to avoid annoying NSBeeps.
}

- (NSPoint)mouseLocation
{
    if (fullScreen) return [NSEvent mouseLocation];
    return [[NSApp mainWindow] mouseLocationOutsideOfEventStream];
}

- (void)copyOpenGLToQuartz
{
    NSSize size = [self frame].size;
    NSBitmapImageRep *glPixels;
    glPixels = [[NSBitmapImageRep alloc] 
		   initWithBitmapDataPlanes:nil
		   pixelsWide:size.width
		   pixelsHigh:size.height
		   bitsPerSample:8
		   samplesPerPixel:3
		   hasAlpha:NO
		   isPlanar:NO
		   colorSpaceName:NSDeviceRGBColorSpace
		   bytesPerRow:0
		   bitsPerPixel:32];
				
    [openGLContext makeCurrentContext];
    [openGLContext flushBuffer];
    
    glPixelStorei(GL_PACK_ALIGNMENT, 4);
    glReadPixels(0, 0, 
		 size.width, size.height, 
		 GL_RGBA, 
		 GL_UNSIGNED_BYTE, 
		 [glPixels bitmapData]);

    [self lockFocus];
    [glPixels drawInRect:[self frame]];
    [glPixels release];
    [self unlockFocus];
    [[self window] flushWindow];
}
					 
@end
