/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <OpenGL/gl.h>
#import "GameUnit.h"


@implementation GameUnit
- (id)initInMap:(GameMap *)map
{
    self = [super init];
    _map = [map retain];
    _omega = 180;
    _maxHitPoints = 100;
    _hitPoints = _maxHitPoints;
    return self;
}

- (void)dealloc {
    [_map release];
    [super dealloc];
}

- (void)setMap:(GameMap *)map
{
    [map retain];
    [_map release];
    _map = map;
}

- (void)setPosition:(float)x :(float)y
{
    _x.x = x;
    _x.y = y;
}

- (void)setOrientation:(float)angle
{
    _angle = angle;
}

- (void)setBoundingRadius:(float)radius
{
    _boundingRadius = radius;
}

- (float)x { return _x.x; }
- (float)y { return _x.y; }
- (float)angle { return _angle; }
- (Vector2 *)position { return &_x; }
- (float)boundingRadius { return _boundingRadius; }
- (BOOL)isDead { return _isDead; }
- (BOOL)canBeHitByBullets { return NO; }

- (BOOL)moveBy:(float)dx :(float)dy
{
    Vector2 delta = { dx, dy };
    _hitSomething = NO;

    // Check for intersection with walls of map.
    int i, numWalls = [_map numberOfWalls];
    Edge *walls = [_map walls];
    for (i = 0; i < numWalls; i++) {
	// If there is a collision, this function will change delta
	// such that the collision is avoided.
	if (check_moving_circle_against_edge(&_x, &delta, _boundingRadius, &walls[i])) {
	    [_map setWallFlagWithIndex:i toValue:YES];
	    _hitSomething = YES;
	}
    }

    vector_add(&_x, &delta, &_x);

    return _hitSomething;
}

- (void)rotateBy:(float)dtheta
{
    _angle += dtheta;
}

- (void)run:(float)deltaTime
{
    if (_hitPoints <= 0) {
	_isDead = YES;
	[self explode];
	return;
    }

    if (_displayHPMeter) {
	_displayTimer -= deltaTime;
	if (_displayTimer <= 0) {
	    _displayHPMeter = NO;
	}
    }

    float dx = _v.x * deltaTime;
    float dy = _v.y * deltaTime;

    if (dx != 0 || dy != 0) {
	[self moveBy:dx :dy];
    }
}

- (void)draw
{
    /* do nothing for the virtual class */
}

- (void)explode
{
    /* do nothing for the virtual class */
}

- (void)takeDamage:(int)damage
{
    _hitPoints -= damage;
    _displayHPMeter = YES;
    _displayTimer = 1;
}

// This is essentially the same as takeDamage: but
// we need another method because
// (a) the name makes more sense
// (b) derived classes can override takeDamage:
// in order to implement things like shields.
- (void)addToLife:(int)points
{
    _hitPoints += points;
    if (_hitPoints > _maxHitPoints) _hitPoints = _maxHitPoints;
    _displayHPMeter = YES;
    _displayTimer = 1;
}

- (void)drawHPMeter
{
    float x = -_boundingRadius;
    float y = -_boundingRadius - 3;
    float dx = 2*_boundingRadius / 10.0;
    float dy = -5;

    glColor3ub(0,0xff,0);
    glBegin(GL_LINES);
    int i, numLines = (10*_hitPoints + _maxHitPoints - 1) / _maxHitPoints;
    for (i = 0; i < numLines; i++) {
	glVertex2f(x,y);
	glVertex2f(x,y+dy);
	x += dx;
    }
    glEnd();
}

@end
