/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <ApplicationServices/ApplicationServices.h>
#import "GameController.h"
#import "GameView.h"
#import "GameMap.h"
#import "ParticleEngine.h"
#import "Ship.h"
#import "Utility.h"
#import "Vector2.h"
#import "EnemyShip.h"
#import "EnemyCrawler.h"
#import "Bullet.h"
#import "Jellybean.h"

typedef struct {
    int mazeWidth;
    int mazeHeight;
    int enemyShipCount;
    float crawlerPercentage;
    int beanCount;
} LevelCode;


@implementation GameController

- (IBAction)startNewGame:(id)sender {
    if (_gameOver) {
	_level = -1;
	_startNewLevel = YES;
	_score = 0;
	_pointsToBeAdded = 0;
	_gameOver = NO;
	[ship release];
	ship = [[Ship alloc] initInMap:nil];
    }
}

- (IBAction)toggleFullScreen:(id)sender {
    [view createNewOpenGLContext:![view fullScreen]];
    [self initializeOpenGL];
}

- (IBAction)toggleShowFPS:(id)sender 
{ 
    [view setShowFPS:![view showFPS]];
    if ([view showFPS]) {
	[sender setTitle:@"Hide Frames Per Second"];
    } else {
	[sender setTitle:@"Show Frames Per Second"];
    }
}

- (IBAction)togglePause:(id)sender
{
    _paused = !_paused; 
    [self tabulaRasa]; 
    if (_paused) {
	char str[] = { 5, 'p', 'a', 'u', 's', 'e' };
	SpeakString(str);
	[sender setTitle:@"Resume"];
    } else {
	char str[] = { 8, 'u', 'n', ' ', 'p', 'a', 'u', 's', 'e' };
	SpeakString(str);
	[sender setTitle:@"Pause"];
    }
}

- (id)init
{
    self = [super init];

    srandom(time(0));

    [self tabulaRasa];

    ship = [[Ship alloc] initInMap:nil];
    bulletList = [[NSMutableArray alloc] initWithCapacity:100];

    [self createMapForCurrentLevel];

    view = [[GameView alloc] initWithFrame:NSZeroRect controller:self];
    [view createNewOpenGLContext];
    [[view window] setTitle:@"Jellybean Fugue"];
    [self initializeOpenGL];

    _zoomFactor = 1.0;
    
    return self;
}

- (void)dealloc
{
    [view release];
    [ship release];
    [unitList release];
    [bulletList release];
    [super dealloc];
}

- (void)createMapForCurrentLevel
{
    static const LevelCode levelCodes[] =
	{ { 3, 3, 2, 0, 5 },         // level 0
	  { 5, 3, 7, 0.2, 10 },      // level 1
	  { 5, 3, 20, 0.4, 20 },     // level 2
	  { 7, 5, 30, 0.2, 20 },     // level 3
	  { 7, 5, 40, 0.2, 15 },     // level 4
	  { 9, 5, 40, 0.2, 25 },     // level 5
	  { 13, 3, 40, 0.2, 25 },    // level 6
	  { 9, 7, 50, 0.2, 30 },     // level 7
	  { 11, 7, 30, 0.5, 200 },   // level 8
	  { 7, 4, 50, 0, 0 },        // level 9
	  { 11, 7, 50, 0.2, 50 },    // level 10
	  { 23, 3, 50, 0.5, 100 } }; // level 11

    LevelCode myCode;
    if (_level > 11) {
	myCode.mazeWidth = 2 + random()%14;
	myCode.mazeHeight = 2 + random()%14;
	int area = myCode.mazeWidth * myCode.mazeHeight;
	myCode.enemyShipCount = 10 + random()%100;
	myCode.crawlerPercentage = randomFloatBetween(0, 0.7);
	myCode.beanCount = random()%(3*area);
    } else {
	myCode = levelCodes[_level];
    }


    if (map) { [map release]; }
    map = [[GameMap alloc] initWithSize:NSMakeSize(myCode.mazeWidth, myCode.mazeHeight) gameController:self];

    [ship setMap:map];
    [ship setPosition:0:0];

    [bulletList removeAllObjects];

    if (unitList) { [unitList release]; }
    unitList = [[NSMutableArray alloc] initWithCapacity:10];

    int i;
    for (i = 0; i < myCode.enemyShipCount; i++) {
	EnemyShip *enemy = [[EnemyShip alloc] initInMap:map];
	NSPoint p = [map randomPointWithinMaze];
	[enemy setPosition:p.x:p.y];
	[enemy setOrientation:randomFloatBetween(0,360)];
	[unitList addObject:enemy];
	[enemy release];
    }
    for (i = 0; i < myCode.crawlerPercentage*[map numberOfWalls]; i++) {
	EnemyCrawler *enemy = [[EnemyCrawler alloc] initInMap:map];
	NSPoint p = [map randomPointWithinMaze];
	[enemy setPosition:p.x:p.y];
	[enemy setOrientation:randomFloatBetween(0,360)];
	[enemy attachToNearestWall];
	[unitList addObject:enemy];
	[enemy release];
    }
    for (i = 0; i < myCode.beanCount; i++) {
	NSPoint p = [map randomPointWithinMaze];
	Vector2 x = { p.x, p.y };
	Jellybean *bean = [Jellybean randomBeanWithPosition:&x inMap:map];
	[unitList addObject:bean];
    }

    Levelbean *bean = [[Levelbean alloc] initInMap:map];
    NSPoint p = [map randomPointWithinMaze];
    [bean setPosition:p.x:p.y];
    [bean setOrientation:randomFloatBetween(0,360)];
    [unitList addObject:bean];
    [bean release];

    _displayMessage = YES;
    _message = [[NSString stringWithFormat:@"Level %d", _level] retain];
    _messageTimer = 2;
    
}

- (void)keyDown:(NSEvent *)event
{
    if ([event isARepeat]) return;  // ignore repeats
    unichar key = [[event charactersIgnoringModifiers] characterAtIndex:0];
    if (key == NSLeftArrowFunctionKey)  { keys[0] = 1; return; }
    if (key == NSRightArrowFunctionKey) { keys[1] = 1; return; }
    if (key == NSUpArrowFunctionKey)    { keys[2] = 1; return; }
    if (key == NSDownArrowFunctionKey)  { keys[3] = 1; return; }
    if (key == ' ') { keys[' '] = 1; return; }
    if (key == 27)  { [NSApp terminate:self]; return; }
    if (key == 'z') { 
	if (_zoomLevel > -25) {
	    _zoomFactor *= 0.9;
	    _zoomLevel--;
	}
	return;
    }
    if (key == 'Z') { 
	if (_zoomLevel < 10) {
	    _zoomFactor *= 1.0/0.9;
	    _zoomLevel++;
	}
	return;
    }
    if (key == 'c' || key == 'C') {
	_cameraFollowsShip = !_cameraFollowsShip;
    }

    if (!([event modifierFlags] & NSCommandKeyMask)) {
	// When the command key is down, cmd+P is handled by
	// the application and we need to make sure we don't
	// accidentally end up calling togglePause: twice --
	// once for cmd+P and once for P.
	if (key == 'p' || key == 'P' ) {
	    [self togglePause:nil]; 
	    return; 
	}
    }
}

- (void)keyUp:(NSEvent *)event 
{ 
    unichar key = [[event charactersIgnoringModifiers] characterAtIndex:0];
    if (key == NSLeftArrowFunctionKey)  { keys[0] = 0; return; }
    if (key == NSRightArrowFunctionKey) { keys[1] = 0; return; }
    if (key == NSUpArrowFunctionKey)    { keys[2] = 0; return; }
    if (key == NSDownArrowFunctionKey)  { keys[3] = 0; return; }
    if (key == ' ') { keys[' '] = 0; return; }
}

- (void)flagsChanged:(NSEvent *)event
{
    if ([event modifierFlags] & NSCommandKeyMask) 
	keys[4] = 1;
    else 
	keys[4] = 0;
}

- (void)initializeOpenGL
{
    glClearColor(0.0f,0.0f,0.0f,0.0f);
    glViewport(0,0,640,480);
    glShadeModel(GL_SMOOTH);
    glEnable(GL_BLEND);
    glEnable(GL_LINE_SMOOTH);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glLineWidth(1.0);
    glPointSize(2.0);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D(-320, 320, -240, 240);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}

// This method makes certain that we are up to date with
// the current key state and time.  It's useful for whenever
// we restart the game after it has been suspended or paused.
- (void)tabulaRasa
{
    int i;
    for (i = 0; i < 128; i++) keys[i] = 0;

    UnsignedWide uwtime = {0, 0};
    Microseconds(&uwtime);
    _time = (double)UnsignedWideToUInt64(uwtime) / 1000000.0;
}

- (void)run
{
    if (_paused) return;

    UnsignedWide uwtime = {0, 0};
    Microseconds(&uwtime);
    double currTime = (double)UnsignedWideToUInt64(uwtime) / 1000000.0;
    double deltaTime = currTime - _time;
    _time = currTime;

    // We avoid problems with long pauses when the system is
    // doing something else by putting a cap on deltaTime.  The
    // maximum value is chosen to allow for a minimum of 15 fps.
    if (deltaTime > 0.066) deltaTime = 0.066;

    if (!_gameOver) {
	if (keys[0]) [ship rotateLeft:deltaTime];
	if (keys[1]) [ship rotateRight:deltaTime];
	if (keys[2]) [ship accelerateForward];
	if (keys[3]) [ship accelerateBackward];
	if (!keys[2] && !keys[3]) [ship brake];
	if (keys[' ']) {
	    [ship fireWeapon];
	    /*
	    if (!SpeechBusy()) {
		if (random()%2 == 0) {
		    char str[] = { 5, 'b', 'a', 'n', 'g', '!' };
		    SpeakString(str);
		} else {
		    char str[] = { 4, 'z', 'a', 'p', '!' };
		    SpeakString(str);
		}
	    }
	    */
	}
	if (keys[4]) [ship activateShield];
	else if (!keys[4]) [ship deactivateShield];
	
	[ship run:deltaTime];
	
	if ([ship isDead]) {
	    _gameOver = YES;
	    _displayMessage = YES;
	    _message = [@"Game Over" retain];
	    _messageTimer = 120;
	}
    }

    int index = [unitList count];
    while (index--) {
	GameUnit *unit = [unitList objectAtIndex:index];
	[unit run:deltaTime];
	if ([unit isDead]) {
	    [unitList removeObjectAtIndex:index];
	}
    }

    index = [bulletList count];
    while (index--) {
	Bullet *bullet = [bulletList objectAtIndex:index];
	[bullet run:deltaTime];
	if ([bullet isDead]) {
	    [bulletList removeObjectAtIndex:index];
	}
    }

    [[ParticleEngine sharedParticleEngine] run:deltaTime];

    // We roll up the score in increments.
    if (_pointsToBeAdded > 500) {
	_score += 100;
	_pointsToBeAdded -= 100;
    } else if (_pointsToBeAdded > 5) {
	_score += 5;
	_pointsToBeAdded -= 5;
    } else if (_pointsToBeAdded > 0) {
	_score += _pointsToBeAdded;
	_pointsToBeAdded = 0;
    }

    if (_startNewLevel) {
	if (!_gameOver) {
	    _level += 1;
	    [self createMapForCurrentLevel];
	}
	_startNewLevel = NO;
    }

    if (_displayMessage) {
	_messageTimer -= deltaTime;
	if (_messageTimer < 0) {
	    _displayMessage = NO;
	    [_message release];
	    _message = nil;
	}
    }

}

- (void)draw
{
    /*
    // Motion blur code
    glClear(GL_DEPTH_BUFFER_BIT);
    glColor4f(0,0,0,0.2);
    glBegin(GL_QUADS);
    glVertex2f(320,240);
    glVertex2f(-320,240);
    glVertex2f(-320,-240);
    glVertex2f(320,-240);
    glEnd();
    */

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();

    glColor3ub(0xff,0xff,0xff);
    drawLEDStringAtPoint([NSString stringWithFormat:@"%d", _score], NSMakePoint(318, 238), 3, 3);
    if (_displayMessage && _message) {
	float alpha = _messageTimer / 0.5;
	if (alpha > 1) alpha = 1;
	glColor4ub(0xff,0xff,0xff,0xff*alpha);
	drawLEDStringAtPoint(_message, NSMakePoint(0,100), 10, 4);
    }

    glScalef(_zoomFactor,_zoomFactor,0);
    if (_cameraFollowsShip) {
	glRotatef(-[ship angle] + 90, 0, 0, 1);
    }
    glTranslatef(-([ship x]), -([ship y]), 0);

    [map draw];

    int index = [unitList count];
    while (index--) [(GameUnit *)[unitList objectAtIndex:index] draw];

    index = [bulletList count];
    while (index--) [(Bullet *)[bulletList objectAtIndex:index] draw];

    [[ParticleEngine sharedParticleEngine] draw];
    if (!_gameOver) [ship draw];

}

- (void)addBullet:(Bullet *)bullet
{
    [bulletList addObject:bullet];
}

- (NSMutableArray *)unitList { return unitList; }
- (Ship *)ship { return ship; }

- (void)addToScore:(int)points
{
    _pointsToBeAdded += points;
}

- (void)startNewLevel { _startNewLevel = YES; }
@end
