/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import "EnemyCrawler.h"
#import "Utility.h"
#import "Jellybean.h"
#import "GameController.h"
#import "EnemyBullet.h"
#import "ParticleEngine.h"


@implementation EnemyCrawler

- (id)initInMap:(GameMap *)map
{
    self = [super initInMap:map];
    _timeToReload = 0.5;

    GLubyte colors[][3] = { {255,0,0}, {0,0,255}, {255,255,0}, 
			    {255,0,255}, {0,255,255} };
    int i = random()%5;
    _color[0] = colors[i][0];
    _color[1] = colors[i][1];
    _color[2] = colors[i][2];

    _maxHitPoints = 15;
    _hitPoints = _maxHitPoints;

    _boundingRadius = 10;

    return self;
}

- (void)attachToNearestWall
{
    if (_v.x == 0 && _v.y == 0) {
	_v.x = 10;
	vector_rotate(&_v, _angle * DEG_TO_RAD, &_v);
    }

    int i, numWalls = [_map numberOfWalls];
    Edge *walls = [_map walls];
    float collisionTime, bestTime;
    Edge *nearestWall = NULL;
    for (i = 0; i < numWalls; i++) {
	if (ray_intersects_edge(&_x, &_v, &(walls[i]), &collisionTime)) {
	    if (nearestWall == NULL || collisionTime < bestTime) {
		nearestWall = &(walls[i]);
		bestTime = collisionTime;
	    }
	}
    }
    if (nearestWall == NULL) return;

    _homeWall = nearestWall;
    edge_normal(_homeWall, &_normal);
    Vector2 temp;    
    vector_sub(&_homeWall->v1, &_x, &temp);
    if (vector_dot(&_normal, &temp) > 0) {
	vector_scale(-1, &_normal, &_normal);
    }

    _angle = RAD_TO_DEG * vector_angle(&_normal);
    
    // Determine initial position of crawler.

    vector_add(&_homeWall->v1, &_homeWall->v2, &_x);
    vector_scale(0.5, &_x, &_x);
    vector_scale(8, &_normal, &temp);
    vector_add(&temp, &_x, &_x);
}

- (BOOL)canBeHitByBullets { return YES; }

- (void)fireWeapon
{
    // Don't fire weapon unless it's had enough time to recharge.
    if (_reloadTimer > 0) return;
    _reloadTimer = _timeToReload;

    float radAngle = _angle * DEG_TO_RAD + _cannonAngle;
    Vector2 vel = { 100, 0 };
    vector_rotate(&vel, radAngle, &vel);

    Vector2 pos = { _boundingRadius, 0 };
    vector_rotate(&pos, radAngle, &pos);
    vector_add(&_x, &pos, &pos);

    [[_map gameController] addBullet:[EnemyBullet bulletInMap:_map
						  withPosition:&pos
						  velocity:&vel
						  orientation:_angle]];
}

- (void)moveLeft
{
    vector_rotate(&_normal, 0.5*M_PI, &_v);
    vector_scale(100, &_v, &_v);
}

- (void)moveRight
{
    vector_rotate(&_normal, -0.5*M_PI, &_v);
    vector_scale(100, &_v, &_v);
}

- (BOOL)moveBy:(float)dx :(float)dy
{
    Vector2 delta = { dx, dy };
    _hitSomething = NO;

    // Check that we are still attached to home wall.
    if (!check_moving_circle_against_edge(&_x, &delta, _boundingRadius, _homeWall)) {
	return NO;
    }

    // Check for intersection with walls of map.
    int i, numWalls = [_map numberOfWalls];
    Edge *walls = [_map walls];
    for (i = 0; i < numWalls; i++) {
	// If there is a collision, this function will change delta
	// such that the collision is avoided.
	if (check_moving_circle_against_edge(&_x, &delta, _boundingRadius, &walls[i])) {
	    [_map setWallFlagWithIndex:i toValue:YES];
	    _hitSomething = YES;
	    if (&walls[i] != _homeWall) return _hitSomething;
	}
    }

    vector_add(&_x, &delta, &_x);

    return _hitSomething;
}

- (void)run:(float)deltaTime
{
    // An enemy crawler without a wall to crawl on is as good as dead.
    if (_homeWall == NULL) {
	_isDead = YES;
	return;
    }

    // Update the various timers.
    _timer -= deltaTime;
    _reloadTimer -= deltaTime;
    // When the state timer goes below zero, it's time to change state.
    if (_timer < 0) {
	_state = random()%4;
	switch (_state) {
	case 0: _timer = randomFloatBetween(0.5, 2); break;
	case 1: case 2: _timer = randomFloatBetween(0,2); break;
	case 3: _timer = randomFloatBetween(0,1);
	}
    }
    // The crawler's actions depend on its current state.
    switch (_state) {
    case 0: _v.x = _v.y = 0; break;
    case 1: [self moveLeft]; break;
    case 2: [self moveRight]; break;
    case 3: [self fireWeapon]; break;
    }

    [super run:deltaTime];

    // When the crawler dies, it leaves behind a jellybean
    // and points get added to the player's score.
    if (_isDead) {
	[[_map gameController] addToScore:150];
	Jellybean *bean = [Jellybean randomBeanWithPosition:&_x inMap:_map];
	[[[_map gameController] unitList] addObject:bean];
    }
}

- (void)draw
{
    glPushMatrix();
    glTranslatef(_x.x, _x.y, 0);

    // Display the HP meter before rotating so that
    // it will always appear below the unit.
    if (_displayHPMeter) {
	[self drawHPMeter];
    }

    glRotatef(_angle, 0, 0, 1);

    glColor3ubv(_color);
    glBegin(GL_LINE_LOOP);
    glVertex2f(6, 5);
    glVertex2f(-8,10);
    glVertex2f(-8,-10);
    glVertex2f(6,-5);
    glEnd();
    
    glBegin(GL_LINES);
    glVertex2f(0,0);
    glVertex2f(16*cos(_cannonAngle),16*sin(_cannonAngle));
    glEnd();

    glPopMatrix();
}

- (void)explode 
{
    [[ParticleEngine sharedParticleEngine] addExplosionAtPoint:&_x withColor:_color];
}

@end
