/* Jellybean Fugue 1.0
   Copyright (c) 2004 Phillip Nguyen

   Jellybean Fugue is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License or (at your option) any later version.  

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Jellybean Fugue; see the file COPYING. If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Contact Info:
   nguyenp@eecs.tulane.edu
*/
#import <OpenGL/gl.h>
#import "GameController.h"
#import "EnemyShip.h"
#import "Bullet.h"
#import "Utility.h"
#import "ParticleEngine.h"

#define BULLET_TIME_TO_LIVE  5


@implementation Bullet

+ (Bullet *)bulletInMap:(GameMap *)map withPosition:(Vector2 *)x velocity:(Vector2 *)v orientation:(float)angle
{
    Bullet *bullet = [[Bullet alloc] initInMap:map
				     withPosition:x
				     velocity:v
				     orientation:angle];

    return [bullet autorelease];
}

- (id)initInMap:(GameMap *)map withPosition:(Vector2 *)x velocity:(Vector2 *)v orientation:(float)angle
{
    self = [super initInMap:map];

    _x = *x;
    _v = *v;
    _angle = angle;

    // We go ahead and determine which wall of the map that this
    // bullet is headed towards.  It may be headed towards multiple
    // walls, so we find the wall that it would hit first.  Then
    // later on when we do collision testing, we need only check 
    // against this one wall, rather than checking all walls in the map.
    int i, numWalls = [map numberOfWalls];
    Edge *walls = [map walls];
    float collisionTime, bestTime;
    _wallOfFate = NULL;
    for (i = 0; i < numWalls; i++) {
	if (ray_intersects_edge(x, v, &(walls[i]), &collisionTime)) {
	    if (_wallOfFate == NULL || collisionTime < bestTime) {
		_wallOfFate = &(walls[i]);
		bestTime = collisionTime;
	    }
	}
    }
    
    return self;
}

- (BOOL)moveBy:(float)dx :(float)dy
{
    Vector2 delta = { dx, dy };
    _hitSomething = NO;

    if (_wallOfFate && check_moving_point_against_edge(&_x, &delta, _wallOfFate)) {
	_hitSomething = YES;
    } else {
	NSArray *unitList = [[_map gameController] unitList];
	int index = [unitList count];
	while (index--) {
	    GameUnit *unit = [unitList objectAtIndex:index];
	    if ([unit canBeHitByBullets]) {
		if (check_moving_point_against_circle(&_x, &delta, [unit position], [unit boundingRadius])) {
		    _hitSomething = YES;
		    [unit takeDamage:1];
		    break;
		}
	    }
	}
    }

    vector_add(&_x, &delta, &_x);

    return _hitSomething;
}

- (void)run:(float)deltaTime
{

    // We make sure that any errant bullets will eventually die.
    _lifetime += deltaTime;
    if (_lifetime > BULLET_TIME_TO_LIVE) {
	_isDead = YES;
	return;
    }

    [super run:deltaTime];
    
    if (_hitSomething) {
	[self explode];
	_isDead = YES;
    }
}

- (void)explode
{
    Vector2 v[3] = { {-10, -5}, {0, 0}, {-10, 5} };
    float radAngle = _angle * DEG_TO_RAD;
    int i;
    for (i = 0; i < 3; i++) {
	vector_rotate(&v[i], radAngle, &v[i]);
	vector_add(&v[i], &_x, &v[i]);
    }
    Edge e[2] = { {v[0],v[1]}, {v[1],v[2]} };

    float theta = randomFloatBetween(M_PI/4, M_PI/2);
    GLubyte red = 192 + random()%64;
    GLubyte green = random()%256;
    for (i = 0; i < 2; i++) {
	Vector2 vel;
	vector_scale(-0.1, &_v, &vel);
	vector_rotate(&vel, theta, &vel);
	[[ParticleEngine sharedParticleEngine] addEdge:&e[i] 
					       withVelocity:&vel
					       color:red:green:0
					       timer:0.6
					       fade:YES];
	theta *= -1;
    }
}

- (void)draw
{
    glPushMatrix();
    glTranslatef(_x.x, _x.y, 0);
    glRotatef(_angle, 0, 0, 1);

    glColor3ub(0xff, 0, 0);
    glBegin(GL_LINE_STRIP);
    glVertex2f(-10, -5);
    glVertex2f(  0,  0);
    glVertex2f(-10,  5);
    glEnd();

    glPopMatrix();
}


@end
