#import <Foundation/Foundation.h>

/*! \file ASWString.h */

/*! \brief An NSString category that adds useful methods. */
@interface NSString (ASWString)

/*! \brief Creates a NSString by concatentating the contents of array.
	\return An NSString. */
- (id) initWithStrings: (NSArray*) array joinedBy: (NSString*) separator;

/*! \brief Creates a NSString by concatentating the contents of array.
	\return An NSString. */
+ (NSString*) stringWithStrings: (NSArray*) array joinedBy: (NSString*) separator;

/*! \brief Creates a NSString that is formatted appropriately for the byte count.
	
	Create a NSString that is formatted appropriately: 32k, 1 MB, 2 GB
	
	\param byteCount The byte count to create a a string for.
	\return An NSString. */
+ (NSString *)stringWithByteCount:(unsigned long long)byteCount;

/*! \brief Convenience method to interpret an NSData buffer as UTF-8 text */
+ (NSString *)stringWithUTF8Data:(NSData *)data;

/*! \brief Returns TRUE if self contains any of the Unicode newline characters
	
	Currently can return false positives. */
- (BOOL)containsNewLine;


/*! \brief Returns TRUE if self contains the substring
	\return TRUE if self contains the substring */
- (BOOL) hasSubstring: (NSString*) substring;

/*! \brief Creates an string suitable for passing the system shell to execute
	
	Pass arguments if any, as NSStrings following the command, and
	terminate with nil. All arguments are escaped so as to not be interpreted
	by the shell. Any instances of NSNull in the command list are ignored. */
+ (NSString *)escapedShellCommand:(NSString *)command, ...;

/*! \brief Similar to Perl's split()
	
	Via CocoaDev.com
	
	\return Array of tokens
 */
- (NSArray *)tokensSeparatedByCharactersFromSet:(NSCharacterSet *)separatorSet;

/*! \brief Returns the total range of the first substring that contains only characters in the set.
	
	See rangeOfCharactersFromSet:options: for more info.
*/
- (NSRange) totalRangeOfCharactersFromSet:(NSCharacterSet*) set options: (unsigned int) mask;

/*! \brief Compares two version strings...
	\return comparison result */
- (NSComparisonResult) versionStringCompare: (NSString*) aString;

/*! \brief Compares two version strings...
	\return comparison result */
- (NSComparisonResult) numericallyOrderedCompare: (NSString*) aString;

/*! \brief  Compares two version strings...
	\return comparison result */
- (NSComparisonResult) caseInsensitiveNumericallyOrderedCompare: (NSString*) aString;

/*! \brief returns a rect value for a given string
	\return rect version of the string*/
- (CGRect)rectValue;

/*! \brief creates a NSString from a NSRect
	\param inRect NSRect to get a string representation from
	\return NSString representation of the NSRect */
+ (id)stringWithRect:(CGRect)inRect;

/*! \brief creates a NSString with the contents of a CFUUID
    \return NSString containing UUID
 */
+ (id) stringWithNewUUID;

#if defined(__IPHONE_3_0) && __IPHONE_3_0 <= __IPHONE_OS_VERSION_MAX_ALLOWED
#import <libxml/tree.h>

/*! \brief get the string as an xmlChar * 
	\return xmlChar * of the string */
- (const xmlChar *)xmlChar;

#elif defined(ASW_NOT_IPHONE) && ASW_NOT_IPHONE
/*! \brief Convenience method to convert a longward OSType into a 4 character string. 
	Returns an empty string if the input osType is 0. */
+ (NSString *)stringWithOSType:(OSType)osType;

/*! \brief Convenience method to convert a longward OSType into a 4 character string. 
	Returns an empty string if the input osType is 0.  */
- (id)initWithOSType:(OSType)osType;

/*! \brief Convenience method to return a 4 character string as an OSType. 
	Returns 0 if the string is empty. */
- (OSType)osTypeValue;
- (NSString *)stringByTruncatingToWidth:(CGFloat)width withAttributes:(NSDictionary *)attributes;
#endif

/*! \brief Convenience method to return a string with all the whitespace trimmed off 
	\return whitespace trimmed string */
- (NSString *)trimWhitespace;

/*! \brief Break the string up into components of length
 
	The remainder is in the last component */
- (NSArray*) componentsWithLength: (NSUInteger) length;

@end

/*! \brief An NSMutableString category that adds useful methods */
@interface NSMutableString (ASWMutableString)

/*! \brief removes any characters in the characterset from the string
	\param inSet character set specifying which characters to remove */
- (void)removeCharactersFromCharacterSet:(NSCharacterSet*)inSet;

@end

/*! \brief NSString category that adds path methods */
@interface NSString (ASWStringPathUtils)

/*! \brief Attempts to resolve aliases in the receiver's path
	
	traverses the path and if it finds that any path component is an alias it will resolve
	that alias and continue to attempt traversal of the remaining path components
	
	\return path with resolved aliases as a string
*/
- (NSString*)resolveAliases;

@end
