/*
 *  ASWFileManager.h
 *  WireTapPro2
 *
 *  Created by Dominic Feira on 12/21/05.
 *  Copyright 2005 __MyCompanyName__. All rights reserved.
 *
 */

#import "ASWFileRef.h"

/*! \file ASWFileManager.h */

/*! \brief An NSFileManager category that adds useful methods for dealing with ASWFileRefs. */
@interface NSFileManager (ASWFileManager)

/*! Copies a source file referenced by an ASWFileRef to the destination directory referenced by an ASWFileRef. The name is enumeratored until a free path is found or maxTries.
	\param source ASWFileRef that references the source file.
	\param dest ASWFileRef that references the destination directory.
	\param name The name the file should be given in the destination.
	\param maxTries Max number of times the copy should be tried.
	\param handler A delegate error handler. Currently not used.
	\return Boolean, TRUE for success. */
- (ASWFileRef*) duplicateFileRef:(ASWFileRef *)source 
					 inDirectory:(ASWFileRef *)dest 
			  withEnumeratedName:(NSString*) name 
						maxTries:(NSUInteger) maxTries
						 handler:(id)handler;

/*! Moves a source file referenced by an ASWFileRef to the destination directory referenced by an ASWFileRef. The name is enumeratored until a free path is found or maxTries.
	\param source ASWFileRef that references the source file.
	\param dest ASWFileRef that references the destination directory.
	\param name The name the file should be given in the destination.
	\param maxTries Number of times to try move before giving up
	\param handler A delegate error handler. Currently not used.
	\return Boolean, TRUE for success.
 */
- (ASWFileRef*)moveFileRef:(ASWFileRef *)source 
			   toDirectory:(ASWFileRef *)dest 
		withEnumeratedName:(NSString*) name 
				  maxTries:(NSUInteger) maxTries
				   handler:(id)handler;


/*! Copies a source file referenced by an ASWFileRef to the destination directory referenced by an ASWFileRef
	\param source ASWFileRef that references the source file.
	\param dest ASWFileRef that references the destination directory.
	\param name The name the file should be given in the destination.
	\param handler A delegate error handler.  Currently not used.
	\return Boolean, TRUE for success.
 */
- (ASWFileRef*)duplicateFileRef:(ASWFileRef *)source 
					inDirectory:(ASWFileRef *)dest 
					   withName:(NSString*) name 
						handler:(id)handler;

/*! Moves a source file referenced by an ASWFileRef to the destination directory referenced by an ASWFileRef
	\param source ASWFileRef that references the source file.
	\param dest ASWFileRef that references the destination directory.
	\param name The name the file should be given in the destination.
	\param handler A delegate error handler. Currently not used.
	\return Boolean, TRUE for success.
 */
- (ASWFileRef*)moveFileRef:(ASWFileRef *)source 
			   toDirectory:(ASWFileRef *)dest 
				  withName:(NSString*) name 
				   handler:(id)handler;

/*! Does a move when possible.  Otherwise performs a copy. */
- (BOOL) fastMovePath: (NSString*) sourcePath toPath: (NSString*) destinationPath handler: (id) handler;

/*! \brief Get an enumerated name in the specified directory.
	\param maxTries  The maximum number of tries before it gives up
	\return The enumerated name or nil if unable to find an unused name within maxTries */
- (NSString*) enumeratedName: (NSString*) name inDirectory: (ASWFileRef*) fileRef 
					maxTries: (NSUInteger) maxTries handler: (id) handler;


/*! \brief Moves a file to the trash. */
- (void)trashFileRef:(ASWFileRef *)source;

/*! \brief Moves a file to the trash. */
- (void)trashFileAtPath: (NSString*) path;

/*! \brief Creates a file at the given path.  
	\param path Path that references the desired file location.
	\param dir BOOL that indicates if the created file is a directory or not.
	\return ASWFileRef*, nil for failure.
*/
- (ASWFileRef*) createFileRefAtPath: (NSString*) path asDirectory: (BOOL) dir;

/*! \brief Creates a file at the given path.  
 \param pathIn Path that references the desired file location.
 \param intermediate BOOL that indicates if all directories in the path should be created if necessary.
 \param error NSError instance if an error occurs
 \return ASWFileRef*, nil for failure.
 */
- (ASWFileRef*) createDirectoryAtPath: (NSString*)pathIn withIntermediateDirectories: (BOOL) intermediate error: (NSError**) error;

/*! \brief Creates a file at the given path.  
	\param name Desried file name
	\param parent Directory to create the file in.
	\param dir BOOL that indicates if the created file is a directory or not.
	\return ASWFileRef*, nil for failure.
*/
- (ASWFileRef*) createFileRefWithName: (NSString*) name 
						  inDirectory: (ASWFileRef*) parent 
						  asDirectory: (BOOL) dir;

/*! \brief Recursively calculates the size of the directory tree under a path
	\param path Path to a directory or single file
*/
- (UInt64)sizeAtPath:(NSString *)path;

/*! \brief Return the path to a temporary file.
	\param baseFileName A string to act as prefix for the URL created.
	\param extension The extension to give the temp file. 
	\return The URL of the temp file.
 */
- (NSURL*) tempFileURLWithBase:(NSString*)baseFileName extension:(NSString*)extension;


@end
