/*
 ASWEditorProxy
 
 Copyright (c) 2009 Ambrosia Software, Inc.
 All Rights Reserved
 
 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:
 1. Redistributions of source code must retain the above copyright
 notice, this list of conditions and the following disclaimer.
 2. Redistributions in binary form must reproduce the above copyright
 notice, this list of conditions and the following disclaimer in the
 documentation and/or other materials provided with the distribution.
 3. The name of the author may not be used to endorse or promote products
 derived from this software without specific prior written permission.
 
 THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/** \brief Proxy object for buffering edits to an object and integrating changes with 
 *         an undo manager.
 *
 *  This class provides a layer to abstract saving changes from an editable document.
 *  This is used in Soundboard by the editor, to buffer changes until the user presses
 *  the Save button. It should just work, as it supports KVO and Cocoa bindings.
 *  
 *  Do NOT use primitives with this; only valid Obj-C types are supported.
 *  
 *  To use:
 *
 *  -# Create with proxyInitWithObject:, supplying the source object 
 *  -# Use ASWEditorProxy in place of your source object. ASWEditorProxy will use Obj-C
 *     introspection/runtime voodoo to call methods on the proxied object as follows:
 *       - if you use setValue:forKey:, setValue:forKeyPath:, setValue:forUndefinedKey:,
 *         or set<Key>: those changes will be stored in the ASWEditorProxy
 *       - if you use a key-value accessor or valueForKey:, the ASWEditorProxy will check
 *         to see if you have changed the value for that key (using the above methods),
 *         returning the new value if you have, and calling the proxied object's accessor
 *         otherwise
 *       - if it hasn't gotten called yet, it'll call the method on the proxied object
 *  -# If the user presses the Save button, call saveChangesToProxiedObject
 *  -# Release the object when the editor is closed.
 */

#import <Foundation/Foundation.h>

@interface ASWEditorProxy : NSObject {
	id proxiedObject;
	NSMutableDictionary *changes;
	
	NSUndoManager *undoManager;
}

/** 
 *  Creates a new proxy object, which refers to obj.
 *  \arg \c obj the object you wish to buffer changes from
 *  \return new ASWEditorProxy
 */
-(id)proxyInitWithObject:(id)obj;

/** 
 *  Optionally set an undo manager for automatic undo/redo of
 *  changes made to the proxy.
 *  \arg \c newUndoManager an NSUndoManager that you want edits registered with 
 *       (typically you'd get this from the NSWindow you're using at the time)
 */
-(void)proxySetUndoManager:(NSUndoManager *)newUndoManager;

/**
 *  Walks through the list of changes made, and calls:
 *  [proxiedObject setValue:<newValue> forKeyPath:<keyPath>]
 *  for every change made.
 */
-(void)saveChangesToProxiedObject;

@end
