//
//  ASWTransitionView.h
//  ASWTransitionView
//
//  Basically a re-engineered version of ASWAnimatedTabView without the tabs and now has view & window resizing ability
//
//  Created by Jeremy Knope on 7/14/09.
//  Copyright 2009 Ambrosia Software, Inc.. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import <QuartzCore/QuartzCore.h>
#import "ASWAnimation.h"

/*! \file ASWTransitionView.h */

/*! \brief Style of transition to use */
typedef enum 
{
	/*! \brief Copy-machine transition */
    ASWTransitionViewCopyMachineTransitionStyle,
	/*! \brief Disintegrate with mask transition */
    ASWTransitionViewDisintegrateWithMaskTransitionStyle,
	/*! \brief Simple cross-disolve transition, the default */
    ASWTransitionViewDissolveTransitionStyle,
	/*! \brief Flash transition */
    ASWTransitionViewFlashTransitionStyle,
	/*! \brief Mod transition */
    ASWTransitionViewModTransitionStyle,
	/*! \brief Page curling transition */
    ASWTransitionViewPageCurlTransitionStyle,
	/*! \brief Watter ripple transition */
    ASWTransitionViewRippleTransitionStyle,
	/*! \brief Swipe transition */
    ASWTransitionViewSwipeTransitionStyle,
	/*! \brief Slide transition, similar to iPhone navigation controller's slide */
    ASWTransitionViewSlideTransitionStyle,
	/*! \brief Used when using a custom CIFilter */
	ASWTransitionViewCustomTransitionStyle,
} ASWTransitionViewTransitionStyle;

typedef enum
{
	ASWDirectionUnknown = -1,
    ASWDirectionBackwards = 0,
    ASWDirectionForwards = 1,
} ASWTransitionViewTransitionDirection;

@class ASWTransitionView;

@protocol ASWTransitionViewDelegate
- (BOOL)transitionView:(ASWTransitionView*)tabView shouldCacheView:(NSView*)view;
- (void)transitionView:(ASWTransitionView*)tabView willCacheView:(NSView*)view;
- (void)transitionView:(ASWTransitionView*)tabView didCacheView:(NSView*)view;

- (void)filter:(CIFilter*)filter settingsForDirection:(NSInteger)direction;
@end


/*! ASWTransitionView
 \brief Allows transitioning between views along with resizing the containing window to fit 
 \ingroup ASWAppKit
 
 The view when created should be set to autosize if you want it to properly size itself along with the window it's in */
@interface ASWTransitionView : NSView
{
	NSTimeInterval						mAnimationDuration;
	ASWAnimationBlockingMode	mAnimationMode;
    ASWTransitionViewTransitionStyle	mTransitionStyle;
    CIFilter					*mTransitionFilter;
    CIImage						*mInputShadingImage;
    CIImage						*mInputMaskImage;
    NSRect						mImageRect;
	ASWAnimation *mAnimation;
	
	NSView						*mFinalContentView;
	//NSTabViewItem				*mFinalTabViewItem;
    NSMutableArray				*mAnimationQueue;
	BOOL						slidingPluginLoaded;
	NSMutableArray				*mCachedViews;
	
	NSMutableArray *mViews;
	NSView *mCurrentView;
	
	id delegate;
	
	// for animation of window resizing
	NSRect mWindowStartFrame;
	NSRect mWindowEndFrame;

	BOOL mAnimates;
}

/*! \brief Adds a view to the end of the view list 
 
 The first view added will be set current and will attempt to resize window if available */
- (void)addView:(NSView *)view;

/*! \brief Selects the given view, animating if requested 
	\param view View to be selected that has previously been added
 
 Selects (animates and makes it visible) the given view if it has already been added via addView: */
- (void)selectView:(NSView *)view;

/*! \brief Resets to first view, no animation */
- (void)resetToFirstView;

/*! \brief Returns the current(selected) view */
- (NSView *)selectedView;

/*! \brief Animates to view at given index if there is one */
- (void)selectViewAtIndex:(NSUInteger)viewIndex;

/*! \brief Animates to the next view in the list if there is one */
- (void)nextView;

/*! \brief Animates to previous view if there is one */
- (void)previousView;

/*! \brief Sets delegate (unfinished) */
- (void)setDelegate:(id)newDelegate;

/*! \brief Returns current delegate */
- (id)delegate;

/*! \brief Sets seconds the transition & window animation should last */
- (void)setTransitionDuration:(NSTimeInterval)durationInSecs;

/*! \brief Sets the style to use, defaults to ASWTransitionViewDissolveTransitionStyle 
	\param newTransitionStyle The transition style to use when changing views */
- (void)setTransitionStyle:(ASWTransitionViewTransitionStyle)newTransitionStyle;

/*! \brief Returns the currently set transition style */
- (ASWTransitionViewTransitionStyle)transitionStyle;

- (BOOL)shouldAnimate;
- (void)setShouldAnimate:(BOOL)shouldAnimate;

@end
