//
//  ASWSelfHealerPanel.h
//  ASWAppKit
//
//  Created by Mike Blaguszewski on 5/15/08.
//  Copyright 2008 Ambrosia Software, Inc.. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import "ASWSelfHealer.h"

/*! \file ASWSelfHealerPanel.h
	High-level (GUI) interface for application integrity checking and self-healing. See
	ASWSelfHealer.h for further documentation. */

/*! GUI class for performing self-healing operations. Can be run modally or as a sheet.
	If no healing is found to be necessary, does not display a dialog at all. */
@interface ASWSelfHealerPanel : NSObject
{
	ASWSelfHealer *_selfHealer;
	NSWindow *_window;
	NSImage *_icon;
	id _didEndDelegate;
	SEL _didEndSelector;
	
	NSString* _internalProblemsAlertTitle;
	NSString* _internalProblemsAlertInformativeText;
	NSString* _externalProblemsAlertTitle;
	NSString* _externalProblemsAlertInformativeText;
	NSString* _problemsAlertFixButtonText;
	NSString* _problemsAlertCancelButtonText;
}
/*! Designated initializer. Attempts to read a plist with expected file attributes, and to
	verify the cryptographic signature on the plist.
	\param path Location of plist. If nil, looks for a SelfHealing.plist in the bundle's
	resource directory (see -bundle method)
	\param signedHash An SHA1 hash of the plist file, signed with Ambrosia code-signing key. If
	nil, uses the value for the key ASWSelfHealingSignature in the bundle's Info.plist.
	\param bundle The bundle used to search for Info.plist and SelfHealing.plist, and to 
	find internal components. Defaults to the main bundle.
	\return a new self healer panel, or nil if plist reading or verification fails */
- (id)initWithPlist:(NSString *)path signature:(NSData *)signedHash bundle:(NSBundle *)bundle;
/*! \brief Icon to use with alerts.
	The default image is the application icon. */
- (NSImage *)icon;
/* \brief Sets the icon to use on alerts. */
- (void)setIcon:(NSImage *)icon;
/*! Check integrity, and if problems are found, present a modal alert. If the user accepts,
	attempt self healing. */
- (ASWSelfHealingResult)runModalIfNeeded;
/*! Check integrity, and if problems are found, present a sheet on the indicated window. If the
	user accepts, attempt self healing.
	\param window Window to attach sheet to
	\param delegate Delegate that receives the didEndSelector. Can be nil.
	\param didEndSelector Selector of the form: - (void)healingDidEnd:(ASWSelfHealer *)healer returnCode:(ASWSelfHealingResult)result */
- (void)beginSheetModalForWindow:(NSWindow *)window delegate:(id)delegate didEndSelector:(SEL)didEndSelector;

- (ASWSelfHealer*) selfHealer;

#pragma mark Customization
- (void) setExternalProblemsAlertTitle: (NSString*) title;
- (NSString *)externalProblemsAlertTitle;

- (void) setExternalProblemsAlertInformativeText: (NSString*) text;
- (NSString *)externalProblemsAlertInformativeText;

- (void) setInternalProblemsAlertTitle: (NSString*) title;
- (NSString *)internalProblemsAlertTitle;

- (void) setInternalProblemsAlertInformativeText: (NSString*) text;
- (NSString *)internalProblemsAlertInformativeText;

- (void) setProblemsAlertFixButtonText: (NSString*) text;
- (NSString *)problemsAlertFixButtonText;

- (void) setProblemsAlertCancelButtonText: (NSString*) text;
- (NSString *)problemsAlertCancelButtonText;
@end
