//
//  ASWPreferencesController.h
//  ASWInterface
//
//  Created by Steve Streza on 12/18/09.
//  Copyright 2009 Ambrosia Software. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class ASWPreferencesViewController;
@class ASWTransitionView;

@protocol ASWPreferencesControllerProtocol

/*! \brief Gets called before the view appears
 *
 *  \note  These get called by ASWPreferencesController as different view
 *         controllers are displayed/hidden. Override in your subclass.
 *         Default implementation does nothing.
 */
-(void)viewWillAppear;

/*! \brief Gets called after the view appears
 *
 *  \note  These get called by ASWPreferencesController as different view
 *         controllers are displayed/hidden. Override in your subclass.
 *         Default implementation does nothing.
 */
-(void)viewDidAppear;

/*! \brief Gets called before the view disappears
 *
 *  \note  These get called by ASWPreferencesController as different view
 *         controllers are displayed/hidden. Override in your subclass.
 *         Default implementation does nothing.
 */
-(void)viewWillDisappear;

/*! \brief Gets called after the view disappears
 *
 *  \note  These get called by ASWPreferencesController as different view
 *         controllers are displayed/hidden. Override in your subclass.
 *         Default implementation does nothing.
 */
-(void)viewDidDisappear;


@end

/** \file ASWPreferencesController.h
 *  \class ASWPreferencesController
 *  \brief A window controller for managing tabs of preference panes
 *
 *  ASWPreferencesController is a window controller for showing a set of
 *  preferences in a series of preference panes. To use it, you need to subclass 
 *  ASWPreferencesViewController for each pane you wish to make. Then, call
 *  +[ASWPreferencesController addController:] or its variants to register
 *  your subclasses. Then, create and present your ASWPreferencesController.
 *  When you show it, it will automatically populate with views from each of
 *  the ASWPreferencesViewController objects added.
 */
@interface ASWPreferencesController : NSWindowController {
	NSToolbar *toolbar;
	
	NSMutableDictionary *identifierDictionary;

	NSUInteger selectedViewIndex;
	
	ASWTransitionView *animatingView;
}

/*! \brief the toolbar for the preferences window */
@property (retain) IBOutlet NSToolbar *toolbar;
@property (retain) NSMutableDictionary *identifierDictionary;

/*! \brief Returns a view controller for a given identifier 
 *  \param ident A string identifying a given view controller
 *  \return the view controller matching the ident
 */
- (ASWPreferencesViewController *)controllerForIdentifier:(NSString *)ident;

/*! \brief Returns the identifier for a given view controller
 *  \param controller A view controller owned by the preferences controller
 *  \return the string identifier which uniquely identifies the view controller
 */
- (NSString *)identifierForController:(ASWPreferencesViewController *)controller;

/*! \brief Adds a view controller to the preferences window */
- (void)addController:(ASWPreferencesViewController *)controller;

/*! \brief makes the current selection the tab with the specified identifier, if it exists
 *         
 *  \param identifier a string representing the desired tab
 */
-(void)selectPreferencesViewWithIdentifier:(NSString*)identifier;

/*! \brief makes the current selection the tab with the specified identifier, if it exists
 *         
 *  \return the identifier associated with the currently selected preferences view
 */
- (NSString*)identifierForSelectedPreferencesView;


/*! \brief gives access to the viewControllers that currently make up the window
 *         
 *  \return the identifier associated with the currently selected preferences view
 */
- (NSArray*)viewControllers;

- (ASWPreferencesViewController*)selectedViewController;

@end
