-----------------------------------------------------------------------------
-- ModBrowserUtilities.lua
-- Suite of shared routines used by various panels of the Mods Browser.
-- Copyright (c) 2010 Firaxis Games, Inc. All rights reserved.
-----------------------------------------------------------------------------

-----------------------------------------------------------------------------
-- Download States
-- Enumeration for Download states.
-----------------------------------------------------------------------------
DownloadStates = {
	[0] = "QUEUED",
	["QUEUED"] = 0,

	[1] = "CONNECTING",
	["CONNECTING"] = 1,

	[2] = "TRANSFERRING",
	["TRANSFERRING"] = 2,

	[3] = "SUSPENDED",
	["SUSPENDED"] = 3,

	[4] = "ERROR",
	["ERROR"] = 4,

	[5] = "WARNING",
	["WARNING"] = 5,

	[6] = "TRANSFERRED",
	["TRANSFERRED"] = 6,

	[7] = "COMPLETE",
	["COMPLETE"] = 7,
	
	[8] = "CANCELED",
	["CANCELED"] = 8,
}

-----------------------------------------------------------------------------
-- FileSizeAsString(bytes)
-- Returns a formatted string for displaying file sizes.
-----------------------------------------------------------------------------
local gbString = Locale.ConvertTextKey("TXT_KEY_MODDING_FILESIZE_GB");
local mbString = Locale.ConvertTextKey("TXT_KEY_MODDING_FILESIZE_MB");
local kbString = Locale.ConvertTextKey("TXT_KEY_MODDING_FILESIZE_KB");
local byteString = Locale.ConvertTextKey("TXT_KEY_MODDING_FILESIZE_B");
function FileSizeAsString(bytes)
	if(bytes > 1073741824) then
		return string.format("%d %s", bytes / 1073741824, gbString);
	elseif(bytes > 1048576) then
		return string.format("%d %s", bytes / 1048576, mbString);
	elseif(bytes > 1024) then
		return string.format("%d %s", bytes / 1024, kbString);
	else
		return string.format("%d %s", bytes, byteString);
	end
end

-----------------------------------------------------------------------------
-- GetDownloadStatus(jobStatus)
-- Returns a localized string describing the status of a download.
-----------------------------------------------------------------------------
local DownloadStatusText = {};
function DownloadStatus(value)
	local text = DownloadStatusText[value];
	if(text == nil) then
		text = Locale.ConvertTextKey("TXT_KEY_MODDING_DOWNLOADSTATUS_" .. value);
		DownloadStatusText[value] = text;
	end
	
	return text;
end

-----------------------------------------------------------------------------
-- GetDownloadButtonStates(jobStatus)
-- Simple routine for determining which buttons should be hidden or shown.
-----------------------------------------------------------------------------
function GetDownloadButtonStates(jobStatus)
	local hidePauseButton = true;
	local hideResumeButton = true;
	local hideCancelButton = true;
	local hideCloseButton = true;
	
	if(jobStatus == 3 or jobStatus == 4) then		-- Suspended or Error (most resume or cancel)
		hideResumeButton = false;
		hideCancelButton = false;
		
	elseif(jobStatus == 6 or jobStatus == 7 or jobStatus == 8) then --Transferred, Complete, or Cancelled
		hideCloseButton = false;
	
	else										-- All other options get Pause and Cancel
		hidePauseButton = false;
		hideCancelButton = false;
	end
	
	return {
		HidePauseButton = hidePauseButton,
		HideResumeButton = hideResumeButton,
		HideCancelButton = hideCancelButton,
		HideCloseButton = hideCloseButton,
	}
end

function GetDownloadDescription(ModID, ModVersion, ModTitle)
	return string.format("%s - (v. %i)", ModTitle, ModVersion);
end

-----------------------------------------------------------------------------
-- GetDownloadFileName(modID, modVersion)
-- Returns the download filename for a specific modid, version.
-----------------------------------------------------------------------------
function GetDownloadFileName(modID, modVersion)
	return string.format("%s-%i.civ5mod", modID, modVersion);
end

-----------------------------------------------------------------------------
-- GetOnlineErrorMessage(errorCode)
-- Returns a localized error message for an error code.
-----------------------------------------------------------------------------
local errorMessages = {
	generic = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_GENERIC"),
	[2] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_SERVICE_DISABLED"),
	[3] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_CONNETION_TIMEOUT"),
	[4] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_CONNECTION"),
	[16] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_ALREADY_RATED"),
	[23] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_REQUEST_CANCELLED"),
	[25] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_HTTP_UNAUTHORIZED"),
	[26] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_HTTP_FORBIDDEN"),
	[27] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_HTTP_FILE_NOT_FOUND"),
	[28] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_HTTP_REQUEST_REJECTED"),
	[29] = Locale.ConvertTextKey("TXT_KEY_MODDING_ERROR_HTTP_SERVER_ERROR"),
		
}
function GetOnlineErrorMessage(errorCode)
	return errorMessages[errorCode] or errorMessages.generic;
end

------------------------------------------------------------------------------
-- Get Mod Category Name
-- Retrieves a potentially localized name of a specific category ID.
------------------------------------------------------------------------------
local modCategoryNames = {};
function GetModCategoryName(catID, defaultName)
	if(modCategoryNames[catID] == nil) then
		local key = "TXT_KEY_MODDING_CATEGORY_" .. catID;
		local name = Locale.ConvertTextKey(key);
			
		if(key == name) then
			name = defaultName;
		end
		
		modCategoryNames[catID] = name;
	end
	return modCategoryNames[catID];
end