--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Utility functions
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

bsutil = {}

gAutoSaveTime = 0

-------------------------------------------------------------------------------
-- User helpers

function bsutil.AutoSave()
	-- Auto-save current game if >5 minutes have passed
	local elapsedTime = UtilCurrentTime() - gAutoSaveTime
--	local remain = (5 * 60 * 1000) - elapsedTime
	local remain = (30 * 1000) - elapsedTime
	if remain <= 0 then
		DebugOut("Auto-saving...")
		bsutil.SaveGame()
	else
		DebugOut("Auto-saving in: " .. remain)
	end
end

function bsutil.SaveGame()
--	if gSim and gSim.player == GetCurrentUserName() and gSim.mode then
	if gSim and gSim.mode then
		assert(gSim.player == GetCurrentUserName())
		bsutil.LogScore()
		gSim:SaveGame()
	end
end

function bsutil.LoadGame(mode)
	assert(mode)
	local oldGame = true		-- returns "false" if this is a NEW game
	if not gSim or gSim.mode ~= mode then
		if gSim then bsutil.SaveGame() end
		Simulator:Reset(GetCurrentUserName())
		
		oldGame = gSim:LoadGame(mode)
		if not oldGame then
			-- Initialize a new game
			if mode == "free" then gSim:ResetFreePlayMode()
			else gSim:ResetEmpireMode()
			end
			gSim:SaveGame()
		end

		assert(gSim and gSim.player == GetCurrentUserName())
		gSim:ProjectProduction()
	end
	return oldGame
end

function bsutil.ResetGame(mode)
	assert(gSim)
	Simulator:Reset(gSim.player)
	if mode == "free" then gSim:ResetFreePlayMode()
	else gSim:ResetEmpireMode()
	end
	gSim:SaveGame()
end

function bsutil.CreateNewUser(name,mode)
	assert(mode)
	CreateNewUser(name)
	Simulator:Reset(name)
	if mode == "free" then gSim:ResetFreePlayMode()
	else gSim:ResetEmpireMode()
	end
	gSim:SaveGame()
end

function bsutil.SwitchUser(index)
	-- Save the current player and switch
	bsutil.SaveGame()
	gSim = nil
	SetCurrentUser(index)
	bsutil.LoadGame(GetCurrentUserMode())
end

function bsutil.LogScore()
	if gSim then
		local score, weeks, money = gSim:GetScore()
		local rank = gSim.rank or 1
		
		-- Prepare server data string: R-WWWWWWWWW-MMMMMMMMM
		local stringData = string.format("%d-%09d-%09d", rank, weeks, money)
		
		if gSim.mode == "empire" then SetCurrentGameMode(0)
		else SetCurrentGameMode(1)
		end
		LogScore(score, stringData, gSim:GetMedalFlags())
	end
end

-------------------------------------------------------------------------------
-- Math helpers

-- floor(n) - returns the largest integer <= n
bsutil.floor = UtilMathFloor

bsutil.srand = UtilMathSeedRandom

-- random(a) - returns a random number (1..n) inclusive
-- random(a,b) - returns a random number (a..b) inclusive
function bsutil.random(a,b)
	if b then return UtilMathRandom(a,b)
	else return UtilMathRandom(1,a)
	end
end

-------------------------------------------------------------------------------
-- Data helpers

function bsutil.ArrayIterator(t)
	local i = 0
	local n = table.getn(t)
	return function()
		i = i + 1
		return (i <= n) and t[i] or nil
	end
end

bsutil.StringToClipboard = UtilToClipboard

-------------------------------------------------------------------------------
-- Format helpers

-- GetVariableString(key, vals1, vals2) - looks up a string and replaces <var> references
-- with values from the given value tables, also appropriately translated
function bsutil.GetVariableString(key, vals1, vals2)
	-- Default to using gSim for data lookups
	if not vals1 then vals1 = gSim
	elseif not vals2 then vals2 = gSim
	end
	
	local s = GetString(key)
	if s == "#####" then s = key
	elseif vals1 or vals2 then
		s = string.gsub(s, "<(%w+)>",
			function(var)
				local s1 = vals1 and vals1[var]
				local s2 = vals2 and vals2[var]
				
				-- Find a string to use, first one wins
				local key = nil
				if s1 and type(s1) == "function" then s1 = s1() end
				if s1 then
					s1 = tostring(s1)
					key = s1
				else
					if s2 and type(s2) == "function" then s2 = s2() end
					if s2 then
						s2 = tostring(s2)
						key = s2
					end
				end
				
				-- Translate the string and return the valid look-up or the original string
				local s = key
				if type(s) == "string" then
					s = GetString(s)
					s = (s == "#####") and key or s
				end
				if s == nil then s = "<"..var..">" end
				return tostring(s)
			end)
	end
	return s
end

function Dollars(n)
	return UtilDollars(n or 0)
end

-------------------------------------------------------------------------------
-- UI Helpers

-- Generic dialogs - create a modal dialog with a function providing the contents
function bsutil.DoDialog(content, fill)
	return DisplayDialog{"ui/dialog.lua", dialogContents = content, dialogFill = fill}
end

-- Animation generation function for inclusion in port layouts
function Animation(t)
	return function() return SimpleAnimation(t) end
end

-- Bitmap generation function for inclusion in port layouts
function BitmapLayer(t)
	return function() return Bitmap(t) end
end

-------------------------------------------------------------------------------
-- Audio helpers

gBackgroundMusicFiles =
{
	"gameplay1.ogg",
	"gameplay2.ogg",
	"gameplay3.ogg",
	"storyloop.ogg",
}

function StopBackgroundMusic()
	StartMusic("")
end

function BackgroundMusicNow(name)
	-- Play at random if nothing supplied
	if not name then
		name = gBackgroundMusicFiles[bsutil.random(4)]
	end
	StartMusic(name)
end

function BackgroundMusic(name, delay)
	-- Play at random if nothing supplied
	if not name then
		name = gBackgroundMusicFiles[bsutil.random(3)]
	end
	
	-- Random 6-10 second delay between songs if none specified
	if not delay then delay = bsutil.random(6,10) end
	
	QueueMusic(name, delay)
end

-------------------------------------------------------------------------------
-- Help helpers

function DisplayHelp(num, force, showAll)
--	local h = PrepareHelp(num,force,showAll)
--	if h then DisplayDialog{"ui/dialog.lua",body=h} end

	local h = "ui/help"..tostring(num)..".lua"
	DisplayDialog{h, num=num, showAll=showAll}
end

function HelpTitle()
	local num = gDialogTable.num
	local showAll = gDialogTable.showAll
	local t = "#"..GetString("help"..tostring(num).."_title")
	if showAll then t = t .. " ("..tostring(num).."/10)" end
	return Text { x=12+32,y=12,w=kMax,h=32, font=helpTitleFont, flags=kVAlignCenter+kHAlignLeft, label=t }
end

function HelpNavButtons()
	local bw = 38
	local xRight = 640-12-bw
	local xMid = xRight-bw
	local xLeft = xMid-bw
	local y = 12 - (bw-32)/2
	local buttons = nil
	if gDialogTable.showAll then
		local num = gDialogTable.num
		local force = true
		local showAll = gDialogTable.showAll
	
		buttons = {}
		
		if num > 1 then
			table.insert(buttons, Button { x=xLeft,y=y, scale=bw/64, graphics=backButtonGraphics, command=function() PopModal("help_dlg") DisplayHelp(num-1,force,showAll) end })
		end
		table.insert(buttons, Button { x=xMid,y=y, scale=bw/64, graphics=exitButtonGraphics, close=true })
		if num < 10 then
			table.insert(buttons, Button { x=xRight,y=y, scale=bw/64, graphics=nextButtonGraphics, command=function() PopModal("help_dlg") DisplayHelp(num+1,force,showAll) end })
		end
		
		buttons = Group(buttons)
	else
		buttons = Button { x=xRight,y=y, scale=bw/64, graphics=yesButtonGraphics, close=true }
	end
	
	return buttons
end

function PrepareHelp(num, force, showAll)
	local current = "help"..tostring(num)
	
--	local yBottom = 415-12-64
	local yBottom = 415-64
	
	local layout =
	{
		x=12,y=12,w=590,h=415,
--		Text { x=190,y=17,w=203,h=30,font=ButtonLabelFont,flags=kVAlignCenter+kHAlignCenter,label=current.."_title" },
		Text { x=100,y=17,w=383,h=30,font=ButtonLabelFont,flags=kVAlignCenter+kHAlignCenter,label=current.."_title" },
		
		AppendStyle { font=helpFont },
		dofile("ui/"..current..".lua"),
	}
	
	if showAll then
		if num > 1 then
			table.insert(layout, Button { x=295-32-64,y=yBottom, graphics=backButtonGraphics, command=function() PopModal("help_dlg") DisplayHelp(num-1,force,showAll) end })
		end
			
		table.insert(layout, Button { x=295-32,y=yBottom, graphics=exitButtonGraphics, close=true })
		table.insert(layout, Text { x=590-64,y=yBottom, w=62,h=64, font=ButtonLabelFont, label="#"..tostring(num).."/9", flags=kVAlignCenter+kHAlignCenter })

		if num < 9 then
			table.insert(layout, Button { x=295+32,y=yBottom, graphics=nextButtonGraphics, command=function() PopModal("help_dlg") DisplayHelp(num+1,force,showAll) end })
		end
	else
		table.insert(layout, Button { x=295-32,y=yBottom, graphics=exitButtonGraphics, close=true })
	end
	
	layout =
	{
--		x=105-12,y=92-12,w=590+24,h=415+24,
x=80,y=80,w=640,h=440,
		color = BrownColor,
		Window(layout),
	}
	return {name="help_dlg", Frame(layout)}
end
