--[[---------------------------------------------------------------------------
	Chocolatier Travel Support Functions
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- Leg base class (a leg is a direct single-vehicle trip from point to point)

LLeg =
{
	__tostring = function(t) return "{Leg:" .. t.startName .. "-" .. t.endName .. "}" end,
	_ByStart = {},
	_ByRoute = {},
}

-------------------------------------------------------------------------------
-- Leg "constructor"

function LLeg:new(t)
	devmessage(t.startName, "Leg defined with no start")
	devmessage(t.endName, "Leg defined with no end")
	if t.startName and t.endName then
		t = t or {} setmetatable(t, self) self.__index = self
		
		-- Keep global tables
		self._ByStart[t.startName] = t
		self._ByRoute[t.startName .. t.endName] = t
		
		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Accessors

function LLeg:ByRoute(startName, endName)
	local r = self._ByRoute[startName .. endName]
	if not r then
		r = self._ByRoute[endName .. startName]
		if r then r = r:CreateReverseLeg()
		else devmessage(r, "Leg from "..startName.." to "..endName.." not available or defined")
		end
	end
	return r
end

-------------------------------------------------------------------------------
-- Paths

function LLeg:GetPath()
	-- "path" will be a set of points including the start/end points and waypoints
	if not self.path then
		-- Start with a straight line
		local startPort = LPort:ByName(self.startName)
		self.path = { skip=true, {startPort.mapx,startPort.mapy} }

		-- Include waypoints if defined
		if self.waypoints then
			for _,p in ipairs(self.waypoints) do table.insert(self.path, p) end
			self.waypoints = nil
		end
		
		local endPort = LPort:ByName(self.endName)
		table.insert(self.path, {endPort.mapx,endPort.mapy} )
	end
	return self.path
end

function LLeg:CreateReverseLeg()
	newLeg = {
		startName = self.endName,
		endName = self.startName,
		vehicle = self.vehicle,
		time = self.time,
		price = self.price,
		path = { skip=true },
	}
	
	local rpath = self:GetPath()
	for _,p in ipairs(rpath) do table.insert(newLeg.path, 1, p) end
	
	return self:new(newLeg)
end

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- Route base class (a route is a complete trip with a number of legs)

LRoute =
{
	__tostring = function(t) return "{Route:" .. t.startName .. "-" .. t.endName .. "}" end,
	_ByStart = {},
	_ByRoute = {},
}

-------------------------------------------------------------------------------
-- Route "constructor"

function LRoute:new(t)
	devmessage(t.startName, "Route defined with no start")
	devmessage(t.endName, "Route defined with no end")
	if t.startName and t.endName then
		t = t or {} setmetatable(t, self) self.__index = self
		
		-- Keep global tables
		self._ByStart[t.startName] = t
		self._ByRoute[t.startName .. t.endName] = t
		
		return t
	else
		return nil
	end
end

function LRoute:Price()
	local p = gSim.travelPrices[self.endName]
	if not p then
		if not self.price then self.price = Simulator.travelcost * self.time end
		p = bsutil.floor(self.price * bsutil.random(80,120) / 100)
		gSim.travelPrices[self.endName] = p
	end
	
	-- Check for special quest variables for free trip
	if LQuest._Variable.airship > 0 then
		p = 0
	elseif LQuest._Variable.FreeTrip and LQuest._Variable.FreeTrip > 0 then
		p = 0
	end
	
	return  p
end

-------------------------------------------------------------------------------
-- Data description functions

function DefineRoute(t)
	t.startName = t.startName or t[1]
	t.endName = t.endName or t[2]
	t[1] = nil
	t[2] = nil

	-- A route with no "via" is a single-leg journey, define the leg
	if not t.via then
		t.time = t.time or 1
		t.price = t.price or t.time * Simulator.travelcost
		local leg = {
			startName = t.startName,
			endName = t.endName,
			waypoints = t.waypoints,
			vehicle = t.vehicle or "boat",
			time = t.time,
			price = t.price,
		}

		-- Last-minute map shifting		
		if leg.waypoints then
			for _,p in ipairs(leg.waypoints) do
				p[2] = p[2] + LPort.mapYOffset
			end
		end
		
		leg = LLeg:new(leg)
		t.waypoints = nil
		t.vehicle = nil
		t.legs = { leg }
	end
	
	-- Create the defined route
	return LRoute:new(t)
end

-------------------------------------------------------------------------------
-- Accessors

function LRoute:ByRoute(startName, endName)
	local r = self._ByRoute[startName .. endName]
	if not r then
		r = self._ByRoute[endName .. startName]
		if r then r = r:CreateReverseRoute()
		else devmessage(r, "Route from "..startName.." to "..endName.." not available or defined")
		end
	end
	return r
end

-------------------------------------------------------------------------------
-- Route calculation

function LRoute:GetLegs()
	if not self.legs then
		local t = 0
		local p = 0

		-- Get the first leg for this trip
		assert(self.via)
		local leg = LLeg:ByRoute(self.startName, self.via)  assert(leg)
		t = t + leg.time
		p = p + leg.price
		self.legs = { leg }
		
		-- Build from there, recursively
		local onward = LRoute:ByRoute(self.via, self.endName)  assert(onward)
		onward = onward:GetLegs()  assert(onward)
		for _,leg in ipairs(onward) do
			table.insert(self.legs, leg)
			t = t + leg.time
			p = p + leg.price
		end

		self.via = nil
		self.time = self.time or t
		self.price = self.price or p
	end
	return self.legs
end

function LRoute:CreateReverseRoute()
	local rev =
	{
		startName = self.endName,
		endName = self.startName,
		time = self.time,
		price = self.price,
		legs = {}
	}
	
	-- Reverse the leg ordering
	local rlegs = self:GetLegs()
	for _,l in ipairs(rlegs) do
		table.insert(rev.legs, 1, LLeg:ByRoute(l.endName, l.startName))
	end
	
	return self:new(rev)
end

function LRoute:PostProcessAll()
	-- Pre-calculate all legs and routings
	for _,l in pairs(LLeg._ByRoute) do l:GetPath() end
	for _,r in pairs(self._ByRoute) do r:GetLegs() end
	
	-- Create reversed versions of any routes that haven't already been reverse-defined
	for _,r in pairs(self._ByRoute) do LRoute:ByRoute(r.endName, r.startName) end
end

-------------------------------------------------------------------------------
-------------------------------------------------------------------------------
-- Travel functions

function Simulator:InitiateTravel(to)
	UtilDisableWindow("map")
	UtilDisableWindow("recipe")
	UtilDisableWindow("pause")
	
	gTravelCharacterEncounter = false
	local p = 0
	gTravelPrice = 0
	local from = self.port
	
	local route = LRoute:ByRoute(from.name, to.name)
	assert(route)

	if LQuest._Variable.airship > 0 then
		-- Airship: free direct travel in half the time
		local time = bsutil.floor(route.time/2 + 0.5)
		route =
		{
			startName = from.name,
			endName = to.name,
			price = 0,
			time = time,
			legs = {
				{ time=time, vehicle="airship", path={ skip=false, {from.mapx,from.mapy},{to.mapx,to.mapy}} }
			},
		}
	else
		-- Make sure paths are prepared
		route:GetLegs()
		
		-- Check price
		p = route:Price()
		if LQuest._Variable.FreeTrip and LQuest._Variable.FreeTrip > 0 then
			p = 0
			LQuest._Variable.FreeTrip = LQuest._Variable.FreeTrip - 1
		end
	end
	
	gTravelPrice = p
	AnimateTravel(route)

	gTravelCharacterEncounter = nil
end

-- Trigger arrival at a port
function Simulator:EnterPort()
	gSim.port:EnterPort()
end

function Simulator:ArriveAtPort(port)
	if type(port) == "string" then port = LPort:ByName(port) end
	
	gTravelPrice = gTravelPrice or 0
	if gTravelPrice <= gSim.money then self:AdjustMoney(-gTravelPrice) end
	gTravelPrice = nil
	
	-- Queue background music
	BackgroundMusic()
	
	PlayerMessage(GetString("arrival", GetString(port.name)))
	self:SetPort(port)
	
	-- Let window manager take care of stuff, and queue up port entry
	table.insert(gCommandQueue,Simulator.EnterPort)
end
