--[[---------------------------------------------------------------------------
	Chocolatier Simulator
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

local kSimulatorVersion = 1
Simulator =
{
	travelcost = 150,				-- base cost per week of travel, global
}

-- Simulator support structures and functions
require("sim/utility.lua")

-- Order may be significant for these includes
require("sim/character.lua")
require("sim/item.lua")
require("sim/building.lua")
require("sim/shop.lua")
require("sim/market.lua")
require("sim/factory.lua")
require("sim/port.lua")
require("sim/producttype.lua")
require("sim/quest.lua")
require("sim/travel.lua")

-------------------------------------------------------------------------------
-- Define a Simulator object, with a "new" constructor
function Simulator:new(t)
	t = t or {} setmetatable(t, self) self.__index = self
	return t
end

function Simulator:Initialize()
	-- Simulator data
	dofile("assets/data/items.lua")
	dofile("assets/data/ports.lua")
	dofile("assets/data/travelers.lua")
	dofile("assets/data/empty.lua")
	dofile("assets/data/routes.lua")
	dofile("assets/data/quests.lua")
	
	-- Convert named Item/ProductType references, collect items by Type
	LItem:PostProcessAll()
	-- Collect (or build) all route information
	LRoute:PostProcessAll()
	-- Process all buildings - convert Character references
	LBuilding:PostProcessAll()
	-- Process Port data - flag Buildings and Item references
	LPort:PostProcessAll()
	-- Convert string-referenced Quests and Characters
	LQuest:PostProcessAll()
end

-- The simulator state, a global table with various additional pieces
-- initialized in other simulator-related data structures
function Simulator:Reset(name)
	gFirstMissWarning = true

	gSim = Simulator:new
	{
--		mode = mode,					-- player's game mode ("empire" or "free") will be set by further function
		player = name,					-- player name
		companyName = nil,				-- company name
		logo = nil,						-- company logo information
		rank = 0,						-- player rank (0="apprentice")
		time = 0,						-- current sim time in weeks
		days = 0,						-- current sim "sub-time" ("days")
		money = 0,						-- current cash balance
		port = nil,						-- current port (not just name)
		quest = nil,					-- currently-active quest
		questTimer = 0,					-- time last quest completed
		haggle = true,					-- is haggling available in current port (one shot per visit)
		messages = {},					-- player message output lines (large amount)
		portPrices = {},				-- current port pricing info (set of item name = price)
		travelPrices = {},				-- current travel pricing info (set of port = price)
		inventory = {},					-- current player inventory (set of item name = count)
		purchasePrices = {},			-- purchase purchase prices (set of item name = price)
		purchasePorts = {},				-- purchase purchase locations (set of item name = port name)
		seen = {},						-- location of where items seen (set of item name = port name)
		seenPrices = {},				-- last prices on items seen (set of item name = price)
		usetime = {},					-- last time ingredient was used, or product was created, for expiration (set of item name = time)
		sold = {},						-- total counts of products sold (product = count)
		medals = {},					-- medals received
		nextTipTime = 0,				-- eligible time for next tip
		tips = {},						-- currently-active pricing tips
		facCount = 0,					-- total number of factories owned
		shopCount = 0,					-- total number of shops owned

		mcWeeks = nil,					-- weeks when player reached MC
		mcMoney = nil,					-- money when player reached MC

		-- Statistics (mostly for FirstPeek)
		minigames = 0,					-- total number of times minigame was played
		totalQuests = 0,				-- total number of quests completed
		totalMade = 0,					-- total number of ALL products made
		totalSold = 0,					-- total number of ALL products sold
		recipeCount = 0,				-- total number of recipes known
		lastQuestEnd = 0,				-- end time (weeks) of previous quest
		shortages = {},					-- ingredient shortage statistics
	}

	-- Clear medals table
	gSim:ResetMedals()
	
	-- Reset temporary game data
	LBuilding:ResetAll()
	LItem:ResetAll()
	LPort:ResetAll()
	LQuest:ResetAll()
end

function Simulator:ResetEmpireMode()
	self.mode = "empire"
	
	self.port = LPort:ByName("sanfrancisco")
	self.port.available = true
	self.port.visited = true
	self.portName = "sanfrancisco"
	
	local f = LFactory:ByPort("sanfrancisco")
	f:MarkOwned()
	f:SetConfiguration("basebars", LFactory.defaultProductionCount)
	
	self:FreshGame()
end

function Simulator:ResetFreePlayMode()
	self:ResetEmpireMode()
	self.mode = "free"
	self.money = 10000
	
--	-- Make all non-hidden ports available
--	for p in LPort:AllPorts() do
--		if not p.hidden then p.available = true end
--	end
	-- Default to SF, NY, and Merida free
	local p = LPort:ByName("sanfrancisco")
	if p then p.available = true end
	p = LPort:ByName("newyork")
	if p then p.available = true end
	p = LPort:ByName("merida")
	if p then p.available = true end
	
	for p in LItem:AllProducts() do
	-- Give the player all flagged recipes
--		if p.free then p:EnableRecipe() end
	-- Give the player ALL recipes
		p:EnableRecipe()
	end
	
	-- Adjust rank
--	self:AdjustRank(3)
end

-------------------------------------------------------------------------------
-- Game Save/Load

function CopyTable(src, dest)
	if type(src) == "table" and type(dest) == "table" then
		for k,v in pairs(src) do
			dest[k] = v
		end
	end
end

function ConvertToString(t, outString)
	for k,v in pairs(t) do
		-- If key is a number, just leave it off
		local key = k.."="
		if type(k) == "number" then key = "" end
		
		if type(v) == "string" then
			table.insert(outString, string.format("%s%q,", key, v))
		elseif type(v) == "number" or type(v) == "boolean" then
			table.insert(outString, string.format("%s%s,", key, tostring(v)))
		elseif type(v) == "table" then
			table.insert(outString, string.format("%s{", key))
			ConvertToString(v, outString)
			table.insert(outString, "},")
		end
	end
end

function Simulator:SaveGame()
	assert(self.player == GetCurrentUserName())
	if self.mode then
		-- Turn the game state into a simple table, convert it to a string, write it to prefs
		local t = self:SaveGameTable()
		local outString = { "return {" }
		ConvertToString(t, outString)
		table.insert(outString, "}")
		local s = table.concat(outString)
--		local s = table.concat(outString, "\n")
--		bsutil.StringToClipboard(s)
	
		gAutoSaveTime = UtilCurrentTime()	-- Note current time for auto-save
		SaveGameString(self.mode, s)		-- Save the resulting string for later load
	end
end

function Simulator:SaveGameTable()
	if self.port then self.portName = self.port.name
	else self.portName = nil
	end
	
	local questName = nil
	local qStartTime = nil
	if self.quest then
		questName = self.quest.name
		qStartTime = self.quest.startTime
	end
	
	-- Start with the basic sim information
	local t =
	{
		version = kSimulatorVersion,
		mode = self.mode,
--		player = self.player,
		companyName = self.companyName,
		logo = self.logo,
		rank = self.rank,
		time = self.time,
		days = self.days,
		money = self.money,
		portName = self.portName,
		questName = questName,
		questTimer = self.questTimer,
		qStartTime = qStartTime,
		haggle = self.haggle,
		portPrices = self.portPrices,
		travelPrices = self.travelPrices,
		inventory = self.inventory,
		purchasePrices = self.purchasePrices,
		purchasePorts = self.purchasePorts,
		seen = self.seen,
		seenPrices = self.seenPrices,
		usetime = self.usetime,
		sold = self.sold,
		nextTipTime = self.nextTipTime,
		tips = self.tips,
		medals = self.medals,
		mcWeeks = self.mcWeeks,
		mcMoney = self.mcMoney,

		minigames = self.minigames,
		totalQuests = self.totalQuests,
		totalSold = self.totalSold,
		totalMade = self.totalMade,
--		recipeCount = self.recipeCount,		-- automatically recalculated on load
		lastQuestEnd = self.lastQuestEnd,
	}

	-- Save Ten Messages
	if gSim.messages then
		t.saveMessages = {}
		local i = table.getn(gSim.messages) - 10
		if i < 1 then i = 1 end
		for i=i,table.getn(gSim.messages) do
			table.insert(t.saveMessages, gSim.messages[i])
		end
	end

	-- First Peek
	if gFirstPeek then
		t.shortages = self.shortages
		realTime = cFP_CurrentTime() - (gSim.questStartTime or 0) + (gSim.questPreviousSession or 0)
		t.questPreviousSession = realTime
	end
	
	-- Save the states of additional sim data, as necessary
	LBuilding:SaveGameTable(t)
	-- Characters: not necessary
	LItem:SaveGameTable(t)
	LPort:SaveGameTable(t)
	-- Product Types: not necessary
	LQuest:SaveGameTable(t)
	-- Routes: not necessary
	
	return t
end

function Simulator:RetrieveGameTable(mode)
	local t = nil
	local s = LoadGameString(mode)			-- Restore a string from disk
	gAutoSaveTime = UtilCurrentTime()		-- Note current time for auto-save
	if s and s ~= "" then
		local f = loadstring(s)
		if type(f) == "function" then t = f() end
	end
	return t
end

function Simulator:LoadGame(mode)
	-- Load string from prefs, convert it to a table, and restore state
	local loaded = false
	if mode then
		local t = self:RetrieveGameTable(mode)
		if t then
			self:LoadGameTable(t)
			loaded = true
			self:AdjustRank(self.rank)
		end

		-- Synchronize available ports in Free Play mode...
		if mode == "free" then
			local story = self:RetrieveGameTable("empire")
			if story then
				for p in LPort:AllPorts() do p.available = false end
				for name,_ in pairs(story.ports) do
					self:EnablePort(name)
				end

				self.rank = story.rank
				self.logo = story.logo
				self.companyName = story.companyName
				if story.quest_variables.airship > 0 then
					LQuest._Variable.airship = story.quest_variables.airship
				end
			end

			-- In free play mode, SF/NY/Merida always available
			self:EnablePort("sanfrancisco")
			self:EnablePort("newyork")
			self:EnablePort("merida")

			-- Release truffle powder regardless of rank...
			local tp = LItem:ByName("truffle_powder")
			for port in LPort:AllPorts() do
				if port.availability["cacao"] then port.availability["truffle_powder"] = tp end
			end
		end
	end
	SetMoneyDisplay(self.money)
	return loaded
end

function Simulator:LoadGameTable(t)
--	self.player = t.player
	self.player = GetCurrentUserName()
	
	self.mode = t.mode
	self.companyName = t.companyName
	self.logo = t.logo
	self.rank = t.rank
	self.time = t.time
	self.days = t.days
	self.money = t.money
	self.portName = t.portName or "sanfrancisco"
	self.port = LPort:ByName(self.portName)
	if t.questName then
		local q = LQuest:ByName(t.questName)
		q:MarkActive(t.qStartTime)
	end
	self.questTimer = t.questTimer
	self.haggle = t.haggle

	self.mcWeeks = t.mcWeeks
	self.mcMoney = t.mcMoney

	-- FirstPeek statistics
	self.minigames = t.minigames
	self.totalQuests = t.totalQuests
	self.totalSold = t.totalSold
	self.totalMade = t.totalMade
	self.lastQuestEnd = t.lastQuestEnd
	if type(t.shortages) == "table" then CopyTable(t.shortages, self.shortages) end

	self.messages = {}
	if type(t.saveMessages) == "table" then
		for _,m in ipairs(t.saveMessages) do PlayerMessage(m) end
	end

	CopyTable(t.portPrices, self.portPrices)
	CopyTable(t.travelPrices, self.travelPrices)
	CopyTable(t.inventory, self.inventory)
	CopyTable(t.purchasePrices, self.purchasePrices)
	CopyTable(t.purchasePorts, self.purchasePorts)
	CopyTable(t.seen, self.seen)
	CopyTable(t.seenPrices, self.seenPrices)
	CopyTable(t.usetime, self.usetime)
	CopyTable(t.sold, self.sold)
	CopyTable(t.medals, self.medals)

	self.nextTipTime = t.nextTipTime
	if type(t.tips) == "table" then
		for _,tip in ipairs(t.tips) do
			local temp = {}
			for k,v in pairs(tip) do
				temp[k] = v
			end
			table.insert(self.tips, temp)
		end
	end
	
	-- Restore the states of additional sim data
	LBuilding:LoadGameTable(t)
	LItem:LoadGameTable(t)
	LPort:LoadGameTable(t)
	LQuest:LoadGameTable(t)

	self:FreshGame(t)
end

function Simulator:FreshGame(t)
	-- Make sure port pricing is set
	if not t then
		self.port:PreparePricing(self.portPrices)
		self:ApplyTipPricing()
	end

	-- First Peek
	if gFirstPeek then
		self.questStartTime = cFP_CurrentTime()
		if t then self.questPreviousSession = t.questPreviousSession end
	end
end

-------------------------------------------------------------------------------
-- Lose condition

function Simulator:CheckLoseCondition()
	local keepPlaying = true
	
	-- Basic fail state: <$200 with no products
	if self.money < 200 and not self:AnyProductsInInventory() then
		-- Check factory production
		self:ProjectProduction()
		local bad = true
		for f in LFactory:AllFactories() do
			if f:Running() then bad = false end
		end
		
		-- Any other ways to get money/stuff?
		-- e.g. active quests ending in current port?
		--      active quests anywhere and ability to travel free (airship)?
		
		if bad then
			local cont = DisplayDialog { "ui/yndialog.lua", body="game_over" }
			if cont == "no" then
				keepPlaying = false
				bsutil.ResetGame(gSim.mode)
				SwapToModal("ui/mainmenu.lua")
			end
		else
			-- At least one factory is running, so give the sim a tick
			self:Tick()
		end
	end
	return keepPlaying
end

-------------------------------------------------------------------------------
-- Simulator ticking and time

-- Here's the score frozen at Master Chocolatier
function Simulator:GetMCScore()
	local weeks = self.mcWeeks or self.time or 1
	if weeks > 999999999 then weeks = 999999999 end
	if weeks <= 0 then weeks = 1 end
	
	local money = self.mcMoney or self.money or 0
	if money > 999999999 then money = 999999999 end
	if money <= 0 then money = 0 end
	
	local score = bsutil.floor(money / weeks + 0.5)
	if score > 999999999 then score = 999999999 end
	
	return score,weeks,money
end

-- Here's the score that continues moving past Master Chocolatier
function Simulator:GetScore()
	-- 4/12/07: We've decided to go back to freezing MC score in Empire mode...
	-- 4/17/07: ...and to make the $ be your high score for Free Play mode
	if self.rank == 4 and gSim.mode == "empire" then
		local score,weeks,money = self:GetMCScore()
		return score,weeks,money
	else
		local weeks = self.time or 1
		if weeks > 999999999 then weeks = 999999999 end
		if weeks <= 0 then weeks = 1 end
		
		local money = self.money or 0
		if money > 999999999 then money = 999999999 end
		if money <= 0 then money = 0 end
		
		local score = money
		if gSim.mode == "empire" then score = bsutil.floor(money / weeks + 0.5) end
		if score > 999999999 then score = 999999999 end
		
		return score,weeks,money
	end
end

function Simulator:GetTimeString()
	local year = self:CurrentYear()
	local week = self.time - 52 * year
	local month = bsutil.floor(week * 12 / 52)
	local day = bsutil.floor((week - month * 52 / 12) * 7)
	if day == 0 then day = 1
	elseif day > 28 then day = 28
	end
	return bsutil.GetVariableString("time", { year=year+1880, day=day, month = "month_" .. (month + 1) })
end

-- return current simulator year (0-based)
function Simulator:CurrentYear()
	return bsutil.floor(self.time / 52)
end

-- return current week (0..51) within the year
function Simulator:CurrentWeek()
	return self.time - 52 * self:CurrentYear()
end

-- return current month (0..11) within the year
function Simulator:CurrentMonth()
	return bsutil.floor(self:CurrentWeek() * 12 / 52)
end

function Simulator:SubTick(days)
	-- Do a "day" sub-tick... when enough days have accumulated, tick off a week
	days = days or 1
	self.days = self.days + days
	local ok = true
	while ok and self.days >= 4 do
		self.days = self.days - 4
		ok = self:Tick()
	end
	if ok then UpdateStandardUI() end
end

function Simulator:Tick(weeks)
	local ok = true
	weeks = weeks or 1
	for i = 1,weeks do
		self.time = self.time + 1
		if self.time/52 == bsutil.floor(self.time/52) then PlayerMessage(GetString("newyear"))
		else PlayerMessage(GetString("tick"))
		end
		
		----------> FACTORY PRODUCTION PROCESSING
		-- Run factories
		local productionStop = false
		for f in LFactory:AllFactories() do
			if f.owned then
				-- Let each factory do its thing, returns number created
				local count = f:ProductionRun()

				-- FirstPeek statistics
				self.totalMade = self.totalMade + count

				-- Notify player if production has stopped
				if count == 0 then productionStop = true end
			end
		end
		if productionStop then PlayerMessage(GetString("factory_stop")) end
		
		----------> INGREDIENT EXPIRATION
		for name,count in pairs(self.inventory) do
			if self.usetime[name] and count > 0 then
				-- After 16 weeks of non-use, expire ingredients over 4 weeks
				local d = self.time - self.usetime[name] - 16
				if d > 0 then
					if d == 1 then count = 3 * count / 4		-- lose 1/4
					elseif d == 2 then count = 2 * count / 3	-- lose another 1/4 (2/3 * 3/4 = 1/2 of orignal)
					elseif d == 3 then count = count / 2		-- another 1/4 (1/2 * 1/2 = 1/4 of original)
					else count = 0 self.usetime[name] = nil		-- all gone
					end
					
					self.inventory[name] = bsutil.floor(count + 0.5)
					PlayerMessage(GetString("ing_expire", GetString(name)))
				end
			end
		end
		
		-- FirstPeek
		if gFirstPeek and self.mode == "empire" then
			if self.time / 6 == bsutil.floor(self.time / 6) then
				FP_DumpSummary()
				FP_DumpInventory()
			end
		end
	end
	
	-- Project future factory production
	self:ProjectProduction()
end

function Simulator:TravelTick(sting)
	-- Player may encounter a fellow traveler at random (when rank > 0)
	-- Make characters mode-of-travel specific?
	if self.rank > 0 and not gTravelCharacterEncounter then
		local b = LBuilding:ByName("travel")
		assert(b)
		local odds = 20
		if b.encounterOdds then odds = b.encounterOdds end
		
		local allow = false
		if bsutil.random(100) <= odds then allow = true end

		if self:QuestsEnabled() then
			-- See if there is an active quest with a quest-ender in this building...
			if self.quest and b:ContainsCharacter(self.quest.ender) then
				-- Don't allow them until end time is met
				allow = (self.quest.goals.endTime < self.time)
			end
		end
		
		if allow then
			-- Only allow one travel encounter per trip (this is reset in InitiateTravel)
			gTravelCharacterEncounter = true
			PauseTravel()
			b:UISelect()
			ResumeTravel()
		end
	end
	
	-- Tick the simulator after interacting with the character
	self:Tick()
--	UpdateStandardUI()

	-- Queue the arrival sting if signalled
	if sting then StartMusic(sting.."_sting.ogg") end
end

function Simulator:GetSeason()
	local season = 0
	local year = bsutil.floor(self.time / 52)
	local week = self.time - year * 52
	if week > 2 and week < 6 then season=1				-- Valentine's Day Feb 14 (day 45 - week 6.5)
	elseif week > 10 and week < 14 then season=2		-- Easter around April 7 (day 97 - week 13.9)
	elseif week > 39 and week < 43 then season=3		-- Halloween October 31 (day 304 - week 43.4)
	elseif week > 48 and week < 52 then season=4		-- Christmas December 25 (day 359 - week 51.3)
	end
	return season
end

-------------------------------------------------------------------------------
-- Factory Production

function Simulator:ProjectProduction()
	-- Collect total product needs
	local needs = {}
	for f in LFactory:AllFactories() do
		if f.owned then f:ProductionNeeds(needs) end
	end
	
	-- Now compute the maximum duration of each ingredient given current inventory
	local weeks = {}
	for name,count in pairs(needs) do
		local inv = gSim.inventory[name] or 0
		if count == 0 then weeks[name] = 0
		else weeks[name] = inv / count
		end
	end
	
	-- And pass the total list back on to the factories
	for f in LFactory:AllFactories() do
		if f.owned then f:EstimateProduction(weeks) end
	end
end

-------------------------------------------------------------------------------
-- Tips

local function RandomAvailablePort()
	-- Pick a location from all available ports
	local portOptions = {}
	for port in LPort:AllPorts() do
		if port ~= gSim.port and port.available then table.insert(portOptions, port) end
	end
	
	-- If completely stuck, offer a tip for the current port
	if table.getn(portOptions) == 0 then return gSim.port
	else return portOptions[bsutil.random(table.getn(portOptions))]
	end
end

local ingUpEventList = { "ev_fire","ev_strike","ev_ing_priceup01","ev_ing_priceup02","ev_ing_priceup03","ev_ing_priceup04","ev_ing_priceup05" }
local ingDownEventList = { "ev_glut","ev_ing_pricedown01","ev_ing_pricedown02","ev_ing_pricedown03","ev_ing_pricedown04","ev_ing_pricedown05" }
function Simulator:GetIngredientTip()
	local tip = {}

	-- Pick an ingredient from all ingredients available at a random port
	tip.port = RandomAvailablePort()
	local available = {}
	for i,_ in pairs(tip.port.availability) do table.insert(available, i) end
	tip.ing = available[bsutil.random(table.getn(available))]
	
	-- Pick tip text
	if bsutil.random(100) > 50 then
		tip.up = true
		tip.key = ingUpEventList[bsutil.random(table.getn(ingUpEventList))]
	else
		tip.down = true
		tip.key = ingDownEventList[bsutil.random(table.getn(ingDownEventList))]
	end
	
	tip.port = tip.port.name
	tip.endTime = self.time + 10
	return tip
end

local prodUpEventList = { "ev_demand","ev_prod_priceup01","ev_prod_priceup02","ev_prod_priceup03","ev_prod_priceup04","ev_prod_priceup05", }
local prodDownEventList = { "ev_tired","ev_prod_pricedown01","ev_prod_pricedown02","ev_prod_pricedown03","ev_prod_pricedown04","ev_prod_pricedown05" }
function Simulator:GetProductTip()
	local tip = {}
	
	-- Pick a recipe and drive demand in a random port
	tip.port = RandomAvailablePort()
	local known = {}
	-- Pick a recipe from all known recipes
--	for i in LItem:AllProducts() do if i.known then table.insert(known, i) end end
	-- Pick a recipe currently in production
	for f in LFactory:AllFactories() do if f.owned and f.product then table.insert(known, f.product) end end
	if table.getn(known) == 0 then tip.prod = "basebars"
	else tip.prod = known[bsutil.random(table.getn(known))]
	end
	
	-- Pick tip text
	if bsutil.random(100) > 50 then
		tip.up = true
		tip.key = prodUpEventList[bsutil.random(table.getn(prodUpEventList))]
	else
		tip.down = true
		tip.key = prodDownEventList[bsutil.random(table.getn(prodDownEventList))]
	end
	
	tip.port = tip.port.name
	tip.prod = tip.prod.name
	tip.endTime = self.time + 10
	return tip
end

function Simulator:GetSeasonalTip(s)
	local tip = {}
	tip.key = "tip_season"..tostring(s)
	return tip
end

function Simulator:GetTip()
	if self.time >= self.nextTipTime then
		local tip = nil
		self.nextTipTime = self.time + 4
		local r = bsutil.random(100)
		local season = self:GetSeason()
		if season > 0 then
			-- seasonal tip possible: 33/33/33
			if r > 66 then tip = self:GetIngredientTip()
			elseif r > 33 then tip = self:GetProductTip()
			else tip = self:GetSeasonalTip(season)
			end
		else
			-- NO seasonal tip possible: 50/50
			if r > 50 then tip = self:GetIngredientTip()
			else tip = self:GetProductTip()
			end
		end
		
		-- Some tips are generic, most are port- or item- specific
		if tip and (tip.port or tip.prod or tip.ing) then
			table.insert(self.tips, tip)
		end
		if tip then return bsutil.GetVariableString(tip.key, tip)
		else return nil
		end
	end
	return nil
end

function Simulator:ApplyTipPricing()
	-- Check and apply tips
	local n = 1
	while n <= table.getn(self.tips) do
		local t = self.tips[n]
		if t.endTime < self.time then
			-- Expired tip
			table.remove(self.tips, n)
		else
			n = n + 1
			if t.port == self.port.name then
				if t.ing then
					local i = LItem:ByName(t.ing)
					if t.up then self.portPrices[t.ing] = bsutil.floor(i.high * 1.2)
					elseif t.down then self.portPrices[t.ing] = bsutil.floor(i.low * .8)
					end
					if self.portPrices[t.ing] < 1 then self.portPrices[t.ing] = 1 end
				elseif t.prod then
					local i = LItem:ByName(t.prod)
					if t.up then self.portPrices[t.prod] = bsutil.floor(i.high * 1.2)
					elseif t.down then self.portPrices[t.prod] = bsutil.floor(i.low * .8)
					end
					if self.portPrices[t.prod] < 1 then self.portPrices[t.prod] = 1 end
				end
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Travel

function Simulator:SetPort(port)
	if type(port) == "string" then port = LPort:ByName(port) end
	
	if self.port ~= port then
		self.port = port
		self.portName = port.name
		
		-- Reset the "day" sub-ticker and "haggle" flag
		self.days = 0
		self.haggle = true

		-- Rebuild the portPrices table for this port
		self.portPrices = {}
		port:PreparePricing(self.portPrices)
		self:ApplyTipPricing()

		-- Rebuild the travelPrices table for this port
		self.travelPrices = {}
-- Allow LRoute to do this "on demand"
--		port:PrepareTravelPricing(self.travelPrices)
	end
end

-------------------------------------------------------------------------------
-- Handy iterator functions

-- Return recipes, in order, known by the player
function KnownRecipes(type)
	local i = 0
	local n = LItem:ProductCount()
	return function()
		i = i + 1
		while (i <= n) do
			local p = LItem:ProductByIndex(i)
			if p.known and (p.type == type or not type) then return p end
			i = i + 1
		end
		return nil
	end
end

-- Return Ingredients, in order, in the player's inventory
function Simulator:PlayerInventoryIngredients()
	local i = 0
	local n = LItem:IngredientCount()
	return function()
		i = i + 1
		while (i <= n) do
			local ing = LItem:IngredientByIndex(i)
			if self.inventory[ing.name] and self.inventory[ing.name] > 0 then return ing end
			i = i + 1
		end
		return nil
	end
end

-- Return Products, in order, in the player's inventory
function Simulator:PlayerInventoryProducts()
	local i = 0
	local n = LItem:ProductCount()
	return function()
		i = i + 1
		while (i <= n) do
			local prod = LItem:ProductByIndex(i)
			if self.inventory[prod.name] and self.inventory[prod.name] > 0 then return prod end
			i = i + 1
		end
		return nil
	end
end

-- Check for products in the player's inventory, return true if there are any
function Simulator:AnyProductsInInventory()
	for name,count in pairs(self.inventory) do
		local i = LItem:ByName(name)
		if i and count > 0 and i.recipe then return true end
	end
	return false
end

-------------------------------------------------------------------------------
-- Adjustments

function Simulator:AdjustMoney(amount, reason)
	self.money = self.money + amount
	SetMoneyDisplayTarget(self.money)
	UpdateScoreUI()
	if reason then
		if amount < 0 then amount = -amount end
		PlayerMessage(bsutil.GetVariableString(reason, {amount=Dollars(amount)}))
	end
end

function Simulator:AdjustRank(newRank, reason)
	self.rank = newRank
	if reason then PlayerMessage(GetString(reason, GetString("rank_"..newRank))) end
	
	if newRank == 3 then
		-- Release truffle powder...
		local tp = LItem:ByName("truffle_powder")
		for port in LPort:AllPorts() do
			if port.availability["cacao"] then port.availability["truffle_powder"] = tp end
		end
	end
end

function Simulator:AdjustInventory(name, amount, reason)
	local n = self.inventory[name] or 0
	self.inventory[name] = n + amount
	if self.inventory[name] == 0 then
		self.inventory[name] = nil
		self.usetime[name] = nil
	end
	
	-- reset expiration time for ingredients
	local i = LItem:ByName(name)
	if not i.recipe and amount > 0 then self.usetime[name] = self.time end
	
	-- if ingredients change, recalculate factory production
	if not i.recipe then self:ProjectProduction() end
	
	if reason and amount > 0 then PlayerMessage(bsutil.GetVariableString(reason, {item=name, amount=amount})) end
end

function Simulator:EnableRecipe(name, reason)
	local i = LItem:ByName(name)
	i:EnableRecipe()
	if reason then PlayerMessage(bsutil.GetVariableString(reason, {item=name})) end
end

function Simulator:EnablePort(name, reason)
	local p = LPort:ByName(name)
	p.available = true
	p.hidden = nil
	if reason then PlayerMessage(bsutil.GetVariableString(reason, {name=name})) end
end

-------------------------------------------------------------------------------
-- Conveniences

function Simulator:GetInventory(name)
	return self.inventory[name] or 0
end

function Simulator:QuestsEnabled()
	return self.mode == "empire"
end
