--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Shop
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	See external documentation.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Shop class definition, derived from LBuilding
LShop = { helpkey=5 } setmetatable(LShop, LBuilding)
function LShop.__tostring(t) return "{Shop:" .. tostring(t.name) .. "}" end

function Shop(t) return LShop:new(t) end

-------------------------------------------------------------------------------
-- Shop purchase

function LShop:MarkOwned()
	-- Record this shop as owned by the player and mark the port
	gSim.shopCount = gSim.shopCount + 1
	self.owned = true
	self.port.shopowned = true
end

-------------------------------------------------------------------------------

function LShop.SellProduct(item, count)
	-- track total numbers sold
	local oldsold = gSim.sold[item.name] or 0
	local newsold = oldsold + count
	gSim.sold[item.name] = newsold

	-- FirstPeek statistics
	gSim.totalSold = gSim.totalSold + count
	
	-- adjust inventory
	gSim:AdjustInventory(item.name, -count)
	
	local unit = item:GetPrice()
	local cost = count * unit
	local s = GetString("sell_notice", tostring(count), GetString(item.name), Dollars(cost), Dollars(unit), GetString(gSim.port.name))
	PlayerMessage(s)
	
	-- some useful messages based on sales
	if oldsold < 1500 and newsold > 1500 then
		PlayerMessage(GetString("shop_demandbottom", GetString(item.name)))
	elseif oldsold < 1000 and newsold > 1000 then
		PlayerMessage(GetString("shop_demandlow", GetString(item.name)))
	elseif oldsold < 500 and newsold > 500 then
		PlayerMessage(GetString("shop_demanddropping", GetString(item.name)))
	elseif oldsold == 0 then
		PlayerMessage(GetString("shop_newproduct", GetString(item.name)))
	end
	
	return cost
end

function LShop.Sell(c)
	local total = 0
	local more = false
	for item in gSim:PlayerInventoryProducts() do
		local inventory = gSim.inventory[item.name] or 0
		local count = tonumber(GetLabel(item.name)) or 0
		if count > inventory then
			more = true
			count = inventory
		end
		if count > 0 then
			total = total + LShop.SellProduct(item, count)
		end
		
		inventory = gSim.inventory[item.name] or 0
		if inventory == 0 then
			SetLabel(item.name, "-")
--			UtilDisableWindow(item.name.."_platter")
		else
			SetLabel(item.name, GetString("sell"))
		end
	end
	if more then
		local s = bsutil.GetVariableString("shop_toomany")
		DisplayDialog { "ui/okdialog.lua", body="#"..s }
--		c:UIModalDialog(bsutil.GetVariableString("shop_toomany"))
	end
	
	local newCount = 0
	for item in gSim:PlayerInventoryProducts() do
		local inventory = gSim.inventory[item.name] or 0
		newCount = newCount + inventory
	end
	
	if total > 0 then
		gSim:AdjustMoney(total)
		
		-- A "day" passes when the player makes a sale
--		gSim:SubTick()
		gTransactionMade = true

		UpdateStandardUI()
		UpdateDynamicWindow("receipt")
		
		return true
	end
	return false
end

function LShop.SellAll()
	local total = 0
	for item in gSim:PlayerInventoryProducts() do
		local inventory = gSim.inventory[item.name] or 0
		if inventory > 0 then
			total = total + LShop.SellProduct(item, inventory)
		end
	end
	if total > 0 then
		gSim:AdjustMoney(total)
		
		-- A "day" passes when the player makes a sale
--		gSim:SubTick()
		gTransactionMade = true
	end
end

function LShop:UISelect()
	local c,q = self:ChooseCharacter()
	
	if gSim:AnyProductsInInventory() then
		if q then c:UIHandleQuest(q) end
		self:UIDoShop(c)
		self:LeaveBuilding(c,q)
	elseif q then
		if q then c:UIHandleQuest(q) end
		self:LeaveBuilding(c,q)
	else
		c:UIModalDialog(GetString("shop_noproducts"))
	end
end

function LShop:UIDoShop(c)
	-- Decide which products to display -- up to 12, sorted by most recently produced then by price
	local productList = {}
	for item in gSim:PlayerInventoryProducts() do table.insert(productList,item) end
--[[
	table.sort(productList, function(p1,p2)
			local v1 = gSim.usetime[p1.name] or 0
			local v2 = gSim.usetime[p2.name] or 0
			v1 = v1 * 10000 + gSim.portPrices[p1.name]
			v2 = v2 * 10000 + gSim.portPrices[p2.name]
			return v1 > v2
		end)
	local maxProducts = 12
	if table.getn(productList) > maxProducts then
		productList[maxProducts+1] = nil
		table.setn(productList,maxProducts)
	end
]]--
	
	SetAmbient(kShopAmbient)
	DisplayDialog { "ui/buysell.lua", character=c, building=self, sound="charopen",
		type="sell", items=productList, background=ShopColor }
	SetAmbient(kPortAmbient)
end
