--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Quests
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A Quest is a set of goals to be achieved and a set of rewards received by
	the player when those goals are met.
	
	NOTE: Characters must be defined before Quests
--]]---------------------------------------------------------------------------

-- Quest class definition
LQuest =
{
	__tostring = function(t) return "{Quest:" .. t.name .. "}" end,
	_ByIndex = {},
	_ByName = {},
	_Forced = {},
	_Variable = {},
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefineQuest(t)
	return LQuest:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global lists

function LQuest:AllQuests()
	return bsutil.ArrayIterator(self._ByIndex)
end

function LQuest:ByName(name)
	return self._ByName[name]
end

function LQuest:ForcedQuests()
	return bsutil.ArrayIterator(self._Forced)
end

-------------------------------------------------------------------------------
-- "constructor"

function LQuest:new(t)
	assert(t.name, "Defining a Quest with no name")
	assert(not self._ByName[t.name], "Quest " .. t.name .. " already defined")
	assert(t.starter, "Defining a Quest with no starter: "..t.name)
	if t.forceComplete then 
		devmessage(not (t.goals and table.getn(t.goals) > 0), "forceComplete Quests should have no goals: "..t.name)
	end
	devmessage(not t.forceComplete or (t.forceComplete and not t.ender) or (t.forceComplete and t.ender == t.starter), "forceComplete Quests should not have a different ender: "..t.name)
	t = t or {} setmetatable(t, self) self.__index = self
	
	t.complete = false
	t.ender = t.ender or t.starter
	
	-- Make sure there are goals...
	t.goals = t.goals or {}
	
	-- Keep global tables
	self._ByName[t.name] = t
	table.insert(self._ByIndex, t)
	if t.forceOffer then table.insert(self._Forced, t) end
	
	-- FOR NOW: Write out a note if this may be eligible for forceComplete
--	if t.starter == t.ender and table.getn(t.goals) == 0 and not t.forceComplete then
--		DebugOut("Consider Quest for forceComplete: "..tostring(t.name))
--	end

	return t
end

-------------------------------------------------------------------------------
-- Data conversion

function LQuest:PostProcessAll()
	for name,q in pairs(self._ByName) do
		-- Cross-reference involved Characters
		if type(q.starter) == "string" then q.starter = LCharacter:ByName(q.starter) end
		if type(q.ender) == "string" then q.ender = LCharacter:ByName(q.ender) end
		q.starter:StartsQuest(q)

		-- Auto-require this quest for followups
		if type(q.followup) == "string" then
			local f = LQuest:ByName(q.followup)
			q.followup = f
			if not f.requirements then f.requirements = {} end
			
			if not f.requirements.quest then
				f.requirements.quest = q.name
			elseif type(f.requirements.quest) == "string" then 
				f.requirements.quest = { f.requirements.quest, q.name }
			else
				table.insert(f.requirements.quest, q.name)
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Quest dialog

-- Quest name key is "qn_<name>"
-- Introductory string is "qi_<name>"
-- Intermediate string is "qm_<name>"
-- Ending string "qe_<name>"
function LQuest:Name() return bsutil.GetVariableString("qn_" .. self.name, self) end
function LQuest:IntroString()
	local key = "qi_"..self.name
	local val = bsutil.GetVariableString(key, self)
	if val == key then return nil
	else return val
	end
end
function LQuest:SummaryString()
	local s = GetString("qs_" .. self.name)
	if s == "#####" then return ""
	else return s
	end
end
function LQuest:IntermediateString() return bsutil.GetVariableString("qm_" .. self.name, self) end
function LQuest:EndString()
	local key = "qe_"..self.name
	local val = bsutil.GetVariableString(key, self)
	if val == key then return nil
	else return val
	end
end

-------------------------------------------------------------------------------
-- Quest evaluation and rewards
-- See external docs for full and updated description
-- money = X		: require this amount of money / give player this amount
-- rank = N			: require the player be of this rank
-- maxrank = N		: require the player be at or below this rank
-- item = { i=c }	: require this many of this item / give player this many of this item
-- recipe = { "x" }	: require that player knows this recipe / give player this recipe
-- port = { "x" }	: require that player have port access / give player port access
-- quest = { "x" }	: require that the player has completed a quest / enable a quest
-- time = x			: require that game time >= x OR x amount of time passes

function LQuest:EvaluateConditions(t)
	if t.money and gSim.money < t.money then return false end
	if t.rank and gSim.rank < t.rank then return false end
	if t.maxrank and gSim.rank > t.maxrank then return false end
	
	if t.time and gSim.time < t.time then return false end
	if t.endTime and gSim.time < t.endTime then return false end
	
	if type(t.item) == "table" then
		for name,count in pairs(t.item) do
			if not gSim.inventory[name] then return false
			elseif gSim.inventory[name] < count then return false
			end
		end
	end
	
	if type(t.recipe) == "string" then
		local i = LItem:ByName(t.recipe)
		if not i.known then return false end
	elseif type(t.recipe) == "table" then
		for _,name in ipairs(t.recipe) do
			local i = LItem:ByName(name)
			if not i.known then return false end
		end
	end
	
	if type(t.port) == "string" then
		local p = LPort:ByName(t.port)
		if not p.available then return false end
	elseif type(t.port) == "table" then
		for _,name in ipairs(t.port) do
			local p = LPort:ByName(name)
			if not p.available then return false end
		end
	end
	
	if type(t.quest) == "string" then
		local q = LQuest:ByName(t.quest)
		if not q.complete then return false end
	elseif type(t.quest) == "table" then
		for _,name in ipairs(t.quest) do
			local q = LQuest:ByName(name)
			if not q.complete then return false end
		end
	end

	for _,f in ipairs(t) do
		if type(f) == "function" and not f() then return false end
	end
	
	return true
end

function LQuest:EvaluateRequirements()
	-- Hack in a quest delay without affecting the quest requirements
	if self.delayTime and gSim.time < self.delayTime then return false
	elseif self.requirements then return self:EvaluateConditions(self.requirements)
	else return true end
end

function LQuest:EvaluateGoals()
	if self.goals then return self:EvaluateConditions(self.goals)
	else return true end
end

function LQuest:ApplyChanges(t,c)
	if t.money then
		if t.money < 0 then gSim:AdjustMoney(t.money)
		else gSim:AdjustMoney(t.money, "quest_money")
		end
	end
	if t.rank then gSim:AdjustRank(t.rank, "promotion") end
	
	if type(t.item) == "table" then
		for name,count in pairs(t.item) do
			gSim:AdjustInventory(name, count, "quest_item")
		end
	end
	
	if type(t.recipe) == "string" then
		local p = LItem:ByName(t.recipe)
		p:EnableRecipe("recipe_learned")
		p.newrecipe=gSim.time
		gCurrentRecipeSelection = p
	elseif type(t.recipe) == "table" then
		for _,name in ipairs(t.recipe) do
			local p = LItem:ByName(name)
			p:EnableRecipe("recipe_learned")
			p.newrecipe=gSim.time
			gCurrentRecipeSelection = p
		end
	end
	
	if type(t.port) == "string" then
		gSim:EnablePort(t.port, "quest_port")
	elseif type(t.port) == "table" then
		for _,name in ipairs(t.port) do
			gSim:EnablePort(name, "quest_port")
		end
	end
	
	assert(not t.quest)
--[[
	if type(t.quest) == "string" then
		local q = LQuest:ByName(t.quest)
		q.available = true
	elseif type(t.quest) == "table" then
		for _,name in ipairs(t.quest) do
			local q = LQuest:ByName(t.quest)
			q.available = true
		end
	end
]]--
	
	for _,f in ipairs(t) do
		if type(f) == "function" then f(c) end
	end

	UpdateStandardUI()
	
	-- Check for medals
	local m = gSim:EvaluateMedals()
	if m then gSim:AwardMedal(m) end
end

function LQuest:ApplyGift()
	if self.gift then
		self:ApplyChanges(self.gift, self.starter)
	end
end

function LQuest:ApplyDenial()
	if self.denial then
		self:ApplyChanges(self.denial, self.starter)
	end
end

function LQuest:ApplyReward()
	if self.reward then
		self:ApplyChanges(self.reward, self.ender)
	end
end

-------------------------------------------------------------------------------
-- Quest activation and completion

function LQuest:Init()
	if self.init then
		if type(self.init) == "function" then
			self.init()
		else
			for _,f in ipairs(self.init) do
				if type(f) == "function" then f() end
			end
		end
	end
end

function LQuest:MarkActive(time)
	assert(not gSim.quest)
	gSim.quest = self
	
	self.startTime = time
	self.active = true
	self.complete = false
	self.delayTime = nil
	
	-- Compute the quest end time, if time is part of this quest
	if self.goals.time then
		self.goals.endTime = self.startTime + self.goals.time
	else
		self.goals.endTime = self.startTime
	end
	gSim.nextHintTime = gSim.time + 10
end

function LQuest:Activate()
	-- Put this in the active quests list and mark its start time
	self:MarkActive(gSim.time)
	if gFirstPeek then
		gSim.questStartTime = cFP_CurrentTime()
		gSim.questPreviousSession = 0
	end
	
--[[
	if not self.silent then
		local s = GetString("qn_"..self.name)
		if s ~= "#####" then
			s = self:Name()
--			PlayerMessage(GetString("quest_accept", s))
		end
	end
]]--
	self:ApplyGift()

	-- Echo quest instruction text to message log and switch ledger
	if gSim.quest then
		questString = gSim.quest:SummaryString()
		if questString and questString ~= "" then
			PlayerMessage(questString)
			ShowLedger("messages")
		end
	end

	-- For forceComplete quests, this is a one-shot story-progressing deal that ends as
	-- soon as it is displayed
	if self.forceComplete then
		self.ender:CompleteQuest(self)
	end
end

function LQuest:Reject()
	-- Player turns down the quest, do anything special
	if self.offeronce then self:MarkComplete() end
	self:ApplyDenial()
end

function LQuest:MarkComplete()
	if self.active then
		assert(gSim.quest == self)
		gSim.quest = nil
		if not self.forceComplete then gSim.questTimer = gSim.time end
	end
	
	self.active = nil
	if not self.repeatable then self.complete = true end
end

function LQuest:Complete()
	-- Record this for First Peek
	gSim.totalQuests = gSim.totalQuests + 1
	if gFirstPeek then
		local NoQuestInterval = self.startTime - gSim.lastQuestEnd
		local realTime = cFP_CurrentTime() - (gSim.questStartTime or 0) + (gSim.questPreviousSession or 0)
		realTime = realTime / 1000
		local totalTime = FP_TotalTimePlayed()
		totalTime = totalTime / 1000
		cFP_DumpEvent("QuestComplete",
			{
				gSim.time,
				"\""..self.name.."\"",
				realTime,
				gSim.time - self.startTime,
				NoQuestInterval,
				totalTime,
				gSim.totalQuests or 0,
				LQuest._Variable.QuestCounter or 0,
			}
		)
		lastQuestEnd = gSim.time
	end

	self:MarkComplete()
	
	if not self.silent then
		local s = self:Name()
--		PlayerMessage(GetString("quest_complete", s))
	end
	self:ApplyReward()
end

-------------------------------------------------------------------------------
-- Game Save/Load/Reset

function LQuest:DoReset()
	self.active = nil
	self.complete = nil
	self.startTime = nil
	self.endTime = nil
	self.delayTime = nil
end

function LQuest:ResetAll()
	for k,_ in pairs(LQuest._Variable) do
		LQuest._Variable[k] = 0
	end
	
	for q in LQuest:AllQuests() do
		q:DoReset()
	end
end

function LQuest:SaveGameTable(t)
	t.quests_complete = {}
	for name,q in pairs(self._ByName) do
		if q.complete then table.insert(t.quests_complete,name) end
	end
	
	t.quest_variables = LQuest._Variable
end

function LQuest:LoadGameTable(t)
--	LQuest:ResetAll()
	if type(t.quests_complete) == "table" then
		for _,name in ipairs(t.quests_complete) do
			local q = LQuest:ByName(name)
			if q then q:MarkComplete() end
		end
	end
	
	if type(t.quest_variables) == "table" then
		for k,v in pairs(t.quest_variables) do
			LQuest._Variable[k] = v
		end
	end
end

-------------------------------------------------------------------------------
-- Special quest functions

function DesignCompanyLogo()
	DoModal("ui/logobuilder.lua")
	-- Rebuild the port to display the new sign(s)
	SwapToModal("ui/portview.lua")
end

function OwnBuilding(name)
	local b = LBuilding:ByName(name)
	return function() b:MarkOwned() end
end

function IsOwned(name)
	local b = LBuilding:ByName(name)
	return function() return b.owned end
end

function HasProduct()
	return function() return gSim:AnyProductsInInventory() end
end

function OfferQuest(questName)
	local name = questName
	return function()
		local q = LQuest:ByName(name)
		devmessage(q and q.starter, "No quest or missing starter for " .. name)
		q.starter:UIHandleQuest(q)
	end
end

function DelayQuest(questName, delay)
	local name = questName
	local d = delay
	return function()
		local quest = LQuest:ByName(name)
		if quest then
			quest.delayTime = gSim.time + d
		else
			local message = "ERROR: Attempt to DelayQuest '"..tostring(name).."' - does not exist"
			DebugOut(message)
			DisplayDialog { "ui/okdialog.lua", body="#"..message }
		end
	end
end

function CharacterDialog(text,char)
	local textKey = text
	local charName = char
	return function(c)
		if charName then c = LCharacter:ByName(charName) end
		c:Converse(textKey)
	end
end

function LedgerMessage(text)
	local t = text
	return function() PlayerMessage(GetString(t)) end
end

function SetLedger(tab)
	return function() ShowLedger(tab) end
end

function ShowRecipes()
	return function() LItem:FullRecipeBook() end
end

function AwardMedal()
	local m = gSim:EvaluateMedals()
	if m then gSim:AwardMedal(m) end
end

function MCFanfare()
	gSim:AwardMedal(table.getn(Simulator.MedalData))
end

function LessMoneyThan(x)
	return function() return gSim.money < x end
end

function MoreMoneyThan(x)
	return function() return gSim.money > x end
end

function TutorialTimeWarp()
	gSim.time = gSim.time + 6
	UpdateStandardUI()
end

function RecipesKnown(type, count)
	local t = ProductType:ByName(type)
	devmessage(t, "RecipesKnown: Unknown type specified")
	local n = count
	return function() return t and t.knownCount >= n end
end

-------------------------------------------------------------------------------
-- Quest variable functions

function SetQuestVar(v,n)
	devmessage(type(v) == "string", "Quest variables must be in quotes")
	LQuest._Variable[v] = LQuest._Variable[v] or 0
	local var = v
	local val = n
	return function() LQuest._Variable[var] = val end
end

function IncrementQuestVar(v)
	devmessage(type(v) == "string", "Quest variables must be in quotes")
	LQuest._Variable[v] = LQuest._Variable[v] or 0
	local var = v
	return function()
		LQuest._Variable[var] = LQuest._Variable[var] and LQuest._Variable[var] + 1 or 1
	end
end

function QuestVarLessThan(v,n)
	devmessage(type(v) == "string", "Quest variables must be in quotes")
	LQuest._Variable[v] = LQuest._Variable[v] or 0
	local var = v
	local val = n
	return function() return LQuest._Variable[var] and LQuest._Variable[var] < val end
end

function QuestVarMoreThan(v,n)
	devmessage(type(v) == "string", "Quest variables must be in quotes")
	LQuest._Variable[v] = LQuest._Variable[v] or 0
	local var = v
	local val = n
	return function() return LQuest._Variable[var] and LQuest._Variable[var] > val end
end

function QuestVarEqualTo(v,n)
	devmessage(type(v) == "string", "Quest variables must be in quotes")
	LQuest._Variable[v] = LQuest._Variable[v] or 0
	local var = v
	local val = n
	return function()
		return LQuest._Variable[var] and LQuest._Variable[var] == val
	end
end

function SetQuestFlag(f)
	return SetQuestVar(f,1)
end

function ClearQuestFlag(f)
	return SetQuestVar(f,0)
end

function QuestFlagOn(f)
	return QuestVarEqualTo(f,1)
end

function QuestFlagOff(f)
	return QuestVarEqualTo(f,0)
end
