--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Product Types
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A n Item is something that can be bought or sold.
	"static" members:
		_ByName: table of all Items keyed by name
		_IngredientsByIndex: array of all Ingredients in order of creation
		_ProductsByIndex: array of all Products in order of creation
--]]---------------------------------------------------------------------------

ProductType =
{
	__tostring = function(i) return "{ProductType:" .. i.name .. "}" end,
	_ByName = {},
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefineProductType(t)
	t.name = t.name or t[1]
	t[1] = nil

	t.products = {}
	t.knownCount = 0
	
	return ProductType:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global lists

function ProductType:ByName(name)
	return self._ByName[name]
end

-------------------------------------------------------------------------------
-- "constructor"

function ProductType:new(t)
	devmessage(t.name, "Product Type defined with no name")
	devmessage(not self._ByName[t.name], "Product Type " .. t.name .. " already defined")
	if t.name and not self._ByName[t.name] then
		t = t or {} setmetatable(t, self) self.__index = self

		-- Keep global tables
		self._ByName[t.name] = t
		
		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Cross-Referencing

function ProductType:ProductsInType()
	return bsutil.ArrayIterator(self.products)
end

function ProductType:AddProduct(i)
	table.insert(self.products, i)
end

-------------------------------------------------------------------------------
-- Tracking

function ProductType:ResetKnownCount()
	self.knownCount = 0
end

function ProductType:IncrementKnownCount()
	self.knownCount = self.knownCount + 1
	
	-- Make sure we're counting right...
--[[	
	if gDevelopmentMode then
		local compare = 0
		for p in self:ProductsInType() do
			if p.known then compare = compare + 1 end
		end
		assert(compare == self.knownCount)
	end
]]--
end

function ProductType:DecrementKnownCount()
	self.knownCount = self.knownCount - 1
end
