--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Ports
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A Port is a possible player destination.
	
	See external documentation.
	
	hooks:
		PreparePricing(itemTable): alters an item pricing table keyed off the
			item name
--]]---------------------------------------------------------------------------

-- Port class definition
LPort =
{
	__tostring = function(t) return "{Port:" .. t.name .. "}" end,
	_ByName = {},
	_ByIndex = {},
	
	-- Last-minute changes to map position -- putting this in a central place
	mapYOffset = 17,
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefinePort(t)
	-- Last-minute map shifting		
	t.mapy = t.mapy + LPort.mapYOffset
	return LPort:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global Port lists

function LPort:AllPorts()
	return bsutil.ArrayIterator(self._ByIndex)
end

function LPort:ByName(portName)
	return self._ByName[portName]
end

-------------------------------------------------------------------------------
-- "constructor"

function LPort:new(t)
	devmessage(t.name, "Port defined with no name")
	devmessage(not self._ByName[t.name], "Port " .. t.name .. " already defined")
	if t.name and not self._ByName[t.name] then
		t = t or {} setmetatable(t, self) self.__index = self
		
		-- If not specifically un-available, port is available (and mark it visited too)
		if t.available == nil then t.available = true end
--		t.visited = t.visited or t.available
		
		-- Default layouts
		t.mapx = t.mapx
		t.mapy = t.mapy
		t.labelx = t.labelx or t.mapx + 34
		t.labely = t.labely or t.mapy

		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)

		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Data post-processing

function LPort:PostProcessAll()
	for _,port in ipairs(self._ByIndex) do
		-- Make sure the port has SOMEthing in the layout
		port.layout = port.layout or {}
		
		-- Flag the Buildings in the port (marked by the availability of SetPort)
		for _,b in ipairs(port.layout) do
			if b.SetPort then b:SetPort(port) end
		end

		-- Mark available ingredients with full data, not just "true"
		if port.availability then
			for key,_ in pairs(port.availability) do
				port.availability[key] = LItem:ByName(key)
			end
		end
	end
end		

-------------------------------------------------------------------------------
-- Handy iterations

-- Iterate over all defined ingredients, in defined order, available at this port
function LPort:AvailableIngredients()
	-- Can't just iterate over self.availabilty because it may not be in the
	-- prescribed order
	local i = 0
	local n = LItem:IngredientCount()
	return function()
		i = i + 1
		while (i <= n) do
			local ing = LItem:IngredientByIndex(i)
			if ing and self.availability[ing.name] then return ing end
			i = i + 1
		end
		return nil
	end
end

-------------------------------------------------------------------------------
-- Base Port Price and Availability randomization

function LPort:PreparePricing(priceTable)
	-- Check for seasonal pricing
	local season = gSim:GetSeason()

	for item in LItem:AllItems() do
		local price = bsutil.random(item.low, item.high)
		
		-- Apply product pricing rules
		-- NOTE: IF THESE RULES CHANGE, TWEAK THE SHOP OWNER FEEDBACK IN LSHOP:SELL
		if item.recipe then
			local sold = gSim.sold[item.name] or 0
			if sold <= 500 then
				-- Tend towards "middle" price over the first 500 sales
				price = price + (item.middle - price) * sold / 500
			elseif sold < 1500 or item.type == "truffle" then
				-- Tend towards the "lowest" price over the next 1000 sales
				price = item.middle + (item.lowest - item.middle) * (sold - 500) / 1000
			elseif item.type == "truffle" then
				-- Freeze truffles at the middle price
				price = item.middle
			else
				-- After selling 1500, the bottom drops out of the market...
				price = bsutil.floor(item.lowcost * .9)
			end
			
			-- Seasonal 10% bump
			if season > 0 then price = price * 1.1 end
			
			-- 20% markup at owned shops
			if self.shopowned then price = price * 1.2 end
			
			price = bsutil.floor(price + 0.5)
			if price > item.high then price = item.high end
		end
		
		priceTable[item.name] = price
	end
	
	-- HACK: For tutorial, guarantee specific prices
	if gSim.mode == "empire" and (LQuest._Variable.tutorprice > 0 or gSim.time == 0) then
		priceTable.sugar = 3
		priceTable.cacao = 10
		priceTable.maha_cherry_infusion = 1295
	end
end

-------------------------------------------------------------------------------
-- Port Travel Price randomization

function LPort:PrepareTravelPricing(priceTable)
	for p in LPort:AllPorts() do
		if p.available and p ~= self then
			local r = LRoute:ByRoute(self.name, p.name)
			if r then
				priceTable[p.name] = Simulator.travelcost * route.time
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Port Entry Quests

function LPort:EnterPort()
	SwapToModal("ui/portview.lua")
	
	-- HACK: Check for "Around the World" when you arrive in Sydney
	if self.name == "sydney" then
		local m = gSim:EvaluateMedals()
		if m then gSim:AwardMedal(m) end
	end
	
--	local ok = gSim:CheckLoseCondition()
end

-------------------------------------------------------------------------------
-- Map Layout

function LPort:LayoutMap()
	local layout = {}

	local kPortIconSpacing = 15
	local kPortIconRadius = 17
	for port in LPort:AllPorts() do
		local yOffset = -const.travelTagHeight/2
		local xOffset = 0
		local xAbsolute = port.mapx
		local yAbsolute = port.mapy
		-- Flip tag if it hits the right edge of the screen
		if xAbsolute + const.travelTagWidth >= const.screenWidth then
			xOffset = -const.travelTagWidth
		end
		
		local mapx = port.mapx - kPortIconRadius
		local mapy = port.mapy - kPortIconRadius
		if (port == gSim.port) then
			table.insert(layout, Bitmap { x=mapx,y=mapy+7,image="control/port_current", })
		end
		
		if port.available then
			if port.factoryowned then
				local f = LFactory:ByPort(port.name)
				if f:Running() then
					table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')", clicksound=kDefaultPopupClickSound,
						SimpleAnimation { animx=16,animy=16, name=port.name, anim="control/factory.xml", fps=4 } })
				else
					table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')", clicksound=kDefaultPopupClickSound,
--						SimpleAnimation { animx=16,animy=16, name=port.name, anim="control/factory_idle.xml", fps=2 } })
						Bitmap { x=0,y=0, image="control/factory_idle_single"} })
				end
			elseif port.shopowned then
				table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')", clicksound=kDefaultPopupClickSound,
					Bitmap { name=port.name, image="control/icon_shop", } })
			elseif not port.visited then
				table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')", clicksound=kDefaultPopupClickSound,
--					Bitmap { name=port.name, image="control/port_new" } })
					SimpleAnimation { animx=16,animy=16, name=port.name, anim="control/port_new.xml", fps=3 } })
			else
				table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')", clicksound=kDefaultPopupClickSound,
					Bitmap { name=port.name, image="control/port_available"} })
			end
		elseif not port.hidden then
			table.insert(layout, Rollover { x=mapx,y=mapy, xAbsolute=xAbsolute,yAbsolute=yAbsolute, xOffset=xOffset,yOffset=yOffset, hotSize=60, target="LPort:ByName('" .. port.name .. "')",
				Bitmap { name=port.name, image="control/port_unavailable" } })
		end

		if not port.hidden then
			-- Position port label
			mapx = port.mapx
			mapy = port.mapy
			local flags = port.labelpos
			if flags == kHAlignLeft then
				mapx = mapx - kPortIconSpacing - 143
				mapy = mapy - 10
				flags = kHAlignRight + kVAlignCenter
			elseif flags == kVAlignBottom then
				mapx = mapx - 75
				mapy = mapy + kPortIconSpacing
				flags = kHAlignCenter + kVAlignTop
			elseif flags == kVAlignTop then
				mapx = mapx - 75
				mapy = mapy - kPortIconSpacing - 20
				flags = kHAlignCenter + kVAlignBottom
			else
				mapx = mapx + kPortIconSpacing
				mapy = mapy - 10
				flags = kHAlignLeft + kVAlignCenter
			end
			if mapx < 0 then mapx = 0 end
			if mapy < 0 then mapy = 0 end
			table.insert(layout, Text { x=mapx+2, y=mapy+2, w=140, h=20, label=port.name, flags=flags, font=portLabelDropFont })
			table.insert(layout, Text { x=mapx+1, y=mapy+1, w=140, h=20, label=port.name, flags=flags, font=portLabelDropFont })
			table.insert(layout, Text { x=mapx, y=mapy, w=140, h=20, label=port.name, flags=flags, font=portLabelFont })
		end
	end
	
	return Group(layout)
end

-------------------------------------------------------------------------------
-- Port informational popup

function LPort:OnPopupClick()
	if self.available then
		if gSim.port ~= self then
			local route = LRoute:ByRoute(gSim.port.name, self.name)
			local price = route:Price()
			if gSim.money >= price or LQuest._Variable.airship > 0 then
				gSim:InitiateTravel(self)
			end
		else
			SwapToModal("ui/portview.lua")
		end
	end
end

function LPort:CreatePopup()
	local label = { "#"..GetString(self.name) }

	-- Travel information
	if not self.available then
		if gSim.mode == "free" then
			table.insert(label, GetString("travel_freedisabled"))
		else
			table.insert(label, GetString("travel_visarequired"))
		end
	elseif gSim.port ~= self then
		local route = LRoute:ByRoute(gSim.port.name, self.name)
		local price = route:Price()
		if LQuest._Variable.airship > 0 then
			-- Airship: free direct travel in half the time
			price=0
			table.insert(label, bsutil.GetVariableString("travel_time", {time=bsutil.floor(route.time/2+.5)} ))
		else
			table.insert(label, bsutil.GetVariableString("travel_time", route))
			table.insert(label, bsutil.GetVariableString("travel_price", {price=Dollars(price)}))
		end
		
		if gSim.money >= price then
			table.insert(label, GetString("travel_click"))
		else
			table.insert(label, GetString("travel_expensive"))
		end
	elseif gSim.port == self then
		table.insert(label, GetString("travel_here"))
	else
		table.insert(label, GetString("travel_enter"))
	end
	
	label = table.concat(label,"\n")
	local layout
	if self.mapx + const.travelTagWidth >= const.screenWidth then
		layout =
		{
			fit=true,
			x=0,y=0, image="image/traveltag", hflip=true, w=210,h=100,
			Text { x=5,y=5,w=kMax-51,h=kMax-5, label=label, font=travelTagFont, flags=kVAlignCenter+kHAlignLeft, },
		}
	else
		layout =
		{
			fit=true,
			x=0,y=0, image="image/traveltag", w=210,h=100,
			Text { x=51,y=5,w=kMax-5,h=kMax-5, label=label, font=travelTagFont, flags=kVAlignCenter+kHAlignLeft, },
		}
	end
	
	return { Bitmap(layout) }
end

-------------------------------------------------------------------------------
-- Port layout

function LayoutPort()
	return gSim.port:UIPortView()
end

function LPort:UIPortView()
	self.visited = true
	local layout =
	{
		x=0,y=0,image="ports/"..self.name,
	}
	
	local y = 5
	if self.layout then
		for _,b in ipairs(self.layout) do
			if type(b) == "function" then
				local x = b()
				if x then
					table.insert(layout, x)
				end
			elseif b.image then
				local n = b.name.."_img"
				table.insert(layout, Rollover { x=b.x,y=b.y, hotSize=10, area=b.area, name=b.name, target="LBuilding:ByName('" .. b.name .. "')", clicksound=kDefaultPopupClickSound, image=b.name.."_img",
--					Bitmap { name=b.name.."_img", scale=b.scale, image="ports/" .. b.image } })
					BitmapTint { name=n, scale=b.scale, image="ports/" .. b.image } })
				UtilPulseImage(n,30)
			else
				table.insert(layout, Button { x=5,y=y, label=b.name, flags = kVAlignCenter + kHAlignCenter,
						command = b:SelectClosure(),
					})
				y = y + const.genButtonHeight
			end
		end
	end
	
	-- port name banner
	table.insert(layout, Bitmap { image = "ports/banner",x=297,y=0,
		Text { x=30,y=17,w=146,h=20,flags=kVAlignTop+kHAlignCenter,label=self.name, font=portNameFont } })

	-- ledger background
	table.insert(layout, Bitmap { image="image/toolbar",x=0,y=369 })
	
	return Bitmap(layout)
end

-------------------------------------------------------------------------------
-- Game Save/Load

function LPort:DoReset()
	self.visited = nil
	self.available = nil
	self.factoryowned = nil
	self.shopowned = nil
	self.availability["truffle_powder"] = nil
end

function LPort:ResetAll()
	for p in LPort:AllPorts() do
		p:DoReset()
	end
end

function LPort:SaveGameTable(t)
	-- Save available ports and port data
	t.ports = {}
	for p in LPort:AllPorts() do
		if p.available then
			-- A table for each available port
			t.ports[p.name] = {}
			local tp = t.ports[p.name]
			
			-- "Visited" flag
			tp.visited = p.visited
			
			-- Building ownership will be handled in building save/load
		end
	end
end

function LPort:LoadGameTable(t)
--	LPort:ResetAll()
	if type(t.ports) == "table" then
		for name,tp in pairs(t.ports) do
			local p = LPort:ByName(name)

			-- By definition, if it's in this list it's available
			p.available = true
			p.visited = tp.visited 
			
			-- Building ownership will be handled in building save/load
		end
	end
end
