--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Medals
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------
assert(Simulator)

-------------------------------------------------------------------------------

-- RANKS
local function Novice() return gSim.rank > 0 end
local function Entrepreneur() return gSim.rank > 1 end
local function Chocolatier() return gSim.rank > 2 end
local function Master() return gSim.rank > 3 end

-- RECIPES
local function Bars()
	local t = ProductType:ByName("bar")
	return t.knownCount >= 5
end
local function Squares()
	local t = ProductType:ByName("square")
	return t.knownCount >= 10
end
local function Infusions()
	local t = ProductType:ByName("infusion")
	return t.knownCount >= 12
end
local function Truffles()
	local t = ProductType:ByName("truffle")
	return t.knownCount >= 14
end

-- FIRST SHOP
local function FirstShop() return gSim.shopCount > 0 end

-- FACTORIES
local function Factories() return gSim.facCount >= 6 end

-- PORTS
local function Ports()
	for p in LPort:AllPorts() do
		if not p.available or not p.visited then return false end
	end
	return true
end

-------------------------------------------------------------------------------
-- Medals:
--   Novice (Rank 1)
--   Bars (10 recipes)
--   Entrepreneur (Rank 2)
--   First Shop
--   Squares (10 recipes)
--   Chocolatier (Rank 3)
--   Factories (6)
--   Infusions (10 recipes)
--   All ports visited
--   Truffles (10 recipes)
--   Master Chocolatier (Rank 4)

Simulator.MedalData =
{
	{ Novice, "novice" },
	{ Bars, "bars" },
	{ Entrepreneur, "entrep" },
	{ FirstShop, "shop" },
	{ Squares, "squares" },
	{ Chocolatier, "choco" },
	{ Factories, "factory" },
	{ Infusions, "infusions" },
	{ Ports, "ports" },
	{ Truffles, "truffles" },
	{ Master,  "master" },
}

-------------------------------------------------------------------------------

function Simulator:GetMedalFlags()
	local flags = 0
	if self.mode == "empire" and gEnableMedals == 1 then
		local bit = 1
		for i=1,table.getn(Simulator.MedalData) do
			if self.medals[i] then flags = flags + bit end
			bit = bit * 2
		end
	end
	return flags
end

function Simulator:ResetMedals()
	for i=1,table.getn(Simulator.MedalData) do self.medals[i] = false end
end

function Simulator:EvaluateMedals()
	if self.mode == "empire" and gEnableMedals == 1 then
		for i,medal in ipairs(Simulator.MedalData) do
			local f = medal[1]
			-- Run evaluation functions on any medals not already owned
			if not self.medals[i] and f() then
				gNewMedal = i
				return i
			end
		end
	end
	return nil
end

function AllMedalsForMCPath()
	-- See if they have all medals required to start the final path
	-- (8 medals -- not including Ports, Truffles, and MC)
	for i=1,8 do
		if not gSim.medals[i] then
			return false
		end
	end
	return true
end

-------------------------------------------------------------------------------

function Simulator:AwardMedal(i)
	self.medals[i] = true

	StopBackgroundMusic()
	if i < table.getn(Simulator.MedalData) then
		DisplayDialog { "ui/reward.lua", medal=i }
	else
		gSim.mcWeeks = gSim.mcWeeks or gSim.time
		gSim.mcMoney = gSim.mcMoney or gSim.money
		DisplayDialog { "ui/reward_finale.lua" }
	end
	BackgroundMusic()
end

-------------------------------------------------------------------------------

function GetMedalRollover(i)
	local n = i
	return
	{
		CreatePopup = function() return DoMedalRollover(n) end,
		OnPopupClick = function() return DoMedalClick(n) end,
	}
end

function DoMedalClick(i)
	-- Show the award dialog if you own this medal
	if gSim.medals[i] then gSim:AwardMedal(i) end
end

function DoMedalRollover(i)
	local h = infoFont[2]
	local key = Simulator.MedalData[i]
	key = key[2]
	local s = "#"..GetString("medal_"..key) .. "\n" .. GetString("medal_"..key.."_desc")
--	return Rectangle
--	{
--		x=0,y=0,w=250,h=124,
	return Bitmap
	{
		x=0,y=0, image="image/plaque",
		AppendStyle{font=infoFont},
		Text {x=2+10,y=2+10,w=246,h=120,label=s, flags=kVAlignCenter+kHAlignCenter },
	}
end

local medalX = { 18, 88, 160, 233, 305, 378, 449, 521, 595, 668, 738 }
function Simulator:LayoutMedals()
	local layout = {}
	if self.mode == "empire" and gEnableMedals == 1 then
		for i=1,table.getn(Simulator.MedalData) do
			local data = Simulator.MedalData[i]
			local image = "image/medal_" .. data[2]
			if self.medals[i] then image = image.."_small"
			else image = image.."_notyet"
			end
			
			table.insert(layout, 
				Rollover { x=medalX[i],y=0, target="GetMedalRollover("..tostring(i)..")", Bitmap { image=image } })

			-- Add "sparkle"
			if gNewMedal == i then
				local x = medalX[i] - 36
				if x >=0 then
					table.insert(layout, 
						SimpleAnimation { x=x,y=0, animx=52,animy=86-25, fps=10, anim="image/sparkle.xml" })
				else
					table.insert(layout, 
						SimpleAnimation { x=0,y=0, animx=52+x,animy=86-25, fps=10, anim="image/sparkle.xml" })
				end
			end
		end
	end
	gNewMedal = nil
	return Group(layout)
end
