--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Market
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Market class definition, derived from LBuilding
LMarket = { helpkey=4 } setmetatable(LMarket, LBuilding)
function LMarket.__tostring(t) return "{Market:" .. tostring(t.name) .. "}" end

function Market(t) return LMarket:new(t) end

-------------------------------------------------------------------------------

function LMarket:Haggle(c)
	local choice = DisplayDialog{"ui/haggle.lua"}
	if choice ~= "cancel" then
		self:ComputeHaggle(c, choice)
		UpdateDynamicWindow("receipt")
	end
end

local function SetNewPrice(name, n)
	if n < gSim.portPrices[name] then
		UtilSetTextColor(name.."P",HaggleGreenColor)
	elseif n > gSim.portPrices[name] then
		UtilSetTextColor(name.."P",HaggleRedColor)
	else
		UtilSetTextColor(name.."P",BrownColor)
	end
	gSim.portPrices[name] = n
	gSim.seenPrices[name] = n
end

function LMarket:ComputeHaggle(c, choice)
	-- The "reasonableness" of a current price is a measure of where the
	-- actual price lies between the item low and high prices
	-- The overall "reasonableness" of prices is the average reasonableness
	local count = 0
	local R = 0
	for item in gSim.port:AvailableIngredients() do
		local target = (gSim.portPrices[item.name] - item.low) / (item.high - item.low)
		R = R + target
		count = count + 1
	end
	R = 100 * R / count
	
	-- Compute in the "haggle factor" for this character -- just a multiplier for R
	if c.haggleFactor then R = R * c.haggleFactor end
	
	-- Now on a scale of 0..100, the lower value of R, the better the prices...
	-- If we roll lower than R, we can get lower prices -- so as prices get
	-- better, our chances of rolling even lower are slim
	local H = bsutil.random(100)
	
	-- Roll is 20 pts under, prices drop
	-- 30 pts or more over, prices rise
	-- SO... if actual price is half way between low and high, R=50
	--  R=100: 80% drop, 20% stay the same
	--  R=50: 30% drop, 50% stay the same, 20% go up
	--  R=30: 10% dop, 50% stay the same, 40% go up
	--  R=0: 0% drop, 30% stay the same, 70% go up
	
	-- Check for special quest variable to induce better haggling...
	if LQuest._Variable.HaggleBonus and LQuest._Variable.HaggleBonus > 0 then
		-- Force haggling lower...
		LQuest._Variable.HaggleBonus = LQuest._Variable.HaggleBonus - 1
		H = 0
	end
	
	local response = "haggle_nada"
	if H < R-20 then
		-- Prices drop toward minimum
		response = "haggle_lower"
		for item in gSim.port:AvailableIngredients() do
			local n = bsutil.floor( (gSim.portPrices[item.name] + item.low) / 2 )
			SetNewPrice(item.name, n)
		end
	elseif H > R+30 then
		response = "haggle_max"
		for item in gSim.port:AvailableIngredients() do
			SetNewPrice(item.name, item.high)
		end
		
		-- Lose haggling rights for now
		UtilDisableWindow("haggle")
		UtilDisableWindow("haggle_label")
		gSim.haggle = false
	end

	response = response..tostring(bsutil.random(10))
	response = "#"..bsutil.GetVariableString(response, { vendor=c.name, market=self.name })
	DisplayDialog{"ui/haggle_response.lua", response=response, choice=choice}
	
	-- Update prices and receipt
	for item in gSim.port:AvailableIngredients() do
		SetLabel(item.name.."P", Dollars(item:GetPrice()))
	end
	UpdateDynamicWindow("receipt")
end

function LMarket.Buy()
	local total = LMarket.TotalSales(gSim.port:AvailableIngredients())
	if total > gSim.money then
		DisplayDialog { "ui/okdialog.lua", body="buy_notenough" }
	elseif total > 0 then
		for item in gSim.port:AvailableIngredients() do
			local count = tonumber(GetLabel(item.name)) or 0
			if count > 0 then
				gSim:AdjustInventory(item.name, count)
				local unit = item:GetPrice()
				local cost = count * unit
				gSim.purchasePrices[item.name] = unit
				gSim.purchasePorts[item.name] = gSim.port.name
				local s = GetString("buy_notice", tostring(count), GetString(item.name), Dollars(cost), Dollars(unit), GetString(gSim.port.name))
				PlayerMessage(s)
			end
			SetLabel(item.name, GetString("buy"))
		end
		
		gSim:AdjustMoney(-total)
		
		-- A "day" passes when the player makes a purchase
--		gSim:SubTick()
		gTransactionMade = true

		UpdateDynamicWindow("receipt")
		UpdateStandardUI()
		return true
	else
		for item in gSim.port:AvailableIngredients() do
			SetLabel(item.name, GetString("buy"))
		end
	end
	return false
end

function LMarket:UISelect()
	local c,q = self:ChooseCharacter()
	if q then c:UIHandleQuest(q) end
	gTransactionMade = nil
	-- Collect available ingredients
	local ingList = {}
	for item in gSim.port:AvailableIngredients() do
		gSim.seen[item.name] = gSim.port.name
		gSim.seenPrices[item.name] = gSim.portPrices[item.name]
		table.insert(ingList, item)
	end

	SetAmbient(kMarketAmbient)
	DisplayDialog { "ui/buysell.lua", character=c, building=self, sound="charopen",
		type="buy", items=ingList, background=MarketColor }
	SetAmbient(kPortAmbient)
	self:LeaveBuilding(c,q)
end
