--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Items
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	An Item is something that can be bought or sold.
	"static" members:
		_ByName: table of all Items keyed by name
		_ByIndex: array of all Items in order of creation
		_IngredientsByIndex: array of all Ingredients in order of creation
		_ProductsByIndex: array of all Products in order of creation
	
	"hidden" members:
		known: true if a Product recipe is known to the player
--]]---------------------------------------------------------------------------

-- Item class definition
LItem =
{
	__tostring = function(i) return "{Item:" .. i.name .. "}" end,
	_ByName = {},
	_ByIndex = {},
	_IngredientsByIndex = {},
	_ProductsByIndex = {},
}

LItem.truePriceMultiplier = .5		-- eventual "median" price multiplier between low and high
LItem.lowestMarkup = 1.15			-- eventual "lowest" markup allowed

-------------------------------------------------------------------------------
-- Functions for data description

function DefineItem(t)
	t.name = t.name or t[1]
	t[1] = nil
	return LItem:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global Item lists

function LItem:AllItems()
	return bsutil.ArrayIterator(self._ByIndex)
end

function LItem:ByName(name)
	return self._ByName[name]
end

-- Ingredients --
function LItem:AllIngredients()
	return bsutil.ArrayIterator(self._IngredientsByIndex)
end

function LItem:IngredientCount()
	return table.getn(self._IngredientsByIndex)
end

function LItem:IngredientByIndex(n)
	return self._IngredientsByIndex[n]
end

-- Products --
function LItem:AllProducts()
	return bsutil.ArrayIterator(self._ProductsByIndex)
end

function LItem:ProductCount()
	return table.getn(self._ProductsByIndex)
end

function LItem:ProductByIndex(n)
	return self._ProductsByIndex[n]
end

-- Sorting --
function ItemSortByDescendingPrice(a,b)
--	return b:GetPrice() < a:GetPrice()
	return b.high < a.high
end

-------------------------------------------------------------------------------
-- "constructor"

function LItem:new(t)
--	devmessage(t.name, "Item defined with no name")
--	devmessage(not self._ByName[t.name], "Item " .. t.name .. " already defined")
	if t.name and not self._ByName[t.name] then
		t = t or {} setmetatable(t, self) self.__index = self

		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		if t.recipe then table.insert(self._ProductsByIndex, t)
		else table.insert(self._IngredientsByIndex, t)
		end
		
		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Post Processing

function LItem:PostProcessAll()
	for i,item in ipairs(self._ByIndex) do
		if type(item.type) == "string" then
			-- Keep track of all products in a given type
			local itemType = ProductType:ByName(item.type)
			if itemType then
				item.type = itemType
				item.type:AddProduct(item)

				-- Calculate product pricing based on cost of ingredients
				item:CalculateRecipeCosts()
			else
--				devmessage(itemType, "Item "..item.name.." - undefined type "..item.type)
			end
		end
	end
end

function LItem:CalculateRecipeCosts()
	self.lowcost = 0
	self.highcost = 0
	for item,count in pairs(self.recipe) do
		local ing = LItem:ByName(item)
		assert(ing and type(ing) == "table")
		self.lowcost = self.lowcost + count * ing.low
		self.highcost = self.highcost + count * ing.high
	end
	
	-- calculate standard markup
	local markup = self.type.markup or 2
	self.lowest = self.lowcost * self.lowestMarkup
	self.low = self.lowcost * markup
	self.high = self.highcost * markup
	-- and "true" price is between low and high
	self.middle = bsutil.floor(self.low + (self.high - self.low) * self.truePriceMultiplier + .5)
	self.low = bsutil.floor(self.low + .5)
	self.high = bsutil.floor(self.high + .5)
end

-------------------------------------------------------------------------------
-- Recipe tracking (cross-reference Item with Recipe)

function LItem:EnableRecipe(reason)
	devmessage(self.recipe, "Item "..self.name.." has no recipe")
	if self.recipe and not self.known then
		self.known = true
		self.type:IncrementKnownCount()
		if reason then PlayerMessage(bsutil.GetVariableString(reason, self)) end
		
		-- FirstPeek
		gSim.recipeCount = gSim.recipeCount + 1
	end
end

-- For development only
function LItem:DisableRecipe()
	if self.recipe and self.known then
		self.known = false
		self.type:DecrementKnownCount()
		
		-- FirstPeek
		gSim.recipeCount = gSim.recipeCount - 1
	end
end

-------------------------------------------------------------------------------
-- Standard UI: Inventory

function LItem:InventoryWindow(xW,yW,wW,hW)
	local more = false

	local h = ledgerFont[2]
	local layout = {}

--[[
	-- Figure out vertical spacing for as many lines as possible
	local lineCount = bsutil.floor(hW / (const.ingIconHeight + h + 2))
	local totalY = lineCount * const.ingIconHeight
	local ySpace = (hW - totalY) / lineCount
	local yTop = yW + (hW - lineCount * (const.ingIconHeight + ySpace) + ySpace) / 2
	local yMax = hW - const.ingIconHeight
	
	-- Display 5 colums of ingredients
--	local ingCols = bsutil.floor((22 + lineCount - 1) / lineCount)
	local ingCols = 5
	local xSpace = bsutil.floor(const.ingIconWidth / 3)
	local ingWidth = const.ingIconWidth + xSpace
	local prodWidth = const.prodIconWidth + xSpace
	local totalX = wW - ingCols * ingWidth
	local prodCols = bsutil.floor(totalX / prodWidth)
	totalX = ingCols * const.ingIconWidth + prodCols * const.prodIconWidth
	xSpace = (wW - totalX) / (ingCols + prodCols)
	ingWidth = const.ingIconWidth + xSpace
	prodWidth = const.prodIconWidth + xSpace
	local xLeft = xW + xSpace / 2

	-- Make way for scrolling
	-- Ingredients	
	local x = xLeft
	local xRight = xLeft + ingCols * ingWidth
	local y = yTop
	local i = 1
	for item in gSim:PlayerInventoryIngredients() do
		if y >= yMax then more = true break end
		if i >= gCurrentLedgerTop then
			local inv = gSim:GetInventory(item.name) or 0
--			table.insert(layout, Bitmap { x=x+xSpace/2,y=y, image="image/icon_sack" })
			table.insert(layout, Rollover { x=x+xSpace/2,y=y, target="LItem:ByName('"..item.name.."')", Bitmap { image="item/"..item.name }})
			table.insert(layout, Text { x=x,y=y+const.ingIconHeight,w=ingWidth,h=h, flags=kVAlignCenter+kHAlignCenter, label="#"..tostring(inv), font=invFont })
		end
		
		x = x + ingWidth
		if x + const.ingIconWidth >= xRight then
			x = xLeft
			y = y + const.ingIconHeight + ySpace
			i = i + 1
			if i == gCurrentLedgerTop then y = yTop end
		end
	end

	xLeft = xRight
	xRight = wW - const.prodIconWidth
	x = xLeft
	y = yTop
	local i = 1
	for item in gSim:PlayerInventoryProducts() do
		if y >= yMax then more = true break end
		if i >= gCurrentLedgerTop then
			local inv = gSim:GetInventory(item.name) or 0
--			table.insert(layout, Bitmap { x=x+xSpace/2,y=y, image="image/icon_crate" })
			table.insert(layout, Rollover { x=x+xSpace/2,y=y, target="LItem:ByName('"..item.name.."')", Bitmap { image="item/"..item.name }})
			table.insert(layout, Text { x=x,y=y+const.prodIconHeight,w=prodWidth,h=h, flags=kVAlignCenter+kHAlignCenter, label="#"..tostring(inv), font=invFont })
		end
		
		x = x + prodWidth
		if x >= xRight then
			x = xLeft
			y = y + const.prodIconHeight + ySpace
			i = i + 1
			if i == gCurrentLedgerTop then y = yTop end
		end
	end
]]--

	-- FINALIZING AND TWEAKING CONSTANTS FOR BETTER LAYOUT
	local yRow1 = 8
	local yRow2 = 75

	-- Lay out 2 rows x 5 columns of ingredients starting at the appropriate row
	local xLeft = 3								-- start of leftmost edge
	local ix = 3								-- half spacing
	local dx = const.sackIconWidth + 2 * ix	-- full column width
	local x = xLeft + ix
	local y = yRow1
	local startIndex = 1 + (gCurrentLedgerTop - 1) * 5
	local i = 0
	local count = 0
	local offsetX = (const.sackIconWidth - const.ingIconWidth) / 2
	local offsetY = (const.sackIconHeight - const.ingIconHeight) / 2
	for item in gSim:PlayerInventoryIngredients() do
		i = i + 1
		if i >= startIndex then
			count = count + 1
			if count == 6 then
				x = xLeft + ix
				y = yRow2
			elseif count == 11 then
				more = true
				break
			end
			
			local inv = gSim:GetInventory(item.name) or 0
			local font = invFont
			if inv > 9999999 then font = invTinyFont
			elseif inv > 99999 then font = invSmallFont end
			if inv > 9999999 then inv = "9999999+" end
			table.insert(layout, Bitmap { x=x+ix,y=y, image="image/icon_sack" })
			table.insert(layout, Rollover { x=x+ix+offsetX,y=y+offsetY, target="LItem:ByName('"..item.name.."')", Bitmap { image="item/"..item.name }})
			table.insert(layout, Text { x=x,y=y+const.sackIconHeight-2,w=dx,h=h, flags=kVAlignTop+kHAlignCenter, label="#"..tostring(inv), font=font })
			x = x + dx
		end
	end

	-- Lay out 2 rows x 3 columns of products starting at the appropriate row
	i = 0
	xLeft = xLeft + dx * 5 + ix
	ix = 1										-- half spacing
	dx = const.crateIconWidth + 2 * ix			-- full column width
	x = xLeft + ix
	y = yRow1
	startIndex = 1 + (gCurrentLedgerTop - 1) * 3
	i = 0
	count = 0
	offsetX = 5		--(const.crateIconWidth - const.prodIconWidth) / 2
	offsetY = 12	--(const.crateIconHeight - const.prodIconHeight) / 2
	for item in gSim:PlayerInventoryProducts() do
		i = i + 1
		if i >= startIndex then
			count = count + 1
			if count == 4 then
				x = xLeft + ix
				y = yRow2
			elseif count == 7 then
				more = true
				break
			end
			
			local inv = gSim:GetInventory(item.name) or 0
			local font = invFont
			if inv > 9999999 then font = invSmallFont end
			if inv > 99999999 then inv = "99999999+" end
			table.insert(layout, Bitmap { x=x+ix,y=y, image="image/icon_crate" })
			table.insert(layout, Rollover { x=x+ix+offsetX,y=y+offsetY, target="LItem:ByName('"..item.name.."')", Bitmap { image="item/"..item.name }})
			table.insert(layout, Text { x=x,y=y+const.crateIconHeight-2,w=dx,h=h, flags=kVAlignTop+kHAlignCenter, label="#"..tostring(inv), font=font })
			x = x + dx
		end
	end

	LedgerScrollButtons(layout, more)
	return layout
end

-------------------------------------------------------------------------------
-- Standard UI: Recipes

local function SelectTab(t)
	t = ProductType:ByName(t)
	gCurrentTypeSelection = t
	UpdateDynamicWindow("recipelist")
	UpdateDynamicWindow("recipe")
end

function LItem:FullRecipeBook()
	local layout =
	{
		x=7,y=0,image = "image/factory_recipes",
		DynamicWindow { x=53,y=50,w=314,h=317, name="recipelist", contents="ui/recipelist.lua", },
		DynamicWindow { x=407,y=35,w=319,h=371, name="recipe", contents="ui/recipeselect.lua", },
		
		SetStyle(tabButtonStyle),
		BeginGroup(),
		Button { x=52,y=0,label="bar", name="bar", command = function() gCurrentRecipeSelection=LItem:ByName("basebars") SelectTab("bar") end, graphics={"control/std_tab_up","control/tab_1_down","control/tab_1_down"} },
		Button { x=226,y=0,label="square", name="square", command = function() gCurrentRecipeSelection=LItem:ByName("xdark_sq") SelectTab("square") end, graphics={"control/std_tab_up","control/tab_2_down","control/tab_2_down"} },
		Button { x=424,y=0,label="infusion", name="infusion", command = function() gCurrentRecipeSelection=LItem:ByName("milk_choc_infusion") SelectTab("infusion") end, graphics={"control/std_tab_up","control/tab_3_down","control/tab_3_down"} },
		Button { x=587,y=0,label="truffle", name="truffle", command = function() gCurrentRecipeSelection=LItem:ByName("dark_truffles") SelectTab("truffle") end, graphics={"control/std_tab_up","control/tab_4_down","control/tab_4_down"} },
	}
	
	if not gCurrentRecipeSelection then
		-- Default to base bars if necessary
		gCurrentRecipeSelection = LItem:ByName("basebars")
	end
	gCurrentTypeSelection = gCurrentRecipeSelection.type

	table.insert(layout, function() UtilSetButtonOn(gCurrentRecipeSelection.type.name) end)
	table.insert(layout, SetStyle(genButtonStyle))
	
	layout =
	{
		StandardUI("reconfigure"),
		Bitmap(layout),
	}

	return DisplayDialog{"ui/dialog.lua", body=layout }
end

-------------------------------------------------------------------------------
-- Popups

function LItem:CreatePopup()
	if self.recipe then return self:CreateProductPopup()
	else return self:CreateIngredientPopup()
	end
end

function LItem:OnPopupClick()
--	if self.recipe and self.known then
	if self.recipe then
		gCurrentRecipeSelection = self
		UpdateDynamicWindow("recipelist")
		UpdateDynamicWindow("recipe")
	end
end

function LItem:CreateIngredientPopup()
	local y = 5
	local h = infoFont[2]
	local count = gSim:GetInventory(self.name) or 0
	local s = "#"..GetString(self.name).." ("..GetString("item_inventory", tostring(count))..")"
	local layout =
	{
		x=0,y=0,w=800,h=600, shrink=true,
		color=PopupColor,
		AppendStyle{font=infoFont},
		Text {x=5,y=y,w=kMax,h=h,label=s },
	}
	if gSim.inventory[self.name] and gSim.inventory[self.name] > 0 and gSim.purchasePorts[self.name] and gSim.purchasePrices[self.name] then
		y = y + h
		local s = "#"..GetString("last_bought", GetString(gSim.purchasePorts[self.name]), Dollars(gSim.purchasePrices[self.name]))
		table.insert(layout, Text { x=5,y=y,w=kMax,h=h,label=s })
	elseif gSim.seen[self.name] and gSim.seen[self.name] ~= gSim.port.name then
		y = y + h
		local s = "#"..GetString("last_seen", GetString(gSim.seen[self.name]), Dollars(gSim.seenPrices[self.name]))
		table.insert(layout, Text { x=5,y=y,w=kMax,h=h,label=s })
	end
	
	return { Rectangle(layout) }
end

function LItem:CreateProductPopup()
	local y = 5
	local h = infoFont[2]
	local layout =
	{
		x=0,y=0,w=800,h=600, shrink=true,
		color=PopupColor,
		AppendStyle{font=infoFont},
		Text {x=5,y=y,w=kMax,h=h,label=self.name },
	}
	if self.known then
		local sold = gSim.sold[self.name] or 0
		local count = gSim:GetInventory(self.name) or 0
		local s = "#"..GetString("item_sold", tostring(sold))
		s = s..", "..GetString("prod_inventory", tostring(count))
		table.insert(layout, Text { x=5,y=y+h,w=kMax,h=h,label=s })
	end
	return { Rectangle(layout) }
end

-------------------------------------------------------------------------------
-- Conveniences...

function LItem:GetPrice()
	return gSim.portPrices[self.name] or 0
end

-------------------------------------------------------------------------------
-- Game Save/Load/Reset

function LItem:ResetAll()
	for p in LItem:AllProducts() do
		p.known = nil
	end
	gSim.recipeCount = 0

	-- Erase counts of known recipes by type	
	for _,t in pairs(ProductType._ByName) do t:ResetKnownCount() end
end

function LItem:SaveGameTable(t)
	-- Dump all enabled recipes
	t.recipes = {}
	for p in LItem:AllProducts() do
		if p.known then t.recipes[p.name] = true end
	end
end

function LItem:LoadGameTable(t)
--	LItem:ResetAll()
	if type(t.recipes) == "table" then
		for name,_ in pairs(t.recipes) do
			local p = LItem:ByName(name)
			if p then p:EnableRecipe() end
		end
	end
end
