--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Factory
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A Factory is a factory, creating products according to the player's
	specifications.
	
	See external data description documentation.
	
	NOTES:
	ProductionRun fills in a table with information about what the factory should do:
		created = { name = count, ... }
		consumed = { name = count, ... }
	AFTER calling ProductionRun, the simulator makes appropriate adjustments
	to the player's inventory.
--]]---------------------------------------------------------------------------

require("sim/building.lua")

-- Factory class definition, derived from LBuilding
LFactory = { helpkey=6 } setmetatable(LFactory, LBuilding)
function LFactory.__tostring(t) return "{Factory:" .. tostring(t.name) .. "}" end

function Factory(t) return LFactory:new(t) end

LFactory.defaultProductionCount = 0

-------------------------------------------------------------------------------
-- Factory tracking

LFactory._FactoriesByIndex = {}
LFactory._ByPort = {}

function LFactory:ByPort(p)
	return LFactory._ByPort[p]
end

function LFactory:SetPort(port)
	LFactory._ByPort[port.name] = self
	LBuilding.SetPort(self,port)
	table.insert(LFactory._FactoriesByIndex, self)
end

function LFactory:AllFactories()
	return bsutil.ArrayIterator(self._FactoriesByIndex)
end

-------------------------------------------------------------------------------
-- Factory purchase

function LFactory:MarkVisited()
	-- Factories actually care if they've been visited
	self.visited = true
end

function LFactory:MarkOwned()
	LBuilding.MarkOwned(self)

	-- Record this factory as available to the player
	gSim.facCount = gSim.facCount + 1
	self.port.factoryowned = true
	self.config = {}
	self.machines = { bar=true }
end

function LFactory:Purchase()
	self:MarkOwned()
	PlayerMessage(bsutil.GetVariableString("factory_acquired", { port=self.port.name }))
	
	-- All factories start with a "bar" machine and possibly others and come
	-- pre-configured to produce 0 basebars per week
	self:SetConfiguration("basebars", LFactory.defaultProductionCount)
	gSim:ProjectProduction()

	-- Check for "six factory" awards and do it	
	local m = gSim:EvaluateMedals()
	if m then gSim:AwardMedal(m) end
end

-------------------------------------------------------------------------------
-- Configuration

function LFactory:SetConfiguration(productName, rate)
	if not productName and self.product then productName = self.product.name end
	if productName then
		rate = rate or self.config[productName]
		rate = rate or LFactory.defaultProductionCount
		
		self.config[productName] = rate
		self.product = LItem:ByName(productName)
		self.rate = rate
	end
end

function LFactory:UIReconfigure(character)
	gCurrentRecipeSelection = self.product
	gCurrentTypeSelection = self.product.type

	local r = LItem:FullRecipeBook()
	if r == "select" then
		-- See if we have machines on hand to make this and offer to sell one...
		local newRecipe = gCurrentRecipeSelection
		if not self.machines[gCurrentRecipeSelection.type.name] then
			if gCurrentRecipeSelection.type.price > gSim.money then
				local t = bsutil.GetVariableString("factory_buymachine_expensive",
					{ type=newRecipe.type.name, price=Dollars(newRecipe.type.price) })
				character:UIModalDialog(t)
				newRecipe = nil
			else
				local t = bsutil.GetVariableString("factory_buymachine",
					{ product=newRecipe.name, type=newRecipe.type.name, price=Dollars(newRecipe.type.price) })
				local r = character:UIModalDialog(t, { "buy", "notnow" })
				if r == "buy" then
					gSim:AdjustMoney(-newRecipe.type.price)
					self.machines[newRecipe.type.name] = true
					PlayerMessage(bsutil.GetVariableString("machine_acquired", { port=self.port.name, type=newRecipe.type.name }))
				else
					newRecipe = nil
				end
			end
		end
		if newRecipe and newRecipe ~= self.product then
			self:SetConfiguration(newRecipe.name)
			PlayerMessage(bsutil.GetVariableString("factory_newrecipe", { port=self.port.name, product=self.product.name }))
			gSim:ProjectProduction()
			UpdateStandardUI()
		end
	end
end

-------------------------------------------------------------------------------
-- Production

function LFactory:DoMiniGame(c)
	-- Suspend background animations
	SetUpdateRate(0)
	StopBackgroundMusic()
	SetAmbient(kFactorySetupAmbient)
	
	-- LAYOUT CONSTANTS FROM CHARTALK STANDARD CHARACTER DIALOG -------------
	local cy = 45
	local cx = 20+3 --0		-- 410 for right
	local rectL = 52
	local rectT = cy+26
	local rectW = 492
	local rectH = 239
	-- (( Usable area: 57,51 350x172 ))
	local textLeft = 187+5		-- rectL+5 for right
	local textWidth = 345
	local textTop = cy+31
	local textHeight = 172+25
	local buttonY = textTop+textHeight - 7
	local x3 = 62
	local x1 = x3 + 62
	local x2 = x3+(rectW-10)/2+19
--	local x1 = 62
--	local x3 = x1+(rectW-10)/2+19
--	local x2 = x3 + 62
	
	local warnOnMiss = false
	if gFirstMissWarning and self.product.name == "basebars" then warnOnMiss = true end
	gFirstMissWarning = nil
	
	local layout =
	{
		name="minigame",
		Bitmap
		{
			x=0,y=0,w=800,h=600,image="image/factory",
			FactoryGame {
				x=0,y=0,w=800,h=600,
				name="factory",
				character=c.name, firstMiss=warnOnMiss,
				port=gSim.port.name,
				product=self.product, prodname=self.product.name, recipe=self.product.recipe, type=self.product.type,
				Text { x=616,y=380,w=170,h=33,flags=kVAlignCenter+kHAlignCenter,label=self.product.name,font={stdFontFace,16,Color(255,121,1)} },

				Text { x=681,y=3,w=38,h=16,flags=kVAlignTop+kHAlignCenter,label="cases",font={stdFontFace,16,Color(255,1,17)} },
				Text { name="counter", x=646,y=15,w=108,h=38, label="#0", flags=kVAlignCenter+kHAlignCenter, font={stdFontFace,35,Color(255,41,77)} },

				Text { x=669,y=65,w=63,h=16,flags=kVAlignTop+kHAlignCenter, label="mini_time", font={stdFontFace,16,Color(255,121,1)} },
				Text { name="timer", x=669,y=77,w=63,h=35,flags=kVAlignCenter+kHAlignCenter, label="#1:00", font={stdFontFace,35,Color(255,173,43)} },

				Text { name="countdown", x=150,y=150,w=300,h=300, label="#4",
					font={ stdFontFace, 200, RedColor }, flags=kVAlignCenter+kHAlignCenter },

--[[
Frame { name="finished", x=47,y=200+33,w=500,h=200-66, color=BrownColor,
	Text { x=0,y=14,w=kMax,h=kMax,flags=kVAlignTop+kHAlignCenter, name="finish_text", label="finished", font=gameAlertFont },
	Text { x=0,y=14,w=kMax,h=kMax,flags=kVAlignTop+kHAlignCenter, name="early_text", label="finished_early", font=gameAlertFont },
	Button { x=110,y=kMax-14-64+4, name="try_again", graphics=genericButtonGraphics,
		command = function()
			-- FirstPeek statistics
			gSim.minigames = gSim.minigames + 1
			RestartGame()
		end },
	Text { x=110+60,y=kMax-14-64+4,w=100,h=64, label="try_again", flags=kHAlignLeft+kVAlignCenter, font=yesNoDialogFont },
	Button { x=310,y=kMax-14-64+4, name="yes", graphics=yesButtonGraphics, command = function() CloseWindow(GetProductCount()) end },
	Text { x=310+60,y=kMax-14-64+4,w=100,h=64, label="ok", flags=kHAlignLeft+kVAlignCenter, font=yesNoDialogFont },
},
]]--
				
				Button { x=679,y=416,name="pause", graphics={"control/fac_btn_pause_up","control/fac_btn_pause_down","control/fac_btn_pause_over"},
					command = function() TogglePause() end },
				Text { name="pause_label", x=639,y=444,w=124,h=24,flags=kVAlignTop+kHAlignCenter,label="pause", font={stdFontFace,22,WhiteColor} },	--Color(255,121,1)} },
				Frame { name="paused", x=47,y=200+33,w=500,h=200-66, color=BrownColor,
					Text { x=0,y=14,w=kMax,h=kMax,flags=kVAlignTop+kHAlignCenter, label="paused", font=gameAlertFont },
--					Text { x=0,y=0,w=kMax,h=kMax,flags=kVAlignCenter+kHAlignCenter,label="paused", font=gameAlertFont },
					Button { x=242,y=kMax-14-64+4, graphics=genericButtonGraphics, command = function() TogglePause() end },
					Text { x=242+60,y=kMax-14-64+4,w=100,h=64, label="resume", flags=kHAlignLeft+kVAlignCenter, font=yesNoDialogFont },
					
					Button { x=178,y=kMax-14-64+4, graphics=helpButtonGraphics, command = function() SetUpdateRate(kDefaultUpdateRate) DisplayHelp(7,true) SetUpdateRate(0) end },
				},


			
			
				-- STANDARD CHARACTER BOX FROM CHARTALK ----------------------
				Window
				{
					name="keeper",
					x=10,y=30,			--x=101,y=10,
					w=597,h=301+cy, fit=true,
					Rectangle { x=52,y=cy+26,w=492,h=239+25, color=CharColor },
					
					Text { x=textLeft,y=textTop,w=textWidth,h=textHeight,
						name="chartext",
						font=charDialogFont, label="factory_start", flags=kVAlignCenter+kHAlignCenter },

					Bitmap { x=0,y=cy+0, image="image/dchar_left" },
					Bitmap { x=64+469,y=cy+0, image="image/dchar_right" },
					Bitmap { x=64,y=cy+0, image="image/dchar_top" },
					Bitmap { x=64,y=cy+235+25, image="image/dchar_bottom" },

					Window
					{
						name="gamestart",
						x=0,y=0,w=kMax,h=kMax, fit=true,
						Button { x=x1,y=buttonY, graphics=exitButtonGraphics, command = function() CloseWindow(-1) end },
						Text { x=x1+60,y=buttonY,w=150,h=64, label="cancel", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
						Button { x=x2,y=buttonY, graphics=yesButtonGraphics,
							command = function()
								-- FirstPeek statistics
								gSim.minigames = gSim.minigames + 1
								SetAmbient{}
								StartGame()
							end },
						Text { x=x2+60,y=buttonY,w=150,h=64, label="start", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
					},

					Window
					{
						name="gameend",
						x=0,y=0,w=kMax,h=kMax, fit=true,
						Button { x=x1,y=buttonY, graphics=genericButtonGraphics,
							command = function()
								-- FirstPeek statistics
								gSim.minigames = gSim.minigames + 1
								RestartGame()
							end },
						Text { x=x1+60,y=buttonY,w=150,h=64, label="try_again", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
						Button { x=x2,y=buttonY, graphics=yesButtonGraphics, command = function() CloseWindow(GetProductCount()) end },
						Text { x=x2+60,y=buttonY,w=150,h=64, label="ok", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
					},
					
					Button { x=x3,y=buttonY, graphics=helpButtonGraphics, command = function() SetUpdateRate(kDefaultUpdateRate) DisplayHelp(7,true) SetUpdateRate(0) end },
--					Text { x=x3+60,y=buttonY,w=150,h=64, label="help", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },

					c:UIAppearance(cx,0),
				},

			},
		}
	}

	local help = nil
--	collectgarbage("stop")
	local count = DisplayDialog{"ui/dialog.lua", sound="charopen", body=layout }
--	collectgarbage("restart")

	if count > 0 then
		self:SetConfiguration(self.product.name, count)
		UpdateDynamicWindow("factory_status")
		BackgroundMusicNow()
		PlayerMessage(bsutil.GetVariableString("factory_ratechange", { port=self.port.name, product=self.product.name, rate=count }))
		gSim:Tick()
		UpdateStandardUI()
	end

	-- Restore background animations
	SetUpdateRate(kDefaultUpdateRate)
	SetAmbient(kPortAmbient)
end

-------------------------------------------------------------------------------
-- Factory Execution

function LFactory:ProductionNeeds(needs)
	-- How many of each ingredient do we need to do a full run at this factory?
	-- Add it to the total supplied
	for name,count in pairs(self.product.recipe) do
		local total = needs[name] or 0
		total = total + count * self.rate
		needs[name] = total
	end
end

function LFactory:EstimateProduction(weeks)
	self.shortage = {}
	if self.owned and self.product then
		-- Given the current supply of ingredients and the needs of all factories,
		-- when will we run out at this factory?
		local max = 999999
		for name,needed in pairs(self.product.recipe) do
			local inv = gSim.inventory[name] or 0
			if inv < needed then
				self.shortage[name] = true
				max = 0
			elseif weeks[name] < max then
				max = weeks[name]
			end
		end
		self.weeksSupply = max
	else
		self.weeksSupply = 0
	end
end

function LFactory:Running()
	return self.owned and self.weeksSupply and self.weeksSupply > 0 and self.rate > 0
end

function LFactory:ProductionRun()
	local prod = 0
	if self.product and self.rate > 0 then
		-- Determine how many products we could create from the current pool of ingredients
		prod = 999999
		for name,needed in pairs(self.product.recipe) do
			local inv = gSim.inventory[name] or 0
			if count == 0 then inv = 0
			else inv = inv / needed
			end
			if inv < prod then prod = inv end
		end
		prod = bsutil.floor(prod)
		
		-- Make sure not to produce more than the factory is set up for...
		if prod > self.rate then prod = self.rate end
		if prod > 0 then
			-- Alter the player's inventory for production at this factory
			local total = gSim.inventory[self.product.name] or 0
			total = total + prod
			gSim.inventory[self.product.name] = total
			
			-- Consume ingredients
			for name,needed in pairs(self.product.recipe) do
				local inv = gSim.inventory[name] or 0
				inv = inv - needed * prod
				gSim.inventory[name] = inv
				gSim.usetime[name] = gSim.time
			end
		else
			-- Production has stopped...
			-- Figure out WHY production has stopped (for FirstPeek)
			if gFirstPeek then
				for name,needed in pairs(self.product.recipe) do
					local inv = gSim.inventory[name] or 0
					if inv < needed then
						local c = gSim.shortages[name] or 0
						gSim.shortages[name] = c + 1
					end
				end
			end
		end
	end
	return prod
end

-------------------------------------------------------------------------------
-- Factory UI

function LFactory:UISelect()
	local c,q = self:ChooseCharacter()
	if q then c:UIHandleQuest(q) end
	
	if not self.owned then
		local bought = self:UIPurchase(c)
		if bought then
			-- If purchased, trick the logo into appearing in the port view
			SwapToModal("ui/portview.lua")
			self:UISelect()
		else
			self:LeaveBuilding(c,q)
		end
	else
		self:UIStatus(c)
		self:LeaveBuilding(c,q)
	end
end

function LFactory:UIPurchase(character)
	if gSim.money >= self.price then
		local purchase = character:UIModalDialog(bsutil.GetVariableString("factory_purchase", {price=Dollars(self.price)}), { "yes", "reject" })
		if purchase == "yes" then
			gSim:AdjustMoney(-self.price)
			self:Purchase()
			UpdateStandardUI()
			return true
		end
	else
		character:UIModalDialog(bsutil.GetVariableString("factory_toomuch", {price=Dollars(self.price)}))
	end
	return false
end


local function ReconfigureButton(f,c)
	f:UIReconfigure(c)
	UpdateDynamicWindow("factory_status")
end

local function MinigameButton(f,c)
	-- Make sure we have correct ingredients
	local needed = 0
	for ing,count in pairs(f.product.recipe) do
		if gSim:GetInventory(ing) < count then needed = needed + 1 end
	end

	if needed == 0 then
		f:DoMiniGame(c)
	else
		local h = charDialogFont[2]
		
		-- From chartalk
		local cy = 45
		local textLeft = 187+5		-- rectL+5 for right
		local textWidth = 345
		local textTop = cy+31
		local textHeight = 172+25 - 3*h

		local buttons = {"ok"}
		local content = {
			Text {
				x=textLeft,y=textTop,w=textWidth,h=textHeight,
				name="bodyText",
				font=charDialogFont, label="factory_noingredients", flags=kVAlignCenter+kHAlignCenter
			}
		}

		local x = textLeft
		local y = textTop + textHeight
		local count = 0
		local half = needed - bsutil.floor(needed / 2)
		for ing,num in pairs(f.product.recipe) do
			if gSim:GetInventory(ing) < num then 
				if needed < 4 then
					-- One column, more or less centered
					x = textLeft + textWidth / 3
					y = textTop + textHeight + count * h
				elseif count >= half then
					x = textLeft + textWidth / 2
					y = textTop + textHeight + (count - half) * h
				else
					x = textLeft
					y = textTop + textHeight + count * h
				end
				
				local scale = h/const.ingIconHeight
				table.insert(content, Rollover { x=x,y=y, target="LItem:ByName('"..ing.."')", Bitmap { image="item/"..ing, scale=scale } })
				table.insert(content, Text { x=x+const.ingIconWidth*scale+2,y=y,w=textWidth/2,h=h, flags=kVAlignCenter+kHAlignLeft, label=ing, font=charDialogFont })
				count = count + 1
			end
		end
		
		DisplayDialog{"ui/chartalk.lua", content=Group(content), buttons=buttons, character=c, sound="charopen"}
	end
end

function LFactory:UIStatus(character)
	gCurrentFactory = self
	local c = character
	local layout =
	{
 		Bitmap
		{
			x=15,y=24,image="image/factory_status", mask="image/factory_status_mask",
			character:UIAppearance(567,1),
			
			-- labels
			AppendStyle { flags=kVHCenter, font=largeLabelFontY },
			Text { x=160-100,y=197,w=200,h=30, label="current_recipe" },
			Text { x=384-100,y=197,w=200,h=30, label="current_output" },
			Text { x=265,y=23,w=239,h=42, label="head_factory", font={ stdFontFace, 35, Color(133,66,6,255) } },

			Text { x=384-35,y=151,w=70,h=30, label="factory_perweek", font=infoFont, flags=kVAlignCenter+kHAlignCenter },

			-- buttons
			Button { x=662,y=288, name="leave", close=true, 
				graphics=yesButtonGraphics,
				command=function() gCurrentFactory=nil end },
			Button { x=662-64,y=288, name="help",
				graphics=helpButtonGraphics,
				command=function() DisplayHelp(self.helpkey,true) end },
			
			Button { x=130-71,y=276-15, name="reconfigure", type=kPush,
				graphics={"control/btn_recipe_up","control/btn_recipe_down","control/btn_recipe_over"},
				command = function() ReconfigureButton(gCurrentFactory,c) end },
			Text { x=80,y=341,w=160,h=25, label="reconfigure", font=largeLabelFontG },

			Button { x=462-21-38,y=219-21-15, type=kPush,
				graphics={"control/btn_minigame_up","control/btn_minigame_down","control/btn_minigame_over"},
				command = function() MinigameButton(gCurrentFactory, c) end },
			Text { x=429,y=341,w=160,h=25, label="minigame", font=largeLabelFontR },
				
			SetStyle(genButtonStyle),
			
--			AppendStyle { flags=kVHCenter, font=smallLabelFontY },
--			Text { x=648-50,y=300-15,w=100,h=30, label="#Help" },
--			Text { x=722-50,y=300-15,w=100,h=30, label="#Exit" },
		
			DynamicWindow { x=0,y=0,w=780,h=388, name="factory_status", contents="ui/factorystatus.lua" },
		},
	}
	layout =
	{
		StandardUI("factory"),
		Bitmap(layout),
	}
	DisplayDialog{"ui/dialog.lua", body=layout, sound="charopen" }
end

function LFactory:StatusPopup(p)
	local f = self:ByPort(p)

	local y = 5
	local h = infoFont[2]

	local r = f.rate or 0
	local r = "#"..tostring(r).." "..GetString("factory_perweek")
	if f.rate == 0 then r = "factory_zero" end
	
	local w = f.weeksSupply or 0
	if f.rate == 0 then w = nil
	elseif w == 0 then w = "factory_out"
	elseif w < 2 then w = "factory_short"
	else w = "#" .. GetString("factory_weeks", tostring(bsutil.floor(w)))
	end

	local layout =
	{
		color=PopupColor,
		x=0,y=0,w=800,h=600, shrink=true,
		AppendStyle{font=infoFont},
--		Text {x=5,y=y,w=kMax,h=h,label=f.port.name },
		Text {x=5,y=y,w=kMax,h=h,label=f.product.name },
		Text {x=5,y=y+h,w=kMax,h=h,label=r },
		Text {x=5,y=y+2*h,w=kMax,h=h,label=w },
	}
	y = y + 3*h
	
	if f.shortage then
		for ing,_ in pairs(f.shortage) do
			table.insert(layout, Bitmap {x=5,y=y, image="item/"..ing, scale=h/const.ingIconHeight })
			local s = ing
			if gSim.seen[ing] then
				local s1 = GetString("last_seen", GetString(gSim.seen[ing]), Dollars(gSim.seenPrices[ing]))
				s = "#"..GetString(s).." - "..s1
			end
			table.insert(layout, Text {x=5+h,y=y,w=kMax,h=h,label=s })
			y = y + h
		end
	end
	
	return { Rectangle(layout) }
end

-------------------------------------------------------------------------------
-- Standard UI: Factory Status

function LFactory:StatusWindow(xW,yW,wW,hW)
	local h = infoFont[2]
	local layout = { AppendStyle{font=infoFont} }
	
	-- NOTE: HARD CODING THIS ORDER, IF FACTORY LOCATIONS CHANGE...
	local ports={"sanfrancisco","newyork","riodejaneiro","london","mahajanga","hongkong"}
	local left = xW + 27
	local x=left
	local y=yW
	local squareW = wW / 3
	local squareH = hW / 2
	local squareDX = squareW
	local squareDY = squareH
	local count = 0
	
	local lx = 32
	local ly = h

	for _,pname in ipairs(ports) do
		local f = self._ByPort[pname]
		local t = { x=x,y=y,w=squareW,h=squareH }
		if f.owned then
			-- Build a rollover for all the factory info
			local r = { x=0,y=0, target="LFactory:StatusPopup('"..f.port.name.."')" }
			
			-- Animated factory status icon with production rate below, location name to right
			if f:Running() then
				table.insert(r, SimpleAnimation { x=0,y=0, animx=16,animy=16, anim="control/factory.xml", fps=4 })
			else
				table.insert(r, SimpleAnimation { x=0,y=0, animx=16,animy=16, anim="control/factory_idle.xml", fps=2 })
			end
			table.insert(r, Text { x=0,y=32,w=32,h=h, label="#"..f.rate, flags=kVAlignBottom+kHAlignCenter })
			table.insert(r, Text { x=32,y=0,w=squareW-32,h=h, label=pname, flags=kVAlignTop+kHAlignLeft })
		
			-- Product icon to right of status icon, below location name
			table.insert(r, Bitmap { x=33,y=h, image="item/"..f.product.name })
			table.insert(t, Rollover(r))
			
			-- Requried ingredients and status below product icon
			-- Count them and make sure they fit in the available space
			local ic = 0
			for _,icount in pairs(f.product.recipe) do ic = ic + icount end
			
			local iw = (squareW - 33) / 6
			local scale = iw / const.ingIconWidth
			local ih = squareH - h - const.prodIconHeight
			if scale * const.ingIconHeight > ih then
				scale = ih / const.ingIconHeight
				iw = scale * const.ingIconWidth
			end
			ih = scale * const.ingIconHeight
			
			-- Insert ingredients horizontally, centered under product icon
			local ix = (squareW - 33 - ic * iw) / 2
			local iy = h + const.prodIconHeight + 1
			for ing,icount in pairs(f.product.recipe) do
				for n=1,icount do
					local inv = gSim.inventory[ing] or 0
--					local tint = Color(255,255,255,255)
--					if inv < icount then tint = Color(255,64,64,255) end
--					table.insert(t, Rollover{x=ix,y=iy, target="LItem:ByName('"..ing.."')",
--						BitmapTint {x=0,y=0,scale=scale,image="item/"..ing, tint=tint} })

--					table.insert(t, Rollover{x=ix,y=iy, target="LItem:ByName('"..ing.."')",
--						Bitmap {x=0,y=0,scale=scale,image="item/"..ing} })
					table.insert(t, Bitmap {x=ix,y=iy, scale=scale,image="item/"..ing })
						
					if inv < icount then
						local s1 = ih/43
						local x1 = ix + (iw - s1 * 36) / 2
						table.insert(t, Bitmap{x=x1,y=iy,scale=s1,image="image/missed_ingredient"})
					end

					ix = ix + iw + 1
				end
			end
		elseif f.visited then
			-- Add "FOR SALE" sign
			local s = "#"..GetString("for_sale")..": "..Dollars(f.price)
			table.insert(t, BitmapTint { x=lx+9,y=ly,image="ports/factory",scale=33/286, tint=Color(92,92,92,255) })
			table.insert(t, Text { x=8,y=ly+33,w=squareW,h=16, label=s, flags=kVAlignTop+kHAlignLeft, font={plainFontFace,13,RedColor} })
			table.insert(t, Text { x=32,y=0,w=squareW-32,h=h, label=pname, flags=kVAlignTop+kHAlignLeft })
		else
			table.insert(t, BitmapTint { x=lx+9,y=ly,image="ports/factory",scale=33/286, tint=Color(92,92,92,255) })
		end
		
		table.insert(layout, Window(t))
		
		count = count + 1
		if count == 3 then
			x = left
			y = y + squareDY
		else
			x = x + squareDX
		end
	end
	
	return layout
end

-------------------------------------------------------------------------------
-- Game Save/Load

function LFactory:DoReset()
	LBuilding.DoReset(self)
	self.config = nil
	self.machines = nil
	self.product = nil
end

function LFactory:DoSaveGame(tb)
	-- Do regular LBuilding save
	LBuilding.DoSaveGame(self, tb)

	if self.owned then
		-- Save factory configurations and machinery
		tb.config = self.config
		tb.machines = self.machines
		tb.product = self.product.name
	end
end

function LFactory:DoLoadGame(tb)
	-- Do regular LBuilding load
	LBuilding.DoLoadGame(self, tb)
	
	-- Restore factory configurations and machinery
	if type(tb) == "table" and tb.owned then
		if type(tb.config) == "table" then self.config = tb.config end
		if type(tb.machines) == "table" then self.machines = tb.machines end
		if type(tb.product) == "string" then self:SetConfiguration(tb.product) end
	end
end
