--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Characters
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A Character is a unique individual NPC in the game.
	
	See external data description documentation.
	
	NOTES:
		startsquests: List of Quests a character starts (used by Quest:new)
--]]---------------------------------------------------------------------------

-- Character class definition
LCharacter =
{
	__tostring = function(t) return "{Character:" .. t.name .. "}" end,
	_ByName = {},
	_ByIndex = {},
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefineCharacter(t)
	return LCharacter:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global Character lists

function LCharacter:AllCharacters()
	return bsutil.ArrayIterator(self._ByIndex)
end

function LCharacter:ByName(name)
	local c = self._ByName[name]
	devmessage(c, "Character Undefined: "..name)
	return c
end

-------------------------------------------------------------------------------
-- "constructor"

function LCharacter:new(t)
	devmessage(t.name, "Character defined with no name")
	devmessage(not self._ByName[t.name], "Character " .. t.name .. " already defined")
	if t.name and not self._ByName[t.name] then
		t = t or {} setmetatable(t, self) self.__index = self

		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		
		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Quest handling

function LCharacter:StartsQuest(quest)
	self.startsquests = self.startsquests or {}
	table.insert(self.startsquests, quest)
end

-------------------------------------------------------------------------------
-- Generic User Interface

function LCharacter:UIAppearance(x,y,font)
	font = font or charNameFontLight
	local image = self.appearance
	if type(image) ~= "string" then image = "chardummy" end
	return Bitmap { x=x,y=y, image="characters/" .. image,
		Text { x=28,y=193,w=142,h=20, font=font, flags=kVAlignCenter+kHAlignCenter, label=self.name } }
end

function LCharacter:UIModalDialog(bodyText, buttons)
	buttons = buttons or {"ok"}
	return DisplayDialog{"ui/chartalk.lua", bodyText=bodyText, buttons=buttons, character=self, sound="charopen"}
end

function LCharacter:UIModalSpecial(bodyText)
	-- UI Modal Dialog without ledger
	local buttons = {"ok"}
	return DisplayDialog{"ui/chartalk.lua", bodyText=bodyText, buttons=buttons, character=self, sound="charopen", noLedger=true}
end

-------------------------------------------------------------------------------
-- Quest User Interface

function LCharacter:CompleteQuest(q)
	-- Complete the quest
	local endString = q:EndString()
	if endString then self:UIModalDialog(endString) end
	q:Complete()
	if q.followup then self:UIHandleQuest(q.followup) end
end

function LCharacter:OfferQuest(q)
	-- Offer the quest to the player
	q:Init()
	local introString = q:IntroString()
	if introString then
		local accept = q.accept or "accept"
		local reject = q.reject or "reject"
		local buttons
		if q.forceAccept then
			buttons = { accept }
		else
			buttons = { accept, reject }
		end
		local r = self:UIModalDialog(introString, buttons )
		if r == accept then q:Activate()
		else q:Reject()
		end
	else
		-- Optional: No quest intro text supplied, make SURE quest is flagged for forceAccept
		devmessage(q.forceAccept, "Quest with no QI_ text MUST be set for forceAccept: " .. tostring(q))
		q:Activate()
	end
end

function LCharacter:UIHandleQuest(q)
	if gSim:QuestsEnabled() then
		if q.active and q:EvaluateGoals() then
--			assert(q.ender == self)
			self:CompleteQuest(q)
		elseif q.active then
			self:UIModalDialog(q:IntermediateString())
		elseif not q.complete then
			self:OfferQuest(q)
		end
	end
end

-------------------------------------------------------------------------------
-- Character Generic Actions

function LCharacter:DoAction(building)
	local t = self.action
	
	-- At rank 0, force all characters to send you back to Evangeline
	-- if QuestCounter == 0, they'll do the same unless they have a quest to offer...
	-- During tutorial, send players back to the tutor
	if gSim.rank == 0 then
		ShowLedger("messages")
		return self:Converse("tutorofftrack")
	end
	
	-- Between tutorial and first quest, give them the novice nudge
	if gSim.mode == "empire" and not gSim.quest and LQuest._Variable.QuestCounter == 0 then
		return self:Converse("qi_novicenudge")
	end
	
	-- After X weeks of non-quest-completion, give a hint every N weeks
	-- Disable generic hints after rank 2
	if gSim.mode == "empire" and gSim.quest and gSim.time >= gSim.nextHintTime then
		gSim.nextHintTime = gSim.time + 12
		local h = "qh_"..gSim.quest.name
		if GetString(h) == "#####" then 
			if gSim.rank <= 2 then h = "qh_nudge"
			else h = nil
			end
		end
		if h then
			ShowLedger("messages")
			return self:Converse(h)
		end
	end
	
	-- Single action, just do it
	if type(t) == "function" then
		return t(self,building)
	end
	
	-- Pick an action at random from the table
	if type(t) == "table" then
		local n = bsutil.random(table.getn(t))
		local a = t[n]
		local ok = false
		if type(a) == "function" then ok = a(self,building) end
		
		-- Until something happens, try everything (else) from the table in order
		local i = 1
		while i <= table.getn(t) and not ok do
			if i ~= n then
				a = t[i]
				if type(a) == "function" then ok = a(self,building) end
			end
			i = i + 1
		end
		
		return ok
	end
	
	return false
end

-------------------------------------------------------------------------------
-- Random conversation

function LCharacter:Converse(s)
	if type(s) == "table" then s = s[bsutil.random(table.getn(s))] end
	if type(s) == "string" then
		self:UIModalDialog(bsutil.GetVariableString(s, self))
		return true
	end
	return false
end

-- For declaration in DefineCharacter
function Speak(t)
	-- (Note to self: t is a static table, ok to use it)
	return function(c) return c:Converse(t) end
end

-------------------------------------------------------------------------------
-- Tips

function LCharacter:GiveTip()
	local s = gSim:GetTip()
	if s then
		self:UIModalDialog(s)
		return true
	end
	return false
end

-- For declaration in DefineCharacter
GiveTip = LCharacter.GiveTip

-------------------------------------------------------------------------------
-- Dice

function LCharacter:PlayDice()
	if gSim.rank < 1 or gSim.money < 500 then return false end

	-- wager between 1/10 and 3/4 of player's money, rounded to 100, max 20,000
	local wager = bsutil.random(gSim.money/10, gSim.money*3/4)
	wager = bsutil.floor(wager/100) * 100
	local maxwager = bsutil.floor(gSim.money * .2)

	-- Randomize then cap pushes likelihood of getting max wager
	
	if wager < 100 then wager = 100
--	elseif wager > 1000000 then wager = 1000000
	elseif wager > 250000 then wager = 250000
	elseif wager > maxwager then wager = maxwager
	end
	wager = bsutil.floor(wager/100) * 100
	
	local diceIntro = bsutil.GetVariableString("dice_intro",{wager=Dollars(wager)})
	
	-- LAYOUT CONSTANTS FROM CHARTALK STANDARD CHARACTER DIALOG -------------
	local cy = 45
	local cx = 20+3 --0		-- 410 for right
	local rectL = 52
	local rectT = cy+26
	local rectW = 492
	local rectH = 239
	-- (( Usable area: 57,51 350x172 ))
	local textLeft = 187+5		-- rectL+5 for right
	local textWidth = 345
	local textTop = cy+31
	local textHeight = 172+25
	local buttonY = textTop+textHeight
	local x1 = 62		-- textLeft
	local x2 = x1+(rectW-10)/2+19
	
	local diceH = 50
	textHeight = textHeight - diceH
	
	local layout =
	{
		StandardUI("dice"),
		Window
		{
			x=101,y=10,w=597,h=301+cy, fit=true,
			Rectangle { x=52,y=cy+26,w=492,h=239+25, color=CharColor },
			
			DiceGame
			{
				x=0,y=0,w=kMax,h=kMax,
				
				Bitmap { x=textLeft+textWidth-110-35,y=textTop+textHeight-58,name="die4",image="image/die1" },
				Bitmap { x=textLeft+textWidth-55-35,y=textTop+textHeight-58,name="die3",image="image/die1" },
				Text { x=0,y=textTop+textHeight-58,w=textLeft+textWidth-115-35,h=50, label="#"..bsutil.GetVariableString("dice_myroll", {character=GetString(self.name)}), flags=kVAlignCenter+kHAlignRight },
				
				Bitmap { x=textLeft+textWidth-110-35,y=textTop+textHeight,name="die2",image="image/die1" },
				Bitmap { x=textLeft+textWidth-55-35,y=textTop+textHeight,name="die1",image="image/die1" },
				Text { x=0,y=textTop+textHeight,w=textLeft+textWidth-115-35,h=50, label="#"..bsutil.GetVariableString("dice_yourroll"), flags=kVAlignCenter+kHAlignRight },

				Text { x=textLeft,y=textTop,w=textWidth,h=textHeight-40, name="intro",
						font=charDialogFont, label="#"..diceIntro,
						flags=kVAlignCenter+kHAlignCenter },
				Text { x=textLeft,y=textTop,w=textWidth,h=textHeight-40, name="win_message",
						font=charDialogFont, label="#"..bsutil.GetVariableString("dice_win"),
						flags=kVAlignCenter+kHAlignCenter },
				Text { x=textLeft,y=textTop,w=textWidth,h=textHeight-40, name="lose_message",
						font=charDialogFont, label="#"..bsutil.GetVariableString("dice_lose"),
						flags=kVAlignCenter+kHAlignCenter },
				Text { x=textLeft,y=textTop,w=textWidth,h=textHeight-40, name="draw_message",
						font=charDialogFont, label="#"..bsutil.GetVariableString("dice_tie"),
						flags=kVAlignCenter+kHAlignCenter },

				Button { x=x2,y=buttonY, name="stop", graphics=genericButtonGraphics, command=function() StopDie() end },
				Text { x=x2+60,y=buttonY,w=150,h=57, name="stop_label", label="dice_stop", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
			},

			Bitmap { x=0,y=cy+0, image="image/dchar_left" },
			Bitmap { x=64+469,y=cy+0, image="image/dchar_right" },
			Bitmap { x=64,y=cy+0, image="image/dchar_top" },
			Bitmap { x=64,y=cy+235+25, image="image/dchar_bottom" },

			Button { x=x2,y=buttonY, name="roll", graphics=yesButtonGraphics,
				command=function()
					UtilDisableWindow("roll")
					UtilDisableWindow("roll_label")
					UtilDisableWindow("leave")
					UtilDisableWindow("leave_label")
					UtilEnableWindow("stop")
					UtilEnableWindow("stop_label")
					RollDice(tostring(wager))
					UpdateStandardUI()
				end
			},
			Text { x=x2+60,y=buttonY,w=150,h=57, name="roll_label", label="dice_roll", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },

			Button { x=x1,y=buttonY, name="leave", close=true, graphics=exitButtonGraphics },
			Text { x=x1+60,y=buttonY,w=150,h=57, name="leave_label", label="leave", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
			
			self:UIAppearance(cx,0),
		},
	}
	
	DisplayDialog{"ui/dialog.lua", sound="charopen", body=layout }
	return true
end

-- For declaration in DefineCharacter
PlayDice = LCharacter.PlayDice

-------------------------------------------------------------------------------
-- Selling an Item

-- For declaration in DefineCharacter
function SellItem(text, item)
	return function(c,b) return c:SellItem(text, item, b) end
end

function LCharacter:SellItem(text, item, building)
	local i=LItem:ByName(item)
	local speech = bsutil.GetVariableString(text, {item=item, price=Dollars(i:GetPrice())})

	gSim.seen[item] = gSim.port.name
	gSim.seenPrices[item] = gSim.portPrices[item]
	
	local function Buy()
		local count = tonumber(GetLabel("count")) or 0
		local cost = i:GetPrice()
		local total = count * cost
		if total > gSim.money then
			DisplayDialog { "ui/okdialog.lua", body="buy_notenough" }
		elseif total > 0 then
			gSim:AdjustInventory(i.name, count)
			local s = GetString("buy_notice", tostring(count), GetString(i.name), Dollars(total), Dollars(cost), GetString(gSim.port.name))
			
			gSim.purchasePrices[i.name] = cost
			gSim.purchasePorts[i.name] = gSim.port.name
			gSim:AdjustMoney(-total)
			PlayerMessage(s)

			-- A "day" passes when the player makes a purchase
			gSim:SubTick()
			CloseWindow()
			return true
		end
		return false
	end

	-- (constants from chartalk.lua)
	local textLeft = 187+5
	local textWidth = 345
	local textTop = 25+31+20+20
	local textHeight = 90
	local textCenter = textLeft + textWidth/2
	local layout = Group
	{
		Text {
			x=textLeft,y=textTop,w=textWidth,h=textHeight,
			name="bodyText",
			font=charDialogFont, label="#"..speech, flags=kVAlignCenter+kHAlignCenter
		},

		DynamicWindow { x=textLeft,y=textTop+textHeight+5,w=textWidth,h=const.ingIconWidth, name="receipt", ontext=true, contents="ui/farmreceipt.lua" },
--		Rollover { x=textCenter - const.ingIconWidth / 2, y=textTop+textHeight+5,
--			target="LItem:ByName('"..item.."')",
--			Bitmap { image="item/"..item } },

		Button { x=textCenter-27,y=textTop+textHeight+5+const.ingIconHeight+5, graphics={"control/register_tag"},
--			Rectangle { x=0,y=0,w=39,h=22, },
			TextEdit { x=0,y=2,w=kMax,h=kMax, length=4, name="count", label=GetString("buy"), ignore=kNumbersOnly, flags=kHAlignCenter+kVAlignCenter, font=marketFont },
			command = function()
				SetLabel("count","")
				ActivateNumberPad("count", textCenter-40-1+13-73+120-9 ,textTop+textHeight+5+const.ingIconHeight+5+147-155+35-5-23+3)
				SetFocus("count")
			end     
		},
	}
	
	local buttonY = 25+31+172+25+20
	local buttonLayout =
	{
		Button { x=62,y=buttonY, name="leave", close=true, graphics=yesButtonGraphics },
		Text { x=62+60,y=buttonY,w=150,h=57, label="leave", font=ButtonLabelFont, flags=kHAlignLeft+kVAlignCenter },
	}
	
	DisplayDialog{"ui/chartalk.lua", content=layout, character=self, sound="charopen",
		buttonLayout=buttonLayout, extra=NumberPad("buy",Buy) }
	return true
end
