--[[---------------------------------------------------------------------------
	Chocolatier Simulator: Building
	Copyright (c) 2006 Big Splash Games, LLC. All Rights Reserved.

	A Building is a visit-able location.
	
	See external documentation.
--]]---------------------------------------------------------------------------

-- Building class definition
LBuilding =
{
	__tostring = function(t) return "{Building:" .. tostring(t.name) .. "}" end,
	_ByName = {},
	_ByIndex = {},
}

-- So classes can inherit without calling LBuilding:new
LBuilding.__index = LBuilding

-------------------------------------------------------------------------------
-- Functions for data description

function Building(t) return LBuilding:new(t) end

function DefineBuilding(t)
	local x = t.type or LBuilding
	t.type = nil
	return x:new(t)
end

-------------------------------------------------------------------------------
-- "static" functions to access global LBuilding lists

function LBuilding:ByName(buildingName)
	return self._ByName[buildingName]
end

-------------------------------------------------------------------------------
-- "constructor"

function LBuilding:new(t)
	devmessage(t.name, "Building defined with no name")
	devmessage(not self._ByName[t.name], "Building "..t.name.." already defined")
	devmessage(t.character, "Building "..t.name.." contains no characters")
	if t.name and not self._ByName[t.name] then
		t = t or {} setmetatable(t, self) self.__index = self
		
		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		
		t.character = t.character or {}
		if type(t.character) == "string" then t.character = { t.character } end

		return t
	else
		return nil
	end
end

-------------------------------------------------------------------------------
-- Building purchase

function LBuilding:Purchase()
	self:MarkOwned()
end

function LBuilding:MarkVisited()
	-- Most buildings don't care...
end

function LBuilding:MarkOwned()
	-- Record this building as owned by the player
	self.owned = true
	self.visited = true
end

-------------------------------------------------------------------------------
-- Data post-processing

function LBuilding:SetPort(port)
	self.port = port
end

-------------------------------------------------------------------------------
-- Character access

function LBuilding:ContainsCharacter(c)
	for _,char in ipairs(self.character) do
		if char == c then return true end
	end
	return false
end

function LBuilding:ChooseCharacter()
	-- Look for quest-enders here (either incomplete or complete quests)
	local choice = nil
	local quest = nil
	
	if gSim:QuestsEnabled() then
		quest = gSim.quest
		
		-- See if there is an active quest with a quest-ender in this building...
		if quest and self:ContainsCharacter(quest.ender) then
			if gSim.time < quest.goals.endTime then quest = nil
			else choice = quest.ender
			end
		elseif not quest and not choice then
			-- No characters ending quests, look for eligible forceOffer quests
			local possible = {}
			for q in LQuest:ForcedQuests() do
				if not q.complete and not q.active and self:ContainsCharacter(q.starter) and q:EvaluateRequirements() then
					table.insert(possible, q)
				end
			end
			
			-- If no eligible forceOffer quests, check quest timer and rank and force a quest if necessary
			-- TODO: TWEAK 15 WEEK TIMER?
			gSim.questTimer = gSim.questTimer or gSim.time
			if gSim.rank >= 3 and gSim.questTimer - gSim.time >= 12 and table.getn(possible) == 0 then
				for _,c in ipairs(self.character) do
					if c.startsquests then
						for _,q in ipairs(c.startsquests) do
							if not q.complete and q:EvaluateRequirements() then
								table.insert(possible, q)
							end
						end
					end
				end
			end

			if table.getn(possible) > 0 then
				quest = possible[bsutil.random(table.getn(possible))]
				choice = quest.starter
			end
		end
	end
	
	if not choice then
		-- No quest-related characters here, pick at random
		quest = nil
		choice = self.character[bsutil.random(table.getn(self.character))]
		
		-- If choosing the ender of the current quest, pick again -- this character has already been ruled out
		-- do something totally definitive to rule out meeting the chosen character
		if gSim:QuestsEnabled() and gSim.quest and gSim.quest.ender == choice and table.getn(self.character) > 1 then
			choice = self.character[1]
			if gSim.quest.ender == choice then choice = self.character[2] end
		end

		-- If the player qualifies for a quest from this character, return that as well
		if gSim:QuestsEnabled() and choice.startsquests and not gSim.quest then
			local possible = {}
			for _,q in ipairs(choice.startsquests) do
				if not q.complete and not q.active and q:EvaluateRequirements() then
					table.insert(possible, q)
				end
			end
			if table.getn(possible) > 0 then
				quest = possible[bsutil.random(table.getn(possible))]
			end
		end
	end

	return choice,quest
end

-------------------------------------------------------------------------------
-- Market/Shop general support (in LBuilding for dual use instead of creating
--   a third sub-class)

function LBuilding.TotalSales(iterator)
	local total = 0
	for item in iterator do
		local s = tonumber(GetLabel(item.name)) or 0
		total = total + s * item:GetPrice()
	end
	return total
end

-------------------------------------------------------------------------------
-- Selection

function LBuilding:SelectClosure()
	return function()
		self:MarkVisited()
		self:UISelect()
	end
end

function LBuilding:UISelect()
	if self.ambient then SetAmbient(self.ambient) end
	local c,q = self:ChooseCharacter()
	if q then
		c:UIHandleQuest(q)
		self:LeaveBuilding(c,q)
	elseif c and c:DoAction(self) then
		UpdateStandardUI()
	else
		self:UISelectEmpty(c)
	end
	SetAmbient(kPortAmbient)
end

function LBuilding:UISelectEmpty(c)
	if self.name == "_empty" then
		if gSim.rank < 4 then
			c:UIModalDialog(GetString("building_empty"))
		else
			c:UIModalDialog(bsutil.GetVariableString("building_empty_mc"))
		end
	else
		local b = LBuilding:ByName("_empty")
		if b then b:UISelect() end
	end
end

function LBuilding:LeaveBuilding()
	-- Look for quest completion on the way out of the building
	if gSim:QuestsEnabled() and gSim.quest and self:ContainsCharacter(gSim.quest.ender) and gSim.quest:EvaluateGoals() then
		gSim.quest.ender:UIHandleQuest(gSim.quest)
	end
	
	UpdateStandardUI()
	gSim:CheckLoseCondition()
end

-------------------------------------------------------------------------------
-- User Interface

function LBuilding:OnPopupClick()
	bsutil.AutoSave()
	self:MarkVisited()
	self:UISelect()
end

function LBuilding:CreatePopup()
	local layout =
	{
		Rectangle
		{
			x=0,y=0,w=800,h=600, shrink=true,
			Text {x=5,y=5,w=kMax,h=buildingTagfont[2], label=self.name, font=buildingTagfont },
		}
	}
	return layout
end

-------------------------------------------------------------------------------
-- Data post-processing

function LBuilding:PostProcessOne()
	-- Convert named Characters to LCharacter objects
	for i,char in ipairs(self.character) do
		if type(char) == "string" then
			self.character[i] = LCharacter:ByName(char)
		end
	end
end

function LBuilding:PostProcessAll()
	for _,b in ipairs(self._ByIndex) do
		b:PostProcessOne()
	end
end

-------------------------------------------------------------------------------
-- Game Save/Load/Reset

function LBuilding:DoReset()
	self.owned = nil
	self.visited = nil
end

function LBuilding:ResetAll()
	for _,b in ipairs(LBuilding._ByIndex) do
		b:DoReset()
	end
end

function LBuilding:DoSaveGame(tb)
	tb.owned = self.owned
end

function LBuilding:DoLoadGame(tb)
	if type(tb) == "table" then
		if tb.owned then self:MarkOwned() end
		self.visited = true
	end
end

function LBuilding:SaveGameTable(t)
	t.buildings = {}
	for _,b in ipairs(self._ByIndex) do
		if b.visited or b.owned then
			t.buildings[b.name] = {}
			local tb = t.buildings[b.name]
			b:DoSaveGame(tb)
		end
	end
end

function LBuilding:LoadGameTable(t)
--	LBuilding:ResetAll()
	if type(t.buildings) == "table" then
		for name,tb in pairs(t.buildings) do
			local b = LBuilding:ByName(name)
			if b then b:DoLoadGame(tb) end
		end
	end
end
