//
//  XLDCueParser.m
//  XLD
//
//  Created by tmkk on 06/06/10.
//  Copyright 2006 tmkk. All rights reserved.
//
// Access to AccurateRip is regulated, see  http://www.accuraterip.com/3rdparty-access.htm for details.

#import "XLDCueParser.h"
#import "XLDController.h"
#import "XLDecoderCenter.h"
#import "XLDTrack.h"
#import "XLDRawDecoder.h"
#import "XLDCustomClasses.h"

static NSString *framesToMSFStr(xldoffset_t frames, int samplerate)
{
	int min = frames/samplerate/60;
	frames -= min*samplerate*60;
	int sec = frames/samplerate;
	frames -= sec*samplerate;
	int f = frames*75/samplerate;
	return [NSString stringWithFormat:@"%02d:%02d:%02d",min,sec,f];
}

static xldoffset_t timeToFrame(int min, int sec, int sector, int samplerate)
{
	xldoffset_t ret;
	ret = min*60*samplerate;
	ret += sec*samplerate;
	ret += sector*samplerate/75;
	return ret;
}

char *fgets_private(char *buf, int size, FILE *fp)
{
	int i;
	char c;
	
	for(i=0;i<size-1;) {
		if(fread(&c,1,1,fp) != 1) break;
		buf[i++] = c;
		if(c == '\n' || c == '\r') {
			break;
		}
	}
	if(i==0) return NULL;
	buf[i] = 0;
	return buf;
}

NSStringEncoding detectEncoding(FILE *fp)
{
	char buf[2048];
	char tmp[2048];
	char *ptr = buf;
	int len = 0;
	int minLength = INT_MAX;
	off_t pos = ftello(fp);
	CFStringRef asciiStr;
	CFStringRef sjisStr;
	CFStringRef cp932Str;
	CFStringRef jisStr;
	CFStringRef eucStr;
	CFStringRef utf8Str;
	int asciiLength;
	int sjisLength;
	int cp932Length;
	int jisLength;
	int eucLength;
	int utf8Length;
	
	
	while(fgets_private(tmp,2048,fp) != NULL) {
		int ret = strlen(tmp);
		len += ret;
		if(len > 2048) {
			len -= ret;
			break;
		}
		memcpy(ptr,tmp,ret);
		ptr += ret;
	}
	
	asciiStr = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len,  kCFStringEncodingASCII,false);
	sjisStr = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len, kCFStringEncodingShiftJIS,false);
	cp932Str = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len, kCFStringEncodingDOSJapanese,false);
	jisStr = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len, kCFStringEncodingISO_2022_JP,false);
	eucStr = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len, kCFStringEncodingEUC_JP,false);
	utf8Str = CFStringCreateWithBytes(NULL, (const UInt8 *)buf, len, kCFStringEncodingUTF8,false);
	
	asciiLength = (asciiStr) ? CFStringGetLength(asciiStr) : INT_MAX;
	sjisLength = (sjisStr) ? CFStringGetLength(sjisStr) : INT_MAX;
	cp932Length = (cp932Str) ? CFStringGetLength(cp932Str) : INT_MAX;
	jisLength = (jisStr) ? CFStringGetLength(jisStr) : INT_MAX;
	eucLength = (eucStr) ? CFStringGetLength(eucStr) : INT_MAX;
	utf8Length = (utf8Str) ? CFStringGetLength(utf8Str) : INT_MAX;
	
	if(asciiLength < minLength) minLength = asciiLength;
	if(sjisLength < minLength) minLength = sjisLength;
	if(cp932Length < minLength) minLength = cp932Length;
	if(jisLength < minLength) minLength = jisLength;
	if(eucLength < minLength) minLength = eucLength;
	if(utf8Length < minLength) minLength = utf8Length;
	
	//NSLog(@"%d,%d,%d,%d,%d,%d\n",asciiLength,sjisLength,cp932Length,jisLength,eucLength,utf8Length);
	
	if(asciiStr) CFRelease(asciiStr);
	if(sjisStr) CFRelease(sjisStr);
	if(cp932Str) CFRelease(cp932Str);
	if(jisStr) CFRelease(jisStr);
	if(eucStr) CFRelease(eucStr);
	if(utf8Str) CFRelease(utf8Str);
	fseeko(fp,pos,SEEK_SET);
	
	if(minLength == INT_MAX) return [NSString defaultCStringEncoding];
	if(minLength == asciiLength) return [NSString defaultCStringEncoding];
	if(minLength == sjisLength) return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingShiftJIS);
	if(minLength == cp932Length) return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingDOSJapanese);
	if(minLength == utf8Length) return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingUTF8);
	if(minLength == eucLength) return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingEUC_JP);
	if(minLength == jisLength) return CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingISO_2022_JP);
	
	return [NSString defaultCStringEncoding];
}

NSString *readMetadataFromLineBuf(char *cuebuf, int pos, NSStringEncoding enc, BOOL robustEncoding)
{
	int j;
	BOOL valid = NO;
	NSString *str;
	while(*(cuebuf+pos)==' ' || *(cuebuf+pos)=='\t' || *(cuebuf+pos)=='"') {
		valid = YES;
		pos++;
		if(*(cuebuf+pos-1)=='"') break;
	}
	if(!valid) return nil;
	j=pos;
	while(*(cuebuf+j)!='"' && *(cuebuf+j)!='\n' && *(cuebuf+j)!='\r' && *(cuebuf+j)!=0) j++;
	if(pos == j) return nil;
	*(cuebuf+j) = 0;
	NSData *dat = [NSData dataWithBytes:cuebuf+pos length:j-pos];
	str = [[NSString alloc] initWithData:dat encoding:enc];
	if(robustEncoding) {
		if(!str) str = [[NSString alloc] initWithData:dat encoding:[NSString defaultCStringEncoding]];
		if(!str) str = [[NSString alloc] initWithData:dat encoding:NSUTF8StringEncoding];
	}
	if(str) return [str autorelease];
	else return nil;
}

static unsigned int getDiscId(NSArray *trackList, xldoffset_t totalFrames)
{
	unsigned int i,cddbDiscId=0;
	int totalTrack = [trackList count];
	
	for(i=0;i<totalTrack;i++) {
		int trackOffset =  [(XLDTrack *)[trackList objectAtIndex:i] index];
		trackOffset /= 588;
		
		int r=0;
		int n=trackOffset/75 + 2;
		while(n>0) {
			r = r + (n%10);
			n = n/10;
		}
		cddbDiscId = cddbDiscId + r;
	}
	
	cddbDiscId = ((cddbDiscId % 255) << 24) | ((totalFrames/588/75 - [(XLDTrack *)[trackList objectAtIndex:0] index]/588/75) << 8) | totalTrack;
	return cddbDiscId;
}

@implementation XLDCueParser

- (id)init
{
	[super init];
	trackList = [[NSMutableArray alloc] init];
	checkList = [[NSMutableArray alloc] init];
	delegate = nil;
	fileToDecode = nil;
	title = nil;
	cover = nil;
	driveStr = nil;
	return self;
}

- (id)initWithDelegate:(id)del
{
	[self init];
	delegate = [del retain];
	return self;
}

- (void)dealloc
{
	[trackList release];
	[checkList release];
	if(delegate) [delegate release];
	if(fileToDecode) [fileToDecode release];
	if(title) [title release];
	if(driveStr) [driveStr release];
	[super dealloc];
}

- (IBAction)checkboxStatusChanged:(id)sender
{
	int i;
	if([sender state] == NSOnState) {
		if([sender shiftKeyPressed]) {
			for(i=0;i<[checkList count];i++) {
				[self checkAtIndex:i];
			}
		}
		else if([sender commandKeyPressed]) {
			NSTableView *table = (NSTableView *)[delegate table];
			for(i=0;i<[checkList count];i++) {
				if([table isRowSelected:i]) [self checkAtIndex:i];
			}
		}
	}
	else {
		if([sender shiftKeyPressed]) {
			for(i=0;i<[checkList count];i++) {
				[self uncheckAtIndex:i];
			}
		}
		else if([sender commandKeyPressed]) {
			NSTableView *table = (NSTableView *)[delegate table];
			for(i=0;i<[checkList count];i++) {
				if([table isRowSelected:i]) [self uncheckAtIndex:i];
			}
		}
	}
}

- (void)clean
{
	int i;
	for(i=0;i<[checkList count];i++) {
		[[checkList objectAtIndex:i] removeFromSuperview];
	}
	if(trackList) [trackList release];
	if(checkList) [checkList release];
	trackList = [[NSMutableArray alloc] init];
	checkList = [[NSMutableArray alloc] init];
	if(fileToDecode) [fileToDecode release];
	if(title) [title release];
	if(cover) [cover release];
	if(driveStr) [driveStr release];
	fileToDecode = nil;
	title = nil;
	cover = nil;
	driveStr = nil;
}

- (BOOL)isCompilationForTracks:(NSArray *)tracks
{
	if(!tracks) return NO;
	if([tracks count] == 0) return NO;
	NSString *albumArtist = [[[tracks objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUMARTIST];
	if(albumArtist && ![albumArtist isEqualToString:@""]) return NO;
	NSString *artist = nil;
	int i;
	for(i=0;i<[tracks count];i++) {
		NSString *str = [[[tracks objectAtIndex:i] metadata] objectForKey:XLD_METADATA_ARTIST];
		if(!str) continue;
		else if([str isEqualToString:@""]) continue;
		else if([str isEqualToString:@" "]) continue;
		else if([str isEqualToString:LS(@"multibyteSpace")]) continue;
		else if(!artist) artist = str;
		else if([artist isEqualToString:str]) continue;
		else return YES;
	}
	return NO;
}

- (BOOL)isCompilation
{
	return [self isCompilationForTracks:trackList];
}

- (NSString *)setTrackData:(NSMutableArray *)tracks forCueFile:(NSString *)file withDecoder:(id)decoder
{
	FILE *fp = fopen([file UTF8String],"rb");
	if(!fp) return nil;
	
	char cuebuf[512];
	int stat = 1,i,len,track=1;
	xldoffset_t gapIdx=0;
	NSString *titleStr = nil;
	NSString *date = nil;
	NSString *genre = nil;
	NSString *albumArtist = nil;
	NSString *catalog = nil;
	NSString *comment = nil;
	NSString *composer = nil;
	int totalDisc = 0;
	int discNumber = 0;
	unsigned int discid = 0;
	NSStringEncoding enc;
	BOOL hasPerformer = NO;
	unsigned char bom[] = {0xEF,0xBB,0xBF};
	unsigned char tmp[3];
	
	if([delegate encoding] == 0xFFFFFFFF) {
		enc = detectEncoding(fp);
	}
	else {
		enc = [delegate encoding];
	}
	int read = 0;
	
	fread(tmp,1,3,fp);
	if(memcmp(tmp,bom,3)) rewind(fp);
	
	while(read < 100*1024 && (fgets_private(cuebuf,512,fp) != NULL)) {
		i=0;
		len = strlen(cuebuf);
		read += len;
		while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
		if(i>len-3) continue;
		if(!strncasecmp(cuebuf+i,"TITLE",5)) {
			titleStr = readMetadataFromLineBuf(cuebuf,i+5,enc,YES);
		}
		else if(!strncasecmp(cuebuf+i,"PERFORMER",9)) {
			albumArtist = readMetadataFromLineBuf(cuebuf,i+9,enc,YES);
		}
		else if(!strncasecmp(cuebuf+i,"CATALOG",7)) {
			catalog = readMetadataFromLineBuf(cuebuf,i+7,enc,YES);
		}
		else if(!strncasecmp(cuebuf+i,"SONGWRITER",10)) {
			composer = readMetadataFromLineBuf(cuebuf,i+10,enc,YES);
		}
		else if(!strncasecmp(cuebuf+i,"DISCNUMBER",10)) {
			discNumber = [readMetadataFromLineBuf(cuebuf,i+10,enc,YES) intValue];
		}
		else if(!strncasecmp(cuebuf+i,"TOTALDISCS",10)) {
			totalDisc = [readMetadataFromLineBuf(cuebuf,i+10,enc,YES) intValue];
		}
		else if(!strncasecmp(cuebuf+i,"REM",3)) {
			i = i + 3;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			if(!strncasecmp(cuebuf+i,"DATE",4)) {
				date = readMetadataFromLineBuf(cuebuf,i+4,enc,YES);
			}
			else if(!strncasecmp(cuebuf+i,"GENRE",5)) {
				genre = readMetadataFromLineBuf(cuebuf,i+5,enc,YES);
			}
			else if(!strncasecmp(cuebuf+i,"COMMENT",7)) {
				comment = readMetadataFromLineBuf(cuebuf,i+7,enc,YES);
			}
			else if(!strncasecmp(cuebuf+i,"DISCNUMBER",10)) {
				discNumber = [readMetadataFromLineBuf(cuebuf,i+10,enc,YES) intValue];
			}
			else if(!strncasecmp(cuebuf+i,"TOTALDISCS",10)) {
				totalDisc = [readMetadataFromLineBuf(cuebuf,i+10,enc,YES) intValue];
			}
			else if(!strncasecmp(cuebuf+i,"DISCID",6)) {
				discid = strtoul([readMetadataFromLineBuf(cuebuf,i+6,enc,YES) UTF8String],NULL,16);
			}
		}
		else if(!strncasecmp(cuebuf+i,"TRACK",5)) break;
	}
	
	rewind(fp);
	fread(tmp,1,3,fp);
	if(memcmp(tmp,bom,3)) rewind(fp);
	
	while(fgets_private(cuebuf,512,fp) != NULL) {
		len = strlen(cuebuf);
		if(stat == 1) {
			i=0;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			if(i>len-4) continue;
			if(!strncasecmp(cuebuf+i,"TRACK",5)) {
				stat = 2;
				if([tracks count]) gapIdx = -1;
				else gapIdx = 0; // index 00 of track1 is always 0:0:0
				XLDTrack *trk = [[XLDTrack alloc] init];
				[[trk metadata] setObject:[NSNumber numberWithInt: track++] forKey:XLD_METADATA_TRACK];
				if(titleStr) [[trk metadata] setObject:titleStr forKey:XLD_METADATA_ALBUM];
				if(genre) [[trk metadata] setObject:genre forKey:XLD_METADATA_GENRE];
				if(albumArtist) [[trk metadata] setObject:albumArtist forKey:XLD_METADATA_ALBUMARTIST];
				if(catalog) [[trk metadata] setObject:catalog forKey:XLD_METADATA_CATALOG];
				if(comment) [[trk metadata] setObject:comment forKey:XLD_METADATA_COMMENT];
				if(composer) [[trk metadata] setObject:composer forKey:XLD_METADATA_COMPOSER];
				if(date) {
					[[trk metadata] setObject:date forKey:XLD_METADATA_DATE];
					if([date length] > 3) {
						int year = [[date substringWithRange:NSMakeRange(0,4)] intValue];
						if(year >= 1000 && year < 3000) [[trk metadata] setObject:[NSNumber numberWithInt:year] forKey:XLD_METADATA_YEAR];
					}
				}
				if(totalDisc) [[trk metadata] setObject:[NSNumber numberWithInt:totalDisc] forKey:XLD_METADATA_TOTALDISCS];
				if(discNumber) [[trk metadata] setObject:[NSNumber numberWithInt:discNumber] forKey:XLD_METADATA_DISC];
				[tracks addObject:trk];
				[trk release];
			}
		}
		else if(stat == 2) {
			i=0;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			//if(i>len-4) continue;
			if(!strncasecmp(cuebuf+i,"INDEX",5)) {
				i = i + 5;
				while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
				if(!strncasecmp(cuebuf+i,"00",2)) {
					i = i + 2;
					while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
					int min = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int sec = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int frame = atoi(cuebuf+i);
					gapIdx = timeToFrame(min,sec,frame,[decoder samplerate]);
				}
				else if(!strncasecmp(cuebuf+i,"01",2)) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					i = i + 2;
					while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
					int min = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int sec = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int frame = atoi(cuebuf+i);
					xldoffset_t idx = timeToFrame(min,sec,frame,[decoder samplerate]);
					[trk setIndex:idx];
					if(gapIdx != -1) [trk setGap:idx-gapIdx];
					if([tracks count] > 1) {
						XLDTrack *trk2 = [tracks objectAtIndex:[tracks count]-2];
						if(gapIdx != -1) [trk2 setFrames:gapIdx-[trk2 index]];
						else [trk2 setFrames:idx-[trk2 index]];
					}
					stat = 1;
				}
			}
			else if(!strncasecmp(cuebuf+i,"TITLE",5)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+5,enc,YES);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_TITLE];
				}
			}
			else if(!strncasecmp(cuebuf+i,"PERFORMER",9)) {
				hasPerformer = YES;
				NSString *str = readMetadataFromLineBuf(cuebuf,i+9,enc,YES);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_ARTIST];
				}
			}
			else if(!strncasecmp(cuebuf+i,"ISRC",4)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+4,enc,YES);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_ISRC];
				}
			}
			else if(!strncasecmp(cuebuf+i,"SONGWRITER",10)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+10,enc,YES);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_COMPOSER];
				}
			}
			else if(!strncasecmp(cuebuf+i,"REM",3)) {
				i = i + 3;
				while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
				if(!strncasecmp(cuebuf+i,"DATE",4)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+4,enc,YES);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_DATE];
						if([str length] > 3) {
							int year = [[str substringWithRange:NSMakeRange(0,4)] intValue];
							if(year >= 1000 && year < 3000) [[trk metadata] setObject:[NSNumber numberWithInt:year] forKey:XLD_METADATA_YEAR];
						}
					}
				}
				else if(!strncasecmp(cuebuf+i,"GENRE",5)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+5,enc,YES);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_GENRE];
					}
				}
				else if(!strncasecmp(cuebuf+i,"COMMENT",7)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+7,enc,YES);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_COMMENT];
					}
				}
			}
			else if(!strncasecmp(cuebuf+i,"FLAGS",5)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+5,enc,YES);
				if(str && [[str lowercaseString] isEqualToString:@"pre"]) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_PREEMPHASIS];
				}
			}
		}
	}
	fclose(fp);
	
	if(!discid) discid = getDiscId(tracks, [decoder totalFrames]);
	
	NSData *coverData = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	for(i=0;i<[tracks count];i++) {
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:[tracks count]] forKey:XLD_METADATA_TOTALTRACKS];
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithUnsignedInt:discid] forKey:XLD_METADATA_FREEDBDISCID];
		if(coverData) [[[tracks objectAtIndex:i] metadata] setObject:coverData forKey:XLD_METADATA_COVER];
		if(i==[tracks count]-1) [[tracks objectAtIndex:i] setSeconds:([decoder totalFrames]-[(XLDTrack *)[tracks objectAtIndex:i] index])/[decoder samplerate]];
		else [[tracks objectAtIndex:i] setSeconds:[[tracks objectAtIndex:i] frames]/[decoder samplerate]];
		if(!hasPerformer && albumArtist) {
			[[[tracks objectAtIndex:i] metadata] setObject:albumArtist forKey:XLD_METADATA_ARTIST];
			[[[tracks objectAtIndex:i] metadata] removeObjectForKey:XLD_METADATA_ALBUMARTIST];
		}
	}
	
	if([delegate canSetCompilationFlag] && [self isCompilationForTracks:tracks]) {
		for(i=0;i<[tracks count];i++) [[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_COMPILATION];
	}
	
	return titleStr;
}

- (id)decoderForCueSheet:(NSString *)file isRaw:(BOOL)raw promptIfNotFound:(BOOL)prompt error:(XLDErr *)error
{
	FILE *fp = fopen([file UTF8String],"rb");
	if(!fp) {
		*error = XLDReadErr;
		return nil;
	}
	
	int i,len;
	char cuebuf[512];
	NSStringEncoding enc;
	unsigned char bom[] = {0xEF,0xBB,0xBF};
	unsigned char tmp[3];
	NSString *path = nil;
	id decoder;
	
	if([delegate encoding] == 0xFFFFFFFF) {
		enc = detectEncoding(fp);
	}
	else {
		enc = [delegate encoding];
	}
	int read = 0;
	
	fread(tmp,1,3,fp);
	if(memcmp(tmp,bom,3)) rewind(fp);
	
	while(read < 100*1024 && (fgets_private(cuebuf,512,fp) != NULL)) {
		i=0;
		len = strlen(cuebuf);
		read += len;
		while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
		if(i>len-3) continue;
		if(!strncasecmp(cuebuf+i,"FILE",4)) {
			NSString *str = readMetadataFromLineBuf(cuebuf,i+4,enc,YES);
			if(str) {
				NSMutableString *mstr = [NSMutableString stringWithString:str];
				[mstr replaceOccurrencesOfString:@"\\" withString:@"/" options:0 range:NSMakeRange(0, [mstr length])];
				path = [[file stringByDeletingLastPathComponent] stringByAppendingPathComponent: [mstr lastPathComponent]];
			}
			break;
		}
	}
	
	if(!path) {
		fclose(fp);
		*error = XLDUnknownFormatErr;
		return nil;
	}
	
	if(![[NSFileManager defaultManager] fileExistsAtPath:path]) {
		if(!prompt) {
			fclose(fp);
			*error = XLDCancelErr;
			return nil;
		}
		NSOpenPanel *op = [NSOpenPanel openPanel];
		[op setCanChooseDirectories:NO];
		[op setCanChooseFiles:YES];
		[op setAllowsMultipleSelection:NO];
		[op setTitle:LS(@"choose file to split")];
		
		int ret = [op runModalForDirectory:[file stringByDeletingLastPathComponent] file:nil types:nil];
		if(ret != NSOKButton) {
			fclose(fp);
			*error = XLDCancelErr;
			return nil;
		}
		path = [op filename];
	}
	if(!raw) {
		decoder = [[delegate decoderCenter] preferredDecoderForFile:path];
		if(![(id <XLDDecoder>)decoder openFile:(char *)[path UTF8String]]) {
			[decoder closeFile];
			fclose(fp);
			*error = XLDReadErr;
			return nil;
		}
	}
	else {
		decoder = [[XLDRawDecoder alloc] initWithFormat:format endian:endian];
		if(![(XLDRawDecoder *)decoder openFile:(char *)[path UTF8String]]) {
			[decoder closeFile];
			[decoder release];
			fclose(fp);
			*error = XLDReadErr;
			return nil;
		}
		[decoder autorelease];
	}
	fclose(fp);
	*error = XLDNoErr;
	return decoder;
}

- (NSString *)setTrackData:(NSMutableArray *)tracks forCueData:(NSString *)cueData withDecoder:(id)decoder
{
	char cuebuf[512];
	int stat = 1,i,len,gapIdx=0,track=1;
	NSString *titleStr = nil;
	NSString *date = nil;
	NSString *genre = nil;
	NSString *albumArtist = nil;
	NSString *catalog = nil;
	NSString *comment = nil;
	NSString *composer = nil;
	int totalDisc = 0;
	int discNumber = 0;
	unsigned int discid = 0;
	NSRange range, subrange;
	BOOL hasPerformer = NO;
	cuebuf[511] = 0;
	
	if([[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS]) totalDisc = [[[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS] intValue];
	if([[decoder metadata] objectForKey:XLD_METADATA_DISC]) discNumber = [[[decoder metadata] objectForKey:XLD_METADATA_DISC] intValue];
	
	range = NSMakeRange(0, [cueData length]);
	while(range.length > 0) {
		subrange = [cueData lineRangeForRange:NSMakeRange(range.location, 0)];
		int lengthToCopy = strlen([[cueData substringWithRange:subrange] UTF8String])+1;
		lengthToCopy = (lengthToCopy > 511) ? 511 : lengthToCopy;
        memcpy(cuebuf,[[cueData substringWithRange:subrange] UTF8String],lengthToCopy);
		range.location = NSMaxRange(subrange);
        range.length -= subrange.length;
		
		i=0;
		len = strlen(cuebuf);
		while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
		if(i>len-3) continue;
		if(!strncasecmp(cuebuf+i,"TITLE",5)) {
			titleStr = readMetadataFromLineBuf(cuebuf,i+5,NSUTF8StringEncoding,NO);
		}
		else if(!strncasecmp(cuebuf+i,"PERFORMER",9)) {
			albumArtist = readMetadataFromLineBuf(cuebuf,i+9,NSUTF8StringEncoding,NO);
		}
		else if(!strncasecmp(cuebuf+i,"CATALOG",7)) {
			catalog = readMetadataFromLineBuf(cuebuf,i+7,NSUTF8StringEncoding,NO);
		}
		else if(!strncasecmp(cuebuf+i,"SONGWRITER",10)) {
			composer = readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO);
		}
		else if(!strncasecmp(cuebuf+i,"DISCNUMBER",10)) {
			discNumber = [readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO) intValue];
		}
		else if(!strncasecmp(cuebuf+i,"TOTALDISCS",10)) {
			totalDisc = [readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO) intValue];
		}
		else if(!strncasecmp(cuebuf+i,"REM",3)) {
			i = i + 3;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			if(!strncasecmp(cuebuf+i,"DATE",4)) {
				date = readMetadataFromLineBuf(cuebuf,i+4,NSUTF8StringEncoding,NO);
			}
			else if(!strncasecmp(cuebuf+i,"GENRE",5)) {
				genre = readMetadataFromLineBuf(cuebuf,i+5,NSUTF8StringEncoding,NO);
			}
			else if(!strncasecmp(cuebuf+i,"COMMENT",7)) {
				comment = readMetadataFromLineBuf(cuebuf,i+7,NSUTF8StringEncoding,NO);
			}
			else if(!strncasecmp(cuebuf+i,"DISCNUMBER",10)) {
				discNumber = [readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO) intValue];
			}
			else if(!strncasecmp(cuebuf+i,"TOTALDISCS",10)) {
				totalDisc = [readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO) intValue];
			}
			else if(!strncasecmp(cuebuf+i,"DISCID",6)) {
				discid = strtoul([readMetadataFromLineBuf(cuebuf,i+6,NSUTF8StringEncoding,NO) UTF8String],NULL,16);
			}
		}
		else if(!strncasecmp(cuebuf+i,"TRACK",5)) break;
	}
	
	range = NSMakeRange(0, [cueData length]);
	while(range.length > 0) {
		subrange = [cueData lineRangeForRange:NSMakeRange(range.location, 0)];
		int lengthToCopy = strlen([[cueData substringWithRange:subrange] UTF8String])+1;
		lengthToCopy = (lengthToCopy > 511) ? 511 : lengthToCopy;
        memcpy(cuebuf,[[cueData substringWithRange:subrange] UTF8String],lengthToCopy);
		range.location = NSMaxRange(subrange);
        range.length -= subrange.length;
		
		len = strlen(cuebuf);
		if(stat == 1) {
			i=0;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			if(i>len-4) continue;
			if(!strncasecmp(cuebuf+i,"TRACK",5)) {
				stat = 2;
				if([tracks count]) gapIdx = -1;
				else gapIdx = 0; // index 00 of track1 is always 0:0:0
				XLDTrack *trk = [[XLDTrack alloc] init];
				[[trk metadata] setObject:[NSNumber numberWithInt: track++] forKey:XLD_METADATA_TRACK];
				if(titleStr) [[trk metadata] setObject:titleStr forKey:XLD_METADATA_ALBUM];
				if(genre) [[trk metadata] setObject:genre forKey:XLD_METADATA_GENRE];
				if(albumArtist) [[trk metadata] setObject:albumArtist forKey:XLD_METADATA_ALBUMARTIST];
				if(catalog) [[trk metadata] setObject:catalog forKey:XLD_METADATA_CATALOG];
				if(comment) [[trk metadata] setObject:comment forKey:XLD_METADATA_COMMENT];
				if(composer) [[trk metadata] setObject:composer forKey:XLD_METADATA_COMPOSER];
				if(date) {
					[[trk metadata] setObject:date forKey:XLD_METADATA_DATE];
					if([date length] > 3) {
						int year = [[date substringWithRange:NSMakeRange(0,4)] intValue];
						if(year >= 1000 && year < 3000) [[trk metadata] setObject:[NSNumber numberWithInt:year] forKey:XLD_METADATA_YEAR];
					}
				}
				if(totalDisc) [[trk metadata] setObject:[NSNumber numberWithInt:totalDisc] forKey:XLD_METADATA_TOTALDISCS];
				if(discNumber) [[trk metadata] setObject:[NSNumber numberWithInt:discNumber] forKey:XLD_METADATA_DISC];
				[tracks addObject:trk];
				[trk release];
			}
		}
		else if(stat == 2) {
			i=0;
			while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
			//if(i>len-4) continue;
			if(!strncasecmp(cuebuf+i,"INDEX",5)) {
				i = i + 5;
				while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
				if(!strncasecmp(cuebuf+i,"00",2)) {
					i = i + 2;
					while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
					int min = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int sec = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int frame = atoi(cuebuf+i);
					gapIdx = timeToFrame(min,sec,frame,[decoder samplerate]);
				}
				else if(!strncasecmp(cuebuf+i,"01",2)) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					i = i + 2;
					while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
					int min = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int sec = atoi(cuebuf+i);
					while(*(cuebuf+i)!=':') i++;
					i++;
					int frame = atoi(cuebuf+i);
					xldoffset_t idx = timeToFrame(min,sec,frame,[decoder samplerate]);
					[trk setIndex:idx];
					if(gapIdx != -1) [trk setGap:idx-gapIdx];
					if([tracks count] > 1) {
						XLDTrack *trk2 = [tracks objectAtIndex:[tracks count]-2];
						if(gapIdx != -1) [trk2 setFrames:gapIdx-[trk2 index]];
						else [trk2 setFrames:idx-[trk2 index]];
					}
					stat = 1;
				}
			}
			else if(!strncasecmp(cuebuf+i,"TITLE",5)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+5,NSUTF8StringEncoding,NO);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_TITLE];
				}
			}
			else if(!strncasecmp(cuebuf+i,"PERFORMER",9)) {
				hasPerformer = YES;
				NSString *str = readMetadataFromLineBuf(cuebuf,i+9,NSUTF8StringEncoding,NO);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_ARTIST];
				}
			}
			else if(!strncasecmp(cuebuf+i,"SONGWRITER",10)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+10,NSUTF8StringEncoding,NO);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_COMPOSER];
				}
			}
			else if(!strncasecmp(cuebuf+i,"ISRC",4)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+4,NSUTF8StringEncoding,NO);
				if(str) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:str forKey:XLD_METADATA_ISRC];
				}
			}
			else if(!strncasecmp(cuebuf+i,"REM",3)) {
				i = i + 3;
				while(*(cuebuf+i)==' ' || *(cuebuf+i)=='\t') i++;
				if(!strncasecmp(cuebuf+i,"DATE",4)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+4,NSUTF8StringEncoding,NO);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_DATE];
						if([date length] > 3) {
							int year = [[str substringWithRange:NSMakeRange(0,4)] intValue];
							if(year >= 1000 && year < 3000) [[trk metadata] setObject:[NSNumber numberWithInt:year] forKey:XLD_METADATA_YEAR];
						}
					}
				}
				else if(!strncasecmp(cuebuf+i,"GENRE",5)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+5,NSUTF8StringEncoding,NO);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_GENRE];
					}
				}
				else if(!strncasecmp(cuebuf+i,"COMMENT",7)) {
					NSString *str = readMetadataFromLineBuf(cuebuf,i+7,NSUTF8StringEncoding,NO);
					if(str) {
						XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
						[[trk metadata] setObject:str forKey:XLD_METADATA_COMMENT];
					}
				}
			}
			else if(!strncasecmp(cuebuf+i,"FLAGS",5)) {
				NSString *str = readMetadataFromLineBuf(cuebuf,i+5,NSUTF8StringEncoding,NO);
				if(str && [[str lowercaseString] isEqualToString:@"pre"]) {
					XLDTrack *trk = [tracks objectAtIndex:[tracks count]-1];
					[[trk metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_PREEMPHASIS];
				}
			}
		}
	}
	
	if(!discid) discid = getDiscId(tracks, [decoder totalFrames]);
	
	NSData *coverData = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	for(i=0;i<[tracks count];i++) {
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:[tracks count]] forKey:XLD_METADATA_TOTALTRACKS];
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithUnsignedInt:discid] forKey:XLD_METADATA_FREEDBDISCID];
		if(coverData) [[[tracks objectAtIndex:i] metadata] setObject:coverData forKey:XLD_METADATA_COVER];
		if(i==[tracks count]-1) [[tracks objectAtIndex:i] setSeconds:([decoder totalFrames]-[(XLDTrack *)[tracks objectAtIndex:i] index])/[decoder samplerate]];
		else [[tracks objectAtIndex:i] setSeconds:[[tracks objectAtIndex:i] frames]/[decoder samplerate]];
		if(!hasPerformer && albumArtist) {
			[[[tracks objectAtIndex:i] metadata] setObject:albumArtist forKey:XLD_METADATA_ARTIST];
			[[[tracks objectAtIndex:i] metadata] removeObjectForKey:XLD_METADATA_ALBUMARTIST];
		}
	}
	
	if([delegate canSetCompilationFlag] && [self isCompilationForTracks:tracks]) {
		for(i=0;i<[tracks count];i++) [[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_COMPILATION];
	}
	
	return titleStr;
}

- (XLDErr)openFile:(NSString *)file
{
	int i;
	XLDErr error;
	NSString *titleStr = nil;
	id decoder = [self decoderForCueSheet:file isRaw:NO promptIfNotFound:YES error:&error];
	if(!decoder) {
		return error;
	}
	
	[self clean];
	rawMode = NO;
	
	titleStr = [self setTrackData:trackList forCueFile:file withDecoder:decoder];
	
	samplerate = [decoder samplerate];
	totalFrames = [decoder totalFrames];
	
	cover = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	if(cover) [cover retain];
	fileToDecode = [[decoder srcPath] retain];
	if(titleStr) {
		title = [titleStr retain];
	}
	else {
		title = [[file lastPathComponent] retain];
	}
	
	[decoder closeFile];
	
	[(NSTableView *)[delegate table] reloadData];
	
	for(i=0;i<[trackList count];i++) {
		XLDButton *button = [[XLDButton alloc] init];
		[button setButtonType:NSSwitchButton];
		[button setState:NSOnState];
		[[button cell] setControlSize:NSSmallControlSize];
		[button setTitle:@""];
		[button setTarget:self];
		[button setAction:@selector(checkboxStatusChanged:)];
		NSRect frame = [(NSTableView *)[delegate table] frameOfCellAtColumn:0 row:i];
		frame.origin.x += 11;
		[button setFrame:frame];
		[checkList addObject:button];
		[(NSTableView *)[delegate table] addSubview:button];
		[button release];
	}
	
	return XLDNoErr;
}

- (XLDErr)openFile:(NSString *)file withRawFormat:(XLDFormat)fmt endian:(XLDEndian)e
{
	int i;
	XLDErr error;
	NSString *titleStr = nil;
	XLDFormat origFmt = format;
	XLDEndian origEndian = endian;
	format = fmt;
	endian = e;
	id decoder = [self decoderForCueSheet:file isRaw:YES promptIfNotFound:YES error:&error];
	if(!decoder) {
		format = origFmt;
		endian = origEndian;
		return error;
	}
	
	[self clean];
	rawMode = YES;
	
	titleStr = [self setTrackData:trackList forCueFile:file withDecoder:decoder];
	
	samplerate = [decoder samplerate];
	totalFrames = [decoder totalFrames];
	rawOffset = 0;
	
	cover = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	if(cover) [cover retain];
	fileToDecode = [[decoder srcPath] retain];
	if(titleStr) {
		title = [titleStr retain];
	}
	else {
		title = [[file lastPathComponent] retain];
	}
	
	[decoder closeFile];
	
	[(NSTableView *)[delegate table] reloadData];
	
	for(i=0;i<[trackList count];i++) {
		XLDButton *button = [[XLDButton alloc] init];
		[button setButtonType:NSSwitchButton];
		[button setState:NSOnState];
		[[button cell] setControlSize:NSSmallControlSize];
		[button setTitle:@""];
		[button setTarget:self];
		[button setAction:@selector(checkboxStatusChanged:)];
		NSRect frame = [(NSTableView *)[delegate table] frameOfCellAtColumn:0 row:i];
		frame.origin.x += 11;
		[button setFrame:frame];
		[checkList addObject:button];
		[(NSTableView *)[delegate table] addSubview:button];
		[button release];
	}
	
	return XLDNoErr;
}

- (XLDErr)openFile:(NSString *)file withCueData:(NSString *)cueData decoder:(id)decoder
{
	int i;
	NSString *titleStr = nil;
	rawMode = NO;
	
	[self clean];
	
	titleStr = [self setTrackData:trackList forCueData:cueData withDecoder:decoder];
	
	samplerate = [decoder samplerate];
	totalFrames = [decoder totalFrames];
	
	cover = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	if(cover) [cover retain];
	fileToDecode = [file retain];
	if(titleStr) {
		title = [titleStr retain];
	}
	else {
		title = [[file lastPathComponent] retain];
	}
	
	[(NSTableView *)[delegate table] reloadData];
	
	for(i=0;i<[trackList count];i++) {
		XLDButton *button = [[XLDButton alloc] init];
		[button setButtonType:NSSwitchButton];
		[button setState:NSOnState];
		[[button cell] setControlSize:NSSmallControlSize];
		[button setTitle:@""];
		[button setTarget:self];
		[button setAction:@selector(checkboxStatusChanged:)];
		NSRect frame = [(NSTableView *)[delegate table] frameOfCellAtColumn:0 row:i];
		frame.origin.x += 11;
		[button setFrame:frame];
		[checkList addObject:button];
		[(NSTableView *)[delegate table] addSubview:button];
		[button release];
	}
	
	return XLDNoErr;
}

- (void)openRawFile:(NSString *)file withTrackData:(NSMutableArray *)arr decoder:(id)decoder
{
	[self openFile:file withTrackData:arr decoder:decoder];
	rawMode = YES;
	format.bps = [decoder bytesPerSample];
	format.channels = [decoder channels];
	format.isFloat = [decoder isFloat];
	format.samplerate = [decoder samplerate];
	endian = [(XLDRawDecoder *)decoder endian];;
	rawOffset = [(XLDRawDecoder *)decoder offset];
}

- (void)openFile:(NSString *)file withTrackData:(NSMutableArray *)arr decoder:(id)decoder
{
	int totalDisc = 0;
	int discNumber = 0;
	unsigned int discid = getDiscId(arr, [decoder totalFrames]);
	[self clean];
	rawMode = NO;
	cover = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	if(cover) [cover retain];
	samplerate = [decoder samplerate];
	totalFrames = [decoder totalFrames];
	fileToDecode = [file retain];
	title = [[file lastPathComponent] retain];
	[trackList addObjectsFromArray:arr];
	
	if([[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS]) totalDisc = [[[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS] intValue];
	if([[decoder metadata] objectForKey:XLD_METADATA_DISC]) discNumber = [[[decoder metadata] objectForKey:XLD_METADATA_DISC] intValue];
	
	int i;
	for(i=0;i<[trackList count];i++) {
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:i+1] forKey:XLD_METADATA_TRACK];
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:[trackList count]] forKey:XLD_METADATA_TOTALTRACKS];
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithUnsignedInt:discid] forKey:XLD_METADATA_FREEDBDISCID];
		if(cover) [[[trackList objectAtIndex:i] metadata] setObject:cover forKey:XLD_METADATA_COVER];
		if(totalDisc) [[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:totalDisc] forKey:XLD_METADATA_TOTALDISCS];
		if(discNumber) [[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:discNumber] forKey:XLD_METADATA_DISC];
		if(i==[trackList count]-1) [[trackList objectAtIndex:i] setSeconds:(totalFrames-[(XLDTrack *)[trackList objectAtIndex:i] index])/samplerate];
		else [[trackList objectAtIndex:i] setSeconds:[[trackList objectAtIndex:i] frames]/samplerate];
	}
	
	if([delegate canSetCompilationFlag] && [self isCompilationForTracks:trackList]) {
		for(i=0;i<[trackList count];i++) [[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_COMPILATION];
	}
	
	[(NSTableView *)[delegate table] reloadData];
	
	for(i=0;i<[trackList count];i++) {
		XLDButton *button = [[XLDButton alloc] init];
		[button setButtonType:NSSwitchButton];
		if([[trackList objectAtIndex:i] enabled]) [button setState:NSOnState];
		else {
			[[[trackList objectAtIndex:i] metadata] setObject:LS(@"(Data Track)") forKey:XLD_METADATA_TITLE];
			[button setState:NSOffState];
			[button setEnabled:NO];
		}
		[[button cell] setControlSize:NSSmallControlSize];
		[button setTitle:@""];
		[button setTarget:self];
		[button setAction:@selector(checkboxStatusChanged:)];
		NSRect frame = [(NSTableView *)[delegate table] frameOfCellAtColumn:0 row:i];
		frame.origin.x += 11;
		[button setFrame:frame];
		[checkList addObject:button];
		[(NSTableView *)[delegate table] addSubview:button];
		[button release];
	}
}

- (NSArray *)trackList
{
	int i;
	for(i=0;i<[checkList count];i++) {
		if([[checkList objectAtIndex:i] state] == NSOnState) [[trackList objectAtIndex:i] setEnabled:YES];
		else [[trackList objectAtIndex:i] setEnabled:NO];
	}
	
	return trackList;
}

- (NSString *)artistForTracks:(NSArray *)tracks
{
	if(!tracks) return @"";
	if([tracks count] == 0) return @"";
	if([[[tracks objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUMARTIST])
		return [[[tracks objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUMARTIST];
	NSString *artist = nil;
	int i;
	for(i=0;i<[tracks count];i++) {
		NSString *str = [[[tracks objectAtIndex:i] metadata] objectForKey:XLD_METADATA_ARTIST];
		if(!str) continue;
		else if([str isEqualToString:@""]) continue;
		else if([str isEqualToString:@" "]) continue;
		else if([str isEqualToString:LS(@"multibyteSpace")]) continue;
		else if(!artist) artist = str;
		else if([artist isEqualToString:str]) continue;
		else return LS(@"Various Artists");
	}
	if(!artist) return @"";
	return artist;
}

- (NSArray *)trackListForExternalCueSheet:(NSString *)file
{
	int i;
	XLDErr error;
	id decoder = [self decoderForCueSheet:file isRaw:NO promptIfNotFound:YES error:&error];
	if(!decoder) return nil;
	NSMutableArray *arr = [[NSMutableArray alloc] init];
	[self setTrackData:arr forCueFile:file withDecoder:decoder];
	for(i=0;i<[arr count];i++) {
		[[arr objectAtIndex:i] setEnabled:NO];
	}
	//[self setPreferredFilenameForTracks:arr];
	[decoder closeFile];
	return [arr autorelease];
}

- (NSArray *)trackListForDecoder:(id)decoder withEmbeddedCueData:(NSString *)cueData
{
	int i;
	NSMutableArray *arr = [[NSMutableArray alloc] init];
	[self setTrackData:arr forCueData:cueData withDecoder:decoder];
	for(i=0;i<[arr count];i++) {
		[[arr objectAtIndex:i] setEnabled:NO];
	}
	//[self setPreferredFilenameForTracks:arr];
	return [arr autorelease];
}

- (NSArray *)trackListForDecoder:(id)decoder withEmbeddedTrackList:(NSArray *)tracks
{
	int i;
	int totalDisc = 0;
	int discNumber = 0;
	unsigned int discid = getDiscId(tracks, [decoder totalFrames]);
	if([[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS]) totalDisc = [[[decoder metadata] objectForKey:XLD_METADATA_TOTALDISCS] intValue];
	if([[decoder metadata] objectForKey:XLD_METADATA_DISC]) discNumber = [[[decoder metadata] objectForKey:XLD_METADATA_DISC] intValue];
	NSData *coverData = [[decoder metadata] objectForKey:XLD_METADATA_COVER];
	
	for(i=0;i<[tracks count];i++) {
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:i+1] forKey:XLD_METADATA_TRACK];
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:[tracks count]] forKey:XLD_METADATA_TOTALTRACKS];
		[[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithUnsignedInt:discid] forKey:XLD_METADATA_FREEDBDISCID];
		if(coverData) [[[tracks objectAtIndex:i] metadata] setObject:coverData forKey:XLD_METADATA_COVER];
		if(totalDisc) [[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:totalDisc] forKey:XLD_METADATA_TOTALDISCS];
		if(discNumber) [[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithInt:discNumber] forKey:XLD_METADATA_DISC];
		if(i==[tracks count]-1) [[tracks objectAtIndex:i] setSeconds:([decoder totalFrames]-[(XLDTrack *)[tracks objectAtIndex:i] index])/[decoder samplerate]];
		else [[tracks objectAtIndex:i] setSeconds:[[tracks objectAtIndex:i] frames]/[decoder samplerate]];
		[[tracks objectAtIndex:i] setEnabled:NO];
	}
	
	if([delegate canSetCompilationFlag] && [self isCompilationForTracks:tracks]) {
		for(i=0;i<[tracks count];i++) [[[tracks objectAtIndex:i] metadata] setObject:[NSNumber numberWithBool:YES] forKey:XLD_METADATA_COMPILATION];
	}
	//[self setPreferredFilenameForTracks:tracks];
	
	return tracks;
}

- (NSString *)lengthOfTrack:(int)track
{
	if(track < [trackList count]-1) return framesToMSFStr([[trackList objectAtIndex:track] frames]+[[trackList objectAtIndex:track+1] gap],samplerate);
	return framesToMSFStr(totalFrames - [(XLDTrack *)[trackList objectAtIndex:track] index],samplerate);
}

- (NSString *)gapOfTrack:(int)track
{
	return framesToMSFStr([[trackList objectAtIndex:track] gap],samplerate);
}

- (NSString *)fileToDecode
{
	return fileToDecode;
}

- (NSString *)title
{
	return title;
}

- (NSString *)artist
{
	return [self artistForTracks:trackList];
}

- (void)checkAtIndex:(int)idx
{
	if(![[checkList objectAtIndex:idx] isEnabled]) return;
	[[checkList objectAtIndex:idx] setState:NSOnState];
}

- (void)uncheckAtIndex:(int)idx
{
	[[checkList objectAtIndex:idx] setState:NSOffState];
}

- (xldoffset_t)totalFrames
{
	return totalFrames;
}

- (XLDFormat)rawFormat
{
	return format;
}

- (XLDEndian)rawEndian
{
	return endian;
}

- (int)rawOffset
{
	return rawOffset;
}

- (BOOL)rawMode
{
	return rawMode;
}

- (NSData *)coverData
{
	return cover;
}

- (void)setCoverData:(NSData *)data
{
	int i;
	if(!trackList) return;
	if(cover) [cover release];
	cover = [data retain];
	for(i=0;i<[trackList count];i++) {
		[[[trackList objectAtIndex:i] metadata] setObject:data forKey:XLD_METADATA_COVER];
	}
}

- (NSMutableData *)cueData
{
	if(!trackList) return nil;
	int i,n=1;
	int offset = 0;
	BOOL removeRedundancy = NO;
	NSMutableData *data = [[NSMutableData alloc] init];
	id obj;
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUM])
		[data appendData:[[NSString stringWithFormat:@"TITLE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUMARTIST])
		[data appendData:[[NSString stringWithFormat:@"PERFORMER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	else if(![[self artist] isEqualToString:@""] && ![[self artist] isEqualToString:LS(@"Various Artists")]) {
		[data appendData:[[NSString stringWithFormat:@"PERFORMER \"%@\"\n",[self artist]] dataUsingEncoding:NSUTF8StringEncoding]];
		removeRedundancy = YES;
	}
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_CATALOG])
		[data appendData:[[NSString stringWithFormat:@"CATALOG %@\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_GENRE])
		[data appendData:[[NSString stringWithFormat:@"REM GENRE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_DATE])
		[data appendData:[[NSString stringWithFormat:@"REM DATE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	else if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_YEAR])
		[data appendData:[[NSString stringWithFormat:@"REM DATE %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_DISC])
		[data appendData:[[NSString stringWithFormat:@"REM DISCNUMBER %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_TOTALDISCS])
		[data appendData:[[NSString stringWithFormat:@"REM TOTALDISCS %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_FREEDBDISCID])
		[data appendData:[[NSString stringWithFormat:@"REM DISCID %08X\n",[obj unsignedIntValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_ALBUM_GAIN])
		[data appendData:[[NSString stringWithFormat:@"REM REPLAYGAIN_ALBUM_GAIN %.2f dB\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_ALBUM_PEAK])
		[data appendData:[[NSString stringWithFormat:@"REM REPLAYGAIN_ALBUM_PEAK %f\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	[data appendBytes:"FILE \"" length:6];
	rangeForCuesheet = NSMakeRange([data length],0);
	[data appendBytes:"\" WAVE\n" length:7];
	for(i=0;i<[trackList count];i++) {
		XLDTrack *track = [trackList objectAtIndex:i];
		if(![[checkList objectAtIndex:i] isEnabled] && ([[checkList objectAtIndex:i] state] == NSOffState)) {
			if(i==0) {
				offset = [track frames];
				if(i+1 < [trackList count]) offset += [[trackList objectAtIndex:i+1] gap];
			}
			continue;
		}
		[data appendData:[[NSString stringWithFormat:@"  TRACK %02d AUDIO\n",n] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_TITLE])
			[data appendData:[[NSString stringWithFormat:@"    TITLE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if((obj=[[track metadata] objectForKey:XLD_METADATA_ARTIST]) && !removeRedundancy)
			[data appendData:[[NSString stringWithFormat:@"    PERFORMER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_COMPOSER])
			[data appendData:[[NSString stringWithFormat:@"    SONGWRITER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_ISRC]) 
			[data appendData:[[NSString stringWithFormat:@"    ISRC %@\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_PREEMPHASIS]) {
			if([obj boolValue]) [data appendData:[@"    FLAGS PRE\n" dataUsingEncoding:NSUTF8StringEncoding]];
		}
		/*if(obj=[[track metadata] objectForKey:XLD_METADATA_GENRE])
			[data appendData:[[NSString stringWithFormat:@"    REM GENRE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_DATE])
			[data appendData:[[NSString stringWithFormat:@"    REM DATE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		else if(obj=[[track metadata] objectForKey:XLD_METADATA_YEAR])
			[data appendData:[[NSString stringWithFormat:@"    REM DATE %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];*/
		if(obj=[[track metadata] objectForKey:XLD_METADATA_COMMENT])
			[data appendData:[[NSString stringWithFormat:@"    REM COMMENT \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_GAIN])
			[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_GAIN %.2f dB\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
		if(obj=[[track metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_PEAK])
			[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_PEAK %f\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
		
		if([track gap] != 0)
			[data appendData:[[NSString stringWithFormat:@"    INDEX 00 %@\n",framesToMSFStr([track index]-[track gap]-offset,44100)] dataUsingEncoding:NSUTF8StringEncoding]];
		[data appendData:[[NSString stringWithFormat:@"    INDEX 01 %@\n",framesToMSFStr([track index]-offset,44100)] dataUsingEncoding:NSUTF8StringEncoding]];
		n++;
	}
	return [data autorelease];
}

- (NSMutableData *)cueDataForSeparatedFiles:(BOOL)HTOA
{
	if(!trackList) return nil;
	int i,n=1;
	BOOL removeRedundancy = NO;
	NSMutableData *data = [[NSMutableData alloc] init];
	id obj;
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUM])
		[data appendData:[[NSString stringWithFormat:@"TITLE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUMARTIST])
		[data appendData:[[NSString stringWithFormat:@"PERFORMER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	else if(![[self artist] isEqualToString:@""] && ![[self artist] isEqualToString:LS(@"Various Artists")]) {
		[data appendData:[[NSString stringWithFormat:@"PERFORMER \"%@\"\n",[self artist]] dataUsingEncoding:NSUTF8StringEncoding]];
		removeRedundancy = YES;
	}
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_CATALOG])
		[data appendData:[[NSString stringWithFormat:@"CATALOG %@\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_GENRE])
		[data appendData:[[NSString stringWithFormat:@"REM GENRE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_DATE])
		[data appendData:[[NSString stringWithFormat:@"REM DATE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
	else if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_YEAR])
		[data appendData:[[NSString stringWithFormat:@"REM DATE %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_DISC])
		[data appendData:[[NSString stringWithFormat:@"REM DISCNUMBER %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_TOTALDISCS])
		[data appendData:[[NSString stringWithFormat:@"REM TOTALDISCS %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_FREEDBDISCID])
		[data appendData:[[NSString stringWithFormat:@"REM DISCID %08X\n",[obj unsignedIntValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_ALBUM_GAIN])
		[data appendData:[[NSString stringWithFormat:@"REM REPLAYGAIN_ALBUM_GAIN %.2f dB\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	if(obj=[[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_ALBUM_PEAK])
		[data appendData:[[NSString stringWithFormat:@"REM REPLAYGAIN_ALBUM_PEAK %f\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
	
	for(i=0;i<[trackList count];i++) {
		XLDTrack *track = [trackList objectAtIndex:i];
		if(![[checkList objectAtIndex:i] isEnabled] && ([[checkList objectAtIndex:i] state] == NSOffState)) {
			continue;
		}
		[data appendData:[[NSString stringWithFormat:@"FILE \"%@.wav\" WAVE\n",[[delegate preferredFilenameForTrack:track index:i+1 baseDir:@"" createSubDir:NO singleImageMode:NO albumArtist:[self artist]] lastPathComponent]] dataUsingEncoding:NSUTF8StringEncoding]];
		if(([track gap] != 0) && (i != 0)) {
			[data appendData:[@"    INDEX 01 00:00:00\n" dataUsingEncoding:NSUTF8StringEncoding]];
		}
		else {
			[data appendData:[[NSString stringWithFormat:@"  TRACK %02d AUDIO\n",n] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_TITLE])
				[data appendData:[[NSString stringWithFormat:@"    TITLE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if((obj=[[track metadata] objectForKey:XLD_METADATA_ARTIST]) && !removeRedundancy)
				[data appendData:[[NSString stringWithFormat:@"    PERFORMER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_COMPOSER])
				[data appendData:[[NSString stringWithFormat:@"    SONGWRITER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_ISRC])
				[data appendData:[[NSString stringWithFormat:@"    ISRC %@\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_PREEMPHASIS]) {
				if([obj boolValue]) [data appendData:[@"    FLAGS PRE\n" dataUsingEncoding:NSUTF8StringEncoding]];
			}
			/*if(obj=[[track metadata] objectForKey:XLD_METADATA_GENRE])
				[data appendData:[[NSString stringWithFormat:@"    REM GENRE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_DATE])
				[data appendData:[[NSString stringWithFormat:@"    REM DATE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			else if(obj=[[track metadata] objectForKey:XLD_METADATA_YEAR])
				[data appendData:[[NSString stringWithFormat:@"    REM DATE %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];*/
			if(obj=[[track metadata] objectForKey:XLD_METADATA_COMMENT])
				[data appendData:[[NSString stringWithFormat:@"    REM COMMENT \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_GAIN])
				[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_GAIN %.2f dB\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[track metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_PEAK])
				[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_PEAK %f\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
			if(([track gap] != 0) && (i == 0)) {
				if(HTOA) {
					[data appendData:[@"    INDEX 00 00:00:00\n" dataUsingEncoding:NSUTF8StringEncoding]];
					[data appendData:[[NSString stringWithFormat:@"    INDEX 01 %@\n",framesToMSFStr([track index],44100)] dataUsingEncoding:NSUTF8StringEncoding]];
				}
				else {
					[data appendData:[[NSString stringWithFormat:@"    PREGAP %@\n",framesToMSFStr([track index],44100)] dataUsingEncoding:NSUTF8StringEncoding]];
					[data appendData:[@"    INDEX 01 00:00:00\n" dataUsingEncoding:NSUTF8StringEncoding]];
				}
			}
			else [data appendData:[@"    INDEX 01 00:00:00\n" dataUsingEncoding:NSUTF8StringEncoding]];
		}
		n++;
		
		if((i != [trackList count]-1) && [[trackList objectAtIndex:i+1] gap]) {
			[data appendData:[[NSString stringWithFormat:@"  TRACK %02d AUDIO\n",n] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_TITLE])
				[data appendData:[[NSString stringWithFormat:@"    TITLE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if((obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_ARTIST]) && !removeRedundancy)
				[data appendData:[[NSString stringWithFormat:@"    PERFORMER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_COMPOSER])
				[data appendData:[[NSString stringWithFormat:@"    SONGWRITER \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_ISRC])
				[data appendData:[[NSString stringWithFormat:@"    ISRC %@\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_PREEMPHASIS]) {
				if([obj boolValue]) [data appendData:[@"    FLAGS PRE\n" dataUsingEncoding:NSUTF8StringEncoding]];
			}
			/*if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_GENRE])
				[data appendData:[[NSString stringWithFormat:@"    REM GENRE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_DATE])
				[data appendData:[[NSString stringWithFormat:@"    REM DATE \"%@\"\n",obj] dataUsingEncoding:NSUTF8StringEncoding]];
			else if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_YEAR])
				[data appendData:[[NSString stringWithFormat:@"    REM DATE %d\n",[obj intValue]] dataUsingEncoding:NSUTF8StringEncoding]];*/
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_GAIN])
				[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_GAIN %.2f dB\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
			if(obj=[[[trackList objectAtIndex:i+1] metadata] objectForKey:XLD_METADATA_REPLAYGAIN_TRACK_PEAK])
				[data appendData:[[NSString stringWithFormat:@"    REM REPLAYGAIN_TRACK_PEAK %f\n",[obj floatValue]] dataUsingEncoding:NSUTF8StringEncoding]];
			[data appendData:[[NSString stringWithFormat:@"    INDEX 00 %@\n",framesToMSFStr([track frames],44100)] dataUsingEncoding:NSUTF8StringEncoding]];
		}
	}
	return [data autorelease];
}

- (NSArray *)trackListForSingleFile
{
	int i;
	NSMutableArray *arr = [[NSMutableArray alloc] init];
	XLDTrack *trk = [[XLDTrack alloc] init];
	xldoffset_t index = -1;
	xldoffset_t length = 0;
	for(i=0;i<[trackList count];i++) {
		XLDTrack *track = [trackList objectAtIndex:i];
		if([[checkList objectAtIndex:i] state] == NSOffState) continue;
		if(index == -1) {
			if((i==0) && ([track gap]!=0)) index = 0;
			else index = [track index];
		}
		if([track frames] == -1) length = -1;
		else {
			length = [track index] + [track frames] - index;
			if(i<[trackList count]-1) length += [[trackList objectAtIndex:i+1] gap];
		}
	}
	
	[trk setIndex:index];
	[trk setFrames:length];
	
	id obj;
	if(obj = [[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_ALBUM])
		[[trk metadata] setObject:obj forKey:XLD_METADATA_ALBUM];
	if(![[self artist] isEqualToString:@""])
		[[trk metadata] setObject:[self artist] forKey:XLD_METADATA_ALBUMARTIST];
	if(obj = [[[trackList objectAtIndex:0] metadata] objectForKey:XLD_METADATA_COMPOSER])
		[[trk metadata] setObject:obj forKey:XLD_METADATA_COMPOSER];
	
	NSMutableData *cue = [self cueData];
	[cue replaceBytesInRange:rangeForCuesheet withBytes:"CDImage.wav" length:11];
	NSString *cueStr = [[NSString alloc] initWithData:cue encoding:NSUTF8StringEncoding];
	
	[[trk metadata] setObject:cueStr forKey:XLD_METADATA_CUESHEET];
	[cueStr release];
	
	if(cover) [[trk metadata] setObject:cover forKey:XLD_METADATA_COVER];
	if(length == -1) [trk setSeconds:(totalFrames-[(XLDTrack *)trk index])/samplerate];
	else [trk setSeconds:length/samplerate];
	
	//NSLog(@"index:%lld, length:%lld",index,length);
	
	[arr addObject:trk];
	[trk release];
	return [arr autorelease];
}

- (NSRange)rangeForCuesheet
{
	return rangeForCuesheet;
}

- (void)enableAllChecks
{
	int i;
	for(i=0;i<[checkList count];i++) {
		if([[checkList objectAtIndex:i] state] == NSOffState) continue;
		[[checkList objectAtIndex:i] setEnabled:YES];
	}
}

- (void)disableAllChecks
{
	int i;
	for(i=0;i<[checkList count];i++) {
		if(![[checkList objectAtIndex:i] isEnabled]) continue;
		[[checkList objectAtIndex:i] setState:NSOnState];
		[[checkList objectAtIndex:i] setEnabled:NO];
	}
}

- (void)setTitle:(NSString *)str
{
	if(title) [title release];
	title = [str retain];
}

- (void)setDriveStr:(NSString *)str
{
	if(driveStr) [driveStr release];
	driveStr = [str retain];
}

- (NSString *)driveStr
{
	return driveStr;
}

- (NSData *)accurateRipData
{
	int i,discId1=0,discId2=0,cddbDiscId=0;
	int totalTrack = [trackList count];
	int totalAudioTrack = [trackList count];
	if(![[checkList objectAtIndex:totalAudioTrack-1] isEnabled] && ![[trackList objectAtIndex:totalAudioTrack-1] enabled]) {
		totalAudioTrack--;
	}
	
	for (i=0;i<totalTrack;i++) {
		int trackOffset =  [(XLDTrack *)[trackList objectAtIndex:i] index];
		trackOffset /= 588;
		
		if(i<totalAudioTrack) {
			discId1 += trackOffset;
			discId2 += (trackOffset ? trackOffset : 1) * (i + 1);
		}
		int r=0;
		int n=trackOffset/75 + 2;
		while(n>0) {
			r = r + (n%10);
			n = n/10;
		}
		cddbDiscId = cddbDiscId + r;
	}
	
	discId1 += totalFrames/588;
	discId2 += totalFrames/588 * (totalAudioTrack+1);
	cddbDiscId = ((cddbDiscId % 255) << 24) | ((totalFrames/588/75 - [(XLDTrack *)[trackList objectAtIndex:0] index]/588/75) << 8) | totalTrack;
	//NSLog([NSString stringWithFormat:@"http://www.accuraterip.com/accuraterip/%01x/%01x/%01x/dBAR-%03d-%08x-%08x-%08x.bin",discId1 & 0xF, discId1>>4 & 0xF, discId1>>8 & 0xF, totalAudioTrack, discId1, discId2, cddbDiscId]);
	NSData *data = [NSData dataWithContentsOfURL:[NSURL URLWithString:[NSString stringWithFormat:@"http://www.accuraterip.com/accuraterip/%01x/%01x/%01x/dBAR-%03d-%08x-%08x-%08x.bin",discId1 & 0xF, discId1>>4 & 0xF, discId1>>8 & 0xF, totalAudioTrack, discId1, discId2, cddbDiscId]]];
	//[data writeToFile:@"/Users/tmkk/AR.dat" atomically:YES];
	return data;
}

- (xldoffset_t)firstAudioFrame
{
	if(![trackList count]) return 0;
	if(![[checkList objectAtIndex:0] isEnabled] && ([[checkList objectAtIndex:0] state] == NSOffState))
		return [(XLDTrack *)[trackList objectAtIndex:1] index];
	else
		return 0;
}

- (xldoffset_t)lastAudioFrame
{
	if(![trackList count]) return 0;
	if(![[checkList objectAtIndex:[checkList count]-1] isEnabled] && ([[checkList objectAtIndex:[checkList count]-1] state] == NSOffState))
		return [(XLDTrack *)[trackList objectAtIndex:[checkList count]-1] index];
	else
		return [(XLDTrack *)[trackList objectAtIndex:[checkList count]-1] index] + [(XLDTrack *)[trackList objectAtIndex:[checkList count]-1] frames];
}

@end
