//
//  XLDConverterTask.m
//  XLD
//
//  Created by tmkk on 07/11/14.
//  Copyright 2007 tmkk. All rights reserved.
//

#import "XLDConverterTask.h"
#import "XLDQueue.h"
#import "XLDOutput.h"
#import "XLDRawDecoder.h"
#import "XLDDefaultOutputTask.h"
#import <sys/time.h>
#import <math.h>

#import <cdio/cdio.h>
#import <cdio/cdda.h>
#import <cdio/paranoia.h>
#import "XLDCDDARipper.h"
#import "XLDCDDAResult.h"
#import "XLDAccurateRipDB.h"

#define REPLAYGAIN_SAMPLE_LENGTH 2205

@implementation XLDConverterTask

- (NSString *)formatStringForOutput:(id)obj
{
	if([NSStringFromClass([obj class]) isEqualToString:@"XLDDefaultOutput"]) {
		switch(pcmType) {
			case 0:
				return @"WAV";
			case 1:
				return @"AIFF";
			case 2:
				return @"PCM (little endian)";
			case 3:
				return @"PCM (big endian)";
			case 4:
				return @"WAVE64";
			default:
				return @"";
		}
	}
	else return [[obj class] pluginName];
}

- (id)init
{
	[super init];
	
	nameField = [[NSTextField alloc] init];
	[nameField setBordered:NO];
	[nameField setEditable:NO];
	[[nameField cell] setWraps:NO];
	[nameField setBackgroundColor:[NSColor controlColor]];
	[nameField setFont:[NSFont boldSystemFontOfSize:13]];
	
	statusField = [[NSTextField alloc] init];
	[statusField setBordered:NO];
	[statusField setEditable:NO];
	[[statusField cell] setWraps:NO];
	[statusField setBackgroundColor:[NSColor controlColor]];
	[statusField setTextColor:[NSColor grayColor]];
	[statusField setFont:[NSFont systemFontOfSize:11]];
	[statusField setStringValue:LS(@"Waiting")];
	
	speedField = [[NSTextField alloc] init];
	[speedField setBordered:NO];
	[speedField setEditable:NO];
	[[speedField cell] setWraps:NO];
	[speedField setBackgroundColor:[NSColor controlColor]];
	[speedField setTextColor:[NSColor darkGrayColor]];
	[speedField setFont:[NSFont systemFontOfSize:9]];
	
	progress = [[NSProgressIndicator alloc] init];
	[progress setControlSize:NSSmallControlSize];
	[progress setIndeterminate:NO];
	[progress setStyle:NSProgressIndicatorBarStyle];
	
	stopButton = [[NSButton alloc] init];
	[stopButton setButtonType:NSMomentaryLightButton];
	[[stopButton cell] setControlSize:NSSmallControlSize];
	[[stopButton cell] setBezelStyle:NSRoundedBezelStyle];
	[stopButton setFont:[NSFont systemFontOfSize:11]];
	[stopButton setAction:@selector(stopConvert:)];
	[stopButton setTarget:self];
	[stopButton setTitle:LS(@"Cancel")];
	
	superview = [[XLDView alloc] init];
	NSRect frame1;
	frame1.origin.x = 10;
	frame1.origin.y = 30;
	frame1.size.width = 385;
	frame1.size.height = 19;
	NSRect frame4 = frame1;
	frame4.origin.y = 19;
	frame4.size.height = 11;
	NSRect frame2 = frame1;
	frame2.origin.y = 4;
	frame2.size.height = 13;
	frame2.size.width -= 100;
	NSRect frame3 = frame2;
	frame3.origin.x += frame2.size.width + 15;
	frame3.origin.y = 4;
	frame3.size.width = 65;
	frame3.size.height += 10;
	NSRect frame5 = frame2;
	frame5.origin.y = 11;
	
	[progress setFrame:frame2];
	[nameField setFrame:frame1];
	[stopButton setFrame:frame3];
	[statusField setFrame:frame5];
	[speedField setFrame:frame4];
	
	[superview addSubview:statusField];
	[superview addSubview:progress];
	[superview addSubview:nameField];
	[superview addSubview:speedField];
	[superview addSubview:stopButton];
	
	queue = nil;
	encoder = nil;
	decoder = nil;
	encoderTask = nil;
	decoderClass = nil;
	encoderArray = [[NSMutableArray alloc] init];
	encoderTaskArray = [[NSMutableArray alloc] init];
	track = nil;
	inFile = nil;
	outDir = nil;
	fixOffset = NO;
	pcmType = 0;
	tagWritable = YES;
	running = NO;
	stopConvert = NO;
	scaleType = 0;
	compressionQuality = 0.7f;
	scaleSize = 300;
	iTunesLib = nil;
	//mountOnEnd = NO;
	useParanoiaMode = YES;
	retryCount = 5;
	offsetCorrectionValue = 0;
	cueData = nil;
	resultObj = nil;
	testMode = NO;
	offsetFixupValue = 0;
	detectOffset = NO;
	//replayGainArray = [[NSMutableArray alloc] init];
	//replayGainSampleBuffer = malloc(sizeof(float)*REPLAYGAIN_SAMPLE_LENGTH*2);
	negativeSideSample = calloc(4704,sizeof(unsigned int));
	positiveSideSample = calloc(4704,sizeof(unsigned int));
	currentTrack = 0;
	totalTrack = 0;
	useOldEngine = NO;
	useC2Pointer = NO;
	outputPathStrArray = nil;
	appendBOM = NO;
	embedImages = YES;
	tmpPathStr = nil;
	dstPathStr = nil;
	tmpPathStrArray = nil;
	moveAfterFinish = NO;
	
	return self;
}

- (id)initWithQueue:(id)q
{
	[self init];
	queue = [q retain];
	
	[queue setMenuForItem:progress];
	[queue setMenuForItem:statusField];
	[queue setMenuForItem:nameField];
	[queue setMenuForItem:speedField];
	[queue setMenuForItem:superview];
	
	return self;
}

- (void)dealloc
{
	//NSLog(@"task dealloc");
	//[self hideProgress];
	[nameField release];
	[statusField release];
	[speedField release];
	[progress release];
	[stopButton release];
	if(queue) [queue release];
	if(encoder) [encoder release];
	if(encoderTask) [encoderTask release];
	if(decoder) [decoder release];
	if(track) [track release];
	if(inFile) [inFile release];
	if(outDir) [outDir release];
	if(iTunesLib) [iTunesLib release];
	if(cueData) [cueData release];
	if(resultObj) [resultObj release];
	[encoderArray release];
	[encoderTaskArray release];
	//[replayGainArray release];
	//free(replayGainSampleBuffer);
	free(negativeSideSample);
	free(positiveSideSample);
	if(outputPathStrArray) [outputPathStrArray release];
	if(tmpPathStr) [tmpPathStr release];
	if(tmpPathStrArray) [tmpPathStrArray release];
	if(dstPathStr) [dstPathStr release];
	[superview release];
	[super dealloc];
}

- (void)resizeImage:(NSData *)dat
{
	if(scaleType == XLDNoScale) return;
	
	NSBitmapImageRep *rep = [NSBitmapImageRep imageRepWithData:dat];
	if(!rep) {
		NSImage *img = [[NSImage alloc] initWithData:dat];
		if(!img) return;
		rep = [NSBitmapImageRep imageRepWithData:[img TIFFRepresentation]];
		[img release];
		if(!rep) return;
	}
	
	int beforeX = [rep pixelsWide];
	int beforeY = [rep pixelsHigh];
	int afterX;
	int afterY;
	
	if((scaleType&0xf) == XLDWidthScale) {
		if(!(scaleType&0x10) && (beforeX <= scaleSize)) return;
		afterY = round((double)beforeY * scaleSize/beforeX);
		afterX = scaleSize;
	}
	else if((scaleType&0xf) == XLDHeightScale) {
		if(!(scaleType&0x10) && (beforeY <= scaleSize)) return;
		afterX = round((double)beforeX * scaleSize/beforeY);
		afterY = scaleSize;
	}
	else if((scaleType&0xf) == XLDShortSideScale) {
		if(beforeX > beforeY) {
			if(!(scaleType&0x10) && (beforeY <= scaleSize)) return;
			afterX = round((double)beforeX * scaleSize/beforeY);
			afterY = scaleSize;
		}
		else {
			if(!(scaleType&0x10) && (beforeX <= scaleSize)) return;
			afterY = round((double)beforeY * scaleSize/beforeX);
			afterX = scaleSize;
		}
	}
	else {
		if(beforeX > beforeY) {
			if(!(scaleType&0x10) && (beforeX <= scaleSize)) return;
			afterY = round((double)beforeY * scaleSize/beforeX);
			afterX = scaleSize;
		}
		else {
			if(!(scaleType&0x10) && (beforeY <= scaleSize)) return;
			afterX = round((double)beforeX * scaleSize/beforeY);
			afterY = scaleSize;
		}
	}
	
	
	NSRect targetImageFrame = NSMakeRect(0,0,afterX,afterY);
	NSImage *targetImage = [[NSImage alloc] initWithSize:targetImageFrame.size];
	[targetImage lockFocus];
	[[NSGraphicsContext currentContext] setImageInterpolation:NSImageInterpolationHigh];
	[rep drawInRect: targetImageFrame];
	[targetImage unlockFocus];
	NSBitmapImageRep *newRep = [NSBitmapImageRep imageRepWithData:[targetImage TIFFRepresentation]];
	NSDictionary *dic = [NSDictionary dictionaryWithObject:[NSNumber numberWithFloat:compressionQuality] forKey:NSImageCompressionFactor];
	NSData *data = [newRep representationUsingType: NSJPEGFileType properties: dic];
	[[track metadata] setObject:data forKey:XLD_METADATA_COVER];
	[targetImage release];
}

- (void)hideProgress
{
	[superview removeFromSuperview];
}

- (void)showProgressInView:(NSTableView *)view row:(int)row
{
	position = row;
	NSRect frame1 = [view frameOfCellAtColumn:0 row:row];
	[superview setFrame:frame1];
	
	if(!running) {
		[progress setHidden:YES];
		[statusField setHidden:NO];
	}
	
	[view addSubview:superview];
}

- (void)beginConvert
{
	running = YES;
	NSString *outputPathStr;
	BOOL hasFormatSeparatedDirectoryStructure = NO;
	if(encoder) encoderTask = [(id <XLDOutput>)encoder createTaskForOutput];
	else {
		int i;
		for(i=0;i<[encoderArray count];i++) {
			id tmpEncoder = [(id <XLDOutput>)[encoderArray objectAtIndex:i] createTaskForOutput];
			[encoderTaskArray addObject:tmpEncoder];
			[tmpEncoder release];
		}
		outputPathStrArray = [[NSMutableArray alloc] init];
	}
	
	if([NSStringFromClass(decoderClass) isEqualToString:@"XLDRawDecoder"]) 
		decoder = [[XLDRawDecoder alloc] initWithFormat:rawFmt endian:rawEndian offset:rawOffset];
	else
		decoder = [[decoderClass alloc] init];
	
	if(encoderTask) {
		if([NSStringFromClass([encoderTask class]) isEqualToString:@"XLDDefaultOutputTask"]) {
			[encoderTask initOutputFormat];
			switch(pcmType) {
				case 0:
					[encoderTask setOutputFormatWAV];
					break;
				case 1:
					[encoderTask setOutputFormatAIFF];
					break;
				case 2:
					[encoderTask setOutputFormatLittleEndianLPCM];
					break;
				case 3:
					[encoderTask setOutputFormatBigEndianLPCM];
					break;
				case 4:
					[encoderTask setOutputFormatW64];
					break;
			}
		}
	}
	else {
		int i;
		for(i=0;i<[encoderTaskArray count];i++) {
			id tmpEncoderTask = [encoderTaskArray objectAtIndex:i];
			if([NSStringFromClass([tmpEncoderTask class]) isEqualToString:@"XLDDefaultOutputTask"]) {
				[tmpEncoderTask initOutputFormat];
				switch(pcmType) {
					case 0:
						[tmpEncoderTask setOutputFormatWAV];
						break;
					case 1:
						[tmpEncoderTask setOutputFormatAIFF];
						break;
					case 2:
						[tmpEncoderTask setOutputFormatLittleEndianLPCM];
						break;
					case 3:
						[tmpEncoderTask setOutputFormatBigEndianLPCM];
						break;
					case 4:
						[tmpEncoderTask setOutputFormatW64];
						break;
				}
			}
		}
	}
	
	if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"]) {
		[(XLDCDDARipper *)decoder setUseOldEngine:useOldEngine];
	}
	
	if(![(id <XLDDecoder>)decoder openFile:(char *)[inFile UTF8String]]) {
		fprintf(stderr,"error: cannot open\n");
		[decoder closeFile];
		[self hideProgress];
		[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
		return;
	}
	
	if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"]) {
		if(useParanoiaMode) [decoder setParanoiaMode:PARANOIA_MODE_FULL];
		//if(useParanoiaMode) [decoder setParanoiaMode:PARANOIA_MODE_VERIFY|PARANOIA_MODE_OVERLAP];
		else [(XLDCDDARipper *)decoder setParanoiaMode:PARANOIA_MODE_DISABLE];
		[(XLDCDDARipper *)decoder setRetryCount:retryCount];
		[(XLDCDDARipper *)decoder setOffsetCorrectionValue:offsetCorrectionValue];
		[(XLDCDDARipper *)decoder setUseC2Pointer:useC2Pointer];
		if(testMode) [(XLDCDDARipper *)decoder setTestMode];
	}
	
	if(testMode) {
		outputPathStr = @"/dev/null";
		moveAfterFinish = NO;
	}
	else {
		if(encoderTask) {
			NSString *desiredFilename = [track desiredFileName];
			if([desiredFilename length] > 240) {
				desiredFilename = [desiredFilename substringToIndex:239];
			}
			outputPathStr = [outDir stringByAppendingPathComponent:[desiredFilename stringByAppendingPathExtension:[encoderTask extensionStr]]];
			if(processOfExistingFiles == 1) {
				if([[NSFileManager defaultManager] fileExistsAtPath:outputPathStr]) {
					[decoder closeFile];
					[self hideProgress];
					[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
					return;
				}
			}
			else if(processOfExistingFiles == 0 || [outputPathStr isEqualToString:inFile]) {
				int i=1;
				while([[NSFileManager defaultManager] fileExistsAtPath:outputPathStr]) {
					outputPathStr = [outDir stringByAppendingPathComponent:[[NSString stringWithFormat:@"%@(%d)",desiredFilename,i] stringByAppendingPathExtension:[encoderTask extensionStr]]];
					i++;
				}
			}
		}
		else {
			int i,j;
			for(i=0;i<[encoderTaskArray count];i++) {
				BOOL duplicatedExt = NO;
				id tmpEncoderTask = [encoderTaskArray objectAtIndex:i];
				NSString *baseName;
				NSString *outDirForThisOutput = outDir;
				NSString *desiredFilenameForThisOutput = [track desiredFileName];
				for(j=0;j<[encoderTaskArray count];j++) {
					if((i!=j) && [[tmpEncoderTask extensionStr] isEqualToString:[[encoderTaskArray objectAtIndex:j] extensionStr]]) duplicatedExt = YES;
				}
				NSRange formatIndicatorRange = [outDir rangeOfString:@"[[[XLD_FORMAT_INDICATOR]]]"];
				if(formatIndicatorRange.location != NSNotFound) {
					outDirForThisOutput = [NSMutableString stringWithString:outDir];
					[(NSMutableString *)outDirForThisOutput replaceOccurrencesOfString:@"[[[XLD_FORMAT_INDICATOR]]]" withString:[self formatStringForOutput:[encoderArray objectAtIndex:i]] options:0 range:NSMakeRange(0, [outDirForThisOutput length])];
					duplicatedExt = NO;
					hasFormatSeparatedDirectoryStructure = YES;
				}
				formatIndicatorRange = [desiredFilenameForThisOutput rangeOfString:@"[[[XLD_FORMAT_INDICATOR]]]"];
				if(formatIndicatorRange.location != NSNotFound) {
					desiredFilenameForThisOutput = [NSMutableString stringWithString:desiredFilenameForThisOutput];
					[(NSMutableString *)desiredFilenameForThisOutput replaceOccurrencesOfString:@"[[[XLD_FORMAT_INDICATOR]]]" withString:[self formatStringForOutput:[encoderArray objectAtIndex:i]] options:0 range:NSMakeRange(0, [desiredFilenameForThisOutput length])];
					duplicatedExt = NO;
					hasFormatSeparatedDirectoryStructure = YES;
				}
				if(duplicatedExt) desiredFilenameForThisOutput = [NSString stringWithFormat:@"%@(%@)",desiredFilenameForThisOutput,[[[encoderArray objectAtIndex:i] class] pluginName]];
				if([desiredFilenameForThisOutput length] > 240) {
					desiredFilenameForThisOutput = [desiredFilenameForThisOutput substringToIndex:239];
				}
				
				baseName = [outDirForThisOutput stringByAppendingPathComponent:desiredFilenameForThisOutput];
				outputPathStr = [baseName stringByAppendingPathExtension:[tmpEncoderTask extensionStr]];
				if(processOfExistingFiles == 1) {
					if([[NSFileManager defaultManager] fileExistsAtPath:outputPathStr]) {
						[encoderTaskArray removeObjectAtIndex:i];
						[encoderArray removeObjectAtIndex:i];
						i--;
						continue;
					}
				}
				else if(processOfExistingFiles == 0 || [outputPathStr isEqualToString:inFile]) {
					j=1;
					while([[NSFileManager defaultManager] fileExistsAtPath:outputPathStr]) {
						outputPathStr = [[NSString stringWithFormat:@"%@(%d)",baseName,j] stringByAppendingPathExtension:[tmpEncoderTask extensionStr]];
						j++;
					}
				}
				//NSLog(outputPathStr);
				[outputPathStrArray addObject:outputPathStr];
			}
			if([encoderTaskArray count] == 0) {
				[decoder closeFile];
				[self hideProgress];
				[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
				return;
			}
			outputPathStr = [outputPathStrArray objectAtIndex:0];
		}
		
		if(cueData) {
			if(encoderTask) {
				NSString *desiredFilename = [track desiredFileName];
				if([desiredFilename length] > 240) {
					desiredFilename = [desiredFilename substringToIndex:239];
				}
				NSString *cuePathStr = [outDir stringByAppendingPathComponent:[desiredFilename stringByAppendingPathExtension:@"cue"]];
				if(processOfExistingFiles != 2 || [cuePathStr isEqualToString:inFile]) {
					int i=1;
					while([[NSFileManager defaultManager] fileExistsAtPath:cuePathStr]) {
						cuePathStr = [outDir stringByAppendingPathComponent:[[NSString stringWithFormat:@"%@(%d)",desiredFilename,i] stringByAppendingPathExtension:@"cue"]];
						i++;
					}
				}
				NSString *name = [outputPathStr lastPathComponent];
				unsigned char bom[] = {0xEF,0xBB,0xBF};
				[cueData replaceBytesInRange:cueRange withBytes:[[name dataUsingEncoding:NSUTF8StringEncoding] bytes] length:[[name dataUsingEncoding:NSUTF8StringEncoding] length]];
				if(appendBOM) [cueData replaceBytesInRange:NSMakeRange(0,0) withBytes:bom length:3];
				[cueData writeToFile:cuePathStr atomically:YES];
				if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"] && resultObj) {
					[resultObj setCuePath:cuePathStr];
					[resultObj setAppendBOM:appendBOM];
				}
			}
			else {
				int i,j;
				NSMutableArray *cuePathStrArray = [[[NSMutableArray alloc] init] autorelease];
				unsigned char bom[] = {0xEF,0xBB,0xBF};
				if(appendBOM) {
					[cueData replaceBytesInRange:NSMakeRange(0,0) withBytes:bom length:3];
					cueRange.location += 3;
				}
				for(i=0;i<[outputPathStrArray count];i++) {
					NSString *cuePathStr;
					NSString *basename;
					if(hasFormatSeparatedDirectoryStructure)
						basename = [[outputPathStrArray objectAtIndex:i] stringByDeletingPathExtension];
					else
						basename = [NSString stringWithFormat:@"%@(%@)",[[outputPathStrArray objectAtIndex:i] stringByDeletingPathExtension],[[[encoderArray objectAtIndex:i] class] pluginName]];
					cuePathStr = [basename stringByAppendingPathExtension:@"cue"];
					if(processOfExistingFiles != 2 || [cuePathStr isEqualToString:inFile]) {
						j=1;
						while([[NSFileManager defaultManager] fileExistsAtPath:cuePathStr]) {
							cuePathStr = [[NSString stringWithFormat:@"%@(%d)",basename,j] stringByAppendingPathExtension:@"cue"];
							j++;
						}
					}
					NSString *name = [[outputPathStrArray objectAtIndex:i] lastPathComponent];
					[cueData replaceBytesInRange:cueRange withBytes:[[name dataUsingEncoding:NSUTF8StringEncoding] bytes] length:[[name dataUsingEncoding:NSUTF8StringEncoding] length]];
					[cueData writeToFile:cuePathStr atomically:YES];
					cueRange.length = [[name dataUsingEncoding:NSUTF8StringEncoding] length];
					[cuePathStrArray addObject:cuePathStr];
				}
				if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"] && resultObj) {
					[resultObj setCuePathArray:cuePathStrArray];
					[resultObj setAppendBOM:appendBOM];
				}
			}
		}
	}
	
	if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"] && resultObj) {
		id obj = [[track metadata] objectForKey:XLD_METADATA_TRACK];
		if(obj) currentTrack = [obj intValue];
		else currentTrack = 0;
		totalTrack = ((XLDCDDAResult *)resultObj)->trackNumber;
		cddaRipResult *result = [resultObj resultForIndex:currentTrack];
		if(currentTrack == 0) result->parent = resultObj;
		[(XLDCDDARipper *)decoder setResultStructure:result];
		if(testMode) result->testEnabled = YES;
		else {
			result->enabled = YES;
			if(encoder) result->filename = [outputPathStr retain];
			else result->filelist = [outputPathStrArray retain];
		}
		
		if(!testMode && [resultObj accurateRipDB] && (currentTrack > 0)) detectOffset = YES;
		if([resultObj logFileName]) {
			if(encoder) [resultObj addLogDirectory:outDir];
			else {
				int i;
				for(i=0;i<[outputPathStrArray count];i++) {
					[resultObj addLogDirectory:[[outputPathStrArray objectAtIndex:i] stringByDeletingLastPathComponent]];
				}
			}
		}
	}
	
	if(encoder && iTunesLib) {
		NSString *tmpStr = iTunesLib;
		NSMutableString *filename = [NSMutableString stringWithString:outputPathStr];
		[filename  replaceOccurrencesOfString:@"\"" withString:@"\\\\\\\"" options:0 range:NSMakeRange(0, [outputPathStr length])];
		iTunesLib = [[NSString alloc] initWithFormat:iTunesLib,filename];
		[tmpStr release];
	}
	
	[decoder seekToFrame:index];
	if([(id <XLDDecoder>)decoder error]) {
		fprintf(stderr,"error: cannot seek\n");
		[decoder closeFile];
		[self hideProgress];
		[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
		return;
	}
	
	/* remove cover art from track temporally */
	NSData *imgBackup = nil;
	if(!embedImages && [[track metadata] objectForKey:XLD_METADATA_COVER]) {
		imgBackup = [[[track metadata] objectForKey:XLD_METADATA_COVER] retain];
		[[track metadata] removeObjectForKey:XLD_METADATA_COVER];
	}
	
	if(tagWritable && [[track metadata] objectForKey:XLD_METADATA_COVER]) {
		[self resizeImage:[[track metadata] objectForKey:XLD_METADATA_COVER]];
		//[[[track metadata] objectForKey:XLD_METADATA_COVER] writeToFile: @"/Users/tmkk/test3.jpg" atomically: NO];
	}
	
	XLDFormat fmt;
	
	fmt.bps = [decoder bytesPerSample];
	fmt.channels = [decoder channels];
	fmt.samplerate = [decoder samplerate];
	fmt.isFloat = [decoder isFloat];
	
	if(encoderTask) {
		if(![(id <XLDOutputTask> )encoderTask setOutputFormat:fmt]) {
			if(imgBackup) {
				[[track metadata] setObject:imgBackup forKey:XLD_METADATA_COVER];
				[imgBackup release];
			}
			[encoderTask closeFile];
			[decoder closeFile];
			[self hideProgress];
			[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
			return;
		}
		[encoderTask setEnableAddTag:tagWritable];
		dstPathStr = [outputPathStr retain];
		if(moveAfterFinish) {
			NSString *tempDir = NSTemporaryDirectory();
			NSString *filePath = [tempDir stringByAppendingPathComponent:@"xld_XXXXXX.tmp"];
			size_t bufferSize = strlen([filePath fileSystemRepresentation]) + 1;
			char *buf = (char *)malloc(bufferSize);
			[filePath getFileSystemRepresentation:buf maxLength:bufferSize];
			mkstemps(buf, 6);
			tmpPathStr = [[NSString alloc] initWithUTF8String:buf];
			free(buf);
		}
		NSString *outFile = moveAfterFinish ? tmpPathStr : outputPathStr;
		if(![encoderTask openFileForOutput:outFile withTrackData:track]) {
			if(imgBackup) {
				[[track metadata] setObject:imgBackup forKey:XLD_METADATA_COVER];
				[imgBackup release];
			}
			fprintf(stderr,"error: cannot write file %s\n",[outFile UTF8String]);
			[encoderTask closeFile];
			[decoder closeFile];
			[self hideProgress];
			[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
			return;
		}
	}
	else {
		int i,fail=0;
		if(moveAfterFinish) {
			NSString *tempDir = NSTemporaryDirectory();
			tmpPathStrArray = [[NSMutableArray alloc] init];
			for(i=0;i<[encoderTaskArray count];i++) {
				NSString *filePath = [tempDir stringByAppendingPathComponent:@"xld_XXXXXX.tmp"];
				size_t bufferSize = strlen([filePath fileSystemRepresentation]) + 1;
				char *buf = (char *)malloc(bufferSize);
				[filePath getFileSystemRepresentation:buf maxLength:bufferSize];
				mkstemps(buf, 6);
				[tmpPathStrArray addObject:[NSString stringWithUTF8String:buf]];
				free(buf);
			}
		}
		NSArray *outArray = moveAfterFinish ? tmpPathStrArray : outputPathStrArray;
		for(i=0;i<[encoderTaskArray count];i++) {
			id tmpEncoderTask = [encoderTaskArray objectAtIndex:i];
			[(id <XLDOutputTask> )tmpEncoderTask setOutputFormat:fmt];
			[tmpEncoderTask setEnableAddTag:tagWritable];
			if(![tmpEncoderTask openFileForOutput:[outArray objectAtIndex:i] withTrackData:track]) {
				fprintf(stderr,"error: cannot write file %s\n",[[outArray objectAtIndex:i] UTF8String]);
				fail++;
			}
		}
		if(fail == [encoderTaskArray count]) {
			if(imgBackup) {
				[[track metadata] setObject:imgBackup forKey:XLD_METADATA_COVER];
				[imgBackup release];
			}
			for(i=0;i<[encoderTaskArray count];i++) [[encoderTaskArray objectAtIndex:i] closeFile];
			[decoder closeFile];
			[self hideProgress];
			[queue convertFinished:self];
			return;
		}
	}

	if(imgBackup) {
		[[track metadata] setObject:imgBackup forKey:XLD_METADATA_COVER];
		[imgBackup release];
	}
	
	[statusField setHidden:YES];
	[progress setHidden:NO];
	[NSThread detachNewThreadSelector:@selector(convert) toTarget:self withObject:nil];
}

- (void)stopConvert:(id)sender
{
	if(running) {
		if([NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"]) [decoder cancel];
		stopConvert = YES;
		[stopButton setEnabled:NO];
		return;
	}
	[self hideProgress];
	[queue convertFinished:self];
}


- (void)updateStatus
{
	[progress setDoubleValue:percent];
	[speedField setStringValue:[NSString stringWithFormat:LS(@"%.1f %%, %.1fx realtime, %d:%02d remaining"),percent,speed,(int)remainingMin,(int)remainingSec]];
}

- (void)queryDB
{
	unsigned int arHash;
	int i,currentOffset,queryResult;
	cddaRipResult *result = [resultObj resultForIndex:currentTrack];
	for(currentOffset=-2352;currentOffset<=2352;currentOffset++) {
		arHash = result->arHash;
		if(currentOffset < 0) {
			arHash = arHash + result->sampleSum*(-currentOffset);
			for(i=0;i<(-currentOffset);i++) {
				arHash += negativeSideSample[2352+currentOffset+i]*(i+1);
			}
			for(i=0;i<(-currentOffset);i++) {
				arHash -= positiveSideSample[2352+currentOffset+i]*(totalFrame+currentOffset+i+1) + positiveSideSample[2352+currentOffset+i]*(-currentOffset);
			}
		}
		else if(currentOffset > 0) {
			for(i=0;i<currentOffset;i++) {
				arHash += positiveSideSample[2352+i]*(totalFrame-currentOffset+i+1);
			}
			for(i=0;i<currentOffset;i++) {
				arHash -= negativeSideSample[2352+i]*(i+1-currentOffset);
			}
			arHash = arHash - result->sampleSum*currentOffset;
		}
		queryResult = [[resultObj accurateRipDB] isAccurateCRC:arHash forTrack:currentTrack];
		if(queryResult != -1) {
			[resultObj registerOffset:currentOffset withConfidence:queryResult];
			[result->detectedOffset setObject:[NSNumber numberWithInt:currentOffset] forKey:[NSNumber numberWithUnsignedInt:arHash]];
		}
	}
}

- (BOOL)writeBuffer:(int *)buffer ForMultipleTasks:(int)ret
{
	int i;
	BOOL result = YES;
	for(i=0;i<[encoderTaskArray count];i++) {
		if(![[encoderTaskArray objectAtIndex:i] writeBuffer:buffer frames:ret]) {
			fprintf(stderr,"error: cannot output sample\n");
			result = NO;
		}
	}
	return result;
}

- (void)convert
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	NSFileManager *fm = [NSFileManager defaultManager];
	struct timeval tv1,tv2,tv_start;
	int lasttrack = 0;
	xldoffset_t framesToCopy = totalFrame;
	xldoffset_t baseline;
	int *buffer = (int *)malloc(8192 * [decoder channels] * 4);
	int samplesperloop;
	BOOL error = NO;
	double speedIdx[20];
	int speedPos = -1;
	int i;
	//replayGainSampleCount = 0;
	speed = 0;
	
	if(encoderTask && [NSStringFromClass([encoderTask class]) isEqualToString:@"XLDAlacOutputTask"]) {
		samplesperloop = 2560;
	}
	else samplesperloop = 8192;
	if(totalFrame == -1) {
		lasttrack = 1;
		framesToCopy = [decoder totalFrames] - index;
		totalFrame = framesToCopy;
		//NSLog(@"%lld,%lld",totalFrame,framesToCopy);
	}
	
	if(detectOffset) {
		int *tmp = malloc(4704*4*2);
		if((currentTrack == 1) || (index == firstAudioFrame)) {
			for(i=0;i<2352;i++) negativeSideSample[i] = 0;
			[decoder seekToFrame:index];
			[decoder decodeToBufferWithoutReport:tmp frames:2352];
			for(i=2352;i<4704;i++)
				negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
			[decoder seekToFrame:index+totalFrame-2352];
			[decoder decodeToBufferWithoutReport:tmp frames:4704];
			for(i=0;i<4704;i++)
				positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		}
		else if((currentTrack == totalTrack) || (index+totalFrame == lastAudioFrame)) {
			[decoder seekToFrame:index-2352];
			[decoder decodeToBufferWithoutReport:tmp frames:4704];
			for(i=0;i<4704;i++)
				negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
			[decoder seekToFrame:index+totalFrame-2352];
			[decoder decodeToBufferWithoutReport:tmp frames:2352];
			for(i=0;i<2352;i++)
				positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
			for(i=2352;i<4704;i++) positiveSideSample[i] = 0;
		}
		else {
			[decoder seekToFrame:index-2352];
			[decoder decodeToBufferWithoutReport:tmp frames:4704];
			for(i=0;i<4704;i++)
				negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
			[decoder seekToFrame:index+totalFrame-2352];
			[decoder decodeToBufferWithoutReport:tmp frames:4704];
			for(i=0;i<4704;i++)
				positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		}
		[decoder seekToFrame:index];
		free(tmp);
	}
	
	if(fixOffset && ![NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"]) {
		int *tmpbuf = (int *)calloc(30*[decoder channels],4);
		if(encoderTask) [encoderTask writeBuffer:tmpbuf frames:30 - index];
		else {
			for(i=0;i<[encoderTaskArray count];i++) [[encoderTaskArray objectAtIndex:i] writeBuffer:tmpbuf frames:30 - index];
		}
		free(tmpbuf);
	}
	
	if(offsetFixupValue > 0) {
		[decoder seekToFrame:index+offsetFixupValue];
		framesToCopy -= offsetFixupValue;
	}
	else if(offsetFixupValue < 0) {
		int size = 0-offsetFixupValue;
		int *tmpbuf = (int *)calloc(size*[decoder channels],4);
		if(encoderTask) [encoderTask writeBuffer:tmpbuf frames:size];
		else {
			for(i=0;i<[encoderTaskArray count];i++) [[encoderTaskArray objectAtIndex:i] writeBuffer:tmpbuf frames:size];
		}
		free(tmpbuf);
		framesToCopy += offsetFixupValue;
	}
	
	baseline = framesToCopy;
	gettimeofday(&tv1,NULL);
	gettimeofday(&tv_start,NULL);
	do {
		NSAutoreleasePool *pool2 = [[NSAutoreleasePool alloc] init];
		if(stopConvert) {
			if(encoderTask) {
				//[encoderTask finalize];
				[encoderTask closeFile];
				if(!testMode) {
					if(moveAfterFinish) [fm removeFileAtPath:tmpPathStr handler:nil];
					else [fm removeFileAtPath:dstPathStr handler:nil];
				}
			}
			else {
				for(i=0;i<[encoderTaskArray count];i++) {
					//[[encoderTaskArray objectAtIndex:i] finalize];
					[[encoderTaskArray objectAtIndex:i] closeFile];
					if(!testMode) {
						if(moveAfterFinish) [fm removeFileAtPath:[tmpPathStrArray objectAtIndex:i] handler:nil];
						else [fm removeFileAtPath:[outputPathStrArray objectAtIndex:i] handler:nil];
					}
				}
			}
			[pool2 release];
			goto finish;
		}
		//NSLog(@"begin decode");
		if(!lasttrack && framesToCopy < samplesperloop) samplesperloop = (int)framesToCopy;
		int ret = [decoder decodeToBuffer:(int *)buffer frames:samplesperloop];
		if([(id <XLDDecoder>)decoder error]) {
			fprintf(stderr,"error: cannot decode\n");
			error = YES;
			[pool2 release];
			break;
		}
		//NSLog(@"%d,%d",ret,samplesperloop);
		//NSLog(@"begin output");
		if(ret > 0) {
			if(encoderTask) {
				if(![encoderTask writeBuffer:buffer frames:ret]) {
					fprintf(stderr,"error: cannot output sample\n");
					error = YES;
					[pool2 release];
					break;
				}
			}
			else {
				if(![self writeBuffer:buffer ForMultipleTasks:ret]) {
					fprintf(stderr,"error: cannot output sample\n");
					error = YES;
					[pool2 release];
					break;
				}
			}
		}
		framesToCopy -= ret;
		//NSLog(@"end output");
		gettimeofday(&tv2,NULL);
		double elapsed1 = tv2.tv_sec-tv1.tv_sec + (tv2.tv_usec-tv1.tv_usec)*0.000001;
		double elapsed2 = tv2.tv_sec-tv_start.tv_sec + (tv2.tv_usec-tv_start.tv_usec)*0.000001;
		if(elapsed1 > 0.25) {
			percent = 100.0*((double)totalFrame-(double)framesToCopy)/(double)totalFrame;
			if(speed == 0.0) {
				speed = (((double)totalFrame-(double)framesToCopy)/(double)[decoder samplerate]) / elapsed2;
			}
			remainingSec = round((((double)framesToCopy)/(double)[decoder samplerate]) / speed);
			remainingMin = floor(remainingSec / 60.0);
			remainingSec = remainingSec - remainingMin*60;
			[self performSelectorOnMainThread:@selector(updateStatus) withObject:nil waitUntilDone:YES];
			tv1 = tv2;
		}
		if(elapsed2 > 1.0) {
			if((speedPos == -1) || (elapsed2 > 20)) {
				speed = (((double)baseline-(double)framesToCopy)/(double)[decoder samplerate]) / elapsed2;
				for(i=0;i<20;i++) speedIdx[i] = speed;
				speedPos = 0;
			}
			else {
				speedIdx[speedPos++] = (((double)baseline-(double)framesToCopy)/[decoder samplerate]) / elapsed2;
				speed = 0;
				for(i=0;i<20;i++) speed += speedIdx[i];
				speed /= 20.0;
				if(speedPos == 20) speedPos = 0;
			}
			baseline = framesToCopy;
			tv_start = tv2;
		}
		
		//NSLog(@"after wrote:%d,%d",ret,samplesperloop);
		if((!lasttrack && !framesToCopy) || (lasttrack && (ret < samplesperloop)) || !ret) {
			if(offsetFixupValue > 0) {
				int *tmpbuf = (int *)calloc(offsetFixupValue*[decoder channels],4);
				if(encoderTask) [encoderTask writeBuffer:tmpbuf frames:offsetFixupValue];
				else {
					for(i=0;i<[encoderTaskArray count];i++) [[encoderTaskArray objectAtIndex:i] writeBuffer:tmpbuf frames:offsetFixupValue];
				}
				free(tmpbuf);
			}
			percent = 100.0*((double)totalFrame-(double)framesToCopy)/(double)totalFrame;
			[progress setDoubleValue:percent];
			[pool2 release];
			break;
		}
		[pool2 release];
	} while(1);
	if(encoderTask) {
		if(!error) [encoderTask finalize];
		[encoderTask closeFile];
		if(error && !testMode) {
			if(moveAfterFinish) [fm removeFileAtPath:tmpPathStr handler:nil];
			else [fm removeFileAtPath:dstPathStr handler:nil];
		}
	}
	else {
		for(i=0;i<[encoderTaskArray count];i++) {
			if(!error) [[encoderTaskArray objectAtIndex:i] finalize];
			[[encoderTaskArray objectAtIndex:i] closeFile];
			if(error && !testMode) {
				if(moveAfterFinish) [fm removeFileAtPath:[tmpPathStrArray objectAtIndex:i] handler:nil];
				else [fm removeFileAtPath:[outputPathStrArray objectAtIndex:i] handler:nil];
			}
		}
	}
	
	if(!error) {
		NSMutableDictionary *attrDic = nil;
		if([[track metadata] objectForKey:XLD_METADATA_CREATIONDATE] || [[track metadata] objectForKey:XLD_METADATA_MODIFICATIONDATE]) {
			attrDic = [NSMutableDictionary dictionary];
			if([[track metadata] objectForKey:XLD_METADATA_CREATIONDATE]) {
				[attrDic setObject:[[track metadata] objectForKey:XLD_METADATA_CREATIONDATE] forKey:NSFileCreationDate];
			}
			if([[track metadata] objectForKey:XLD_METADATA_MODIFICATIONDATE]) {
				[attrDic setObject:[[track metadata] objectForKey:XLD_METADATA_MODIFICATIONDATE] forKey:NSFileModificationDate];
			}
		}
		if(encoder) {
			if(attrDic) {
				if(moveAfterFinish) [fm changeFileAttributes:attrDic atPath:tmpPathStr];
				else [fm changeFileAttributes:attrDic atPath:dstPathStr];
			}
			if(moveAfterFinish) [fm movePath:tmpPathStr toPath:dstPathStr handler:nil];
			if(iTunesLib) system([iTunesLib UTF8String]);
		}
		else {
			for(i=0;i<[outputPathStrArray count];i++) {
				if(attrDic) {
					if(moveAfterFinish) [fm changeFileAttributes:attrDic atPath:[tmpPathStrArray objectAtIndex:i]];
					else [fm changeFileAttributes:attrDic atPath:[outputPathStrArray objectAtIndex:i]];
				}
				if(moveAfterFinish) [fm movePath:[tmpPathStrArray objectAtIndex:i] toPath:[outputPathStrArray objectAtIndex:i] handler:nil];
				if(iTunesLib) {
					NSMutableString *filename = [NSMutableString stringWithString:[outputPathStrArray objectAtIndex:i]];
					[filename replaceOccurrencesOfString:@"\"" withString:@"\\\\\\\"" options:0 range:NSMakeRange(0, [filename length])];
					system([[NSString stringWithFormat:iTunesLib,filename] UTF8String]);
				}
			}
		}
		if(detectOffset) [self performSelectorOnMainThread:@selector(queryDB) withObject:nil waitUntilDone:YES];
	}
	
finish:
	free(buffer);
	[decoder closeFile];
	if(resultObj && [NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"]) {
		cddaRipResult *result = [resultObj resultForIndex:currentTrack];
		if(testMode) {
			result->testFinished = YES;
		}
		else {
			result->finished = YES;
			if(stopConvert) result->cancelled = YES;
		}
	}
	[progress removeFromSuperview];
	[speedField removeFromSuperview];
	[stopButton removeFromSuperview];
	if(error) {
		[superview setTag:1];
		[statusField setTextColor:[NSColor redColor]];
		[statusField setFont:[NSFont boldSystemFontOfSize:[[statusField font] pointSize]]];
		[statusField setStringValue:LS(@"Error has occured")];
		[statusField setHidden:NO];
	}
	else if(stopConvert) {
		[superview setTag:0];
		[statusField setFont:[NSFont boldSystemFontOfSize:[[statusField font] pointSize]]];
		[statusField setStringValue:LS(@"Cancelled")];
		[statusField setHidden:NO];
	}
	else {
		[superview setTag:0];
		[statusField setTextColor:[NSColor colorWithDeviceRed:0.25 green:0.35 blue:0.7 alpha:1]];
		[statusField setFont:[NSFont boldSystemFontOfSize:[[statusField font] pointSize]]];
		[statusField setStringValue:LS(@"Completed")];
		[statusField setHidden:NO];
	}
	//[self hideProgress];
	[queue performSelectorOnMainThread:@selector(convertFinished:) withObject:self waitUntilDone:NO];
	[pool release];
}

- (void)setFixOffset:(BOOL)flag
{
	fixOffset = flag;
}

- (void)setIndex:(xldoffset_t)idx
{
	index = idx;
}

- (void)setTotalFrame:(xldoffset_t)frame
{
	totalFrame = frame;
}

- (void)setDecoderClass:(Class)dec
{
	decoderClass = dec;
}

- (void)setEncoder:(id)enc
{
	if(encoder) [encoder release];
	encoder = [enc retain];
}

- (void)setEncoders:(id)enc
{
	[encoderArray removeAllObjects];
	[encoderArray addObjectsFromArray:enc];
}

- (void)setPCMType:(int)type
{
	pcmType = type;
}

- (void)setRawFormat:(XLDFormat)fmt
{
	rawFmt = fmt;
}

- (void)setRawEndian:(XLDEndian)e
{
	rawEndian = e;
}

- (void)setRawOffset:(int)offset
{
	rawOffset = offset;
}

- (void)setInputPath:(NSString *)path
{
	if(inFile) [inFile release];
	inFile = [path retain];
}

- (NSString *)outputDir
{
	return outDir;
}

- (void)setOutputDir:(NSString *)path
{
	if(outDir) [outDir release];
	outDir = [path retain];
}

- (void)setTagWritable:(BOOL)flag
{
	tagWritable = flag;
}

- (void)setTrack:(XLDTrack *)t
{
	track = [t retain];
	NSMutableString *titleStr = [NSMutableString stringWithString:[track desiredFileName]];
	NSRange formatIndicatorRange = [titleStr rangeOfString:@"[[[XLD_FORMAT_INDICATOR]]]"];
	if(formatIndicatorRange.location != NSNotFound) {
		[titleStr replaceOccurrencesOfString:@"[[[XLD_FORMAT_INDICATOR]]]" withString:LS(@"Multiple Formats") options:0 range:NSMakeRange(0, [titleStr length])];
	}
	if(testMode) [nameField setStringValue:[NSString stringWithFormat:@"(Test) %@",titleStr]];
	else [nameField setStringValue:titleStr];
}

- (BOOL)isActive
{
	return running;
}

- (void)setScaleType:(XLDScaleType)type
{
	scaleType = type;
}

- (void)setCompressionQuality:(float)quality
{
	compressionQuality = quality;
}

- (void)setScaleSize:(int)pixel
{
	scaleSize = pixel;
}

- (void)setiTunesLib:(NSString *)lib
{
	if(iTunesLib) {
		[iTunesLib release];
		iTunesLib = nil;
	}
	if(!lib || [lib isEqualToString:@""]) return;
	
	if([lib isEqualToString:@"library playlist 1"]) {
		iTunesLib = [[NSString alloc] initWithFormat:@"/usr/bin/osascript \
			-e \"tell application \\\"iTunes\\\"\" \
			-e \"add ((POSIX file(\\\"%%@\\\")) as alias) to %@\" \
			-e \"end tell\""\
		,lib];
	}
	else {
		iTunesLib = [[NSString alloc] initWithFormat:@"/usr/bin/osascript \
			-e \"tell application \\\"iTunes\\\"\" \
			-e \"if (exists playlist \\\"%@\\\") is false then\" \
			-e \"make new user playlist with properties {name:\\\"%@\\\"}\" \
			-e \"end if\" \
			-e \"add ((POSIX file(\\\"%%@\\\")) as alias) to playlist \\\"%@\\\"\" \
			-e \"end tell\""\
		,lib,lib,lib];
	}
}

- (BOOL)isAtomic
{
	return [NSStringFromClass(decoderClass) isEqualToString:@"XLDCDDARipper"];
	//return NO;
}
/*
- (void)setMountOnEnd
{
	mountOnEnd = YES;
}
*/
- (void)useParanoiaMode:(BOOL)flag
{
	useParanoiaMode = flag;
	
}

- (void)setOffsetCorrectionValue:(int)value
{
	offsetCorrectionValue = value;
}

- (void)setRetryCount:(int)value
{
	retryCount = value;
}

- (void)setCuesheet:(NSMutableData *)dat withNameRange:(NSRange)range
{
	if(cueData) [cueData release];
	cueData = [dat retain];
	cueRange = range;
}

- (void)setResultObj:(id)obj
{
	if(resultObj) [resultObj release];
	resultObj = [obj retain];
}

- (id)resultObj
{
	return resultObj;
}

- (void)setTestMode
{
	testMode = YES;
}

- (void)setOffsetFixupValue:(int)value
{
	offsetFixupValue = value;
}

- (void)setUseOldEngine:(BOOL)flag
{
	useOldEngine = flag;
}

- (void)setFirstAudioFrame:(xldoffset_t)frame
{
	firstAudioFrame = frame;
}

- (void)setLastAudioFrame:(xldoffset_t)frame
{
	lastAudioFrame = frame;
}

- (void)setUseC2Pointer:(BOOL)flag
{
	useC2Pointer = flag;
}

- (NSView *)progressView
{
	return superview;
}

- (int)position
{
	return position;
}

- (void)setAppendBOM:(BOOL)flag
{
	appendBOM = flag;
}

- (void)setProcessOfExistingFiles:(int)value
{
	processOfExistingFiles = value;
}

- (void)setEmbedImages:(BOOL)flag
{
	embedImages = flag;
}

- (void)setMoveAfterFinish:(BOOL)flag
{
	moveAfterFinish = flag;
}

@end
