//
//  XLDAccurateRipChecker.m
//  XLD
//
//  Created by tmkk on 08/08/22.
//  Copyright 2008 tmkk. All rights reserved.
//

#import <sys/time.h>
#import "XLDAccurateRipChecker.h"
#import "XLDecoderCenter.h"
#import "XLDDecoder.h"
#import "XLDTrack.h"
#import "XLDController.h"
#import "XLDAccurateRipDB.h"

static NSString *framesToMSFStr(xldoffset_t frames, int samplerate)
{
	int min = frames/samplerate/60;
	frames -= min*samplerate*60;
	int sec = frames/samplerate;
	frames -= sec*samplerate;
	int f = frames*75/samplerate;
	return [NSString stringWithFormat:@"%02d:%02d:%02d",min,sec,f];
}

static int intSort(id num1, id num2, void *context)
{
    int v1 = [num1 intValue];
    int v2 = [num2 intValue];
	
    if (v1 < v2)
        return NSOrderedDescending;
    else if (v1 > v2)
        return NSOrderedAscending;
    else
        return NSOrderedSame;
}

@implementation XLDAccurateRipChecker

- (id)init
{
	unsigned int i, k, value;
	[super init];
	[NSBundle loadNibNamed:@"ARChecker" owner:self];
	delegate = nil;
	database = nil;
	decoder = nil;
	running = NO;
	stop = NO;
	detectedOffset = [[NSMutableDictionary alloc] init];
	trackList = [[NSMutableArray alloc] init];
	negativeSideSample = calloc(4704,sizeof(unsigned int));
	positiveSideSample = calloc(4704,sizeof(unsigned int));
	rg = (replaygain_t *)malloc(sizeof(replaygain_t));
	gain_init_analysis(rg,44100);
	
	for(i = 0; i < 256; ++i){
		value = i;
		for(k = 0; k < 8; ++k){
			if(value & 1)
				value = 0xEDB88320 ^ (value >> 1);
			else
				value >>= 1;
		}
		crc32Table[i] = value;
	}	
	
	return self;
}

- (id)initWithTracks:(NSArray *)tracks totalFrames:(xldoffset_t)frame
{
	[self init];
	results = (checkResult *)malloc(sizeof(checkResult)*([tracks count]));
	trackNumber = [tracks count];
	int i;
	for(i=0;i<trackNumber;i++) {
		[trackList addObject:[tracks objectAtIndex:i]];
		results[i].crc32 = 0xFFFFFFFF;
		results[i].crc32_eac = 0xFFFFFFFF;
		results[i].arHash = 0;
		results[i].sampleSum = 0;
		results[i].enabled = NO;
		results[i].cancelled = NO;
		results[i].index = [(XLDTrack *)[tracks objectAtIndex:i] index];
		if(i == (trackNumber-1))
			results[i].length = frame - [(XLDTrack *)[tracks objectAtIndex:i] index];
		else
			results[i].length = [(XLDTrack *)[tracks objectAtIndex:i] frames] + [(XLDTrack *)[tracks objectAtIndex:i+1] gap];
		results[i].detectedOffset = [[NSMutableDictionary alloc] init];
	}
	crc32_global = 0xFFFFFFFF;
	crc32_eac_global = 0xFFFFFFFF;
	
	totalFrames = frame - results[0].index;
	
	return self;
}

- (void)dealloc
{
	int i;
	for(i=0;i<trackNumber;i++) [results[i].detectedOffset release];
	if(results) free(results);
	if(database) [database release];
	if(delegate) [delegate release];
	if(decoder) [decoder release];
	[detectedOffset release];
	[trackList release];
	free(negativeSideSample);
	free(positiveSideSample);
	free(rg);
	[super dealloc];
}

- (void)startCheckingForFile:(NSString *)path withDecoder:(id)decoderObj
{
	decoder = [decoderObj retain];
	if(![(id <XLDDecoder>)decoder openFile:(char *)[path UTF8String]]) {
		fprintf(stderr,"error: cannot open\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	if(([(id <XLDDecoder>)decoder samplerate] != 44100) || ([(id <XLDDecoder>)decoder channels] != 2)) {
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[decoder seekToFrame:results[0].index];
	if([(id <XLDDecoder>)decoder error]) {
		fprintf(stderr,"error: cannot seek\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[o_progress setDoubleValue:0.0];
	[o_panel center];
	[o_message setStringValue:LS(@"calculating hash...")];
	[o_panel setTitle:[path lastPathComponent]];
	[o_panel makeKeyAndOrderFront:nil];
	
	running = YES;
	[NSThread detachNewThreadSelector:@selector(check) toTarget:self withObject:nil];
}

- (void)startOffsetCheckingForFile:(NSString *)path withDecoder:(id)decoderObj
{
	decoder = [decoderObj retain];
	if(![(id <XLDDecoder>)decoder openFile:(char *)[path UTF8String]]) {
		fprintf(stderr,"error: cannot open\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	if(([(id <XLDDecoder>)decoder samplerate] != 44100) || ([(id <XLDDecoder>)decoder channels] != 2)) {
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[decoder seekToFrame:results[0].index];
	if([(id <XLDDecoder>)decoder error]) {
		fprintf(stderr,"error: cannot seek\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[o_progress setDoubleValue:0.0];
	[o_panel center];
	[o_message setStringValue:LS(@"detecting offset...")];
	[o_panel setTitle:[path lastPathComponent]];
	[o_panel makeKeyAndOrderFront:nil];
	
	running = YES;
	[NSThread detachNewThreadSelector:@selector(checkOffset) toTarget:self withObject:nil];
}

- (void)startReplayGainScanningForFile:(NSString *)path withDecoder:(id)decoderObj
{
	decoder = [decoderObj retain];
	if(![(id <XLDDecoder>)decoder openFile:(char *)[path UTF8String]]) {
		fprintf(stderr,"error: cannot open\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	if(([(id <XLDDecoder>)decoder samplerate] != 44100) || ([(id <XLDDecoder>)decoder channels] != 2)) {
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[decoder seekToFrame:results[0].index];
	if([(id <XLDDecoder>)decoder error]) {
		fprintf(stderr,"error: cannot seek\n");
		[decoder closeFile];
		[delegate accurateRipCheckDidFinish:self];
		return;
	}
	
	[o_progress setDoubleValue:0.0];
	[o_panel center];
	[o_message setStringValue:LS(@"scanning replaygain...")];
	[o_panel setTitle:[path lastPathComponent]];
	[o_panel makeKeyAndOrderFront:nil];
	
	running = YES;
	[NSThread detachNewThreadSelector:@selector(scanReplayGain) toTarget:self withObject:nil];
}

- (void)updateStatus
{
	[o_progress setDoubleValue:percent];
}

- (void)calculateCRCForTrack:(int)currentTrack withBuffer:(int *)buffer length:(int)ret
{
	int i;
	for(i=0;i<ret;i++) {
		unsigned int sample = ((buffer[i*2] >> 16)&0xffff) | (buffer[i*2+1] & 0xffff0000);
		results[currentTrack-1].crc32 = (results[currentTrack-1].crc32 >> 8) ^ crc32Table[(results[currentTrack-1].crc32 ^ (sample)) & 0xFF];
		results[currentTrack-1].crc32 = (results[currentTrack-1].crc32 >> 8) ^ crc32Table[(results[currentTrack-1].crc32 ^ (sample>>8)) & 0xFF];
		results[currentTrack-1].crc32 = (results[currentTrack-1].crc32 >> 8) ^ crc32Table[(results[currentTrack-1].crc32 ^ (sample>>16)) & 0xFF];
		results[currentTrack-1].crc32 = (results[currentTrack-1].crc32 >> 8) ^ crc32Table[(results[currentTrack-1].crc32 ^ (sample>>24)) & 0xFF];
		if(buffer[i*2] != 0) {
			results[currentTrack-1].crc32_eac = (results[currentTrack-1].crc32_eac >> 8) ^ crc32Table[(results[currentTrack-1].crc32_eac ^ (sample)) & 0xFF];
			results[currentTrack-1].crc32_eac = (results[currentTrack-1].crc32_eac >> 8) ^ crc32Table[(results[currentTrack-1].crc32_eac ^ (sample>>8)) & 0xFF];
		}
		if(buffer[i*2+1] != 0) {
			results[currentTrack-1].crc32_eac = (results[currentTrack-1].crc32_eac >> 8) ^ crc32Table[(results[currentTrack-1].crc32_eac ^ (sample>>16)) & 0xFF];
			results[currentTrack-1].crc32_eac = (results[currentTrack-1].crc32_eac >> 8) ^ crc32Table[(results[currentTrack-1].crc32_eac ^ (sample>>24)) & 0xFF];
		}
		
		crc32_global = (crc32_global >> 8) ^ crc32Table[(crc32_global ^ (sample)) & 0xFF];
		crc32_global = (crc32_global >> 8) ^ crc32Table[(crc32_global ^ (sample>>8)) & 0xFF];
		crc32_global = (crc32_global >> 8) ^ crc32Table[(crc32_global ^ (sample>>16)) & 0xFF];
		crc32_global = (crc32_global >> 8) ^ crc32Table[(crc32_global ^ (sample>>24)) & 0xFF];
		if(buffer[i*2] != 0) {
			crc32_eac_global = (crc32_eac_global >> 8) ^ crc32Table[(crc32_eac_global ^ (sample)) & 0xFF];
			crc32_eac_global = (crc32_eac_global >> 8) ^ crc32Table[(crc32_eac_global ^ (sample>>8)) & 0xFF];
		}
		if(buffer[i*2+1] != 0) {
			crc32_eac_global = (crc32_eac_global >> 8) ^ crc32Table[(crc32_eac_global ^ (sample>>16)) & 0xFF];
			crc32_eac_global = (crc32_eac_global >> 8) ^ crc32Table[(crc32_eac_global ^ (sample>>24)) & 0xFF];
		}
	}
}

- (void)guessOffsetForTrack:(int)track negativeSidePtr:(unsigned int *)negativePtr positiveSidePtr:(unsigned int *)positivePtr
{
	int j;
	for(currentOffset=-2352;currentOffset<=2352;currentOffset++) {
		unsigned int arHash = results[track-1].arHash;
		if(currentOffset < 0) {
			arHash = arHash + results[track-1].sampleSum*(-currentOffset);
			for(j=0;j<(-currentOffset);j++) {
				arHash += negativePtr[2352+currentOffset+j]*(j+1);
			}
			for(j=0;j<(-currentOffset);j++) {
				arHash -= positivePtr[2352+currentOffset+j]*(results[track-1].length+currentOffset+j+1) + positivePtr[2352+currentOffset+j]*(-currentOffset);
			}
		}
		else if(currentOffset > 0) {
			for(j=0;j<currentOffset;j++) {
				arHash += positivePtr[2352+j]*(results[track-1].length-currentOffset+j+1);
			}
			for(j=0;j<currentOffset;j++) {
				arHash += negativePtr[2352+j]*currentOffset;
			}
			for(j=0;j<currentOffset;j++) {
				arHash -= negativePtr[2352+j]*(j+1);
			}
			arHash = arHash -  results[track-1].sampleSum*currentOffset;
		}
		
		int queryResult = [database isAccurateCRC:arHash forTrack:track];
		if(queryResult != -1) {
			//NSLog(@"Track %d: offset: %d (confidence %d)",currentTrack-1,currentOffset,queryResult);
			[results[track-1].detectedOffset setObject:[NSNumber numberWithInt:currentOffset] forKey:[NSNumber numberWithUnsignedInt:arHash]];
			if([detectedOffset objectForKey:[NSNumber numberWithInt:currentOffset]]) {
				if(queryResult<[[detectedOffset objectForKey:[NSNumber numberWithInt:currentOffset]] intValue]) continue;
			}
			[detectedOffset setObject:[NSNumber numberWithInt:queryResult] forKey:[NSNumber numberWithInt:currentOffset]];
		}
	}
}

- (void)calculateOffsetForTrack:(int)currentTrack withBuffer:(int *)buffer length:(int)ret currentFrame:(xldoffset_t)currentFrame
{
	int i;
	for(i=0;i<ret;i++) {
		unsigned int sample = ((buffer[i*2] >> 16)&0xffff) | (buffer[i*2+1] & 0xffff0000);
		
		/* offset detection */
		if(currentTrack & 1) {
			if((currentTrack <= trackNumber) && (currentFrame >=results[currentTrack-1].length-2352)) {
				negativeSideSample[2352+currentFrame - results[currentTrack-1].length] = sample;
				if(currentTrack == trackNumber) negativeSideSample[4704+currentFrame - results[currentTrack-1].length] = 0;
			}
			if(currentFrame < 2352) {
				positiveSideSample[2352+currentFrame] = sample;
			}
		}
		else {
			if((currentTrack <= trackNumber) && (currentFrame >= results[currentTrack-1].length-2352)) {
				positiveSideSample[2352+currentFrame - results[currentTrack-1].length] = sample;
				if(currentTrack == trackNumber) positiveSideSample[4704+currentFrame - results[currentTrack-1].length] = 0;
			}
			if(currentFrame < 2352) {
				negativeSideSample[2352+currentFrame] = sample;
			}
		}
		if((currentTrack > 1) && (currentFrame == 2352)) {
			unsigned int *negativeSideSamplePtr,*positiveSideSamplePtr;
			if(currentTrack & 1) {
				negativeSideSamplePtr = negativeSideSample;
				positiveSideSamplePtr = positiveSideSample;
			}
			else {
				negativeSideSamplePtr = positiveSideSample;
				positiveSideSamplePtr = negativeSideSample;
			}
			[self guessOffsetForTrack:currentTrack-1 negativeSidePtr:negativeSideSamplePtr positiveSidePtr:positiveSideSamplePtr];
		}
		else if((currentTrack == trackNumber) && (currentFrame == results[currentTrack-1].length-1)) {
			unsigned int *negativeSideSamplePtr,*positiveSideSamplePtr;
			if(currentTrack & 1) {
				negativeSideSamplePtr = positiveSideSample;
				positiveSideSamplePtr = negativeSideSample;
			}
			else {
				negativeSideSamplePtr = negativeSideSample;
				positiveSideSamplePtr = positiveSideSample;
			}
			[self guessOffsetForTrack:currentTrack negativeSidePtr:negativeSideSamplePtr positiveSidePtr:positiveSideSamplePtr];
		}
		/* offset detection end */
		
		if((currentTrack == 1) && (currentFrame < 5*588-1)) ++currentFrame;
		else if((currentTrack == trackNumber) && (currentFrame >= results[currentTrack-1].length - 5*588)) ++currentFrame;
		else {
			results[currentTrack-1].arHash += ++currentFrame * sample;
			results[currentTrack-1].sampleSum += sample;
		}
	}
	
}

- (void)check
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	struct timeval tv1,tv2;
	xldoffset_t framesToCopy = results[0].length;
	xldoffset_t currentFrame = 0;
	int *buffer = (int *)malloc(588 * 2 * 4);
	int currentTrack = 1;
	
	results[0].enabled = YES;
	
	gettimeofday(&tv1,NULL);
	do {
		if(stop) {
			results[currentTrack-1].cancelled = YES;
			goto finish;
		}
		int ret = [decoder decodeToBuffer:(int *)buffer frames:588];
		if([(id <XLDDecoder>)decoder error]) {
			fprintf(stderr,"error: cannot decode\n");
			break;
		}
		if(ret > 0) {
			[self calculateOffsetForTrack:currentTrack withBuffer:buffer length:ret currentFrame:currentFrame];
			[self calculateCRCForTrack:currentTrack withBuffer:buffer length:ret];
			gain_analyze_samples_interleaved_int32(rg,buffer,ret,2);
		}
		framesToCopy -= ret;
		currentFrame += ret;
		
		gettimeofday(&tv2,NULL);
		double elapsed1 = tv2.tv_sec-tv1.tv_sec + (tv2.tv_usec-tv1.tv_usec)*0.000001;
		if(elapsed1 > 0.25) {
			percent = 100.0*(((double)results[currentTrack-1].index + (double)currentFrame - (double)results[0].index))/(double)totalFrames;
			[self performSelectorOnMainThread:@selector(updateStatus) withObject:nil waitUntilDone:YES];
			tv1 = tv2;
		}
		
		if(!framesToCopy) {
			results[currentTrack-1].trackGain = PINK_REF-gain_get_title(rg);
			results[currentTrack-1].peak = peak_get_title(rg);
			currentFrame = 0;
			currentTrack++;
			if(currentTrack <= trackNumber) {
				framesToCopy = results[currentTrack-1].length;
				results[currentTrack-1].enabled = YES;
			}
		}
		
		if(currentTrack > trackNumber) {
			[o_progress setDoubleValue:100.0];
			break;
		}
	} while(1);
	
finish:
		free(buffer);
	[decoder closeFile];
	[o_panel close];
	[delegate performSelectorOnMainThread:@selector(accurateRipCheckDidFinish:) withObject:self waitUntilDone:NO];
	[pool release];
}

- (unsigned int)detectOffsetForTrack:(int)currentTrack buffer:(int *)buffer length:(int)ret currentFrame:(xldoffset_t)currentFrame;
{
	int i;
	unsigned int sampleSum = 0;
	for(i=0;i<ret;i++) {
		unsigned int sample = ((buffer[i*2] >> 16)&0xffff) | (buffer[i*2+1] & 0xffff0000);
		if((currentTrack == 1) && (currentFrame < 5*588-1)) ++currentFrame;
		else if((currentTrack == trackNumber) && (currentFrame >= results[currentTrack-1].length - 5*588)) ++currentFrame;
		else {
			results[currentTrack-1].arHash += ++currentFrame * sample;
			sampleSum += sample;
		}
	}
	return sampleSum;
}

- (void)fillBufferForTrack:(int)currentTrack
{
	int i;
	int *tmp = malloc(4704*4*2);
	if(currentTrack == 1) {
		for(i=0;i<2352;i++) negativeSideSample[i] = 0;
		[decoder seekToFrame:results[currentTrack-1].index];
		[decoder decodeToBuffer:tmp frames:2352];
		for(i=2352;i<4704;i++)
			negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		[decoder seekToFrame:results[currentTrack].index-2352];
		[decoder decodeToBuffer:tmp frames:4704];
		for(i=0;i<4704;i++)
			positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
	}
	else if(currentTrack == trackNumber) {
		[decoder seekToFrame:results[currentTrack-1].index-2352];
		[decoder decodeToBuffer:tmp frames:4704];
		for(i=0;i<4704;i++)
			negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		[decoder seekToFrame:results[currentTrack-1].index+results[currentTrack-1].length-2352];
		[decoder decodeToBuffer:tmp frames:2352];
		for(i=0;i<2352;i++)
			positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		for(i=2352;i<4704;i++) positiveSideSample[i] = 0;
	}
	else {
		[decoder seekToFrame:results[currentTrack-1].index-2352];
		[decoder decodeToBuffer:tmp frames:4704];
		for(i=0;i<4704;i++)
			negativeSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
		[decoder seekToFrame:results[currentTrack].index-2352];
		[decoder decodeToBuffer:tmp frames:4704];
		for(i=0;i<4704;i++)
			positiveSideSample[i] = ((tmp[i*2] >> 16)&0xffff) | (tmp[i*2+1] & 0xffff0000);
	}
	free(tmp);
}

- (void)checkOffset
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	xldoffset_t framesToCopy;
	xldoffset_t currentFrame = 0;
	struct timeval tv1,tv2;
	int *buffer = (int *)malloc(588 * 2 * 4);
	int i;
	int currentTrack = 1;
	unsigned int sampleSum = 0;
	
	int minLength = 0x7FFFFFFF;
	for(i=1;i<trackNumber-1;i++) {
		if((minLength > results[i].length) && [database hasValidDataForTrack:i+1]) {
			minLength = results[i].length;
			currentTrack = i+1;
		}
	}
	
	framesToCopy = results[currentTrack-1].length;
	
	[self fillBufferForTrack:currentTrack];
	
	[decoder seekToFrame:results[currentTrack-1].index];
	
	gettimeofday(&tv1,NULL);
	do {
		if(stop) {
			goto finish;
		}
		int getLength = (framesToCopy < 588) ? framesToCopy : 588;
		int ret = [decoder decodeToBuffer:(int *)buffer frames:getLength];
		if([(id <XLDDecoder>)decoder error]) {
			fprintf(stderr,"error: cannot decode\n");
			break;
		}
		
		if(ret > 0) {
			sampleSum += [self detectOffsetForTrack:currentTrack buffer:buffer length:ret currentFrame:currentFrame];
		}
		framesToCopy -= ret;
		currentFrame += ret;
		
		gettimeofday(&tv2,NULL);
		double elapsed1 = tv2.tv_sec-tv1.tv_sec + (tv2.tv_usec-tv1.tv_usec)*0.000001;
		if(elapsed1 > 0.25) {
			percent = 100.0*(double)currentFrame/(double)results[currentTrack-1].length;
			[self performSelectorOnMainThread:@selector(updateStatus) withObject:nil waitUntilDone:YES];
			tv1 = tv2;
		}
		
		if(!framesToCopy) {
			break;
		}
		if(!ret) break;
	} while(1);
	
	for(currentOffset=-2352;currentOffset<=2352;currentOffset++) {
		unsigned int arHash = results[currentTrack-1].arHash;
		if(currentOffset < 0) {
			arHash = arHash + sampleSum*(-currentOffset);
			for(i=0;i<(-currentOffset);i++) {
				arHash += negativeSideSample[2352+currentOffset+i]*(i+1);
			}
			for(i=0;i<(-currentOffset);i++) {
				arHash -= positiveSideSample[2352+currentOffset+i]*(results[currentTrack-1].length+currentOffset+i+1) + positiveSideSample[2352+currentOffset+i]*(-currentOffset);
			}
		}
		else if(currentOffset > 0) {
			for(i=0;i<currentOffset;i++) {
				arHash += positiveSideSample[2352+i]*(results[currentTrack-1].length-currentOffset+i+1);
			}
			for(i=0;i<currentOffset;i++) {
				arHash -= negativeSideSample[2352+i]*(i+1-currentOffset);
			}
			arHash = arHash - sampleSum*currentOffset;
		}
		
		int queryResult = [database isAccurateCRC:arHash forTrack:currentTrack];
		if(queryResult != -1) {
			[detectedOffset setObject:[NSNumber numberWithInt:queryResult] forKey:[NSNumber numberWithInt:currentOffset]];
		}
	}
	
finish:
		free(buffer);
	[decoder closeFile];
	[o_panel close];
	[delegate performSelectorOnMainThread:@selector(offsetCheckDidFinish:) withObject:self waitUntilDone:NO];
	[pool release];
}

- (void)scanReplayGain
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	struct timeval tv1,tv2;
	xldoffset_t framesToCopy = results[0].length;
	xldoffset_t currentFrame = 0;
	int *buffer = (int *)malloc(588 * 2 * 4);
	int currentTrack = 1;
	
	results[0].enabled = YES;
	
	gettimeofday(&tv1,NULL);
	do {
		if(stop) {
			results[currentTrack-1].cancelled = YES;
			goto finish;
		}
		int ret = [decoder decodeToBuffer:(int *)buffer frames:588];
		if([(id <XLDDecoder>)decoder error]) {
			fprintf(stderr,"error: cannot decode\n");
			break;
		}
		if(ret > 0) {
			[self calculateCRCForTrack:currentTrack withBuffer:buffer length:ret];
			gain_analyze_samples_interleaved_int32(rg,buffer,ret,2);
		}
		framesToCopy -= ret;
		currentFrame += ret;
		
		gettimeofday(&tv2,NULL);
		double elapsed1 = tv2.tv_sec-tv1.tv_sec + (tv2.tv_usec-tv1.tv_usec)*0.000001;
		if(elapsed1 > 0.25) {
			percent = 100.0*(((double)results[currentTrack-1].index + (double)currentFrame - (double)results[0].index))/(double)totalFrames;
			[self performSelectorOnMainThread:@selector(updateStatus) withObject:nil waitUntilDone:YES];
			tv1 = tv2;
		}
		
		if(!framesToCopy) {
			results[currentTrack-1].trackGain = PINK_REF-gain_get_title(rg);
			results[currentTrack-1].peak = peak_get_title(rg);
			currentFrame = 0;
			currentTrack++;
			if(currentTrack <= trackNumber) {
				framesToCopy = results[currentTrack-1].length;
				results[currentTrack-1].enabled = YES;
			}
		}
		
		if(currentTrack > trackNumber) {
			[o_progress setDoubleValue:100.0];
			break;
		}
	} while(1);
	
finish:
		free(buffer);
	[decoder closeFile];
	[o_panel close];
	[delegate performSelectorOnMainThread:@selector(replayGainScanningDidFinish:) withObject:self waitUntilDone:NO];
	[pool release];
}

- (void)setAccurateRipDB:(id)db
{
	if(database) [database release];
	database = [db retain];
}

- (void)setDelegate:(id)del
{
	if(delegate) [delegate release];
	delegate = [del retain];
}

- (IBAction)cancel:(id)sender
{
	stop = YES;
}

- (NSString *)logStr
{
	if(!results) return nil;
	if(!running) return nil;
	int i,j;
	NSMutableString *out = [[NSMutableString alloc] init];
	[out appendString:[NSString stringWithFormat:@"X Lossless Decoder version %@ (%@)\n\n",[[[NSBundle mainBundle] infoDictionary] valueForKey:@"CFBundleShortVersionString"],[[[NSBundle mainBundle] infoDictionary] valueForKey:@"CFBundleVersion"]]];
	[out appendString:@"XLD AccurateRip checking logfile\n\n"];
	[out appendString:@"TOC of the selected file\n"];
	[out appendString:@"     Track |   Start  |  Length  | Start sector | End sector \n"];
	[out appendString:@"    ---------------------------------------------------------\n"];
	for(i=0;i<trackNumber;i++) {
		[out appendString:[NSString stringWithFormat:@"      % 3d  | %@ | %@ |   % 7lld    |  % 7lld   \n",i+1,framesToMSFStr(results[i].index,44100),framesToMSFStr(results[i].length,44100),results[i].index/588,(results[i].index+results[i].length)/588-1]];
	}
	[out appendString:@"\n"];
	
	if([detectedOffset count]) {
		NSArray *confidenceList = [[detectedOffset allValues] sortedArrayUsingFunction:intSort context:NULL];
		NSArray *offsetList = [detectedOffset allKeys];
		int n=1;
		int previousConfidence = -1;
		for(j=0;j<[confidenceList count];j++) {
			int confidence = [[confidenceList objectAtIndex:j] intValue];
			if(confidence == previousConfidence) continue;
			for(i=0;i<[offsetList count];i++) {
				int value = [[detectedOffset objectForKey:[offsetList objectAtIndex:i]] intValue];
				if((value == confidence) && [[offsetList objectAtIndex:i] intValue]) {
					if(n==1) [out appendString:@"List of alternate offset correction values\n"];
					NSString *str = [NSString stringWithFormat:@"    (%d) %d (confidence %d)\n",n++,[[offsetList objectAtIndex:i] intValue],confidence];
					[out appendString:str];
				}
			}
			previousConfidence = confidence;
		}
		[out appendString:@"\n"];
	}
	
	if(!stop) {
		float albumGain = PINK_REF-gain_get_album(rg);
		float albumPeak = peak_get_album(rg);
		for(i=0;i<[trackList count];i++) {
			[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:albumGain] forKey:XLD_METADATA_REPLAYGAIN_ALBUM_GAIN];
			[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:albumPeak] forKey:XLD_METADATA_REPLAYGAIN_ALBUM_PEAK];
		}
		[out appendString:@"All Tracks\n"];
		[out appendString:[NSString stringWithFormat:@"    Album gain             : %.2f dB\n",albumGain]];
		[out appendString:[NSString stringWithFormat:@"    Peak                   : %f\n",albumPeak]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash             : %08X\n",crc32_global^0xFFFFFFFF]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash (skip zero) : %08X\n\n",crc32_eac_global^0xFFFFFFFF]];
	}
	
	for(i=0;i<trackNumber;i++) {
		if(!results[i].enabled) continue;
		[out appendString:[NSString stringWithFormat:@"Track %02d\n",i+1]];
		if(results[i].cancelled) {
			[out appendString:@"    (cancelled by user)\n\n"];
			continue;
		}
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:results[i].trackGain] forKey:XLD_METADATA_REPLAYGAIN_TRACK_GAIN];
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:results[i].peak] forKey:XLD_METADATA_REPLAYGAIN_TRACK_PEAK];
		[out appendString:[NSString stringWithFormat:@"    Track gain             : %.2f dB\n",results[i].trackGain]];
		[out appendString:[NSString stringWithFormat:@"    Peak                   : %f\n",results[i].peak]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash             : %08X\n",results[i].crc32^0xFFFFFFFF]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash (skip zero) : %08X\n",results[i].crc32_eac^0xFFFFFFFF]];
		[out appendString:[NSString stringWithFormat:@"    AccurateRip signature  : %08X\n",results[i].arHash]];
		if(!database || ![database hasValidDataForTrack:i+1])
			[out appendString:@"        ->Track not present in AccurateRip database.\n\n"];
		else {
			int queryResult = [database isAccurateCRC:results[i].arHash forTrack:i+1];
			if(queryResult != -1)
				[out appendString:[NSString stringWithFormat:@"        ->Accurately ripped! (confidence %d)\n\n",queryResult]];
			else {
				if([results[i].detectedOffset count]) {
					int confidence = -1;
					unsigned int hash;
					int additionalOffset;
					NSArray *hashArr = [results[i].detectedOffset allKeys];
					for(j=0;j<[hashArr count];j++) {
						queryResult = [database isAccurateCRC:[[hashArr objectAtIndex:j] unsignedIntValue] forTrack:i+1];
						if(queryResult > confidence) {
							confidence = queryResult;
							hash = [[hashArr objectAtIndex:j] unsignedIntValue];
							additionalOffset = [[results[i].detectedOffset objectForKey:[NSNumber numberWithUnsignedInt:hash]] intValue];
						}
					}
					[out appendString:[NSString stringWithFormat:@"        ->Accurately ripped! (confidence %d)\n",confidence]];
					[out appendString:[NSString stringWithFormat:@"          (matched with the different offset correction value;\n           calculated using an additional offset of %d;\n           the signature after correction is: %08X)\n",additionalOffset,hash]];
				}
				else [out appendString:@"        ->Rip may not be accurate.\n\n"];
			}
		}
	}
	
	[out appendString:@"End of status report\n"];
	
	return [out autorelease];
}

- (NSString *)logStrForReplayGainScanner
{
	if(!results) return nil;
	if(!running) return nil;
	int i;
	NSMutableString *out = [[NSMutableString alloc] init];
	[out appendString:[NSString stringWithFormat:@"X Lossless Decoder version %@ (%@)\n\n",[[[NSBundle mainBundle] infoDictionary] valueForKey:@"CFBundleShortVersionString"],[[[NSBundle mainBundle] infoDictionary] valueForKey:@"CFBundleVersion"]]];
	[out appendString:@"XLD ReplayGain scanning logfile\n\n"];
	[out appendString:@"TOC of the selected file\n"];
	[out appendString:@"     Track |   Start  |  Length  | Start sector | End sector \n"];
	[out appendString:@"    ---------------------------------------------------------\n"];
	for(i=0;i<trackNumber;i++) {
		[out appendString:[NSString stringWithFormat:@"      % 3d  | %@ | %@ |   % 7lld    |  % 7lld   \n",i+1,framesToMSFStr(results[i].index,44100),framesToMSFStr(results[i].length,44100),results[i].index/588,(results[i].index+results[i].length)/588-1]];
	}
	[out appendString:@"\n"];
	
	if(!stop) {
		float albumGain = PINK_REF-gain_get_album(rg);
		float albumPeak = peak_get_album(rg);
		for(i=0;i<[trackList count];i++) {
			[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:albumGain] forKey:XLD_METADATA_REPLAYGAIN_ALBUM_GAIN];
			[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:albumPeak] forKey:XLD_METADATA_REPLAYGAIN_ALBUM_PEAK];
		}
		[out appendString:@"All Tracks\n"];
		[out appendString:[NSString stringWithFormat:@"    Album gain             : %.2f dB\n",albumGain]];
		[out appendString:[NSString stringWithFormat:@"    Peak                   : %f\n",albumPeak]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash             : %08X\n",crc32_global^0xFFFFFFFF]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash (skip zero) : %08X\n\n",crc32_eac_global^0xFFFFFFFF]];
	}
	
	for(i=0;i<trackNumber;i++) {
		if(!results[i].enabled) continue;
		[out appendString:[NSString stringWithFormat:@"Track %02d\n",i+1]];
		if(results[i].cancelled) {
			[out appendString:@"    (cancelled by user)\n\n"];
			continue;
		}
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:results[i].trackGain] forKey:XLD_METADATA_REPLAYGAIN_TRACK_GAIN];
		[[[trackList objectAtIndex:i] metadata] setObject:[NSNumber numberWithFloat:results[i].peak] forKey:XLD_METADATA_REPLAYGAIN_TRACK_PEAK];
		[out appendString:[NSString stringWithFormat:@"    Track gain             : %.2f dB\n",results[i].trackGain]];
		[out appendString:[NSString stringWithFormat:@"    Peak                   : %f\n",results[i].peak]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash             : %08X\n",results[i].crc32^0xFFFFFFFF]];
		[out appendString:[NSString stringWithFormat:@"    CRC32 hash (skip zero) : %08X\n\n",results[i].crc32_eac^0xFFFFFFFF]];
	}
	
	[out appendString:@"End of status report\n"];
	
	return [out autorelease];
}

- (NSDictionary *)detectedOffset
{
	return detectedOffset;
}

- (BOOL)cancelled
{
	return stop;
}

@end
