/*
 * L I B
 *
 *    #            #
 *    ##############
 *           #######
 *      ########
 *       #####
 *    #       #### 
 *    ##############
 *    #            #
 *         ####
 *      ##########
 *    ##          # 
 *    #            #
 *    #            #
 *     ###      ###
 *       ########
 *    #            #
 *    ##############
 *    #            #
 *    #            #
 *    #           ##
 *     ####    ####
 *       ########
 *                  I N F O v1.1 beta
 *
 * Retrieve useful info. data from MOD, IT, S3M & XM files.
 *
 * Please take a look to the included documentation ;) 
 *
 * -= Coded by Marco Trillo <toad@arsystel.com> on 2005 =-
 * -= Parts of code were inspired on LibModPlug`s load_it.cpp file, thanks to its authors =-
 * 
 * All of this code is under Public Domain. Please feel free to use it in your distribution; sell it;
 * modify & remodify it; or just drop it to the trash =) in both source & binary forms.
 *
 * If you have a bugfix or an improvement you can send it to the author: thx =) 
 *
 * *** DISCLAIMER ***
 * This software is provided "as is", WITHOUT ANY WARRANTY,
 * either expressed or implied, including but not limited to the implied
 * warranties of merchantability and/or fitness for a particular purpose.
 * The author shall NOT be held liable for ANY damage to you, your
 * computer, or to anyone or anything else, that may result from its use,
 * or misuse. Basically, you use it at YOUR OWN RISK.
 *
 * But for any problem using this software you can contact the author, and we`ll try to fix it ;) 
 * 
 */
 
#include <config.h>

#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <unistd.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <sys/types.h>

/* Converts LittleEndian memory data to be used in BigEndian platforms */

#ifdef WORDS_BIGENDIAN
#define ENDIANRESOLVE(o) ((((o) & 0xFF) << 24) | (((o) & 0xFF00) << 8) | (((o) & 0xFF0000) >> 8) | (((o) & 0xFF000000) >> 24))
#else
#define ENDIANRESOLVE(o) (unsigned long)(o)
#endif

/* Converts BigEndian memory data to be used in LittleEndian platforms */

#ifndef WORDS_BIGENDIAN
#define ENDIANINVERSE(o) ((((o) & 0xFF) << 24) | (((o) & 0xFF00) << 8) | (((o) & 0xFF0000) >> 8) | (((o) & 0xFF000000) >> 24))
#else
#define ENDIANINVERSE(o) (unsigned long)(o)
#endif

#include "libmodinfo.h"

/* Internal function used to completely retrieve a file, storing it to a buffer. */
char* xm_retr_file(int fd,int *len) {
	char* buffer;
	int buflen=0;
	int bytes_out=0;
	
	buffer=(char*)malloc(1024);
	while ((bytes_out=read(fd,buffer+buflen,1024)) != 0) {
		if ( bytes_out == -1 ) break;
		buflen+=bytes_out;
		buffer=(char*)realloc(buffer,1024+buflen);
		if ( buffer==NULL ) return NULL;
	}
	buffer=(char*)realloc(buffer,buflen);
	if ( buffer==NULL ) return NULL;
	
	*len=buflen;
	return buffer;
}

/* Number of instruments of a MODULE */
unsigned char number_instruments(MODULE* modfile) {
	unsigned char ins=0;
	char *buffer;
	short type;
	int modlen;
	
	type=modfile->type;
	modlen=modfile->buflen;
	
	switch( type ) {
		case TYPE_SCREAM_3: case TYPE_IMPULSE:
			if ( modlen > (28+4+4) ) {
				buffer=modfile->buf+28+4+2;
				/*
				 * As max number of instruments/samples is 240 on S3M & IT,
				 * we can safely use a 1byte unsigned char (max number 255)
				 */
				ins=(unsigned char)*buffer;
			} else {
				ins=0;
			}
			break;
		case TYPE_AMIGA_MOD: ins=31; break;
		case TYPE_FT2:
			if ( modlen > (60+14) ) {
				buffer=modfile->buf+60+12;
				ins=(unsigned char)*buffer;
			} else ins=0;
			break;
	}
	
	return ins;
}

/* Number of orders of a MODULE */
unsigned short number_orders(MODULE* modfile) {
	unsigned long ord=0;
	char *buffer;
	short type;
	int modlen;
	
	type=modfile->type;
	modlen=modfile->buflen;
	
	switch( type ) {
		case TYPE_SCREAM_3: case TYPE_IMPULSE:
			if ( modlen > (28+4+2) ) {
				buffer=modfile->buf+28+4;
				memcpy(&ord,buffer,2);
				/* Resolve endianness problems */
				ord=ENDIANRESOLVE(ord);
			} else {
				ord=0;
			}
			break;
		case TYPE_AMIGA_MOD:
			if ( modlen > 950 ) {
				buffer=modfile->buf+950;
				ord=(unsigned long)(*buffer);
			} else ord=0;
			break;
		case TYPE_FT2:
			if ( modlen > (60+12) ) {
				buffer=modfile->buf+60+10;
				memcpy(&ord,buffer,2);
				ord=ENDIANRESOLVE(ord);
			} else ord=0;
			break;
			
	}
	
	/*
	As NumOrders field size is 2 bytes, we can use an "unsigned short"
	*/
	return (unsigned short)ord;
}

/* Loads & typedetects a module to a MODULE struct */
int load_module(MODULE* modfile,char* obuffer,int obuflen) {
	char *buf;
	short type;
	int fd;
	char *buffer;
	int buflen;
	unsigned long K;
	
	if ( obuflen == USE_FILENAME ) {
		fd=open(obuffer,O_RDONLY);
		if ( fd == -1 ) return -1;
		buffer=xm_retr_file( fd,&buflen );
		close( fd );
		modfile->live=1;
	} else {
		buffer=obuffer;
		buflen=obuflen;
		modfile->live=0;
	}
	
	type=TYPE_UNKNOWN;
	
	/* XM`s identifier is 'Extended Module: ', starting at offset 0 */
	if ( buflen > 17 ) {
		buf=buffer;
		if ( memcmp(buffer,"Extended Module: ",17) == 0 ) {
			type=TYPE_FT2;
		}
	}
	
	/* IT`s identifier is 'IMPM', starting at offset 0 */
	if ( buflen > 4 ) {
		buf=buffer;
		if ( *buf == 'I' && *(buf+1) == 'M' && *(buf+2) == 'P' && *(buf+3) == 'M' ) {
			type=TYPE_IMPULSE;
		}
	}
	
	/* S3M`s identifier is 'SCRM', starting at offset 44 (28+4+2*6) */
	if ( buflen > (28+4+2*6+4) ) {
		buf=buffer+28+4+2*6;
		if ( *buf == 'S' && *(buf+1) == 'C' && *(buf+2) == 'R' && *(buf+3) == 'M' ) {
			type=TYPE_SCREAM_3;
		}
	}
	
	/* MOD has a lot of identifiers, starting at offset 1080 */
	if ( buflen > 1084 ) {
		buf=buffer+1080;
		memcpy(&K,buf,4);
		/* For LittleEndian processors */
		K=ENDIANINVERSE(K);
		switch ( K ) {
			/* Thanks to the Modfil11.txt for this identifiers!! =) */
			case 0x4d2e4b2e:
			case 0x46544c34:
			case 0x4d214b21:
			case 0x3443484e:
			case 0x3643484e:
			case 0x3843484e:
			case 0x4f435441:
			case 0x464c5434:
			case 0x43443831:
			case 0x464c5438:
				type=TYPE_AMIGA_MOD;
				break;
		}
	}
	
	modfile->type=type;
	modfile->buflen=buflen;
	modfile->buf=buffer;
	
	if ( type==TYPE_UNKNOWN ) return -1;
	
	return 0;
}
	
/* Returns the name of the "numinst" instument as a null-terminated string */
char* return_instrument_name(MODULE* mod,int numinst) {
	char* buffer;
	char* bufpos;
	char* insname;
	unsigned short orders;
	unsigned char instruments;
	unsigned long Ypos=0,hlen,G=0;
	int D,numsamples,E;
	
	orders=number_orders( mod );
	instruments=number_instruments( mod );
	
	if ( orders < 1 ) return NULL;
	if ( instruments < numinst+1 ) return NULL;
	
	insname=NULL;

	switch( mod->type ) {
		case TYPE_SCREAM_3:
			if ( mod->buflen > (28+4+2*32+orders+2*instruments) ) {
				buffer=mod->buf+28+4+2*32+orders;
				buffer+=numinst*2;
				memcpy(&Ypos,buffer,2);
				/* Resolve endianness problems */
				Ypos=ENDIANRESOLVE(Ypos);
				bufpos=Ypos*16+mod->buf;
				bufpos=bufpos+1+8+1+3+3+2*16;
				insname=(char*)malloc(strlen(bufpos)+1);
				strcpy(insname,bufpos);
			}
			break;
		case TYPE_IMPULSE:
			/* IT offset read based on LibModPlug`s load_it.cpp */
			if ( mod->buflen > (16*4+2*64+orders+4*instruments) ) {
				buffer=mod->buf+(16*4+2*64+orders);
				buffer+=numinst*4;
				memcpy(&Ypos,buffer,4);
				/* Resolve endianness problems */
				Ypos=ENDIANRESOLVE(Ypos);
				bufpos=mod->buf+Ypos;
				bufpos+=2*16;
				insname=(char*)malloc(strlen(bufpos)+1);
				strcpy(insname,bufpos);
			}
			break;
		case TYPE_AMIGA_MOD:
			if ( mod->buflen > 20+30+30*numinst ) {
				buffer=mod->buf+20;
				buffer+=numinst*30;
				insname=(char*)malloc(23);
				for ( Ypos=0;Ypos<22;++Ypos ) {
					insname[Ypos]=*(buffer+Ypos);
				}
				insname[22]=0;
			}
			break;
		case TYPE_FT2:
		
			/* =========================================
			 * FastTracker II XM Instrument name Reader 
			 * =========================================
			 *
			 * Implementing this format required a lot of debug time to me! =( 
			 * It may contain some bugs. If the XM_DEBUG variable is defined, extra info. will be
			 * sent to stderr. 
			 * 
			 * Note that the official XM documentation contain some errors, checkout the XMFORMAT.txt
			 * file for more info.
			 * 
			 * -- Marco <toad@arsystel.com> March 24 2005 */
			 
			if ( mod->buflen > 60+20+(9+64)*orders ) {
				buffer=mod->buf+60+20+256;
				
				/* Processing patterns... */
				for( D=0;D<orders;++D ) {
					buffer+=7;
					memcpy(&Ypos,buffer,2);
					Ypos=ENDIANRESOLVE(Ypos);
					buffer+=2+Ypos;
					memset(&Ypos,0,sizeof(Ypos));
				}
				
#ifdef XM_DEBUG
				fprintf(stderr,"START the process!\n\tnuminst: %d\n\tbuffer: %X\n",numinst,buffer);
#endif
				/* Processing instruments... */
				for ( D=0;D<numinst;++D ) {
				
#ifdef XM_DEBUG
					fprintf(stderr,"Processing instruments...\n\tLooptime: %d\n\tbuffer: %X\n",D,buffer);
#endif					
					memcpy(&Ypos,buffer+27,2);
					Ypos=ENDIANRESOLVE(Ypos);
					numsamples=Ypos;
#ifdef XM_DEBUG
					fprintf(stderr,"\tnumsamples: %d\n",numsamples);
#endif
					memset(&Ypos,0,sizeof(Ypos));
					
					/* Processing samples... */
					if ( numsamples > 0 ) {
#ifdef XM_DEBUG
						fprintf(stderr,"Processing samples...\n");
#endif	
						memcpy(&Ypos,buffer+29,4);
						Ypos=ENDIANRESOLVE(Ypos);
						hlen=Ypos;
#ifdef XM_DEBUG
						fprintf(stderr,"\tSample header len: %d\n",hlen);
#endif
						memset(&Ypos,0,sizeof(Ypos));
						buffer+=0x107;
#ifdef XM_DEBUG
						fprintf(stderr,"\tBuffer: %X\n",buffer);
#endif
						for ( E=0;E<numsamples;++E ) {
							memcpy(&Ypos,buffer,4);
							Ypos=ENDIANRESOLVE(Ypos);
							buffer+=0x28;
#ifdef XM_DEBUG
							fprintf(stderr,"\tSample num: %d\n\tSample length: %d\n",E,Ypos);
#endif
							G+=Ypos;
							memset(&Ypos,0,sizeof(Ypos));
						}
						buffer+=G;
						G=0;
					} else {
#ifdef XM_DEBUG
						fprintf(stderr,"Zero-sample instrument\n\tOffset: %X\n\tInstrument: %d\n",(buffer-mod->buf),D);
#endif
						memcpy(&Ypos,buffer,4);
						Ypos=ENDIANRESOLVE(Ypos);
#ifdef XM_DEBUG
						fprintf(stderr,"\tInslen: %d (%X)\n",Ypos,Ypos);
#endif
						buffer+=Ypos;
						memset(&Ypos,0,sizeof(Ypos));
					}
				}
				insname=NULL;
				
				/* Fetch the instrument name! */
				buffer+=4;
				insname=(char*)malloc(23);
				for ( D=0;D<22;++D ) {
					insname[D]=buffer[D];
				}
				insname[22]=0;
			}
			break;
	}
	
	return insname;
}

/* Retrieves the packed text message (if one) on IT files; NULL-terminated and in UNIX-format for newlines */
char* retrieve_message(MODULE* modfile,unsigned long* length) {
	short type;
	char* buffer;
	char *realbuffer;
	int i;
	unsigned long tt=0;
	unsigned long len=0;
	
	realbuffer=NULL;
	type=modfile->type;
	if ( type == TYPE_IMPULSE && modfile->buflen > (2*16+14+2+6+2+4) ) {
		buffer=modfile->buf+(2*16)+14;
		memcpy(&tt,buffer,2);
		tt=ENDIANRESOLVE(tt);
		if ((tt & 0x01)) {
			buffer+=2+6;
			memcpy(&len,buffer,2);
			len=ENDIANRESOLVE(len);
			buffer+=2;
			memcpy(&tt,buffer,4);
			tt=ENDIANRESOLVE(tt);
			buffer=modfile->buf+tt;
			if ( modfile->buflen > tt+len ) {
				realbuffer=(char*)malloc(1);
				/* Translate DOS to UNIX string format */
				for ( i=0;i<len;++i ) {
					if ( *(buffer+i) == '\r' ) realbuffer[i]='\n';
					else realbuffer[i]=*(buffer+i);
					realbuffer=(char*)realloc(realbuffer,i+1+1);
				}
			}
			realbuffer[len]=0;
		}
		
	}
	
	if ( length!=NULL ) *length=len;
	return realbuffer;
}

/* Fetch the module name! */
char* module_name(MODULE* modfile) {
	char *nam;
	short i=0;
	
	nam=NULL;
	switch ( modfile->type ) {
		case TYPE_SCREAM_3:
			nam=modfile->buf;
			break;
		case TYPE_IMPULSE:
			nam=modfile->buf+4;
			break;
		case TYPE_AMIGA_MOD:
			/* MOD format doesn`t assure that name will be NULL-terminated, so we need to "check" it */
			while (i<20) {
				nam=modfile->buf+i;
				if ( *nam == 0 ) break;
				++i;
			}
			if ( i==20 ) {
				nam=(char*)malloc(21);
				for ( i=0;i<20;++i ) {
					nam[i]=*(modfile->buf+i);
				}
				nam[20]=0;
			} else nam=modfile->buf;
			break;
		case TYPE_FT2:
			if ( modfile->buflen > 17+20 ) {
				nam=(char*)malloc(21);
				memcpy(nam,modfile->buf+17,20);
				nam[20]=0;
			}
			break;
			
	}
	
	return nam;
}

/* -- Very simple function which returns module type -- */
short module_type(MODULE* modfile) {
	return modfile->type;
}

/* -- Free the module buffer -- */
void module_free(MODULE* modfile) {
	if ( modfile->live == 1 ) free(modfile->buf);
	modfile->live=-1;
	modfile->buf=NULL;
	modfile->buflen=0;
	return;
}

/* ============== 
 * T H E   E N D 
 * ==============
 */
