#! /bin/sh
#  
#  Name
#    curls.sh
#  
#  Synopsis
#    curls.sh -h hostName -c cgiBinName -f fileName | -F dirName
#  
#  Example
#    curls.sh -h stranger -c Scripts 
#           -F  E:/Netscape/ns-home/docs/WebObjects
#  
#  Description
#    This script modifies the hostname and cgi-bin designations
#    in URLs marked with the "NeXTInfo=hostCustomize" attribute.   
#    If you use the example invocation above on this URL:
#
#    <A href="http://localhost/cgi-bin/someFile.html" NextInfo = hostCustomize>
#       some text </A>
#
#    The resulting URL will be:
#
#    <A href="http://stranger/Scripts/someFile.html" NextInfo = hostCustomize>
#       some text </A>
#
#    If you give this script a fileName (-f), it makes the modifications 
#    on that file.  If you give it a directory (-F), it makes the modifications
#    on all .html files in the directory and its subdirectories
#       
#
#  Return value
#    0  Successful completion
#    1  Abnormal termination
#  
#  
#  
######################################################################  
CMDNAME=`basename $0`
USAGE="Usage: $CMDNAME [-d] -h hostname -c cgiBinName -f fileName | -F dirName"
D=                       
HOSTNAME=               
CGIBINNAME=                
FILENAME=                
DIRNAME=                
RECURSIVE=                
SUBSTRING="NeXTInfo=hostCustomize"     # marker that we search for
SYSTEMTYPE=
CURLS=

#
# Parse command options.
#
while [ $# -gt 0 ] 
do
    case $1 in
	-d) D=TRUE      # Debugging on
	    shift
	    ;;
	-h) HOSTNAME=$2
	    shift 2
	    ;;
	-c) CGIBINNAME=$2
	    shift 2
	    ;;
	-f) FILENAME=$2
	    shift 2
	    ;;
	-F) RECURSIVE=TRUE      # directory name
	    DIRNAME=$2
	    shift 2
	    ;;
	-*) echo "$USAGE" 1>&2
	    exit 1
	    ;;
	*)  break
	    ;;
    esac
done

if [ "$D" = "TRUE" ] 
then 
    echo "hostname=$HOSTNAME, cgi-bin=$CGIBINNAME, filename=$FILENAME, dirName=$DIRNAME\n"
fi

if [ "$HOSTNAME" = "" ] || [ "$CGIBINNAME" = "" ] 
then 
    echo "$USAGE"
    exit 1
fi

if [ "$FILENAME" = "" ] 
then 
    if [ "$DIRNAME" = "" ]
    then
	echo "curls.sh: you must specify a file or directory name."
	exit 1
    fi
fi

# shell function for finding system type
systemType() {

    _SYSTEMTYPE=
    
    { uname; } >errorMsg.$$ 2>&1  # no /dev/null on NT
    # do not put any commands between the previous line and the next one
    SUCCESS=$?
    rm -f errorMsg.$$
    
    if [ $SUCCESS -eq 0 ] 
    then
	case `uname -s` in
		Windows_NT )    _SYSTEMTYPE="NT"
			    ;;
		HP-UX )    _SYSTEMTYPE="HP"
			    ;;
		SunOS )    case `uname -r` in
				4*)  _SYSTEMTYPE="SUNBSD"    
				    ;;
				5*)  _SYSTEMTYPE="SOLARIS"   
				    ;;
			    esac
			    ;;
	    esac
    else
	# now double-check that its a Mach system
	if [ -d "/NextLibrary" ]
	then
	    _SYSTEMTYPE="MACH"
	fi
    fi
    echo $_SYSTEMTYPE
}

SYSTEMTYPE=`systemType`

if [ "$SYSTEMTYPE" = "" ] 
then 
    echo "Unknown system type"
else
    case $SYSTEMTYPE in
	NT )    
	    CURLS="$NEXT_ROOT/Library/Executables/sh curls.sh"
	    ;;
	MACH )    
	    CURLS="curls.sh"
	    ;;
	HP )    
	    CURLS="curls.sh"
	    ;;
	SUNBSD | SOLARIS )    
	    CURLS="curls.sh"
	    ;;
    esac
fi



if [ "$FILENAME" = "" ] 
then 
    find $DIRNAME -type f -name "*.html" -exec grep -li "$SUBSTRING" {} \; -exec $CURLS -h $HOSTNAME -c $CGIBINNAME -f {} \;

else
    OLDIFS=$IFS
    IFS=
    echo "\n" >> $FILENAME  #add a newline as a precaution: input file may not end in one.
    while read LINE
    do
        case "$LINE" in
            *$SUBSTRING* ) 
	        LINE=`echo "$LINE" | sed -e "s%//localhost/cgi-bin/%//$HOSTNAME/$CGIBINNAME/%g"`
 	      ;;
        esac    
        echo "$LINE"
    done <$FILENAME >$FILENAME.$$
    IFS=$OLDIFS
    rm -f $FILENAME
    mv $FILENAME.$$ $FILENAME
fi

exit 0




