const CONFIG = {
	"AUDIO_RING_BUFFER_SAMPLES" : 2 * 48000 * 4 // 4s
};



const STATE = {
	"SAMPLE_TIME_INDEX" : 0,
	"AUDIO_LATENCY" : 1,
	"AUDIO_LATENCY_COMPENSATED" : 2,
	"AUDIO_LATENCY_OFFSET" : 3
};



class CastPlayWorkletProcessor extends AudioWorkletProcessor {
	constructor() {
		super();
		
		this.state = null;
		this.sampleTimeIndex = 0;
		
		this.audioLatency = 0;
		this.audioLatencyCompensated = false;
		this.audioLatencyOffset = 0;
		
		this.port.onmessage = this.message.bind(this);
	}
	
	
	
	dump(obj) {
		for (const key in obj)
			this.log(obj + "." + key + "=" + obj[key]);
	}
	
	log(msg) {
		const numsg = "[worklet] " + msg;
		console.log(numsg);
		this.port.postMessage({ type : "LOG", message : numsg });
	}
	
	
	
	message(event) {
		switch(event.data.type) {
			case "SET_BUFFERS": {
				const buffers = event.data;
				this.state = new Uint32Array(buffers.state);
				this.samples = new Float32Array(buffers.samples);
				break;
			}
			
			default: {
				this.log("message type unknown: " + event.data.type);
				break;
			}
		}
	}
	
	
	
	process(inputs, outputs, parameters) {
		if (this.state) {
			this.audioLatency = Atomics.load(this.state, STATE.AUDIO_LATENCY);
			this.audioLatencyCompensated = Atomics.load(this.state, STATE.AUDIO_LATENCY_COMPENSATED);
			this.audioLatencyOffset = Atomics.load(this.state, STATE.AUDIO_LATENCY_OFFSET);
		
			if (this.audioLatencyCompensated) {
				const audioSampleTime = this.sampleTimeIndex * 128;
				const output = outputs[0];
			
				let base = Math.round(((audioSampleTime + this.audioLatency - this.audioLatencyOffset) * 2) % CONFIG.AUDIO_RING_BUFFER_SAMPLES);
				for (let chdex = 0; chdex < output.length; chdex += 1) {
					const channel = output[chdex];
					let head = base + chdex;
				
					for (let index= 0; index< channel.length; index += 1) {
						channel[index] = this.samples[head];
						this.samples[head] = 0;
					
						if ((head += 2) >= CONFIG.AUDIO_RING_BUFFER_SAMPLES)
							head = 0;
					}
				}
			}
		
			this.sampleTimeIndex += 1;
			Atomics.store(this.state, STATE.SAMPLE_TIME_INDEX, this.sampleTimeIndex);
		}
		
		return true;
	}
}



registerProcessor('castplay-worklet-processor', CastPlayWorkletProcessor);
