'use strict';



var player = player || {};



player = function()
{
	this.castReceiverManager = cast.receiver.CastReceiverManager.getInstance();
	this.castReceiverManager.onReady = this.castReady.bind(this);
	this.castReceiverManager.onSenderDisconnected = this.castSenderDisconnected.bind(this);
	
	this.messageBus = this.castReceiverManager.getCastMessageBus("urn:x-cast:com.rogueamoeba.CastPlay",cast.receiver.CastMessageBus.MessageType.JSON);
	this.messageBus.onMessage = this.castMessage.bind(this);
	
	this.audioRingBufferSamples = 2 * 48000 * 4; // 4s
	this.audioRingBuffer = new Float32Array(this.audioRingBufferSamples);
	
	this.deviceSpecifics = {};
	this.audioLatency = 0;
	this.audioLatencyCompensated = false;
	this.audioLatencyLedger = [];
	this.audioLatencyOffset = 0;
	this.audioOffset = 48000 * 2;	// 2s (default) +/- audioLatencyOffset
	this.audioSampleTime = 0;
	
	this.model = null;
	this.ownerID = null;
	this.audioStartHandle = null;
	this.audioComputeHandle = null;
	this.audioContext = null;
	this.audioFrames = 8192;		// 170ms (minimum for CCv1)
	this.audioSource = null;
	this.audioNode = null;
	
	this.socket = null;
	
	this.badgeIconURL = null;
	this.albumImageURL = null;
}

player.prototype.start = function()
{
	this.castReceiverManager.start({statusText: "Airfoil: Loading..."});
}

player.prototype.log = function(msg)
{
	var payload = { type:"LOG", message:msg };
	this.messageBus.broadcast(payload);
	console.log(msg);
}



player.prototype.castReady = function(event)
{
	this.castReceiverManager.setApplicationState("Airfoil: Ready");
}

player.prototype.castSenderDisconnected = function(event)
{
	if (this.ownerID && (this.ownerID == event.senderId))
		this.audioStop();
	
	if (this.castReceiverManager.getSenders().length == 0)
		window.close();
}

player.prototype.castMessage = function(event)
{
 	switch(event.data.type)
 	{
 		case "CONNECT":
 		{
 			var error = null;
			var reply = { requestId:event.data.requestId,
						  responseType:event.data.type };
			
 			if (this.ownerID)
 			{
 				error = "Device Busy";
 			}
 			else
 			{
 				var format = event.data.format;
 				if (!format || (format.channels != 2) || (format.format != "lpcm") || (format.rate != 48000))
 				{
 					error = "Format Not Supported";
 				}
 			}
 			
			if (error)
			{
				reply.error = error;
				this.log(error);
			}
			
			this.messageBus.send(event.senderId,reply);
			if (!error)
			{
				var model = event.data.model;
 				if (model && (model != "<unknown>"))
 				{
 					this.model = model;
 					this.log("client provided model: " + model);
 				}
 				
	 			this.ownerID = event.senderId;
	 			
	 			var predelay = event.data.predelay;
	 			if (!predelay || (predelay < 0))
	 				predelay = 0;
	 			
	 			this.log("client provided predelay: " + predelay);
				this.audioStartHandle = setTimeout(function(player)
				{
					player.audioStartHandle = null;
					player.audioStart(event.data.url,event.data.format);
				}, (predelay * 1000), this);
 			}
 			break;
 		}
 		
 		case "GET_SUPPORTED_FORMATS":
 		{
 			var reply = { displaySupported:!this.isCastForAudioDevice(),
 						  formats:[{ channels:2, format:"lpcm", rate:48000 }],
 						  requestId:event.data.requestId,
 						  responseType:event.data.type };
 			this.messageBus.send(event.senderId,reply);
 			break;
 		}
 	}
}



player.prototype.audioStart = function(url,format)
{
	this.log("starting audio...");
	
	this.deviceSpecifics = this.getDeviceSpecifics();
	this.audioLatency = this.deviceSpecifics.latency;
	this.audioLatencyCompensated = false;
	this.audioLatencyLedger = [];
	this.audioLatencyOffset = 0;
	this.audioSampleTime = 0;
	
	// Create and play silence to a temporary audio context to preheat the
	// audio context machinery and underlying audio hardware. This causes
	// our real follow on audio context to start up nearly instantly.
	{
		var nucontext = new AudioContext();
		
		var sampleRate = nucontext.sampleRate;
		var buffer = nucontext.createBuffer(2,512,sampleRate);
		
		var node = nucontext.createBufferSource();
		node.buffer = buffer;
		node.loop = true;
		node.connect(nucontext.destination);
		node.start(0);
		
		while(nucontext.currentTime < 1);
		
		node.stop(0);
		node.disconnect(0);
	}
	
	this.audioContext = new AudioContext();
	
	this.audioNode = this.audioContext.createScriptProcessor(this.audioFrames,2,2);
	this.audioNode.onaudioprocess = this.audioProcess.bind(this);
	this.audioNode.connect(this.audioContext.destination);
	
	this.audioComputeHandle = setInterval(function(player) { player.audioCompute(); }, 200, this);
	
	this.socket = new WebSocket(url);
	this.socket.binaryType = 'arraybuffer';
	this.socket.onopen = this.socketOpen.bind(this);
	this.socket.onclose = this.socketClose.bind(this);
	this.socket.onmessage = this.socketMessage.bind(this);
	
	this.log("started audio: " + url + " " + JSON.stringify(format));
	
	this.castReceiverManager.setApplicationState("Airfoil: Playing");
}

player.prototype.audioStop = function()
{
	this.log("stop audio");
	this.castReceiverManager.setApplicationState("Airfoil: Ready");
	
	if (this.audioStartHandle)
	{
		clearTimeout(this.audioStartHandle);
		this.audioStartHandle = null;
	}
	
	if (this.audioComputeHandle)
	{
		clearInterval(this.audioComputeHandle);
		this.audioComputeHandle = null;
	}
	
	if (this.socket)
	{
		this.socket.close();
		this.socket = null;
	}
	
	this.ownerID = null;
	this.audioContext = null;
	this.audioNode = null;
}

player.prototype.audioCompute = function()
{
	var length = this.audioLatencyLedger.length;
	if (length >= this.getAudioLatencyLedgerSize())
	{
		var median = 0;
		for (var index = 0;index < length;index += 1)
			median += this.audioLatencyLedger[index];
		
		median = Math.round(median / length);
		this.audioLatencyLedger.sort(function(a,b) { return Math.abs(a - median) > Math.abs(b - median); });
		
		length = Math.min(length, 3);
		median = 0;
		
		for (var index = 0;index < length;index += 1)
			median += this.audioLatencyLedger[index];
		
		median = Math.round(median / length);
		var previousAudioLatency = this.audioLatency;
		var updatedAudioLatency = Math.round(this.deviceSpecifics.latency + median);
		
		if (!this.audioLatencyCompensated || (Math.abs(updatedAudioLatency - previousAudioLatency) >= this.audioFrames))
		{
			this.log("latency compensation: " + previousAudioLatency + " -> " + updatedAudioLatency);
			this.audioLatency = updatedAudioLatency;
			this.audioLatencyCompensated = true;
		}
		
		this.audioLatencyLedger = [];
	}
}

player.prototype.audioProcess = function(event)
{
	var computedSampleTime = this.audioContext.currentTime * this.audioContext.sampleRate;
	var computedSampleDelta = Math.round(computedSampleTime - this.audioSampleTime + this.audioFrames);
	this.audioLatencyLedger.push(computedSampleDelta);
	
	if (this.audioLatencyCompensated)
	{
		var channel0 = event.outputBuffer.getChannelData(0);
		var channel1 = event.outputBuffer.getChannelData(1);
		
		var head = Math.round(((this.audioSampleTime + this.audioLatency - this.audioLatencyOffset) * 2) % this.audioRingBufferSamples);
		for (var index = 0;index < this.audioFrames;index += 1)
		{
			channel0[index] = this.audioRingBuffer[head + 0];
			channel1[index] = this.audioRingBuffer[head + 1];
	
			this.audioRingBuffer[head + 0] = 0;
			this.audioRingBuffer[head + 1] = 0;
	
			if ((head += 2) >= this.audioRingBufferSamples)
				head = 0;
		}
	}
	
	this.audioSampleTime += this.audioFrames;
}



player.prototype.packetDispatch = function(props,data,flipped)
{
	switch(props.type)
	{
		case "AUDIO_LPCM":
		{
			this.packetAudioLPCM(props,data,flipped);
			break;
		}
		
		case "SET_BADGE":
		{
			this.packetSetBadge(props,data);
			break;
		}
		
		case "SET_LATENCY_OFFSET":
		{
			this.packetSetLatencyOffset(props);
			break;
		}
		
		case "SET_METADATA":
		{
			this.packetSetMetadata(props,data);
			break;
		}
		
		case "SET_VOLUME_LEVEL":
		{
			this.packetSetVolumeLevel(props);
			break;
		}
		
		case "SET_VOLUME_MUTED":
		{
			this.packetSetVolumeMuted(props);
			break;
		}

		case "TIMESTAMP":
		{
			this.packetTimestamp(props);
			break;
		}

		default:
		{
			this.log("packet type unknown: " + props.type);
			break;
		}
	}
}

player.prototype.packetAudioLPCM = function(props,data,flipped)
{
	if (!data)
	{
		this.log("AUDIO_LPCM: data invalid");
		return;
	}
	
	var timestamp = props.timestamp;
	if (!timestamp)
	{
		this.log("AUDIO_LPCM: timestamp invalid");
		return;
	}
	
	var frames = props.frames;
	if (!frames)
	{
		this.log("AUDIO_LPCM: frames invalid");
		return;
	}
	
	var framesDataSize = frames * 2 * 2;
	if (data.byteLength != framesDataSize)
	{
		this.log("AUDIO_LPCM: frames size mismatch: " + data.byteLength + " vs " + framesDataSize);
		return;
	}
	
	// ArrayBufferView assumes the underlying data is native endian. We can
	// use the getXXX routines in DataView to compensate for this, but they
	// have a lot of overhead and are very slow when used to process large
	// arrays. Manually detecting the mismatch and swapping the bytes
	// ourselves is an order of magnitude faster.
	//
	if (flipped)
	{
		var bytes = new Uint8Array(data.buffer,data.byteOffset,framesDataSize);
		for (var index = 0;index < framesDataSize;index += 2)
		{
			var tmp = bytes[index];
			bytes[index] = bytes[index + 1];
			bytes[index + 1] = tmp;
		}
	}
	
	var samples = new Int16Array(data.buffer,data.byteOffset,(frames * 2));
	var head = (timestamp * 2) % this.audioRingBufferSamples;
	var sampleCount = frames * 2;
	var norm = 1.0 / 32768.0;
	
	for (var index = 0;index < sampleCount;index += 1)
	{
		this.audioRingBuffer[head] = samples[index] * norm;
		if ((head += 1) >= this.audioRingBufferSamples)
			head = 0;
	}
	
	if (props.status)
	{
		var theSequenceID = props.sequenceID;
		if (!theSequenceID)
		{
			this.log("AUDIO_LPCM: sequenceID invalid");
			return;
		}
		
		var reply = { type : "STATUS", sequenceID : theSequenceID };
		this.socket.send(JSON.stringify(reply));
	}
}

player.prototype.packetSetBadge = function(props,data)
{
	delete props.type;
	this.log("badge: " + JSON.stringify(props) + " data: " + data.byteLength + " bytes");
	
	var label = props.label ? props.label : "";
	document.getElementById('badge').setAttribute('label',props.label);
	
	var iconType = props.icon;
	var iconBytes = new Uint8Array(data.buffer,data.byteOffset);
	if (iconType && (data.byteLength > 0))
	{
		var blob = new Blob([iconBytes],{ type : iconType });
		var url = URL.createObjectURL(blob);
		
		document.getElementById('badge-icon').src = url;
		
		if (this.badgeIconURL)
			URL.revokeObjectURL(this.badgeIconURL);
		
		this.badgeIconURL = url;
	}
}

player.prototype.packetSetLatencyOffset = function(props)
{
	var offset = props.offset;
	offset = isNaN(offset) ? 0.0 : Math.max(-1.0,Math.min(1.0,offset));
	this.audioLatencyOffset = Math.round(48000.0 * offset);
	this.log("latency offset: " + offset + " (" + this.audioLatencyOffset + ")");
}

player.prototype.packetSetMetadata = function(props,data)
{
	delete props.type;
	this.log("metadata: " + JSON.stringify(props) + " data: " + data.byteLength + " bytes");
	
	var album = props.album ? props.album : "";
	var artist = props.artist ? props.artist : "";
	var title = props.title ? props.title : "";
	
	var artworkType = props.artwork;
	var artworkBytes = new Uint8Array(data.buffer,data.byteOffset);
	
	document.getElementById('body').classList.add('hide-all');
	var handle = setInterval(function(player)
	{
		if (artworkType && (data.byteLength > 0))
		{
			var blob = new Blob([artworkBytes],{ type : artworkType });
			var url = URL.createObjectURL(blob);
			
			if (document.querySelector('.background-artwork'))
				document.querySelector('.background-artwork').src = url;
			
			document.querySelector('.album-artwork').src = url;
			
			if (player.albumImageURL)
				URL.revokeObjectURL(player.albumImageURL);
			
			player.albumImageURL = url;
		}
		
		document.querySelector('.meta-album').innerHTML = album;
		document.querySelector('.meta-artist').innerHTML = artist;
		document.querySelector('.meta-track').innerHTML = title;
		
		clearInterval(handle);
		var handle2 = setInterval(function()
		{
			clearInterval(handle2);
			document.getElementById('body').classList.remove('hide-all');
		}, 250);
	}, 500, this);
}

player.prototype.packetSetVolumeLevel = function(props)
{
	var level = props.level;
	level = isNaN(level) ? 1.0 : Math.max(0.0,Math.min(1.0,level));
	this.castReceiverManager.setSystemVolumeLevel(level);
	this.log("volume level: " + level);
}

player.prototype.packetSetVolumeMuted = function(props)
{
	var muted = props.muted;
	this.castReceiverManager.setSystemVolumeMuted(muted);
	this.log("volume muted: " + muted);
}

player.prototype.packetTimestamp = function(props)
{
	var theSequenceID = props.sequenceID;
	if (!theSequenceID)
	{
		this.log("TIMESTAMP: sequenceID invalid");
		return;
	}
	
	var theTimestamp = this.audioOffset + (this.audioContext.currentTime * this.audioContext.sampleRate);
	var reply = { type : "TIMESTAMP", sequenceID : theSequenceID, timestamp : theTimestamp };
	this.socket.send(JSON.stringify(reply));
}



player.prototype.socketOpen = function()
{
	this.log("socket connected");
}

player.prototype.socketClose = function(event)
{
	this.log("socket closed");
}

player.prototype.socketMessage = function(event)
{
	var packet = new DataView(event.data);
	var opcode = packet.getInt32(0);			//  0-3: packet opcode
	var packetSize = packet.getInt32(4);		//  4-7: packet size
	
	if (event.data.byteLength != packetSize)
	{
		this.log("packet buffer size mismatch: " + event.data.byteLength + " vs " + packetSize);
		return;
	}
	
	switch(opcode)
	{
		case 1: // Format1
		{
			var headerNE = new Int32Array(event.data);
			var flipped = event.data.byteLength != headerNE[1];
			
			var propsDataSize = packet.getInt32(8);			//  8-12: props data size
			var dataSize = packet.getInt32(12);				// 12-15: data size
			
			var roundedPropsDataSize = (propsDataSize + 3) & ~3;
			var roundedDataSize = (dataSize + 3) & ~3;
			var computedPacketSize = 16 + roundedPropsDataSize + roundedDataSize;
			
			if (computedPacketSize != packetSize)
			{
				this.log("packet content size mismatch: " + computedPacketSize + " vs " + packetSize);
				return;
			}
			
			var props = null;
			var data = null;
			
			if (propsDataSize > 0)
			{
				var propsBytes = new Uint8Array(event.data,16,propsDataSize);
				props = JSON.parse(stringFromUTF8Array(propsBytes));
			}
			
			if (!props)
			{
				this.log("packet properties invalid");
				return;
			}
			
			if (dataSize > 0)
				data = new DataView(event.data,(16 + roundedPropsDataSize));
			
			this.packetDispatch(props,data,flipped);
			break;
		}
		
		default:
		{
			this.log("packet opcode unknown: " + opcode);
			break;
		}
	}
}



player.prototype.getAudioLatencyLedgerSize = function()
{
	var secondsPerBuffer = this.audioFrames / 48000;
	return Math.round((this.audioLatencyCompensated ? 5 : 2.5) / secondsPerBuffer);
}

player.prototype.getDeviceSpecifics = function()
{
	if (this.model && (this.model == "Chromecast Ultra"))
	{
		return { type : "Chromecast Ultra",
				 latency : 53568 };
	}
	
	if (this.isCastForAudioDevice() || (this.model && (this.model == "Chromecast Audio")))
	{
		return { type : "CCa",
				 latency : 12416 };
	}
	
	if (window.navigator.hardwareConcurrency == 1)
	{
		return { type : "CCv1",
				 latency : 26240 };
	}
	
	return { type : "CCv2",
			 latency : 61248 };
}

player.prototype.isCastForAudioDevice = function()
{
	if (this.castReceiverManager)
	{
		var deviceCapabilities = this.castReceiverManager.getDeviceCapabilities();
		if (deviceCapabilities)
		{
			return deviceCapabilities['display_supported'] === false;
		}
	}
	
	return false;
};



function stringFromUTF8Array(data)
{
	const extraByteMap = [ 1, 1, 1, 1, 2, 2, 3, 0 ];
	var count = data.length;
	var str = "";
	
	for (var index = 0;index < count;)
	{
		var ch = data[index++];
		if (ch & 0x80)
		{
			var extra = extraByteMap[(ch >> 3) & 0x07];
			if (!(ch & 0x40) || !extra || ((index + extra) > count))
				return null;
			
			ch = ch & (0x3F >> extra);
			for (;extra > 0;extra -= 1)
			{
				var chx = data[index++];
				if ((chx & 0xC0) != 0x80)
					return null;
				
				ch = (ch << 6) | (chx & 0x3F);
			}
		}
		
		str += String.fromCharCode(ch);
	}
	
	return str;
}
