const CONFIG = {
	"AUDIO_RING_BUFFER_SAMPLES" : 2 * 48000 * 4 // 4s
};



const STATE = {
	"SAMPLE_TIME_INDEX" : 0,
	"AUDIO_LATENCY" : 1,
	"AUDIO_LATENCY_COMPENSATED" : 2,
	"AUDIO_LATENCY_OFFSET" : 3
};



class CastPlayReceiver extends Object {
	constructor() {
		super();
		
		this.cast = cast.framework.CastReceiverContext.getInstance();
		this.cast.addCustomMessageListener("urn:x-cast:com.rogueamoeba.CastPlay", this.castMessage.bind(this));
	
		this.cast.addEventListener(cast.framework.system.EventType.READY, this.castReady.bind(this));
		this.cast.addEventListener(cast.framework.system.EventType.SENDER_DISCONNECTED, this.castSenderDisconnected.bind(this));
		
		this.deviceSpecifics = {};
		this.audioFormat = null;
		this.audioLatency = 0;
		this.audioLatencyCompensated = false;
		this.audioLatencyLedger = [];
		this.audioLatencyOffset = 0;
		this.audioOffset = 48000 * 2;	// 2s (default) +/- audioLatencyOffset
		this.audioSampleTime = 0;
	
		this.debugEnabled = false;
		this.debugAudioExpectedTimestamp = 0;
	
		this.model = null;
		this.ownerID = null;
		this.audioWorkletSupport = false;
		this.audioStartTimestamp = null;
		this.audioStartHandle = null;
		this.audioComputeHandle = null;
		this.audioContext = null;
		this.audioFrames = 8192;		// 170ms (minimum for CCv1)
		this.audioNode = null;
		this.audioRingBuffer = null;
		this.audioSharedRingBuffer = null;
		
		this.socket = null;
		this.socketURL = null;
	
		this.badgeIconURL = null;
		this.albumImageURL = null;
	}



	start() {
		const options = new cast.framework.CastReceiverOptions();
	
		options.customNamespaces = { "urn:x-cast:com.rogueamoeba.CastPlay" : cast.framework.system.MessageType.JSON };
		options.disableIdleTimeout = true;
		options.statusText = "Airfoil: Loading...";
	
		this.cast.start(options);
	}



	dump(obj) {
		for (const key in obj)
			this.log(obj + "." + key + "=" + obj[key]);
	}

	log(msg) {
		this.broadcast({ type : "LOG", message : msg });
		console.log(msg);
	}



	broadcast(payload) {
		this.cast.sendCustomMessage("urn:x-cast:com.rogueamoeba.CastPlay", undefined, payload);
	}

	send(sender, payload) {
		this.cast.sendCustomMessage("urn:x-cast:com.rogueamoeba.CastPlay", sender, payload);
	}



	castReady(event) {
		this.cast.setApplicationState("Airfoil: Ready");
	}


	castSenderDisconnected(event) {
		if (this.ownerID && (this.ownerID == event.senderId))
			this.audioStop();
	
		if (this.cast.getSenders().length == 0)
			window.close();
	}



	castMessage(event) {
		switch(event.data.type) {
			case "CONNECT": {
				const reply = { requestId : event.data.requestId, responseType : event.data.type };
				let error = null;
			
				if (this.ownerID) {
					error = "Device Busy";
				}
				else {
					const format = event.data.format;
					if (!format || (format.channels != 2) || (format.format != "lpcm") || (format.rate != 48000)) {
						error = "Format Not Supported";
					}
				}
			
				if (error) {
					reply.error = error;
					this.log(error);
				}
			
				this.send(event.senderId, reply);
				if (!error) {
					const model = event.data.model;
					if (model && (model != "<unknown>")) {
						this.model = model;
						this.log("client provided model: " + model);
					}
				
					this.ownerID = event.senderId;
				
					let predelay = event.data.predelay;
					if (!predelay || (predelay < 0))
						predelay = 0;
				
					this.log("client provided predelay: " + predelay);
					this.audioStartHandle = setTimeout(function(player) {
						player.audioStartHandle = null;
						player.audioStart(event.data.url, event.data.format);
					}, (predelay * 1000), this);
				}
				break;
			}
		
			case "GET_SUPPORTED_FORMATS": {
				const reply = { displaySupported : !this.isCastForAudioDevice(),
								formats : [{ channels : 2, format : "lpcm", rate : 48000 }],
								requestId : event.data.requestId,
								responseType : event.data.type };
				this.send(event.senderId, reply);
				break;
			}
		}
	}



	audioStart(url, format) {
		this.deviceSpecifics = this.getDeviceSpecifics();
		this.audioFormat = format;
		this.socketURL = url;
	
		this.audioLatency = this.deviceSpecifics.latency;
		this.audioLatencyCompensated = false;
		this.audioLatencyLedger = [];
		this.audioLatencyOffset = 0;
		this.audioSampleTime = 0;
	
		// Create and play silence to an initial (ie, temporary) audio context to preheat
		// the audio context machinery and underlying audio hardware. This causes the
		// real follow on audio context to start up nearly instantly.
		this.audioContext = new AudioContext();
		
		this.audioNode = this.audioContext.createBufferSource();
		this.audioNode.buffer = this.audioContext.createBuffer(2, 512, this.audioContext.sampleRate);
		this.audioNode.loop = true;
		this.audioNode.connect(this.audioContext.destination);
		this.audioNode.start(0);
		
		this.audioStartTimestamp = (new Date()).getTime();
		this.audioStartHandle = setInterval(function(player) { player.audioStart2(); }, 250, this);
		this.audioStart2();
	}
	
	audioStart2() {
		while(true) {
			if (this.audioContext.currentTime > 1)
				break;
		
			if ((((new Date()).getTime()) - this.audioStartTimestamp) > 2000) {
				this.log("ABORTING AUDIO PREFLIGHT: WebAudio Clock Appears Non-Functional !!!");
				break;
			}
		
			return;
		}
	
		clearInterval(this.audioStartHandle);
		this.audioStartHandle = null;
	
		this.audioNode.stop(0);
		this.audioNode.disconnect();
		this.audioNode = null;
	
		this.audioContext.close();
		this.audioContext = null;
	
		// Start real audio context.
		this.audioContext = new AudioContext();
		this.audioStartTimestamp = (new Date()).getTime();
		
		if (this.audioContext.audioWorklet &&
			(typeof this.audioContext.audioWorklet.addModule !== "undefined") &&
			(typeof window.SharedArrayBuffer !== "undefined")) {
		
			this.log("AudioWorklet Supported");
			this.audioWorkletSupport = false;	// disabled
		}
		
		if (this.audioWorkletSupport) {
			// AudioWorklet path
			this.log("Using AudioWorklet");
			this.audioContext.audioWorklet.addModule("processor.js").then(() => {
				this.audioSharedRingBuffer = new SharedArrayBuffer(CONFIG.AUDIO_RING_BUFFER_SAMPLES * 4);
				this.audioRingBuffer = new Float32Array(this.audioSharedRingBuffer);
				
				this.audioNode = new CastPlayWorkletNode(this.audioContext, this.audioSharedRingBuffer);
				this.audioNode.port.onmessage = (event) => {
					switch(event.data.type) {
						case "LOG": {
							this.broadcast({ type : "LOG", message : event.data.message });
						}
					}
				}
				
				this.audioStart3();
			});
		}
		else {
			// ScriptProcessor path
			this.log("Using ScriptProcessor");
			this.audioNode = this.audioContext.createScriptProcessor(this.audioFrames, 2, 2);
			this.audioNode.onaudioprocess = this.audioScriptProcessorProcess.bind(this);
			this.audioRingBuffer = new Float32Array(CONFIG.AUDIO_RING_BUFFER_SAMPLES);
			this.audioStart3();
		}
	}

	audioStart3() {
		this.audioNode.connect(this.audioContext.destination);
		this.audioComputeHandle = setInterval(function(player) { player.audioCompute(); }, 200, this);
		
		this.socket = new WebSocket(this.socketURL);
		this.socket.binaryType = 'arraybuffer';
		this.socket.onopen = this.socketOpen.bind(this);
		this.socket.onclose = this.socketClose.bind(this);
		this.socket.onmessage = this.socketMessage.bind(this);
	
		this.log("started audio: " + this.socketURL + " " + JSON.stringify(this.audioFormat));
	
		this.cast.setApplicationState("Airfoil: Playing");
	}



	audioStop() {
		this.cast.setApplicationState("Airfoil: Ready");
	
		if (this.audioStartHandle) {
			clearTimeout(this.audioStartHandle);
			this.audioStartHandle = null;
		}
	
		if (this.audioComputeHandle) {
			clearInterval(this.audioComputeHandle);
			this.audioComputeHandle = null;
		}
	
		if (this.socket) {
			this.socket.close();
			this.socket = null;
		}
	
		if (this.audioContext)
			this.audioContext.close();
	
		if (this.audioNode)
			this.audioNode.disconnect();
		
		if (this.audioRingBuffer)
			this.audioRingBuffer = null;
		
		if (this.audioSharedRingBuffer)
			this.audioSharedRingBuffer = null;
		
		this.ownerID = null;
		this.audioContext = null;
		this.audioNode = null;
	}



	audioCompute() {
		if (this.audioWorkletSupport) {
			const workletSampleTime = Atomics.load(this.audioNode.state, STATE.SAMPLE_TIME_INDEX) * 128;
			const contextSampleTime = this.audioContext.currentTime * this.audioContext.sampleRate;
			const computedSampleDelta = Math.round(contextSampleTime - workletSampleTime);
			this.audioLatencyLedger.push(computedSampleDelta);
			
			if (this.debugEnabled) {
				this.log("computedSampleDelta: " + computedSampleDelta);
			}
		}
		
		let length = this.audioLatencyLedger.length;
		if (length >= this.getAudioLatencyLedgerSize()) {
			let median = 0;
			for (let index = 0; index < length; index += 1)
				median += this.audioLatencyLedger[index];
		
			median = Math.round(median / length);
			this.audioLatencyLedger.sort(function(a, b) { return Math.abs(a - median) > Math.abs(b - median); });
		
			length = Math.min(length, 3);
			median = 0;
		
			for (let index = 0; index < length; index += 1)
				median += this.audioLatencyLedger[index];
		
			median = Math.round(median / length);
			const previousAudioLatency = this.audioLatency;
			const updatedAudioLatency = Math.round(this.deviceSpecifics.latency + median);
		
			if (!this.audioLatencyCompensated || (Math.abs(updatedAudioLatency - previousAudioLatency) >= this.audioFrames)) {
				this.log("latency compensation: " + previousAudioLatency + " -> " + updatedAudioLatency);
				this.audioLatency = updatedAudioLatency;
				this.audioLatencyCompensated = true;
			}
		
			this.audioLatencyLedger = [];
		}
		
		if (this.audioWorkletSupport) {
			Atomics.store(this.audioNode.state, STATE.AUDIO_LATENCY, this.audioLatency);
			Atomics.store(this.audioNode.state, STATE.AUDIO_LATENCY_COMPENSATED, this.audioLatencyCompensated);
			Atomics.store(this.audioNode.state, STATE.AUDIO_LATENCY_OFFSET, this.audioLatencyOffset);
		}
	}



	audioScriptProcessorProcess(event) {
		if (this.audioContext) {
			const computedSampleTime = this.audioContext.currentTime * this.audioContext.sampleRate;
			const computedSampleDelta = Math.round(computedSampleTime - this.audioSampleTime + this.audioFrames);
			this.audioLatencyLedger.push(computedSampleDelta);
		
			if (this.audioLatencyCompensated) {
				const channel0 = event.outputBuffer.getChannelData(0);
				const channel1 = event.outputBuffer.getChannelData(1);
		
				let head = Math.round(((this.audioSampleTime + this.audioLatency - this.audioLatencyOffset) * 2) % CONFIG.AUDIO_RING_BUFFER_SAMPLES);
				for (let index = 0; index < this.audioFrames; index += 1) {
					channel0[index] = this.audioRingBuffer[head + 0];
					channel1[index] = this.audioRingBuffer[head + 1];
	
					this.audioRingBuffer[head + 0] = 0;
					this.audioRingBuffer[head + 1] = 0;
	
					if ((head += 2) >= CONFIG.AUDIO_RING_BUFFER_SAMPLES)
						head = 0;
				}
			}
	
			if (this.debugEnabled) {
				const nutime = this.audioSampleTime + this.audioLatency - this.audioLatencyOffset;
				const offsetMS = 1000 * ((this.debugAudioExpectedTimestamp - nutime) / 48000);
				this.log("processd: " + nutime + "-" + (nutime + this.audioFrames - 1) + " (" + offsetMS + " ms)");
			}
	
			this.audioSampleTime += this.audioFrames;
		}
	}



	packetDispatch(props, data, flipped) {
		switch(props.type) {
			case "AUDIO_LPCM": {
				this.packetAudioLPCM(props, data, flipped);
				break;
			}
		
			case "SET_BADGE": {
				this.packetSetBadge(props, data);
				break;
			}
		
			case "SET_LATENCY_OFFSET": {
				this.packetSetLatencyOffset(props);
				break;
			}
		
			case "SET_METADATA": {
				this.packetSetMetadata(props, data);
				break;
			}
		
			case "SET_VOLUME_LEVEL": {
				this.packetSetVolumeLevel(props);
				break;
			}
		
			case "SET_VOLUME_MUTED": {
				this.packetSetVolumeMuted(props);
				break;
			}

			case "TIMESTAMP": {
				this.packetTimestamp(props);
				break;
			}

			default: {
				this.log("packet type unknown: " + props.type);
				break;
			}
		}
	}



	packetAudioLPCM(props, data, flipped) {
		if (!data) {
			this.log("AUDIO_LPCM: data invalid");
			return;
		}
	
		const timestamp = props.timestamp;
		if (!timestamp) {
			this.log("AUDIO_LPCM: timestamp invalid");
			return;
		}
	
		const frames = props.frames;
		if (!frames) {
			this.log("AUDIO_LPCM: frames invalid");
			return;
		}
	
		const framesDataSize = frames * 2 * 2;
		if (data.byteLength != framesDataSize) {
			this.log("AUDIO_LPCM: frames size mismatch: " + data.byteLength + " vs " + framesDataSize);
			return;
		}
	
		// ArrayBufferView assumes the underlying data is native endian. We can
		// use the getXXX routines in DataView to compensate for this, but they
		// have a lot of overhead and are very slow when used to process large
		// arrays. Manually detecting the mismatch and swapping the bytes
		// ourselves is an order of magnitude faster.
		//
		if (flipped) {
			const bytes = new Uint8Array(data.buffer, data.byteOffset, framesDataSize);
			for (let index = 0; index < framesDataSize; index += 2) {
				const tmp = bytes[index];
				bytes[index] = bytes[index + 1];
				bytes[index + 1] = tmp;
			}
		}
	
		if (this.debugEnabled) {
			const nustamp = timestamp + frames;
			const oldest = nustamp - (CONFIG.AUDIO_RING_BUFFER_SAMPLES / 2);
			this.log("received: " + oldest + "-" + nustamp + " (" + (this.debugAudioExpectedTimestamp - timestamp) + ")");
			this.debugAudioExpectedTimestamp = nustamp;
		}
	
		const samples = new Int16Array(data.buffer, data.byteOffset, (frames * 2));
		let head = (timestamp * 2) % CONFIG.AUDIO_RING_BUFFER_SAMPLES;
		const sampleCount = frames * 2;
		const norm = 1.0 / 32768.0;
	
		for (let index = 0; index < sampleCount; index += 1) {
			this.audioRingBuffer[head] = samples[index] * norm;
			if ((head += 1) >= CONFIG.AUDIO_RING_BUFFER_SAMPLES)
				head = 0;
		}
	
		if (props.status) {
			const theSequenceID = props.sequenceID;
			if (!theSequenceID) {
				this.log("AUDIO_LPCM: sequenceID invalid");
				return;
			}
		
			const reply = { type : "STATUS", sequenceID : theSequenceID };
			this.socket.send(JSON.stringify(reply));
		}
	}



	packetSetBadge(props, data) {
		delete props.type;
		this.log("badge: " + JSON.stringify(props) + " data: " + data.byteLength + " bytes");
	
		const label = props.label ? props.label : "";
		document.getElementById('badge').setAttribute('label', props.label);
	
		const iconType = props.icon;
		const iconBytes = new Uint8Array(data.buffer, data.byteOffset);
		if (iconType && (data.byteLength > 0)) {
			const blob = new Blob([iconBytes], { type : iconType });
			const url = URL.createObjectURL(blob);
		
			document.getElementById('badge-icon').src = url;
		
			if (this.badgeIconURL)
				URL.revokeObjectURL(this.badgeIconURL);
		
			this.badgeIconURL = url;
		}
	}



	packetSetLatencyOffset(props) {
		let offset = props.offset;
		offset = isNaN(offset) ? 0.0 : Math.max(-1.0, Math.min(1.0, offset));
		this.audioLatencyOffset = Math.round(48000.0 * offset);
		this.log("latency offset: " + offset + " (" + this.audioLatencyOffset + ")");
	}



	packetSetMetadata(props, data) {
		delete props.type;
		this.log("metadata: " + JSON.stringify(props) + " data: " + data.byteLength + " bytes");
	
		const album = props.album ? props.album : "";
		const artist = props.artist ? props.artist : "";
		const title = props.title ? props.title : "";
	
		const artworkType = props.artwork;
		const artworkBytes = new Uint8Array(data.buffer, data.byteOffset);
	
		document.getElementById('body').classList.add('hide-all');
		let handle = setInterval(function(player) {
			if (artworkType && (data.byteLength > 0)) {
				const blob = new Blob([artworkBytes], { type : artworkType });
				const url = URL.createObjectURL(blob);
			
				if (document.querySelector('.background-artwork'))
					document.querySelector('.background-artwork').src = url;
			
				document.querySelector('.album-artwork').src = url;
			
				if (player.albumImageURL)
					URL.revokeObjectURL(player.albumImageURL);
			
				player.albumImageURL = url;
			}
		
			document.querySelector('.meta-album').innerHTML = album;
			document.querySelector('.meta-artist').innerHTML = artist;
			document.querySelector('.meta-track').innerHTML = title;
		
			clearInterval(handle);
			let handle2 = setInterval(function() {
				clearInterval(handle2);
				document.getElementById('body').classList.remove('hide-all');
			}, 250);
		}, 500, this);
	}



	packetSetVolumeLevel(props) {
		let level = props.level;
		level = isNaN(level) ? 1.0 : Math.max(0.0, Math.min(1.0, level));
		this.cast.setSystemVolumeLevel(level);
		this.log("volume level: " + level);
	}



	packetSetVolumeMuted(props) {
		const muted = props.muted;
		this.cast.setSystemVolumeMuted(muted);
		this.log("volume muted: " + muted);
	}



	packetTimestamp(props) {
		const theSequenceID = props.sequenceID;
		if (!theSequenceID) {
			this.log("TIMESTAMP: sequenceID invalid");
			return;
		}
	
		const theTimestamp = this.audioOffset + (this.audioContext.currentTime * this.audioContext.sampleRate);
		const reply = { type : "TIMESTAMP", sequenceID : theSequenceID, timestamp : theTimestamp };
		this.socket.send(JSON.stringify(reply));
	}



	socketOpen() {
		this.log("socket connected");
	}

	socketClose(event) {
		this.log("socket closed");
	}



	socketMessage(event) {
		const packet = new DataView(event.data);
		const opcode = packet.getInt32(0);			//  0-3: packet opcode
		const packetSize = packet.getInt32(4);		//  4-7: packet size
	
		if (event.data.byteLength != packetSize) {
			this.log("packet buffer size mismatch: " + event.data.byteLength + " vs " + packetSize);
			return;
		}
	
		switch(opcode) {
			case 1: { // Format1
				const headerNE = new Int32Array(event.data);
				const flipped = event.data.byteLength != headerNE[1];
			
				const propsDataSize = packet.getInt32(8);			//  8-12: props data size
				const dataSize = packet.getInt32(12);				// 12-15: data size
			
				const roundedPropsDataSize = (propsDataSize + 3) & ~3;
				const roundedDataSize = (dataSize + 3) & ~3;
				const computedPacketSize = 16 + roundedPropsDataSize + roundedDataSize;
			
				if (computedPacketSize != packetSize) {
					this.log("packet content size mismatch: " + computedPacketSize + " vs " + packetSize);
					return;
				}
			
				let props = null;
				let data = null;
			
				if (propsDataSize > 0) {
					const propsBytes = new Uint8Array(event.data, 16, propsDataSize);
					props = JSON.parse(stringFromUTF8Array(propsBytes));
				}
			
				if (!props) {
					this.log("packet properties invalid");
					return;
				}
			
				if (dataSize > 0)
					data = new DataView(event.data, (16 + roundedPropsDataSize));
			
				this.packetDispatch(props, data, flipped);
				break;
			}
		
			default: {
				this.log("packet opcode unknown: " + opcode);
				break;
			}
		}
	}



	getAudioLatencyLedgerSize() {
		const secondsPerBuffer = this.audioFrames / 48000;
		return Math.round((this.audioLatencyCompensated ? 5 : 2.5) / secondsPerBuffer);
	}



	getDeviceSpecifics() {
		if (this.model && (this.model == "Chromecast Ultra"))
			return { type : "Chromecast Ultra", latency : 53568 };
	
		if (this.model && (this.model.includes("Lenovo Smart Display")))
			return { type : "Lenovo Smart Display", latency : 13616 };
	
		if (this.isCastForAudioDevice() || (this.model && (this.model == "Chromecast Audio")))
			return { type : "CCa", latency : 12416 };
	
		if (window.navigator.hardwareConcurrency == 1)
			return { type : "CCv1", latency : 26240 };
	
		return { type : "CCv2", latency : 61248 };
	}



	isCastForAudioDevice() {
		if (this.cast) {
			const deviceCapabilities = this.cast.getDeviceCapabilities();
			if (deviceCapabilities)
				return deviceCapabilities['display_supported'] === false;
		}
	
		return false;
	};
}



// ================
// ================



function stringFromUTF8Array(data) {
	const extraByteMap = [ 1, 1, 1, 1, 2, 2, 3, 0 ];
	const count = data.length;
	let str = "";
	
	for (let index = 0; index < count;) {
		let ch = data[index++];
		if (ch & 0x80) {
			const extra = extraByteMap[(ch >> 3) & 0x07];
			if (!(ch & 0x40) || !extra || ((index + extra) > count))
				return null;
			
			ch = ch & (0x3F >> extra);
			for (;extra > 0;extra -= 1) {
				const chx = data[index++];
				if ((chx & 0xC0) != 0x80)
					return null;
				
				ch = (ch << 6) | (chx & 0x3F);
			}
		}
		
		str += String.fromCharCode(ch);
	}
	
	return str;
}



// ================
// ================



class CastPlayWorkletNode extends AudioWorkletNode {
	constructor(context, sharedSampleBuffer) {
		super(context, 'castplay-worklet-processor');
		
		const sharedStateBuffer = new SharedArrayBuffer(16);
		this.state = new Uint32Array(sharedStateBuffer);
		
		this.port.postMessage({ type : "SET_BUFFERS",
								samples : sharedSampleBuffer,
								state : sharedStateBuffer  });
	}
}
