#
# Analyse the system to diagnose installation failures, etc. Generates a nice HTML report for 
# displaying to the user and for sending to the AgileBits support team.
#
if RUBY_VERSION >= "1.9.0"
  require 'fileutils'
else  
  require 'ftools'
end
require 'date'
require 'logger'
require 'cgi'
require 'erb'
require 'base64'

HOME = File.expand_path("~")
CONTAINER = File.expand_path("~/Library/Containers/com.agilebits.onepassword-osx")
KNOWN_CONFIGURATION_ISSUES = []
BASE_GUIDE_URL = "http://help.agilebits.com/1Password3"
DROPBOX_PATH = ""

RESULT_TEMPLATE = <<result
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
  "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html>
<head>
	<title>1Password Diagnostics Report for <%= %x{whoami} %></title>
	<style type="text/css">
	  .indent { margin-left: 25px; }
	  .log { margin-left: 25px; }
	  h5.header {text-transform:uppercase;}
	</style>
</head>
<body>
	<h3>1Password Diagnostics Report for <%= %x{whoami} %></h3>

	<p>Generated: <%= Time.now() %></p>
	<p>This information will be used by AgileBits to help resolve any issues you may have.<\/p>

	<h3>Configuration</h3>
	<%= configuration.add_brs %>

	<h3>Summary</h3>
	<%= summary.add_brs %>

	<h3>System Details</h3>

	<h5>Table of Contents</h5>
  <ol>
  <%= @@table_of_contents %>
  </ol>
	<%= details.add_brs %>
	<h3>Logs</h3>
	<%= logs.add_brs %>
</body>
</html>
result

class String
  def add_brs
    return self.gsub(/([^>])\n/, "\\1<br/>")
  end
  
  def htmlize
    CGI::escapeHTML(self)
  end
end

def processStarted?(name, global=nil)
  only_us = "^$USER.*" if !global
  return `ps -Ajc | grep -v grep | grep "#{only_us}#{name}"` =~ /#{name}$/ ? true : false
end

def nsUserDefault(bundle_path, key, multi=nil)
  if multi
    nsDefault = `defaults read "#{bundle_path}" "#{key}" 2>&1`
    return nil if nsDefault =~ /The domain\/default pair of \(.*\) does not exist/m 
  else
    nsDefault = `defaults read "#{bundle_path}" 2>&1 | grep "#{key}"`
    nsDefault = $1 if nsDefault =~ / = (.*);$/
    nsDefault = $1 if nsDefault =~ /"(.*)"/
  end
  return nsDefault.chomp
end

def ns1PUserDefault(key, multi=nil, helper='')
  if APPSTORE
    return nsUserDefault("#{CONTAINER}#{helper}/Data/Library/Preferences/com.agilebits.onepassword-osx#{helper}", key, multi)
  else
    return nsUserDefault('ws.agile.1Password', key, multi)
  end
end

def print(s)
  @@result += s.to_s + "\n"
end

class Print
  def Print.header(txt)
    print "<h5 class='header' id='header_#{txt.hash.abs}'>#{txt}</h5>"
    @@table_of_contents += "<li><a href='#header_#{txt.hash.abs}'>#{txt}</a></li>"
  end
  
  def Print.start_section(txt=nil)
    Print.header(txt) unless txt.nil?
    print "<div class='indent'>"
  end
  
  def Print.end_section
    print "</div>"
  end
  
  def Print.writable(file)
    print "#{file} writable?: #{File.stat(File.expand_path(file)).writable?}"
  end

  def Print.exists(file)
    print "#{file} exists?: #{File.exists?(File.expand_path(file))}"
  end

  def Print.dir(dir, sort_option="", silent=nil)
    dir = File.expand_path(dir)
    
    print("#{dir} is symlinked to #{File.readlink(dir)}") if File.symlink?(dir)
    
    unless (File.exists?(dir))
      print "#{dir} does not exist" unless silent
      return
    end
    # dir.gsub!(/\s/, "\\ ")
    cmd = "ls -FaBl@WeH#{sort_option} \"#{dir}\""
    print "<pre>#{cmd}\n#{`#{cmd}`}</pre>"
  end
  
  def Print.stat(dir, silent=nil)
    dir = File.expand_path(dir)
    unless (File.exists?(dir))
      print "#{dir} does not exist" unless silent
      return
    end
    # dir.gsub!(/\s/, "\\ ")
    cmd = "stat \"#{dir}\""
    print "<pre>#{cmd}\n#{`#{cmd}`}</pre>"
  end
  
  def Print.file_contents(filename, silent=nil)
    if File.exists?(filename) && File.file?(filename)
      filename = File.expand_path(filename)
      file = File.open(filename) rescue nil
    
      unless file 
        print "Could not load file #{filename}"
        return
      end
      print "<i><u>Contents of #{filename}</u></i>\n"
      print file.read() + "\n\n"
    else
      print "File [#{filename}] does not exist." unless silent
    end
  end
  
  def Print.print_all_files_in_folder(folder)
    dir = File.expand_path(folder)
    unless File.exists?(dir)
      print "Folder &lt;#{folder}&gt; does not exist"
      return
    end
    
    files = Dir["#{dir}/*"]
    files.each { |f|
      Print.file_contents(f) if File.should_print?(f)
    }
  end
  
  def Print.exception(clazz, exc)
    Print.header("Exception in #{clazz.class}")
    print $!.message + "<br />"
    print $!.backtrace * "<br/>"
  end
end

class BaseTestClass 
  def run_tests
    begin
      tests
    rescue
      Print.exception(self, $!)
    end
  end
end

class Sys < BaseTestClass
  attr :lsregister_dump
  
  def initialize
    lsregister = "/System/Library/Frameworks/ApplicationServices.framework/Frameworks/LaunchServices.framework/Support/lsregister"
    unless File.exists?(lsregister)
      lsregister = "/System/Library/Frameworks/CoreServices.framework/Frameworks/LaunchServices.framework/Support/lsregister"
    end
    @lsregister_dump = `#{lsregister} -dump`
  end
  
  def tests
    Print.start_section "System Information"
    print "User home folder: #{HOME}"
    print `sw_vers`
    print `system_profiler SPHardwareDataType`.gsub(/^\s*Serial Number.*$/, "").gsub(/^\s*Hardware UUID:.*$/, "")
    Print.end_section

    Print.start_section "User &amp; Group Details"
    id_output = `id`
    print "Output from 'id' command: #{id_output}"
    dscl_output = `dscl . -read /Groups/wheel GroupMembership`
    print "Output from 'dscl' command: #{dscl_output}" if id_output =~ /wheel/
    Print.end_section

    Print.start_section("OpenSSL Information")
    print `openssl version`
    Print.end_section
    
    Print.start_section("Bonjour Information")
    print processStarted?('mDNSResponder', 't') ? "mDNSResponder Bonjour process is running." : "mDNSResponder Bonjour process is not running."
    Print.end_section
    
    Print.start_section("Potentially Relevant Currently Running Processes")
    print %x{ps -Ajc | grep -i 1P}
    print %x{ps -Ajc | grep -i Quicksilver}
    print %x{ps -Ajc | grep -i virus}
    Print.end_section
    
    Print.start_section("Available Disk Space")
    print "<pre>#{`df -h`}</pre>"
    Print.end_section
    
    Print.start_section("Dropbox Information")
    # print find_app("Dropbox") == nil ? "Could not locate Dropbox.app" : "Dropbox installed at #{find_app("Dropbox")}"
    dropbox_process = processStarted?('Dropbox') ? "Dropbox is running. Process information: #{%x{ps -Ajc | grep -i ^$USER.*Dropbox}}" : "Dropbox process is not started"
    print dropbox_process
    DROPBOX_PATH << get_dropbox_path
    print "Dropbox location: #{DROPBOX_PATH}"
    if File.exist?(DROPBOX_PATH)
      print "Dropbox 1Password configuration file: 'stat \"#{DROPBOX_PATH}\"' &rarr;  " << %x{stat "#{DROPBOX_PATH}"} unless DROPBOX_PATH =~ /Could not locate/
      Print.file_contents(File.join(DROPBOX_PATH, ".ws.agile.1Password.settings")) unless DROPBOX_PATH =~ /Could not locate/
      acct_dropbox_path = File.expand_path("~/Dropbox")
      if DROPBOX_PATH != acct_dropbox_path and DROPBOX_PATH != File.join(acct_dropbox_path, "1Password")
        print "<p>Dropbox location does <b>not</b> support 1Password 3.9 or higher storing its data file here.</p>"
        KNOWN_CONFIGURATION_ISSUES << "The Dropbox location of #{DROPBOX_PATH} will not be accessible by 1Password 3.9 or higher due to security sandbox rules -- only #{acct_dropbox_path} or #{acct_dropbox_path}/1Password is permitted."
      end
      if File.symlink?(acct_dropbox_path)
        print "<p>Home Dropbox location of #{acct_dropbox_path} is a symlink</p>"
        KNOWN_CONFIGURATION_ISSUES << "The home Dropbox location of #{acct_dropbox_path} is a symlink. This location may not be accessible by 1Password 3.9 or higher due to security sandbox rules."
      end
    else
      print "Dropbox location was not found"
      if dropbox_process =~ / is running/
        print "<p>Dropbox is running, but using an unknown location</p>"
        KNOWN_CONFIGURATION_ISSUES << "The Dropbox client software is running, but with an unknown location. This location may not be accessible by 1Password 3.9 or higher due to security sandbox rules."
      end
    end
    Print.end_section
    
  end
  
  def get_dropbox_path
    preferred_path, old_path = File.expand_path("~/.dropbox/host.dbx"), File.expand_path("~/.dropbox/host.db")
    return non_standard_dropbox_location? unless (File.exist?(preferred_path) or File.exist?(old_path))
    path_file = File.exist?(preferred_path) ? preferred_path : old_path
    lines = File.readlines(File.expand_path(path_file))
    return Base64.decode64(lines[1])
  end
  
  def non_standard_dropbox_location?
    path = ""
    path = $1 if `sqlite3 ~/.dropbox/config.db "select * from config where key='dropbox_path'"` =~ /dropbox_path\|(.*)\n/
    path = File.expand_path("~/Dropbox") if path.empty?
    return path
  end

  def Sys.is_lion?
    `sw_vers` =~ /ProductVersion:\t10.7/
  end

  def Sys.is_snow_leopard?
    `sw_vers` =~ /ProductVersion:\t10.6/
  end
  
  def Sys.is_leopard?
    `sw_vers` =~ /ProductVersion:\t10.5/
  end
  
  def Sys.is_tiger?
    `sw_vers` =~ /ProductVersion:\t10.4/
  end
  
  def find_app(name)
    result = $1 if @lsregister_dump =~ /^.*path:\s*(.*#{name})\.app$/
    return result
  end
end

class Summary

  def Summary.summary
    begin
      _summary
    rescue
      Print.exception(self, $!)
    end
  end

  def Summary._summary
    result = ""
    
    result << "1Password Version: #{OnePasswd.new.version}\n"
    sys_ver = "Mac OS X Version: "
    `sw_vers` =~ /\s*ProductVersion:\s*([0-9\.]+)$/i
    sys_ver << $1
    `sw_vers` =~ /\s*BuildVersion:\s*(\w+)$/i
    sys_ver << " (#{$1})\n"
    result << sys_ver
    
    if !APPSTORE
      result << "Active Keychain Format: "
      if Keychain.new.keychain_type =~ /html/i
        result << "Agile Keychain"
      else
        result << "OS X Keychain"
      end
      result << "\n"
    end
    keychain = Keychain.new    

    result << "Data File location: #{keychain.agileKeychainFolder}\n" if keychain.agile_keychain?
    keychain_data = File.expand_path(keychain.agileKeychainFolder) << "/data/default"

    opfiles = Dir["#{keychain_data}/*[^)].1password"]
    result << "Number of 1Password files: #{opfiles.length}\n" if Keychain.new.agile_keychain?
    
    # Is this user an admin?
    id_output = `id`
    id_output =~ /uid=\d+\(([^\)]+)\)/
    shortname = $1
    if id_output =~ /groups=.*?admin/
      result << "User #{shortname} is in admin group"
    else
      result << "User #{shortname} is <b><i>not</i></b> in admin group"
    end
    result << "\n\n"
    
    # What InputManagers do they have installed?
    result << "Installed extensions: "
    result << "<div class='indent'>"
    extension_locations = ["/Library/InputManagers", "#{HOME}/Library/InputManagers", "/Library/Internet Plug-Ins", "#{HOME}/Library/Internet Plug-Ins"]
    begin
      extension_locations.each do |path|
      if File.exist?(path)
        dir = Dir.new(path).entries.reject { |entry| entry =~ /^\./ }
        unless dir.length == 0
          result << "<h5>#{path}</h5>"
          result << "<ul>"
          dir.each { |entry|
            KNOWN_CONFIGURATION_ISSUES << "EvernoteSafariClipperPlugin is installed. This has been known to cause the 1P button on the Safari toolbar to not stay in the position you choose. If you are having this issue, please uninstall the EvernoteSafariClipperPlugin." if entry =~ /EvernoteSafariClipperPlugin/
            result << "<li>" << entry << "</li>"
          }
          result << "</ul>"
        end
      end
    end
    result << "</div>"
    rescue
    end
    result << "Firewall Software"
    result << "<div class='indent'>"
    little_snitch_installed = File.exist? '/Library/LaunchDaemons/at.obdev.littlesnitchd.plist'

    result << "Little Snitch installed? #{little_snitch_installed ? "<b>Yes</b>" : "No"}<br />"
    result << "Intego VirusBarrier installed? #{VirusBarrier.installed? ? "<b>Yes</b>" : "No"}<br />"
    result << "Intego VirusBarrier started? <b>#{VirusBarrier.started?}</b><br />" if VirusBarrier.installed?
    result << "Hands Off! installed? #{HandsOff.installed? ? "<b>Yes</b>" : "No"}<br />"
    result << "Hands Off! started? <b>#{HandsOff.started?}</b><br />" if HandsOff.installed?
    if little_snitch_installed or VirusBarrier.installed? or HandsOff.installed?
      m = "Firewall software can cause issues with software updates, and syncing with Dropbox or 1Password on Apple devices such as iPhone, iPod touch, and iPad. Be sure to approve 1Password for all outbound network activity."
      KNOWN_CONFIGURATION_ISSUES << "#{m} Please refer to <a href='#{BASE_GUIDE_URL}/outbound_connections.html'>this help document</a> for more information."
    end

    result << "</div>"
    result << "</div>"
    
    # Trusteer Rapport
    #
    result << "<br />"
    result << "Trusteer Rapport"
    result << "<div class='indent'>"
    result << "Rapport installed? #{TrusteerRapport.installed? ? "<b>Yes</b> " : "No"}<br />"
    result << "Rapport started? <b>#{TrusteerRapport.started?}</b><br />" if TrusteerRapport.installed? 

    if TrusteerRapport.installed?
        m = "Trusteer Rapport is installed and can cause web browser crashing on startup when not updated."
        KNOWN_CONFIGURATION_ISSUES << "#{m}"
    end

    result << "</div>"

    result.gsub!(/, \]/, "]")
    
    return result
    
  end
end

class Configuration

  def Configuration.summary
    begin
      _summary
    rescue
      Print.exception(self, $!)
    end
  end

  # Detect common configuration errors
  def Configuration._summary
    errors = []

    keychain = Keychain.new
    errors << "Settings show you are using the Agile Keychain, but no Agile Keychain exists at <code>#{keychain.agileKeychainFolder}</code>." if keychain.agile_keychain? and !File.exist?(File.expand_path(keychain.agileKeychainFolder))
    
    errors << "Your OS X user account has an invalid shortname. Short names can only contain lowercase letters and an invalid shortname can cause many different problems with 1Password. Apple has a technote about how to <a href=\"http://support.apple.com/kb/ht1428\">change the shortname</a> for your account." if %x{whoami}.chomp =~ /[^a-z]/
    errors << "Your OS X user account belongs to the 'wheel' super user group. Please refer to <a href='#{BASE_GUIDE_URL}/fix_wheel_group.html'>User Belongs to Wheel Group</a>" if user_in_wheel? 
    
    errors << "Your OS X user account is launching processes as the 'root' super user. Please refer to <a href='#{BASE_GUIDE_URL}/running_as_root.html'>Keychain Owned by Root</a> to see a description of the problem and how to fix it." if running_as_root? 
    
    errors << "The old 1Passwd Internet Plugin is installed which can prevent the 1Password icon from appearing in Safari and other WebKit browsers. Please use Finder to move this folder into the Trash: '#{File.expand_path('~/Library/Internet Plug-Ins/1PasswdInternetPlugin00.plugin')}'. Once in the trash, restart the browser and the icon should appear." if Configuration.old_internet_plugin_installed?

    errors << "The Bonjour process mDNSResponder is not running. This can cause problems with synching to 1Password for iPhone OS." unless processStarted?('mDNSResponder', 't')
    
    errors << "Safari is running in 32-bit mode which can prevent the 1Password extension from working properly. Please use Finder and open the /Applications folder, ctrl-click on Safari, and deselect the 'Open in 32 bit mode' checkbox.  Refer to <a href='http://support.agilebits.com/kb/browser-extensions/helper-error-after-upgrading-to-safari-514'>this help document</a> for additional information." if Configuration.open_safari_in_32_bit_mode?
    
    if APPSTORE
      errors << "The 1Password Helper process is not running. This can cause problems with the 1Password browser extensions. Please run the 1Password application, and choose the 1Password > Preferences menu option to review your Extensions settings." unless processStarted?('1Password Helper')
    else
      errors << "The 1PasswordAgent process is not running. This can cause problems with the 1Password browser extensions. Please run the 1Password application, and see if that corrects the issue." unless processStarted?('1PasswordAgent')
    end
    errors << "An out-of-date 1PasswordAgent process is still running. This can cause problems with the 1Password browser extensions." if processStarted?('1Password Helper') and processStarted?('1PasswordAgent')

    errors << "One or more 3rd-party SIMBL bundle is installed. These can interfere with the proper behavior of the 1Password browser extensions. Please refer to <a href='#{BASE_GUIDE_URL}/browser_software_conflicts.html'>this help document</a> for additional information." if Dir["/Library/Application Support/SIMBL/Plugins/*"].length > 0 or Dir["#{HOME}/Library/Application Support/SIMBL/Plugins/*"].length > 0

    resetFolderPermissionsSteps = "Please download and run the <a href='http://cdn.agilebits.com/ms/tools/reset_folder_permissions.zip'>standalone 1Password Troubleshooting application</a>, and choose the Reset Folder Permissions option."
    if APPSTORE
      theTouchables = [File.expand_path("#{CONTAINER}/Data/Library/Preferences/com.agilebits.onepassword-osx.plist"), File.expand_path("#{CONTAINER}-helper/Data/Library/Preferences/com.agilebits.onepassword-osx-helper.plist"), Keychain.new.agileKeychainFolder, File.join(Keychain.new.agileKeychainFolder, "data/default")]
    else
      theTouchables = [File.expand_path('~/Library/Preferences/ws.agile.1Password.plist'), File.expand_path('~/Library/LaunchAgents/ws.agile.1PasswordAgent.plist'), File.expand_path('~/Library/Preferences/ws.agile.1Password.plist'), File.expand_path('~/Library/Application Support/1Password'), Keychain.new.agileKeychainFolder, File.join(Keychain.new.agileKeychainFolder, "data/default")]
    end
    theTouchables.each do |touchable|
      r = `touch -c "#{touchable}" 2>&1`
      errors << "Permission denied for #{touchable.inspect}. #{resetFolderPermissionsSteps}" if r =~ /Permission denied/i
      errors << "File or folder missing for #{touchable.inspect}. #{resetFolderPermissionsSteps}" if r =~ /No such file or directory/
    end

    KNOWN_CONFIGURATION_ISSUES.each { |e| errors << e }
    
    return "<p>No known configuration issues detected.</p>" if errors.empty? 

    result = "<ol>"
    errors.each { |e|
      result += "<li>#{e}</li>"
    }
    result += "</ol>"

  end
  
  def Configuration.old_internet_plugin_installed?
    `ls -la "#{HOME}/Library/Internet Plug-Ins"` =~ /1PasswdInternetPlugin00/
  end
  
  def Configuration.open_safari_in_32_bit_mode?
    arch = %x{defaults read ./com.apple.LaunchServices LSArchitecturesForX86_64v2}.chomp
    arch.split(/\n    \);\n/).each {
      if arch =~ /"com\.apple\.Safari".*?i386.*?\)/m
        return true 
      else
        return false 
      end
    }
    return false
  end
  
  def Configuration.safari_code_signature_invalid?
    return false if Sys.is_tiger?

    sig = `/usr/bin/codesign -vvvvv /Applications/Safari.app 2>&1`
    
    return false if sig == "/Applications/Safari.app: a sealed resource is missing or invalid\n/Applications/Safari.app/Contents/Resources/CacheController.nib/objects.nib: resource added\n/Applications/Safari.app/Contents/Resources/ErrorConsole.nib/objects.nib: resource added\n/Applications/Safari.app/Contents/Resources/RenderTree.nib/objects.nib: resource added\n/Applications/Safari.app/Contents/Resources/Snippet.nib/objects.nib: resource added\n/Applications/Safari.app/Contents/Resources/ViewTree.nib/objects.nib: resource added\n"
    
    return sig =~ /valid on disk/ ? false : true
  end
  
  def Configuration.user_in_wheel?
    return false if Sys.is_tiger?
    return `id` =~ /\(wheel\)/
  end

  def Configuration.running_as_root?
    return false if Sys.is_tiger?
    return true if `ps -Aj | grep -v grep | grep 1Password` =~ /^root/
    return false
  end
end

class Simbl < BaseTestClass
  def version
    begin
      info = File.new("#{HOME}/Library/InputManagers/SIMBL/SIMBL.bundle/Contents/Info.plist") 
      lines = info.read
      lines =~ /<key>CFBundleShortVersionString<\/key>$\s*<string>(.*?)<\/string>/i
      return $1
    rescue 
      return "not installed"
    end
  end
  
  def tests
    Print.start_section("Input Managers")
    Print.stat('/Library/InputManagers', 't')
    
    unless !APPSTORE and Sys.is_tiger?
      print("xattr /Library/InputManagers")
      print(`xattr -l "/Library/InputManagers" 2>/dev/null`)
    end
    
    Print.dir('/Library/InputManagers', 't', 't')
    Print.dir('/Library/InputManagers/1PasswdIM', 't', 't') if !APPSTORE
    
    if !APPSTORE
      unless Sys.is_tiger?
        print("xattr /Library/InputManagers/1PasswdIM")
        print(`xattr -l /Library/InputManagers/1PasswdIM 2>/dev/null`)
      else
        Print.dir("#{HOME}/Library/InputManagers/1PasswdIM", 't', 't')
      end
    end
    
    Print.stat('/Library/Application Support')
    Print.stat('/Library/Application Support/SIMBL', 't')
    Print.dir("/Library/Application Support/SIMBL/Plugins", 't', 't')

    Print.stat("#{HOME}/Library")
    Print.stat("#{HOME}/Library/Application Support")
    Print.stat("#{HOME}/Library/InputManagers", 't')
    Print.dir("#{HOME}/Library/InputManagers", 't', 't')
    Print.dir("#{HOME}/Library/Application Support/SIMBL", 't', 't')
    Print.dir("#{HOME}/Library/Application Support/SIMBL/Plugins", 't', 't')
    Print.end_section

    Print.start_section("Other Items Of Interest")
    Print.dir("/tmp", 'd')
    # This is for iCloud, and has been known to sometimes cause hangs during file activities
    # http://www.macworld.com/article/163227/2011/10/fix_a_lion_file_opening_hang_in_mac_os_x_10_7_2.html
    # We'll log it for a while and if it never happens for 1P we can remove it
    Print.dir("#{HOME}/Library/Application Support/Ubiquity", '' 't')
    Print.end_section
  end
end

class App
  def initialize(appname)
    @appname = appname
  end
end

class Browser
  def name
    raise "Must override name method in subclass. Should return application's full name; i.e. `Safari.app`"
  end
  
  def bundle_path
    raise "Must override bundle_path in subclass. Should be the bundle path used by NSUserDefaults."
  end
  
  def version
    begin
      File.new("/Applications/#{name}/Contents/Info.plist").read =~ /<key>CFBundleShortVersionString<\/key>$\s*<string>(.*?)<\/string>/i
      return $1
    rescue 
      return "#{name} not installed in /Applications folder."
    end
  end
  
  def build_number
    begin
      File.new("/Applications/#{name}/Contents/Info.plist").read =~ /<key>CFBundleVersion<\/key>$\s*<string>(.*?)<\/string>/i
      return $1.to_f
    rescue 
      return -1.0
    end
  end
  
  def min_supported
    begin
      File.new("#{COCOA_BUNDLE_PATH}/Contents/Resources/SupportedBrowsers.plist").read =~ /<key>#{bundle_path.gsub('.','\.')}<\/key>.*?<key>MinBundleVersion<\/key>.*?<string>(.*?)<\/string>/ixm
      return $1.to_f
    rescue Exception => e
      return -1.0
    end
  end
  
  def max_supported
      begin
        File.new("#{COCOA_BUNDLE_PATH}/Contents/Resources/SupportedBrowsers.plist").read =~ /<key>#{bundle_path.gsub!('.', '\.')}<\/key>.*?<key>MaxBundleVersion<\/key>.*?<string>(.*?)<\/string>/ixm
        return $1.to_f
      rescue Exception => e
        return -1.0
      end
  end
  
  def build_number_valid?(build=-1)
    begin
      raise if min_supported == -1.0 || max_supported == -1.0
      return build != -1 && build >= min_supported && build <= max_supported
    rescue
      return false
    end
  end
  
  def toolbar_identifier
    begin
      File.new("#{COCOA_BUNDLE_PATH}/Contents/Resources/SupportedBrowsers.plist").read =~ /<key>#{bundle_path.gsub('.','\.')}<\/key>.*?<key>ToolbarIdentifier<\/key>.*?<string>(.*?)<\/string>/ixm
      return $1
    rescue
      return "Could not locate toolbar identifier for #{bundle_path}"
    end
  end
  
  def toolbar_button_index
    r = `defaults read #{bundle_path} ws.agile.1Password.toolbarButtonIndex 2>&1`
    return r unless r =~ /does not exist/
    return "No toolbar index for #{name}."
  end
  
  def extension_loaded
    r = `defaults read #{bundle_path} ws.agile.1PasswordExtensionLoaded 2>&1`
    return r unless r =~ /does not exist/
    return "Extension for #{name} has not been loaded"
  end
  
  def hide_toolbar_button
    r = `defaults read #{bundle_path} ws.agile.1Password.hideToolbarButton 2>&1`
    return r unless r =~ /does not exist/
    return "ws.agile.1Password.hideToolbarButton not set for #{name}"
  end
  
  def toolbar_configuration
    r = `defaults read #{bundle_path} "NSToolbar Configuration #{toolbar_identifier}" 2>&1` unless toolbar_identifier =~ /Could not locate toolbar identifier/
    return r unless r =~ /does not exist/
    return "Could not load toolbar configuration for #{name}."
  end
  
end    

class Safari < Browser
  
  def name
    'Safari.app'
  end
  
  def bundle_path
    'com.apple.Safari'
  end
  
  def toolbar_installed?
    tb = `defaults read #{bundle_path} "NSToolbar Configuration SafariToolbarIdentifier" 2>&1`
    return tb =~ /OPToolbar1Passwd/
  end

  def javascript_enabled
    js = `defaults read #{bundle_path} "WebKitJavaScriptEnabled" 2>&1`
    return "1" if js == "1\n" or js =~ / does not exist/
    return "0" if js == "0\n"
    return js
  end

  def privatebrowsing_enabled
    r = `defaults read #{bundle_path} "WebKitPrivateBrowsingEnabled" 2>&1`
    return "1" if r == "1\n"
    return "0" if r == "0\n" or r =~ / does not exist/
    return r
  end

  def button_installed_automatically?
    r = `defaults read #{bundle_path} OPToolbar1PasswdDisplayed 2>&1`
    return false if r == ""
    return r == "1\n"
  end
  
  def use_single_process_windows?
    r = `defaults read #{bundle_path} "DebugNewWindowsUseSingleProcessWebKit" 2>&1`
    return "1" if r == "1\n"
    return "0" if r == "0\n" or r =~ / does not exist/
    return r
  end
  
  def autofill
    `defaults read #{bundle_path} | grep AutoFill`
  end
  
  def tests
    Print.start_section("Safari")
    print "Version: #{version}"
    if !APPSTORE
      if build_number_valid?(build_number)
        print "Build #{build_number} is in supported range of #{min_supported} to #{max_supported}."
      else
        print "Build #{build_number} is not supported by built-in extensions. Build number must be between #{min_supported} and #{max_supported}."
      end
      print "Toolbar button index: #{toolbar_button_index}"
      print "Built-in extension loaded: #{extension_loaded}" if !APPSTORE
    end
    print "Toolbar Configuration:"
    Print.start_section
    print "#{toolbar_configuration}"
    Print.end_section
    print "Safari AutoFill settings:"
    Print.start_section
    print "#{autofill}"
    Print.end_section
    print "Install Extension Updates Automatically: #{nsUserDefault(bundle_path, 'InstallExtensionUpdatesAutomatically')}"
    js = javascript_enabled
    print "JavaScript enabled: #{js}"
    if js == "0"
      Print.start_section
      print "<p>JavaScript-based extension will <b>NOT</b> function without JavaScript enabled</p>"
      KNOWN_CONFIGURATION_ISSUES << "1Password's Safari extension will not function without JavaScript enabled. In Safari, choose the Safari > Preferences > Security pane, and check the 'Enable JavaScript' box."
      Print.end_section
    end
    r = privatebrowsing_enabled
    print "Private Browsing enabled: #{r}"
    if r == "1"
      Print.start_section
      print "<p>Having private browsing enabled <b>may</b> interfere with extension behavior</p>"
      KNOWN_CONFIGURATION_ISSUES << "Having private browsing enabled may interfere with Safari extension behavior."
      Print.end_section
    end
    
    use_single_process_windows = use_single_process_windows?
    print "DebugNewWindowsUseSingleProcessWebKit: #{use_single_process_windows}"
    if use_single_process_windows == "1"
      Print.start_section
      msg = "1Password's Safari extension will not function when using Single Process WebKit windows (<a href='http://stormchild.tumblr.com/post/10414883514/how-to-stop-safari-5-from-unexpectedly-reloading-pages'>see this post for details</a>). To fix, click the Debug menu in Safari and enable <i>Use Multi-process Windows</i>."
      print msg
      KNOWN_CONFIGURATION_ISSUES << msg
      Print.end_section
    end
    
    if File.exists?("/usr/bin/codesign")
      sig = `/usr/bin/codesign -vvvvv /Applications/Safari.app 2>&1`
      print "Safari Signature: #{sig}"
    end
    Print.end_section
  end
  
end

class JSE
  def overall
    Print.start_section("Overall Browser Settings")
    if File.exists?("/var/db/.AccessibilityAPIEnabled")
      print "Access for assistive devices is enabled"
    else
      print "Access for assistive devices is disabled"
    end
    Print.end_section
  end

  def safari
    Print.start_section("New Safari Extension")
    print "Installed Safari Extensions"
    Print.start_section
    Print.dir("#{HOME}/Library/Safari/Extensions")
    f = "#{HOME}/Library/Safari/Extensions/Extensions.plist"
    if File.exists?(f)
      unless %x{ls -lo "#{f}" 2>/dev/null}.chomp =~ /^-rw(.*) #{%x{whoami}.chomp} /
        m = "Your Safari extensions configuration has an improper file ownership and/or permissions for #{f}."
        print "<p>#{m}</p>"
        KNOWN_CONFIGURATION_ISSUES << "#{m} You may have difficulties installing, updating or uninstalling Safari extensions until this is corrected. It always needs to be owned by your own OS X account, and allow both read and write access."
      end
    end
    Print.end_section

    print "1Password Extension Definition"
    Print.start_section
    inst_ext = `defaults read #{HOME}/Library/Safari/Extensions/Extensions 'Installed Extensions' 2>&1 | grep -B 3 -A 3 1Password`
    print "\n#{inst_ext}\n"
    inst_ext.split("\n").each { |line| 
        if line =~ /"Archive File Name" = "(.*)";/
          name = $1
          print "Extension Archive: #{name}"
          r = `xar -x -C /tmp -f '#{HOME}/Library/Safari/Extensions/#{name}' 1Password.safariextension/Info.plist` rescue nil
          print "\nVersion: #{`defaults read /tmp/1Password.safariextension/Info CFBundleShortVersionString`}\n"
          r = `rm -R /tmp/1Password.safariextension` rescue nil
        end
      }
    Print.end_section

    print "Safari Databases"
    Print.start_section
    Print.dir("#{HOME}/Library/Safari/Databases", 't')
    Print.end_section
    
    print "1Password Safari database folder"
    Print.start_section
    Print.dir("#{HOME}/Library/Safari/Databases/safari-extension_com.agilebits.onepassword-safari-2bua8c4s2c_0", 't')
    Print.end_section
    
    print "Databases.db entry for OnePassword"
    Print.start_section
    database_info = `sqlite3 -line ~/Library/Safari/Databases/Databases.db "select * from Databases where name = 'OnePassword';" 2>/dev/null` rescue nil
    print "\n#{database_info}"
    path = nil
    path = $1 if database_info =~ /path = (.*)\s/
    
    if path
      show_database_info("~/Library/Safari/Databases/safari-extension_com.agilebits.onepassword-safari-2bua8c4s2c_0/#{path}")
    end
    Print.end_section
    Print.end_section
  end
  
  def chrome
    Print.start_section("Chrome Extension")
    print "Installed Chrome Extensions"
    Print.start_section
    Print.dir("#{HOME}/Library/Application\ Support/Google/Chrome/Default/Extensions/")
    manifests = Dir["#{HOME}/Library/Application\ Support/Google/Chrome/Default/Extensions/**/manifest.json"]
    manifests.each { |manifest| 
      lines = File.readlines(manifest) 
      path, name, version = nil, nil, "(undetermined)"
      path = $1 if manifest =~ /\/Extensions\/(.*)\/manifest.json/
      lines.each { |line| 
        if line =~ /"name":\s*"(.*)"\s*,/
          name = $1
        end
        if line =~ /"version":\s*"(.*)"\s*/
          version = $1
        end
      }
      print "#{path} --> #{name} #{version}" if name
    }
    Print.end_section

    print "\nChrome Databases"
    Print.start_section
    Print.dir("#{HOME}/Library/Application\ Support/Google/Chrome/Default/databases", 't')
    Print.end_section
    
    print "Databases.db entry for OnePassword"
    Print.start_section
    database_info = `sqlite3 -line ~/Library/Application\\ Support/Google/Chrome/Default/databases/Databases.db "select * from Databases where name = 'OnePassword';" 2>/dev/null` rescue nil
    print "\n#{database_info}"
    Print.end_section
    
    op_folder, op_file = nil, nil
    if database_info =~ /origin = (.*)\s/
      op_folder = $1 
      Print.dir("#{HOME}/Library/Application\ Support/Google/Chrome/Default/databases/#{op_folder}", 't')
    end
    
    op_file = $1 if database_info =~ /id = (.*)\s/
    if op_folder && op_file 
      show_database_info("#{HOME}/Library/Application Support/Google/Chrome/Default/databases/#{op_folder}/#{op_file}")
    end
    
    Print.end_section
  end
  
  def firefox
    Print.start_section("New Firefox Extension")
    print "\nProfiles and OnePassword Database"
    Print.start_section
    Print.dir("#{HOME}/Library/Application\ Support/Firefox/Profiles", 't')
    Print.end_section
    
    show_database_info("~/Library/Application Support/Firefox/Profiles/OnePassword.sqlite")
    Print.end_section
  end
  
  def show_database_info(path)
    return unless path
    path.gsub!(/ /, "\\ ")
    
    Print.start_section("OnePassword Database Information")
    item_count = `sqlite3 #{path} "select count(*) from items;"  2>/dev/null` rescue nil
    print "# of database items = #{item_count}"

    keys_count = `sqlite3 #{path} "select count(*) from keys;"  2>/dev/null` rescue nil
    print "# of keys = #{keys_count}"

    config_info = `sqlite3 #{path} "select * from config;"  2>/dev/null` rescue nil
    Print.start_section("Config information")
    print "#{config_info}"
    Print.end_section

    schema = `sqlite3 #{path} ".schema"  2>/dev/null` rescue nil
    Print.start_section("Schema")
    print "#{schema}"
    Print.end_section
    Print.end_section
  end
  
  def tests
    overall
    safari
    chrome
    firefox
  end
end

class OmniWeb < Browser
  
  def name
    'OmniWeb.app'
  end
  
  def bundle_path
    'com.omnigroup.OmniWeb5'
  end
  
  def button_in_toolbar?
    tb = `defaults read #{bundle_path} "NSToolbar Configuration BrowserWindow" 2>&1`
    return tb =~ /OPToolbar1Passwd/
  end

  def button_installed_automatically?
    r = `defaults read #{bundle_path} 2>&1 | grep OPToolbar1PasswdDisplayed | awk "{print $3}"`
    return false if r == ""
    return r == "1\n"
  end
  
  def autofill
    `defaults read #{bundle_path} 2>&1 | grep AutoFill`
  end
  
  def tests
    Print.start_section("OmniWeb")
    print "Version: #{version}"
    print "Toolbar button index: #{toolbar_button_index}"
    print "Extension loaded: #{extension_loaded}"
    print "Toolbar Configuration:"
    Print.start_section
    print "#{toolbar_configuration}"
    Print.end_section
    print "OmniWeb AutoFill settings:"
    Print.start_section
    print "#{autofill}"
    Print.end_section
    Print.end_section
  end
end

class NetNewsWire < Browser
  def name
    'NetNewsWire.app'
  end
  
  def bundle_path
    'com.ranchero.NetNewsWire'
  end
  
  def tests
    Print.start_section("NetNewsWire")
    print "Version: #{version}"
    Print.end_section
  end
end

class Fluid < Browser
  def name
    'Fluid.app'
  end
  
  def bundle_path
    'com.fluidapp.Fluid'
  end
  
  def tests
    Print.start_section("Fluid")
    print "Version: #{version}"
    
# Because we're using wildcards, we cannot enclose the wildcard portion inside the double quotes
    instances = `ls "#{HOME}/Library/Preferences/"com.fluidapp.Fluid*.*.plist 2>/dev/null`.split("\n")
    if instances.length < 1
      print "No Fluid instances created."
	  Print.end_section
      return
    end
    
    instances.each { |instance| 
      name = $1 if instance =~ /com\.fluidapp\.FluidInstance\.(.*)?\.plist/
      name = $1 if instance =~ /com\.fluidapp\.FluidApp\.(.*)?\.plist/
#      path = "com.fluidapp.FluidInstance.#{name}"
      print "Fluid instance #{name}"
    }
    Print.end_section
  end
end

class Firefox < Browser
  attr_accessor :profiles, :profile_ini_contents
  def name
    "Firefox.app"
  end
  
  def bundle_path
    "org.mozilla.Firefox"
  end
  
  def javascript_enabled(path)
    return "0" if path == ""
    return "" if !File.exists?(path)
    js = `grep javascript\.enabled "#{path}/prefs.js" 2>&1`
    return "1" if js !=~ /false/
    return "0"
  end

  def load_profiles
    begin
      info = File.open("#{HOME}/Library/Application Support/Firefox/Profiles.ini") 
      lines = info.read
      @profile_ini_contents = lines
      
      result = ""
      @profiles = []
      current = {}
      lines.split("\n").each { |l|
        l = l.strip
        result = "#{result}\n    #{l.inspect}"
        
        if l =~ /^\[(.*)\]$/
          @profiles << current unless current.empty?
          current = {}
        end
        
        unless current.nil?
          current[:name] = $1 if l =~ /^Name=(.*)/
          current[:path] = $1 if l =~ /^Path=(.*)/
          current[:relative] = $1 == "1" if l =~ /^IsRelative=(.*)/
          current[:default] = $1 if l =~ /^Default=(.*)/
        end
      }
      @profiles << current unless current.nil?
    rescue
      print("Error occured while reading the Profiles.ini file: #{$!}")
      KNOWN_CONFIGURATION_ISSUES << "Problem reading Firefox Profiles.ini file: #{$!}"
    end
    
  end

  def tests
    Print.start_section("Firefox")
    print "Version: #{version}"
    unless File.exists?("#{HOME}/Library/Application Support/Firefox")
      print "#{HOME}/Application Support/Firefox does not exist"
      Print.end_section
      return
    end

    load_profiles
    print "Profiles.ini contents:<br/><blockquote>#{@profile_ini_contents}</blockquote>"
    
    print "Application Support Files"
    Print.start_section
    Print.dir("#{HOME}/Library/Application Support/Firefox")
    Print.end_section
    
    print "Profile Files"
    Print.start_section
    Print.dir("#{HOME}/Library/Application Support/Firefox/Profiles", 't')
    Print.end_section

    if @profiles.nil?
      print "No Profiles found"
    else
      @profiles.each { |profile| 
        print "Profile '#{profile[:name]}' details"
        Print.start_section
        path = profile[:relative] ? "#{HOME}/Library/Application Support/Firefox/#{profile[:path]}" : profile[:path]
        if path.nil?
          print "An unspecified path was encountered for this profile -- contents cannot be included in the listing"
          KNOWN_CONFIGURATION_ISSUES << "Your Firefox profile '#{profile[:name]}' has an unspecified path. This may result in the browser from being able to connect to the 1PasswordAgent."
          Print.end_section
          print ""
          next
        end
        path = File.expand_path(path)
        if !profile[:relative]
          print "Externally located profile detected: #{path}"
          Print.stat("#{path}/..")
          if profile[:default] == "1"
            KNOWN_CONFIGURATION_ISSUES << "Your Firefox profile '#{profile[:name]}' is located in a non-default location. This will cause the 1Password Firefox extension to report an error and it will not function. To correct this, move your profile back to the default location."
          else
            KNOWN_CONFIGURATION_ISSUES << "Your Firefox profile '#{profile[:name]}' is located in a non-default location. This will cause the 1Password Firefox extension to report an error and it will not function if you use this profile. To correct this, move this profile back to the default location."
          end
        end
        if javascript_enabled("#{path}") == 0
          Print.start_section
          print "<p>JavaScript-based extension will <b>NOT</b> function without JavaScript enabled</p>"
          KNOWN_CONFIGURATION_ISSUES << "1Password's Firefox extension will not function without JavaScript enabled in the '#{profile[:name]}' profile. In Firefox, choose the Firefox > Preferences > Content pane, and check the 'Enable JavaScript' box."
          Print.end_section
        end

        print "1Password Extension File Contents\n"
        if File.directory?("#{path}/extensions/firefox3@1password.com")
          print "firefox3@1password.com is a folder ???"
          KNOWN_CONFIGURATION_ISSUES << "#{path}/extensions/firefox3@1password.com is a folder."
        else
          Print.file_contents("#{path}/extensions/firefox3@1password.com", 't')
        end
        if File.directory?("#{path}/extensions/firefox4@1password.com")
          print "firefox4@1password.com is a folder ???"
          KNOWN_CONFIGURATION_ISSUES << "#{path}/extensions/firefox4@1password.com is a folder."
        else
          Print.file_contents("#{path}/extensions/firefox4@1password.com", 't')
        end
        if File.directory?("#{path}/extensions/firefox5@1password.com")
          print "firefox5@1password.com is a folder ???"
          KNOWN_CONFIGURATION_ISSUES << "#{path}/extensions/firefox5@1password.com is a folder."
        else
          Print.file_contents("#{path}/extensions/firefox5@1password.com", 't')
        end
        Print.start_section
        print ""
        if File.directory?("#{path}/extensions/onepassword@agilebits.com")
          t = `grep "em:version" "#{path}/extensions/onepassword@agilebits.com/install.rdf"`
          if t =~ /<em:version>(.*)<\/em:version>/
            print "1Password extension version: #{$1}"
          else
            print "Unable to determine the 1Password extension version! (#{t})"
          end
        end
        Print.end_section
        print ""

        print "Installed Extensions"
        Print.dir("#{path}/extensions", 't')
        print ""
        print "Enabled Extensions"
        ee = `cp "#{path}/extensions.sqlite" /tmp; sqlite3 "/tmp/extensions.sqlite" "select id from addon where userDisabled=0 and appDisabled=0;"; rm /tmp/extensions.sqlite`
        print "<br />#{ee}<br />"
        KNOWN_CONFIGURATION_ISSUES << "FoxyProxy add-on has been known to interfere with 1Password's Firefox extension. Please disable FoxyProxy if you see an \"Agent not connected.\" error." if Dir["#{path}/extensions/foxyproxy*@*"].length > 0
        KNOWN_CONFIGURATION_ISSUES << "NoScript add-on has been known to interfere with 1Password's Firefox extension. Please disable NoScript if you cannot successfully use Go & Fill from the 1Password application." if ee =~ /\{73a6fe31-595d-460b-a920-fcc0f8843232\}/

        print "Root Profile Folder Contents"
        Print.dir(path, 't')
        Print.end_section
      }
    end

    print "Crash Logs"
    Print.start_section
    Print.dir("#{HOME}/Library/Application Support/Firefox/Crash Reports", 'tr')
    print "<b>Crash Logs from #{HOME}/Library/Application Support/Firefox/Crash Reports/pending</b>\n"
    Dir.glob("#{HOME}/Library/Application Support/Firefox/Crash Reports/pending/*").reject { |item| item =~ /\.dmp$/ }.each { |file| Print.file_contents(file) }

    print "<b>Crash Logs from #{HOME}/Library/Application Support/Firefox/Crash Reports/submitted</b>\n"
    Dir.glob("#{HOME}/Library/Application Support/Firefox/Crash Reports/submitted/*").reject { |item| item =~ /\.dmp$/ }.each { |file| Print.file_contents(file) }

    Print.end_section
    Print.end_section
  end
end

class Chrome < Browser
  def name
    "Google Chrome.app"
  end

  def bundle_path
    "org.mozilla.Chrome"
  end

  def javascript_enabled
    t = "#{HOME}/Library/Application Support/Google/Chrome/Default/Preferences"
    return "" if !File.exists?(t)
    js = `grep -A 30 \"default_content_settings\" "#{t}" 2>&1 | grep \"javascript\"`
    return "0" if js =~ / 2/
    return "1"
  end

  def tests
    Print.start_section("Google Chrome")
    major_version = $1 if version =~ /^(\d*).*/
    print "Version: #{version}"
    js = javascript_enabled
    print "JavaScript enabled: #{js}"
    if major_version.to_i >= 15 && js == "0"
      Print.start_section
      print "<p>JavaScript-based extension will <b>NOT</b> respond to keyboard shortcuts without JavaScript enabled</p>"
      KNOWN_CONFIGURATION_ISSUES << "1Password's Google Chrome extension will not be able to respond to keyboard shortcuts without JavaScript enabled. In Google Chrome, choose the Chrome > Preferences menu option, click on 'Under the Hood' in the sidebar and then click on the 'Content Settings' button. There, check the 'Allow all sites to run JavaScript (recommended)' radio button."
      Print.end_section
    end
    Print.end_section
  end
end

class Chromium < Browser
  def name
    "Chromium.app"
  end

  def bundle_path
    "org.chromium.Chromium"
  end

  def javascript_enabled
    t = "#{HOME}/Library/Application Support/Chromium/Default/Preferences"
    return "" if !File.exists?(t)
    js = `grep -A 30 \"default_content_settings\" "#{t}" 2>&1 | grep \"javascript\"`
    return "0" if js =~ / 2/
    return "1"
  end

  def tests
    Print.start_section("Chromium")
    print "Version: #{version}"
    js = javascript_enabled
    print "JavaScript enabled: #{js}"
    # For now we can ignore this, as Chromium's JavaScript being disabled seems to not impact extensions -- just web pages
    if js == "0" && js == ""
      Print.start_section
      print "<p>JavaScript-based extension will <b>NOT</b> function without JavaScript enabled</p>"
      KNOWN_CONFIGURATION_ISSUES << "1Password's Google Chrome extension will not function without JavaScript enabled. In Chromium, choose the Chromium > Preferences menu option, click on 'Under the Hood' in the sidebar and then click on the 'Content Settings' button. There, check the 'Allow all sites to run JavaScript (recommended)' radio button."
      Print.end_section
    end
    Print.end_section
  end
end

class OnePasswd < BaseTestClass
  def version
    begin
      info = File.new("#{COCOA_BUNDLE_PATH}/Contents/Info.plist")
      lines = info.read
      build = $1 if lines =~ /<key>CFBundleVersion<\/key>$\s*<string>(.*?)<\/string>/i
      version = $1 if lines =~ /<key>CFBundleShortVersionString<\/key>$\s*<string>(.*?)<\/string>/i
      
      return "#{version} (build ##{build})"
    rescue
      return "not installed in #{COCOA_BUNDLE_PATH}"
    end
  end

  def tests()
    Print.start_section("1Password Settings")
    print "Version: #{version}"
    print "Actual Bundle path from Cocoa: #{COCOA_BUNDLE_PATH}"
    if APPSTORE
      print "\n"
      print "Conceal passwords: #{ns1PUserDefault('HidePasswords') == '1' ? 'Yes' : 'No'}"
      print "Show item counts: #{ns1PUserDefault('showItemCounts') == '1' ? 'Yes' : 'No'}"
      case ns1PUserDefault('mainWindowLayout')
        when "0" then print("Layout: Shelves")
        when "1" then print("Layout: Widescreen")
        when "2" then print("Layout: Traditional")
        else print("Layout: Unknown")
      end
      print "System idle time issue: #{ns1PUserDefault('SystemIdleTimeIsBroken', false, '-helper') == '1' ? 'Yes' : 'No'}"
      print "Show Wi-Fi sync authentication: #{ns1PUserDefault('ShowWiFiSyncAuthAutomatically') == '0' ? 'No' : 'Yes'}"
      print "Wi-Fi sync handshake timeout: #{ns1PUserDefault('WiFiSync.HandshakeTimeout').empty? ? '5' : ns1PUserDefault('WiFiSync.HandshakeTimeout')}"
      print "Declined to use Dropbox: #{ns1PUserDefault('DeclinedToUseDropboxPath') == '1' ? 'Yes' : 'No'}"
    else
      print "1Password folder (as per User Defaults): #{ns1PUserDefault('AppPath')}"
      print "Build # (as per User Defaults): #{ns1PUserDefault('Build')}"
    end
    Print.start_section("Architectures")
    print %x{file "#{COCOA_BUNDLE_PATH}/Contents/MacOS/1Password"}
    Print.end_section
    Print.start_section("Preferences")
    if APPSTORE
      print "Sidebar: Browser Extensions: #{ns1PUserDefault('DisplaySidebarBrowserExtensions') == '' ? 'Yes' : 'No'}"
      print "Sidebar: Folders: #{ns1PUserDefault('DisplaySidebarFolders') == '' ? 'Yes' : 'No'}"
      print "Sidebar: Tags: #{ns1PUserDefault('DisplaySidebarTags') == '' ? 'Yes' : 'No'}"
      print "Sidebar: Generated Passwords: #{ns1PUserDefault('DisplaySidebarPasswords') == '' ? 'No' : 'Yes'}"
      print "\n"
      print "Lock after idle: #{ns1PUserDefault('LockOnIdle', false, '-helper') == '1' ? 'Yes' : 'No'}"
      print "Lock inactivity time: #{ns1PUserDefault('LockTimeout', false, '-helper').empty? ? '20' : ns1PUserDefault('LockTimout', false, '-helper')}"
      print "Lock when sleeping: #{ns1PUserDefault('LockOnSleep', false, '-helper') == '1' ? 'Yes' : 'No'}"
      print "Lock when screen saver: #{ns1PUserDefault('LockOnScreenSaver', false, '-helper') == '1' ? 'Yes' : 'No'}"
      print "Enable Universal Unlock: #{ns1PUserDefault('EnableUniversalUnlock') == '1' ? 'Yes' : 'No'}"
      print "Clear clipboard: #{ns1PUserDefault('ClearPasteboardAfterTimeout', false, '-helper') == '1' ? 'Yes' : 'No'}"
      print "Clear clipboard time: #{ns1PUserDefault('PasteboardClearTimeout', false, '-helper').empty? ? '90' : ns1PUserDefault('PasteboardClearTimeout', false, '-helper')}"
      print "\n"
      t = ns1PUserDefault('KeepHelperRunning')
      print "Keep helper running: #{t == '1' ? 'Yes' : '<b>No</b><br />'}"
      KNOWN_CONFIGURATION_ISSUES << "The 1Password Helper background process is not set for automatic starting. This will prevent access to keyboard shortcuts, Universal Unlock and data syncing of items between 1Password and the browsers. Please choose the 1Password > Preferences > Extensions pane and <b>enable</b> the checkbox for 'Keep helper running ...'." if t != '1'
      print "Show helper icon in menu bar: #{ns1PUserDefault('ShowStatusItem') == '1' ? 'Yes' : 'No'}"
      keychain_data = File.expand_path(Keychain.new.agileKeychainFolder) << "/config/use-thumbnails"
      print "\n"
      print "Download icons and previews: #{File.exists?(keychain_data) ? 'Yes' : 'No'}"
      print "\n"
      print "Disable [Un]Lock Animation: #{ns1PUserDefault('DisableCoreAnimation') == '1' ? 'Yes' : 'No'}"
    else
      Print.start_section("Extensions")
# This next item needs to "wildcard" the settings, rather than use ns1PUserDefault multiple times to list them all out individually
      print "<div style='padding-left:25px;'>#{`defaults read ws.Agile.1Password | grep 'Extension Enabled'`}</div>"
      Print.end_section
      print "Autosave: #{ns1PUserDefault('autosave') == '0' ? 'No' : 'Yes'}"
      print "Autosubmit: #{ns1PUserDefault('autosubmit') == '0' ? 'No' : 'Yes'}"
      print "DisableAutoSavePopup: #{ns1PUserDefault('disableAutoSavePopup') == '1' ? 'Yes' : 'No'}"
      print "UpdateLastChecked: #{ns1PUserDefault('AGUpdateLastChecked')}"
      print "Thumbnail logging: #{ns1PUserDefault('VerboseThumbnailLogging') == '1' ? 'Yes' : 'No'}"
      Print.end_section
    end
    Print.end_section
    
    if !APPSTORE
      Print.start_section("1Password Folder Contents")
      Print.dir("#{HOME}/Library/Application Support/1Password")
      Print.end_section
      Print.start_section("1Password Versioned Extensions")
      Print.dir("#{HOME}/Library/Application Support/1Password/Extensions")
      Print.end_section
    end
    
    Print.start_section("Installed Scripting Additions")
    print "~/Library/ScriptingAdditions"
    
    Print.dir(File.expand_path('~/Library/ScriptingAdditions'))
    
    print %x{file "#{HOME}/Library/ScriptingAdditions/1Password Addition.osax/Contents/MacOS/1Password Addition"} if File.exist? "#{HOME}/Library/ScriptingAdditions/1Password Addition.osax/Contents/MacOS/1Password Addition"
    print %x{file "#{HOME}/Library/ScriptingAdditions/1Password Addition.osax/Contents/MacOS/1Password Addition"} if File.exist? "#{HOME}/Library/ScriptingAdditions/1Password Addition Leopard.osax/Contents/MacOS/1Password Addition Leopard"
    
    print "/Library/ScriptingAdditions"
    Print.dir('/Library/ScriptingAdditions')
    Print.end_section
    
    Print.start_section("Installed Internet Plugins")
    Print.dir('/Library/Internet Plug-Ins', 't')
    Print.dir("#{HOME}/Library/Internet Plug-Ins", 't')
    Print.end_section

if !APPSTORE
    license_seen = false
    Print.start_section("License information from User Defaults")
    t = ns1PUserDefault("License", true) # `defaults read ws.agile.1Password License`
    license_seen = true if !t.nil?
    print "#{t}"
    Print.end_section

    Print.start_section("License Files")
    Print.dir("#{HOME}/Library/Application Support/1Passwd/License", 't', 't')
    Print.dir("#{HOME}/Library/Application Support/1Password/License", 't', 't')
    Print.end_section
    
    Print.start_section("License File Contents")
    dir = File.expand_path("~/Library/Application Support/1Password/License")
    files = Dir["#{dir}/*.plist"] + Dir["#{dir}/*.awslicense"] + Dir["#{File.expand_path("~/Library/Application Support/1Passwd/License")}/*.plist"] + Dir["#{File.expand_path("~/Library/Application Support/1Passwd/License")}/*.awslicense"]
    files.each { |plist|
      print "#{plist} contents:"
      print "<pre>"
      print File.new(plist).readlines.to_s.htmlize
      print "</pre>"
      license_seen = true if %x{grep "<string>3</string>" "#{plist}"}.chomp != ""
    }
    Print.end_section
    KNOWN_CONFIGURATION_ISSUES << "No 1Password 3 for Mac license was found. You cannot add more then 20 items in trial mode. Please visit the <a href='https://agilebits.com/store'>Agile Online Store</a> to purchase a software license, or run the 1Password application and choose the 1Password > License... menu option to make your license known to 1Password." if !license_seen
    verify_installation
end
  end
  
  def verify_installation(manifest_file_path = "#{COCOA_BUNDLE_PATH}/.manifest")
    Print.start_section("1Password installation validation")
    is_valid = true
    
    manifest = File.readlines(manifest_file_path) rescue []
    if manifest.length == 0
      print "Manifest file (#{manifest_file_path}) not found. Skipping validation."
      Print.end_section
      return
    end
    expected_files = []
    
    manifest.each do |line|
      fields = line.strip.split(',')
      if fields.length > 2
      file, expected_size = fields.slice(0...-1).join(',').strip, fields[-1].strip
      else
      file, expected_size = fields[0].strip, fields[1].strip
      end
      if file.to_s.length == 0 || expected_size.to_s.length == 0
        print "Ignoring unexpected line in manifest: #{line.inspect}"
        next
      end

      expected_files << file
      unless File.exists?(File.join(COCOA_BUNDLE_PATH, file))
        print "#{file.inspect} is missing"
        is_valid = false
      end
    
      actual_size = File.size(File.join(COCOA_BUNDLE_PATH, file)).to_s rescue ""
      unless expected_size == actual_size
        print "#{file.inspect} is #{actual_size.inspect} but should be #{expected_size.inspect}"
        is_valid = false
      end
    end
    
    Dir.chdir(COCOA_BUNDLE_PATH) do
      (Dir["**/*"]-expected_files).each do |f| 
        next unless File.file?(f)
        next if f =~ /Growl.framework/
        print "Found unexpected file: #{f.inspect}"
        is_valid = false
      end
    end

    print is_valid ? "Manifest validation passed: all files exist, their sizes match, and no extra files were found." : "Validation was not 100%. Inconsistencies listed above."

    Print.end_section
  end
end

class Keychain < BaseTestClass
  
  def agile_keychain?
    return keychain_type =~ /AGHtmlDatabase/i
  end
  
  def keychain_type
    if !APPSTORE
      result = ns1PUserDefault('databaseClass')
      result = ns1PUserDefault('defaultDatabaseClass') if result == ""
      result = nil if result.length == 0
    end
    result ||= "AGHtmlDatabase"
    return result
  end
  
  def legacyOSXKeychainBackupFolder
    return "#{HOME}/Library/Keychains/backup"
  end

  def agileBackupFolder
    if APPSTORE
      path = "#{CONTAINER}-helper/Data/Library/Application Support/1Password/Backups"
    else
      path = ns1PUserDefault('keychainBackupFolderPath')
      return "#{HOME}/Library/Application Support/1Password/Backups" if path.length == 0
    end
    return path
  end
  
  def agileKeychainFolder
    if APPSTORE
      folder = ns1PUserDefault('DatabasePath', false, '-helper')
      folder = "#{CONTAINER}-helper/Data/Documents/1Password.agilekeychain" if folder.nil? or folder.length == 0
    else
      folder = ns1PUserDefault('AgileKeychainLocation')
      folder = "#{HOME}/Library/Application Support/1Password/1Password.agilekeychain" if folder.length == 0
    end
    return File.expand_path(folder) 
  end

  def tests()
    if APPSTORE
      Print.start_section("Data File Information")
      print("Location: #{agileKeychainFolder}")
      print "\n"
    else
      Print.start_section("Keychain Information")
      print("Keychain type: #{keychain_type}")
      print("Raw defaults value for databaseClass: #{ns1PUserDefault("databaseClass").inspect}")
      print("Raw defaults value for defaultDatabaseClass: #{ns1PUserDefault("defaultDatabaseClass").inspect}")
      print("AgileKeychainLocation: #{agileKeychainFolder}")
      print("LockAgileKeychainOnIdle: #{ns1PUserDefault('LockAgileKeychainOnIdle') == '0' ? 'No' : 'Yes'}")
      print("AgileKeychainLockTimeout: #{ns1PUserDefault('AgileKeychainLockTimeout').empty? ? '20' : ns1PUserDefault("AgileKeychainLockTimeout")}")
      print("LockAgileKeychainOnSleep: #{ns1PUserDefault('LockAgileKeychainOnSleep') == '0' ? 'No' : 'Yes'}")
      print("LockAgileKeychainOnScreenSaver: #{ns1PUserDefault('LockAgileKeychainOnScreenSaver' == '0' ? 'No' : 'Yes')}")
### TODO: doesn't exist?
      print("LockWhenAllAppsClosed: #{ns1PUserDefault('LockWhenAllAppsClosed')}")
      print("AlwaysStart1PasswordLocked: #{ns1PUserDefault('AlwaysStart1PasswordLocked') == '0' ? 'No' : 'Yes'}")
      print("UnlockAllAppsTogether: #{ns1PUserDefault('UnlockAllAppsTogether') == '0' ? 'No' : 'Yes'}")
    end
    
    KNOWN_CONFIGURATION_ISSUES << "The Dropbox client is <b>not</b> running in this account, yet the 1Password data file seems to be located in a Dropbox folder. This will result in no syncing of the data into or out of this computer." if DROPBOX_PATH.length > 0 and agileKeychainFolder =~ /^#{DROPBOX_PATH}/ and !processStarted?('Dropbox')

    keysfile = "#{agileKeychainFolder}/data/default/1password.keys"
    if File.exists?(keysfile)
      keys = File.open(keysfile).read
      print("Encryption key IDs: #{$1.gsub(/\s+/, '').gsub(/</, '&lt;').gsub(/>/, '&gt;')}") if keys =~ /<dict>(.*?)<key>list/m
    else
      print("File [#{keysfile}] does not exist.")
    end 

    if !APPSTORE
      Print.stat("#{HOME}/Library/Keychains")
      Print.dir("#{HOME}/Library/Keychains", 't')
    end

    Print.stat(agileKeychainFolder)
    Print.dir(agileKeychainFolder, 't')
    Print.dir("#{agileKeychainFolder}/a/default/thumb")
    Print.dir("#{agileKeychainFolder}/data", 't')
    Print.dir("#{agileKeychainFolder}/data/default", 't')

    Print.start_section("Data File Backups")
    
    if APPSTORE
      if ns1PUserDefault('BackupEnabled', false, '-helper') == '0'
        print "<p>1Password data file backups are <b>not</b> enabled.</p>"
        KNOWN_CONFIGURATION_ISSUES << "1Password data file backups are not enabled. This could lead to data loss if anything happens to the data file itself and no other backups (such as Time Machine) are available."
      end
      t = ns1PUserDefault('BackupFrequency', false, '-helper')
    else
      t = ns1PUserDefault('keychainBackupFrequency')
    end
    case t
      when "0" then print("Backup Frequency: Daily")
      when "1" then print("Backup Frequency: Weekly")
      when "2" then print("Backup Frequency: Monthly")
      else print("Backup Frequency: Daily")
    end
    
    if APPSTORE
      print("Number of Backups to Keep: #{ns1PUserDefault('BackupFilesToKeep', false, '-helper').nil? ? 50 : ns1PUserDefault('BackupFilesToKeep', false, '-helper')}")
    else
      print("Number of Backups to Keep: #{ns1PUserDefault('keychainRotationFrequency').empty? ? 50 : ns1PUserDefault('keychainRotationFrequency')}")
    end
    if !APPSTORE and File.exists?(legacyOSXKeychainBackupFolder)
      Print.stat(legacyOSXKeychainBackupFolder)
      Print.dir(legacyOSXKeychainBackupFolder, 't')
    end
    if !File.exists?(agileBackupFolder)
      print "<p>The selected 1Password backups location of #{agileBackupFolder} does <b>not</b> exist.</p>"
      KNOWN_CONFIGURATION_ISSUES << "The 1Password backups location does not exist. This could lead to data loss if anything happens to the data file itself and no other backups (such as Time Machine) are available."
    else
      if `test -w '#{agileBackupFolder}' || echo 'protected'` =~ /protected/
        print "<p>The selected 1Password backups location is <b>not</b> writable.</p>"
        KNOWN_CONFIGURATION_ISSUES << "The 1Password backups location is not writable. This could lead to data loss if anything happens to the data file itself and no other backups (such as Time Machine) are available."
      end
      Print.stat(agileBackupFolder)
      Print.dir(agileBackupFolder, 't')
    end
    Print.end_section

    # Often unresponsive / causes hangs
    # Print.start_section("Other Data File Recovery Sources")
    # print %x{mdfind -name .agilekeychain | grep ".agilekeychain" | grep -v "#{agileBackupFolder}"}
    # Print.end_section

    Logs.printLog(:file => "#{HOME}/Library/Preferences/com.apple.security.plist") if !APPSTORE
    Print.end_section
  end
end

class OnePasswdAgent < BaseTestClass
  
  def OnePasswdAgent.buildnum
    return $1 if OnePasswdAgent.full_version =~ /.*\(build #(\d+)\)/
  end
  
  def OnePasswdAgent.full_version(agent_install_path='~/Library/Application Support/1Password/Agent/1PasswordAgent.app')
    begin
      lines = File.new(File.expand_path("#{agent_install_path}/Contents/Info.plist")).read rescue nil
      build = $1 if lines =~ /<key>CFBundleVersion<\/key>$\s*<string>(.*?)<\/string>/i
      version = $1 if lines =~ /<key>CFBundleShortVersionString<\/key>$\s*<string>(.*?)<\/string>/i
      return "#{version} (build ##{build})"
    rescue
      return "not installed in #{path}"
    end
  end

  def tests()
    Print.start_section("1Password Agent")
    print "Agent started? #{processStarted?("1PasswordAgent")}"
    
    Print.start_section("Launch Agents Information")
    f = File.expand_path('~/Library/LaunchAgents')
    resetFolderPermissionsSteps = "Please run 1Password and choose the Help > Troubleshooting > Reset Folder Permissions menu option. You will need to reboot your computer after this step."
    if File.exists?(f)
      Print.stat(f)
      unless %x{ls -lo "#{f}" 2>/dev/null}.chomp =~ /^-rw(.*) #{%x{whoami}.chomp} /
        m = "Your have an improper file ownership and/or permissions for the #{f} folder."
        print "<p>#{m}</p>"
        KNOWN_CONFIGURATION_ISSUES << "#{m} You may have difficulties installing, updating or automatically starting the 1PasswordAgent until this is corrected. It always needs to be owned by your own OS X account, and allow both read and write access. #{resetFolderPermissionsSteps}"
      end
    else
      m = "Your #{f} folder is missing."
      print "<p>#{m}</p>"
      KNOWN_CONFIGURATION_ISSUES << "#{m} You may have difficulties installing, updating or automatically starting the 1PasswordAgent until this is corrected. #{resetFolderPermissionsSteps}"
    end
    Print.stat(File.expand_path('~/Library/LaunchAgents/ws.agile.1PasswordAgent.plist'))
    print "~/Library/LaunchAgents has the following ACLs: [" +  Dir.new("#{HOME}/Library/LaunchAgents").list_acls + "]" rescue nil
    print "/Library/LaunchAgents has the following ACLs: [" + Dir.new("/Library/LaunchAgents").list_acls + "]" rescue nil
    Print.end_section
    
    Print.start_section("1PasswordAgent details")
    print "Installed version: #{OnePasswdAgent.full_version}"
    print "Version in #{COCOA_BUNDLE_PATH}: #{OnePasswdAgent.full_version("#{COCOA_BUNDLE_PATH}/Contents/Executables/1PasswordAgent.app")}"
    Print.stat File.expand_path('~/Library/Application Support/1Password')
    Print.stat File.expand_path('~/Library/Application Support/1Password/Agent')
    Print.stat File.expand_path('~/Library/Application Support/1Password/Agent/1PasswordAgent.app')
    Print.stat File.expand_path('~/Library/Application Support/1Password/Agent/1PasswordAgent.app/Contents/MacOS/1PasswordAgent')
    Print.end_section
    
    Print.start_section("launchctl list ws.agile.1PasswordAgent")
    print `launchctl list ws.agile.1PasswordAgent 2>&1`
    Print.end_section

    Print.start_section("Contents of ws.agile.1PasswordAgent.plist")
    print `cat "#{HOME}/Library/LaunchAgents/ws.agile.1PasswordAgent.plist" 2>&1`.htmlize
    Print.end_section
    if !Sys.is_leopard? and !Sys.is_tiger?
      user_id = `id -u`.chomp
      Print.start_section("Contents of com.apple.launchd.peruser.#{user_id}/overrides")
      print `cat "/var/db/launchd.db/com.apple.launchd.peruser.#{user_id}/overrides.plist" 2>&1`.htmlize
      Print.dir("/var/db/launchd.db/com.apple.launchd.peruser.#{user_id}")
      Print.end_section
    end
    Print.end_section
  end
end

class OnePasswdHelper < BaseTestClass
  def tests()
    Print.start_section("1Password Helper")
    print "Helper started? #{processStarted?("1Password Helper")}"
    Print.end_section
    user_id = `id -u`.chomp
    Print.start_section("Contents of com.apple.launchd.peruser.#{user_id}/overrides")
    print `cat "/var/db/launchd.db/com.apple.launchd.peruser.#{user_id}/overrides.plist" 2>&1`.htmlize
    Print.dir("/var/db/launchd.db/com.apple.launchd.peruser.#{user_id}")
    Print.end_section
  end
end

class Logs
  def Logs.collect
    begin
      _collect
    rescue
      Print.exception(self, $!)
    end
  end
  
  def Logs._collect
    logFolder = File.expand_path('~/Library/Logs/1Password')
    Print.header("Logs from #{logFolder}")
    
    if File.exists?(logFolder)
      Dir.foreach(logFolder) do |x| 
        next if x == "." || x == ".." || x == "keychain_backup.log" || x == "firefox-import-log.plist"
        printLog(:file => "#{logFolder}/#{x}") if File.should_print?("#{logFolder}/#{x}")
      end
    end

    Print.header("Logs from MAS version")
    printLog(:file => "#{CONTAINER}/Data/Library/Logs/1Password.log")
    printLog(:file => "#{CONTAINER}-helper/Data/Library/Logs/1Password Helper.log")

    crashFolder = File.expand_path("#{HOME}/Library/Logs/CrashReporter")
    Print.header("Crash Logs")
    Print.dir(crashFolder, 't')
    if File.exists?(crashFolder)
      Dir.foreach(crashFolder) do |log|
        if log =~ /securityd/ || log =~ /Safari/ || log =~ /irefox/ || log =~ /1Password/ || log =~ /DEVONa/ || log =~ /OmniW/ || log =~ /WebKit/ || log =~ /icab/i
          printLog(:file => "#{crashFolder}/#{log}", :unlimited => true) if File.should_print?("#{crashFolder}/#{log}")
        end
      end
    end

    # Print.header("Mobile Device Crash Logs")
    # Dir[File.expand_path('~/Library/Logs/CrashReporter/MobileDevice/*/*/1Password*')].each
    
    printLog(:file => '/var/log/system.log')

    Print.header("Firewall Logs")
    printLog(:file => '/var/log/ipfw.log')
    printLog(:file => '/var/log/appfirewall.log')
  end
  
    def Logs.printLog(opts)
        filename = File.expand_path(opts[:file])
        unlimited = opts[:unlimited] == true
        
        unless File.exists?(filename)
            Print.header("File #{filename} does not exist.")
        else
            Print.header("Contents of #{filename}")
            print "<pre class='log'>"
            if unlimited
                print `cat "#{filename}" 2>&1`.htmlize
            else
                print `tail -400 "#{filename}" 2>&1`.htmlize
            end
            print "</pre>"
        end
    end
end

class File
  def File.should_print?(filename)
    result = false
    
    f = File.new(filename)
    begin
      t = Time.now-5184000 # 60*24*60*60 Days*Hours*Minutes*Seconds
      n = f.mtime <=> t
      result = (n >= 0)
    rescue
      result = false
    ensure
      return result
    end
  end
end

class Dir
  def list_acls
    # Found here: http://www.macosxhints.com/comment.php?mode=view&cid=101947
    # There is one *very* edge case that will give a false ACL, but I don't think that's something we should worry about.
    `ls -alde "#{self.path}" 2> /dev/null | awk '{if(match($0,"^ [0-9]+:")>0){if(prevline!=""){print prevline;prevline=""}print $0}else{prevline=$0}}'`
  end
end

class VirusBarrier
  def self.installed?
    File.exist?("/Library/LaunchDaemons/com.intego.commonservices.daemon.plist") || File.exist?("/Library/LaunchDaemons/com.intego.task.manager.daemon.plist") || File.exist?("/Library/LaunchDaemons/com.intego.task.manager.daemon") || File.exist?("/Library/LaunchDaemons/com.intego.commonservices.icalserver.plist") || File.exist?("/Library/LaunchAgents/com.intego.task.manager.notifier.plist") || File.exist?("/Library/Intego/IntegoStatusItem.bundle/Contents/Resources/IntegoStatusItemEnabler") || File.exist?("/Library/StartupItems/IntegoCommon")
  end
  
  def self.started?
    %x{ps -Ajc | grep -i virusbarrier}.chomp != ""
  end
end

class HandsOff
  def self.installed?
    File.exist?("/Library/LaunchDaemons/com.metakine.handsoff.daemon.plist") || File.exist?("/Library/LaunchAgents/com.metakine.handsoff.agent.plist")
  end

  def self.started?
    %x{ps -Ajc | grep -i handsoff}.chomp != ""
  end
end

# Trusteer Rapport Test Module
class TrusteerRapport
    def self.installed?
        # Test if the rapportd daemon is installed
        File.exist?("/Library/Rapport/bin/rapportd")
    end

    def self.started?
        # List all items in launchd that are running, grep for rapportd
        %x{launchctl list | grep com.trusteer.rapport.rapportd}.chomp != ""
    end
end

if ARGV[0] =~ /BUNDLE_PATH=(.*)/
  COCOA_BUNDLE_PATH = $1
elsif Dir.pwd =~ /(1P.*?\.app)/
  COCOA_BUNDLE_PATH = Dir.pwd.gsub!('/Contents/Scripts','')
else
  COCOA_BUNDLE_PATH = "/Applications/1Password.app"
end
COCOA_BUNDLE_PATH = "/Applications/1Password.app" if COCOA_BUNDLE_PATH =~ /1Password Troubleshooting(.*).app$/

t = processStarted?("1Password Helper") or (File.exists?(CONTAINER) and !processStarted?("1PasswordAgent"))
t = true if ARGV[1] =~ /APPSTORE=t/
t = false if ARGV[1] =~ /APPSTORE=f/
APPSTORE = t

@@result = ""
@@table_of_contents = ""
@@sys = Sys.new
@@sys.tests
errlog_folder = File.expand_path("~/Library/Logs/1Password")
if RUBY_VERSION >= "1.9.0"
  FileUtils.makedirs(errlog_folder)
else
  File.makedirs(errlog_folder)
end
@@logger = Logger.new("#{errlog_folder}/diagnostics.log")
begin
  OnePasswd.new.tests
  Keychain.new.tests
  if APPSTORE
    OnePasswdHelper.new.tests
  else
    OnePasswdAgent.new.tests
  end
  Simbl.new.tests
  Safari.new.tests
  Firefox.new.tests
  Chrome.new.tests
  Chromium.new.tests
  ## TODO: Chrome Canary
  ## TODO: RockMelt
  JSE.new.tests
  OmniWeb.new.tests
  NetNewsWire.new.tests
  Fluid.new.tests
  
  details = @@result
  @@result = ""
  Logs.collect

  logs = @@result

  summary = Summary.summary
  configuration = Configuration.summary

  template = ERB.new(RESULT_TEMPLATE)
  template = template.result(binding)
  folder = File.expand_path("#{HOME}/Library/Application Support/1Password/Diagnostics")
  if ARGV.to_s =~ /-O=stdout/
    STDOUT.write template
  else
    if RUBY_VERSION >= "1.9.0" 
      FileUtils.makedirs(folder)
    else
      File.makedirs(folder)
    end
    File.open("#{folder}/1Password Troubleshooting.html", 'w') {|f| f.write(template) }
    %x{open "#{folder}/1Password Troubleshooting.html"} if(ENV["TM_FILEPATH"])
  end
rescue Exception => e
  @@logger.error(e.message)
  @@logger.error(e.backtrace.join("\n"))
  puts "Generation failed. See #{errlog_folder}/diagnostics.log for details"
end
