/*
 * gcr.c
 * GCR encoding/decoding utility
 *
 */
#include <stdio.h>
#include "hardware.h"
#include "yae.h"

/*
 * raw track
 * this is obtained by
 * 14.31818MHz / 14 / 32 / 5; difference is due to stretched 65th cycles
 *
 */
#ifndef RAW_TRACK_BYTES
#  define RAW_TRACK_BYTES 6378
#endif

#define DOS_TRACK_BYTES 4096 
#define DOS_TRACK_BYTES_13 3328 
#define RAW_TRACK_BITS (RAW_TRACK_BYTES*8) 

static BYTE	GCR_encoding_table[64] = {
	0x96, 0x97, 0x9A, 0x9B, 0x9D, 0x9E, 0x9F, 0xA6,
	0xA7, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB2, 0xB3,
	0xB4, 0xB5, 0xB6, 0xB7, 0xB9, 0xBA, 0xBB, 0xBC,
	0xBD, 0xBE, 0xBF, 0xCB, 0xCD, 0xCE, 0xCF, 0xD3,
	0xD6, 0xD7, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE,
	0xDF, 0xE5, 0xE6, 0xE7, 0xE9, 0xEA, 0xEB, 0xEC,
	0xED, 0xEE, 0xEF, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6,
	0xF7, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF };

static BYTE	GCR_encoding_table_13[32] = {
	0xAB, 0xAD, 0xAE, 0xAF, 0xB5, 0xB6, 0xB7, 0xBA,
        0xBB, 0xBD, 0xBE, 0xBF, 0xD6, 0xD7, 0xDA, 0xDB,
        0xDD, 0xDE, 0xDF, 0xEA, 0xEB, 0xED, 0xEE, 0xEF,
        0xF5, 0xF6, 0xF7, 0xFA, 0xFB, 0xFD, 0xFE, 0xFF };

static BYTE	GCR_decoding_table_13_prelim[128] = {
        0x11, 0xCA, 0xD0, 0xFD, 0xE6, 0x46, 0xD0, 0x02,
        0xE6, 0x47, 0x38, 0xE9, 0x01, 0xD0, 0xF0, 0x60,
        0x01, 0x30, 0x28, 0x24, 0x20, 0x1E, 0x1D, 0x1C,
	0x1C, 0x1C, 0x1C, 0x1C, 0x70, 0x2C, 0x26, 0x22,
	0x1F, 0x1E, 0x1D, 0x1C, 0x1C, 0x1C, 0x1C, 0x1C,
	0x00, 0x00, 0x00, 0x00, 0x01, 0x08, 0x10, 0x18,
	0x02, 0x03, 0x04, 0x05, 0x06, 0x20, 0x28, 0x30,
	0x07, 0x09, 0x38, 0x40, 0x0A, 0x48, 0x50, 0x58,
	0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x11, 0x12, 0x13,
	0x14, 0x15, 0x16, 0x17, 0x19, 0x1A, 0x1B, 0x1C,
	0x1D, 0x1E, 0x21, 0x22, 0x23, 0x24, 0x60, 0x68,
	0x25, 0x26, 0x70, 0x78, 0x27, 0x80, 0x88, 0x90,
	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x31,
	0x32, 0x33, 0x98, 0xA0, 0x34, 0xA8, 0xB0, 0xB8,
	0x35, 0x36, 0x37, 0x39, 0x3A, 0xC0, 0xC8, 0xD0,
	0x3B, 0x3C, 0xD8, 0xE0, 0x3E, 0xE8, 0xF0, 0xF8 };

static BYTE	GCR_decoding_table[256];
static BYTE	GCR_decoding_table_13[256];

static	int	Swap_Bit[4] = { 0, 2, 1, 3 }; /* swap lower 2 bits */
static BYTE	GCR_buffer[410];
static BYTE	GCR_buffer2[411];

static int	Position=0;
static BYTE	*Track_Nibble;

/* physical sector no. to DOS 3.3 logical sector no. table */
static int	Logical_Sector[16] = {
	0x0, 0x7, 0xE, 0x6, 0xD, 0x5, 0xC, 0x4,
	0xB, 0x3, 0xA, 0x2, 0x9, 0x1, 0x8, 0xF };

/* physical sector no. to DOS 3.2 usual sector position table
   does no real logical -> physical mapping, that is 1:1 in DOS 3.2 */
static int	Logical_Sector_13[13] = {
	0x0, 0xA, 0x7, 0x4, 0x1, 0xB, 0x8, 0x5,
	0x2, 0xC, 0x9, 0x6, 0x3 };

static int	Physical_Sector[16];

/* static function prototypes */

static void init_GCR_table(void);
static BYTE gcr_read_nibble(void);
static void gcr_write_nibble( BYTE );
static void decode62( BYTE* );
static int  decode53( BYTE* );
static void encode62( BYTE* );
static void encode53( BYTE* );
static void FM_encode( BYTE );
static BYTE FM_decode(void);
static void write_sync( int );
static int  read_address_field( int*, int*, int* );
static int  read_address_field_13( int*, int*, int* );
static void write_address_field( int, int, int );
static void write_address_field_13( int, int, int );
static int  read_data_field(void);
static int  read_data_field_13(void);
static void write_data_field(void);
static void write_data_field_13(void);

#define FM_ENCODE(x)	gcrWriteNibble( ((x) >> 1) | 0xAA );\
			gcrWriteNibble( (x) | 0xAA )

static void init_GCR_table(void)
{
	static int	initialized = 0;
	int		i;

	if ( !initialized ) {	
	   for( i = 0; i < 64; i++ )
	      GCR_decoding_table[GCR_encoding_table[i]] = i;
	   for( i = 0; i < 128; i++ ) {
              GCR_decoding_table_13[i] = 0;
	      GCR_decoding_table_13[i+128] = GCR_decoding_table_13_prelim[i]; }
	   for( i = 0; i < 16; i++ )
	      Physical_Sector[Logical_Sector[i]] = i;
	   initialized = 1;
	}
}

static BYTE gcr_read_nibble(void)
{ 
	BYTE	data;
	
	data = Track_Nibble[Position++];
	if ( Position >= RAW_TRACK_BYTES )
	   Position = 0;	
	return data;
}

static void gcr_write_nibble( BYTE data )
{
	Track_Nibble[Position++] = data;
	if ( Position >= RAW_TRACK_BYTES )
	   Position = 0;
}

static void decode62( BYTE *page )
{
	int	i, j;

	/* get 6 bits from GCR_buffer & 2 from GCR_buffer2 */
	for( i = 0, j = 86; i < 256; i++ ) {
	  if ( --j < 0 ) j = 85;
	  page[i] = (GCR_buffer[i] << 2) | Swap_Bit[GCR_buffer2[j] & 0x03];
	  GCR_buffer2[j] >>= 2;
	}
}

static int decode53( BYTE *page )
{
    int	i;
    BYTE a, x, y, v26, v27, c;
    
    i = 0;
    y = 0x9a;
    a = 0;
 loop_b922:
    y--;
    v26 = y;
    do { y = GCR_buffer2[i]; i++; } while ( ! (y & 0x80));
    a ^= GCR_decoding_table_13[y];
    y = v26;
    GCR_buffer[y + 0x100] = a;
    if ( y ) goto loop_b922;
 loop_b934:
    v26 = y;
    do { y = GCR_buffer2[i]; i++; } while ( ! (y & 0x80));
    a ^= GCR_decoding_table_13[y];
    y = v26;
    GCR_buffer[y] = a;
    y++;
    if ( y ) goto loop_b934;
    do { y = GCR_buffer2[i]; i++; } while ( ! (y & 0x80));
    if ( a != GCR_decoding_table_13[y] ) {
	return 0;
    }

    x = 0x32;
    y = 0;
 loop_b9c5:
    a = GCR_buffer[x + 0x100];
    a >>= 3;
    v27 = a;
    a >>= 1;
    v26 = a;
    a >>= 1;
    a |= GCR_buffer[x];
    page[y] = a;
    y++;
    a = GCR_buffer[x + 0x133];
    a >>= 3;
    c = a & 1;
    a >>= 1;
    v27 = (v27 << 1) + c;
    c = a & 1;
    a >>= 1;
    v26 = (v26 << 1) + c;
    a |= GCR_buffer[x + 0x33];
    page[y] = a;
    y++;
    a = GCR_buffer[x + 0x166];
    a >>= 3;
    c = a & 1;
    a >>= 1;
    v27 = (v27 << 1) + c;
    c = a & 1;
    a >>= 1;
    v26 = (v26 << 1) + c;
    a |= GCR_buffer[x + 0x66];
    page[y] = a;
    y++;
    a = v26;
    a &= 0x07;
    a |= GCR_buffer[x + 0x99];
    page[y] = a;
    y++;
    a = v27;
    a &= 0x07;
    a |= GCR_buffer[x + 0xcc];
    page[y] = a;
    y++;
    x--;
    if (! (x & 0x80) ) goto loop_b9c5;
    a = GCR_buffer[0x199];
    a >>= 3;
    a |= GCR_buffer[0xff];
    page[y] = a;

    return 1;
}

static void encode62( BYTE *page )
{
	int	i, j;

	/* 86 * 3 = 258, so the first two byte are encoded twice */
	GCR_buffer2[0] = Swap_Bit[page[1]&0x03];
	GCR_buffer2[1] = Swap_Bit[page[0]&0x03];

	/* save higher 6 bits in GCR_buffer and lower 2 bits in GCR_buffer2 */
	for( i = 255, j = 2; i >= 0; i--, j = j == 85? 0: j + 1 ) {
	   GCR_buffer2[j] = (GCR_buffer2[j] << 2) | Swap_Bit[page[i]&0x03];
	   GCR_buffer[i] = page[i] >> 2;
	}
	 
	/* clear off higher 2 bits of GCR_buffer2 set in the last call */
	for( i = 0; i < 86; i++ )
	   GCR_buffer2[i] &= 0x3f;
}

static void encode53( BYTE *page )
{
    int i;
    BYTE a, x, y, c, v26, v27, v2a;

    x = 50;
    y = 0;
 loop_b804:
    a = page[y];
    v26 = a;
    a >>= 3;
    GCR_buffer[x] = a;
    y++;
    a = page[y];
    v27 = a;
    a >>= 3;
    GCR_buffer[x + 0x33] = a;
    y++;
    a = page[y];
    v2a = a;
    a >>= 3;
    GCR_buffer[x + 0x66] = a;
    y++;
    a = page[y];
    c = a & 1;
    a >>= 1;
    v2a = (v2a << 1) + c;
    c = a & 1;
    a >>= 1;
    v27 = (v27 << 1) + c;
    c = a & 1;
    a >>= 1;
    v26 = (v26 << 1) + c;
    GCR_buffer[x + 0x99] = a;
    y++;
    a = page[y];
    c = a & 1;
    a >>= 1;
    v2a = (v2a << 1) + c;
    c = a & 1;
    a >>= 1;
    v27 = (v27 << 1) + c;
    c = a & 1;
    a >>= 1;
    GCR_buffer[x + 0xcc] = a;
    a = v26;
    a = ((a << 1) + c) & 0x1f;
    GCR_buffer[x + 0x100] = a;
    a = v27 & 0x1f;
    GCR_buffer[x + 0x133] = a;
    a = v2a & 0x1f;
    GCR_buffer[x + 0x166] = a;
    y++;
    x--;
    if ( ! (x & 0x80) ) goto loop_b804;
    a = page[y];
    x = a;
    a &= 0x07;
    GCR_buffer[0x199] = a;
    a = x;
    a >>= 3;
    GCR_buffer[0xff] = a;

    i = 0;
    v26 = GCR_buffer[0x100];
    a = 0;
    y = 0x9a;
    goto loop_b8a9;
 loop_b8a6:
    a = GCR_buffer[y+0x100];
 loop_b8a9:
    a ^= GCR_buffer[y+0xff];
    GCR_buffer2[i] = GCR_encoding_table_13[a]; i++;
    y--;
    if ( y ) goto loop_b8a6;
    a = v26;
 loop_b8be:
    a ^= GCR_buffer[y];
    GCR_buffer2[i] = GCR_encoding_table_13[a]; i++;
    a = GCR_buffer[y];
    y++;
    if ( y ) goto loop_b8be;
    GCR_buffer2[i] = GCR_encoding_table_13[a]; i++;

}

/*
 * write an FM encoded value, used in writing address fields 
 */
static void FM_encode( BYTE data )
{
	gcr_write_nibble( (data >> 1) | 0xAA );
	gcr_write_nibble( data | 0xAA );
}

/*
 * return an FM encoded value, used in reading address fields 
 */
static BYTE FM_decode(void)
{
	int		tmp; 

	/* C does not specify order of operand evaluation, don't
	 * merge the following two expression into one
	 */
	tmp = (gcr_read_nibble() << 1) | 0x01;
	return gcr_read_nibble() & tmp;
}

static void write_sync( int length )
{
	while( length-- )
	   gcr_write_nibble( 0xFF );
}

/*
 * read_address_field: try to read a address field in a track
 * returns 1 if succeed, 0 otherwise
 */
static int read_address_field( int *volume, int *track, int *sector )
{
	int	max_try;
	BYTE	nibble;

	max_try = 100; 
	while( --max_try ) {
	   nibble = gcr_read_nibble(); 
	   check_D5:	
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	   goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0x96 )
	      goto check_D5;
	   *volume = FM_decode();
	   *track = FM_decode();
	   *sector = FM_decode();
	   return ( *volume ^ *track ^ *sector ) == FM_decode() &&
	      gcr_read_nibble() == 0xDE;
	}
	return 0;
}

static int read_address_field_13( int *volume, int *track, int *sector )
{
	int	max_try;
	BYTE	nibble;

	max_try = 100; 
	while( --max_try ) {
	   nibble = gcr_read_nibble(); 
	   check_D5:	
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	   goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xB5 )
	      goto check_D5;
	   *volume = FM_decode();
	   *track = FM_decode();
	   *sector = FM_decode();
	   return ( *volume ^ *track ^ *sector ) == FM_decode() &&
	      gcr_read_nibble() == 0xDE;
	}
	return 0;
}

static void write_address_field( int volume, int track, int sector )
{
	/*
	 * write address mark
	 */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0x96 );

	/*
	 * write Volume, Track, Sector & Check-sum
	 */
	FM_encode( volume );
	FM_encode( track );
	FM_encode( sector );
	FM_encode( volume ^ track ^ sector );

	/*
	 * write epilogue
	 */
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

static void write_address_field_13( int volume, int track, int sector )
{
	/*
	 * write address mark
	 */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xB5 );

	/*
	 * write Volume, Track, Sector & Check-sum
	 */
	FM_encode( volume );
	FM_encode( track );
	FM_encode( sector );
	FM_encode( volume ^ track ^ sector );

	/*
	 * write epilogue
	 */
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

/*
 * read_data_field: read_data_field into GCR_buffers, return 0 if fail
 */
static int read_data_field(void)
{
	int	i, max_try;
	BYTE	nibble, checksum;

	/*
	 * read data mark
	 */
	max_try = 32;
	while( --max_try ) {
	   nibble = gcr_read_nibble();
	check_D5:
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	      goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble == 0xAD )
	      break;
	}
	if ( !max_try ) /* fails to get address mark */
	   return 0;

 	for( i = 0x55, checksum = 0; i >= 0; i-- ) {
	   checksum ^= GCR_decoding_table[gcr_read_nibble()]; 
	   GCR_buffer2[i] = checksum;
	}

	for( i = 0; i < 256; i++ ) {
	   checksum ^= GCR_decoding_table[gcr_read_nibble()]; 
	   GCR_buffer[i] = checksum;
	}

	/* verify sector checksum */
	if ( checksum ^ GCR_decoding_table[gcr_read_nibble()] )
	   return 0; 

	/* check epilogue */
	return gcr_read_nibble() == 0xDE && gcr_read_nibble() == 0xAA;
}

static int read_data_field_13(void)
{
	int	i, max_try;
	BYTE	nibble;

	/*
	 * read data mark
	 */
	max_try = 32;
	while( --max_try ) {
	   nibble = gcr_read_nibble();
	check_D5:
	   if ( nibble != 0xD5 )
	      continue;
	   nibble = gcr_read_nibble();
	   if ( nibble != 0xAA )
	      goto check_D5;
	   nibble = gcr_read_nibble();
	   if ( nibble == 0xAD )
	      break;
	}
	if ( !max_try ) { /* fails to get address mark */
	    for( i = 0; i <= 410; i++ ) { /* zero-fill fresh INITed sector */
		GCR_buffer2[i] = GCR_encoding_table_13[0];
	    }
	    return 1;
	}

 	for( i = 0; i <= 410; i++ ) {
	    GCR_buffer2[i] = gcr_read_nibble();
	}

	/* check epilogue 
	   return gcr_read_nibble() == 0xDE && gcr_read_nibble() == 0xAA; */
	if ( gcr_read_nibble() != 0xDE || gcr_read_nibble() != 0xAA ) {
	    return 0;
	}

	return 1;
}

static void write_data_field(void)
{
	int	i;
	BYTE	last, checksum;

	/* write prologue */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xAD );

	/* write GCR encode data */
 	for( i = 0x55, last = 0; i >= 0; i-- ) {
	   checksum = last^ GCR_buffer2[i];
	   gcr_write_nibble( GCR_encoding_table[checksum] );
	   last = GCR_buffer2[i];
	}
	for( i = 0; i < 256; i++ ) {
	   checksum = last ^ GCR_buffer[i];
	   gcr_write_nibble( GCR_encoding_table[checksum] );
	   last = GCR_buffer[i];
	}

	/* write checksum and epilogue */
	gcr_write_nibble( GCR_encoding_table[last] );
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

static void write_data_field_13(void)
{
	int	i;

	/* write prologue */
	gcr_write_nibble( 0xD5 );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xAD );

	/* write GCR encode data incl. checksum */
 	for( i = 0; i <= 410; i++ ) {
	   gcr_write_nibble( GCR_buffer2[i] );
	}

	/* write epilogue */
	gcr_write_nibble( 0xDE );
	gcr_write_nibble( 0xAA );
	gcr_write_nibble( 0xEB );
}

void SectorsToNibbles( BYTE *sectors, BYTE *nibbles, int volume, int track )
{
	int	i;

	init_GCR_table();
	Track_Nibble = nibbles;
	Position = 0;

	write_sync( RAW_TRACK_BYTES );
	for( i = 0; i < 16; i ++ ) {
	   encode62( sectors + Logical_Sector[i] * 0x100 );
	   write_sync( 26 );
	   write_address_field( volume, track, i );
	   write_sync( 10 );
	   write_data_field();
	}
}

void SectorsToNibbles_13( BYTE *sectors, BYTE *nibbles, int volume, int track )
{
	int	i;

	init_GCR_table();
	Track_Nibble = nibbles;
	Position = 0;

	write_sync( RAW_TRACK_BYTES );
	for( i = 0; i < 13; i ++ ) {
	   encode53( sectors + Logical_Sector_13[i] * 0x100 );
	   write_sync( 49 );
	   write_address_field_13( volume, track, Logical_Sector_13[i] );
	   write_sync( 10 );
	   write_data_field_13();
	}
}

int NibblesToSectors( BYTE *nibbles, BYTE *sectors, int volume, int track )
{
	int	i, scanned[16], max_try, sectors_read;
	int	vv, tt, ss;	/* volume, track no. and sector no. */
	FILE	*fp;

	init_GCR_table();
	Track_Nibble = nibbles;
	Position = 0;

	for( i = 0; i < 16; i++ )
	   scanned[i] = 0;
	sectors_read = 0;

	max_try = 200;
	while( --max_try ) { 
	   if ( !read_address_field( &vv, &tt, &ss ) )
	      continue;

	   if ( (volume && vv != volume ) || tt != track || ss < 0 || ss > 15 ){
	      fprintf(stderr, "phy sector %d address field invalid\n", ss );
	      continue;	/* invalid values for vv, tt and ss, try again */
	   }

	   ss = Logical_Sector[ss];
	   if ( scanned[ss] )	/* sector has been read */
	      continue;

	   if ( read_data_field() ) {
	      decode62( sectors + ss * 0x100 );
	      scanned[ss] = 1;	/* this sector's ok */ 
	      sectors_read++;
	   }
	   else {
	      fprintf(stderr, "fail reading data field of logical sector %d\n", ss );
	   }
	}

	/* if has failed to read any one sector, report error */
	if ( sectors_read == 16 )
	   return 1;
	else {
	   fprintf(stderr, "sectos_read = %d\n",sectors_read);
	   for( i = 0; i < 16; i++ )
	      if ( !scanned[i] )
	         fprintf(stderr, "sector %d(%d) corrupted\n", i, Physical_Sector[i] );
	   if((fp = fopen( ".track", "wb"))!=NULL) {
	      fwrite( nibbles, 1, RAW_TRACK_BYTES, fp );
	      fclose(fp);
	   }
	   return 0;
	}
}

int NibblesToSectors_13( BYTE *nibbles, BYTE *sectors, int volume, int track )
{
	int	i, scanned[13], max_try, sectors_read;
	int	vv, tt, ss;	/* volume, track no. and sector no. */
	FILE	*fp;

	init_GCR_table();
	Track_Nibble = nibbles;
	Position = 0;

	for( i = 0; i < 13; i++ )
	   scanned[i] = 0;
	sectors_read = 0;

	max_try = 200;
	while( --max_try ) { 
	   if ( !read_address_field_13( &vv, &tt, &ss ) )
	      continue;

	   if ( (volume && vv != volume ) || tt != track || ss < 0 || ss > 12 ){
	      fprintf(stderr, "sector %d address field invalid\n", ss );
	      continue;	/* invalid values for vv, tt and ss, try again */
	   }

	   if ( scanned[ss] )	/* sector has been read */
	      continue;
	   
	   if ( read_data_field_13() && decode53( sectors + ss * 0x100 ) ) {
	      scanned[ss] = 1;	/* this sector's ok */ 
	      sectors_read++;
	   }
	   else {
	      fprintf(stderr, "fail reading data field of sector %d\n", ss );
	   }
	}
	
	/* if has failed to read any one sector, report error */
	if ( sectors_read == 13 )
	   return 1;
	else {
	   fprintf(stderr, "sectos_read = %d\n",sectors_read);
	   for( i = 0; i < 13; i++ )
	      if ( !scanned[i] )
	         fprintf( stderr, "sector %d corrupted\n", i );
	   if((fp = fopen( ".track", "wb"))!=NULL) {
	      fwrite( nibbles, 1, RAW_TRACK_BYTES, fp );
	      fclose(fp);
	   }
	   return 0;
	}
}
