/* */
/*	aftp.c - Apple File Transfer Program - Adjusts disk images */
/* */
/*	Written by Galen C. Hunt [email: gchunt@cc.dixie.edu] */
/*	Released into the public domain with no warranties. */
/* */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
/* #include <dos.h> */
#include "aftp.h"
#define XGS 2

int		text_mode = MODE_TEXT;
int		read_only = 0;
int		filesystem = SIMOS_DOS;
int		order = SIMOS_DOS;

/*	Open The Disk Image: */
static	byte	raw_dos[SECTORS_PER_TRACK] = {0,13,11,9,7,5,3,1,14,12,10,8,6,4,2,15};
static	byte	raw_prodos[SECTORS_PER_TRACK] = {0,2,4,6,8,10,12,14,1,3,5,7,9,11,13,15};
static	byte	raw_andru[SECTORS_PER_TRACK] = {0,14,13,12,11,10,9,8,7,6,5,4,3,2,1,15};

disk  	disk_open(char *name, char *mode)
{
	disk 	d;
	FILE *	fp;

	if (!(fp = fopen(name, mode)))
	{
		printf("Unable to open image file: %s\n", name);
		return NULL;
	}
	if (d = alloc(sizeof(*d)))
	{
		int		i;
		word	w;

		memset(d, 0, sizeof(*d));
		d->name = name;
		d->file = fp;


	/* begin ajl mods */
	fseek(fp, 0, SEEK_END);
	switch(order) {
	case SIMOS_DOS:
		d->blocks = ftell(fp) / 512;
		d->data_pos = 0;
		d->use_tracks = 1;
		for (i = 0; i < SECTORS_PER_TRACK; i++)
		{
		d->interleave[i] = i;
		d->order[i] = raw_andru[i];
		}
		break;
	case SIMOS_PRODOS:
		d->blocks = ftell(fp) / 512;
		d->data_pos = 0;
		d->use_tracks = 0;
		for (i = 0; i < SECTORS_PER_TRACK; i++)
		{
		d->interleave[i] = i;
		d->order[i] = i;
		}
		break;
	case XGS:
		d->data_pos = 1076;
		d->blocks = (ftell(fp) - 1076) / 512;
		d->use_tracks = 0;
		for (i = 0; i < SECTORS_PER_TRACK; i++)
		{
		d->interleave[i] = i;
		d->order[i] = i;
		}
		break;
	}
	switch(filesystem) {
	case SIMOS_DOS:
		d->type = SIMOS_DOS;
		break;
	case SIMOS_PRODOS:
		d->type = SIMOS_PRODOS;
		break;
	}

		/* end ajl mods */

		d->buffer_dirty = 0;
		d->buffer_block = -1;
		d->buffer_track = -1;
		d->buffer_sector = -1;
		d->buffer_offset1 = -1;
		d->buffer_offset2 = -1;
		
		switch (d->type)
		{
	case SIMOS_PRODOS:
			d->os = &os_pro;
			break;
		case SIMOS_DOS:
		d->os = &os_dos;
			break;
		}
		if (d->os_data = alloc(d->os->os_data_size))
			if ((*d->os->init)(d) < 0)
			{
				free(d->os_data);
				d->os_data = NULL;
			}
		if (!d->os_data)
		{
			free(d);
			d = NULL;
		}
	}
	return d;
}

int		disk_read_block(disk d, long block)
{
	if (block < 0 || block >= d->blocks)
	{
		printf("disk_read_block: bad block %ld of %ld\n", block, d->blocks);
		return -1;
	}
	if (d->buffer_block != block)
	{
		disk_flush(d);
		d->buffer_track = -1;
		d->buffer_sector = -1;
		d->buffer_block = block;

		if (d->use_tracks)
		{
			long	linear;
			long	s;

			linear = block * 2;
			s = linear % SECTORS_PER_TRACK;
			s = (long)(linear - s) + (long)d->order[s];
			d->buffer_offset1 = d->data_pos + (long)BYTES_PER_SECTOR * s;

			linear = block * 2 + 1;
			s = linear % SECTORS_PER_TRACK;
			s = (long)(linear - s) + (long)d->order[s];
			d->buffer_offset2 = d->data_pos + (long)BYTES_PER_SECTOR * s;
		}
		else
		{
			d->buffer_offset1 = d->data_pos + (long)BYTES_PER_BLOCK * (long)d->buffer_block;
			d->buffer_offset2 = d->buffer_offset1 + BYTES_PER_SECTOR;
		}
		fseek(d->file, d->buffer_offset1, SEEK_SET);
		fread(d->buffer, 1, BYTES_PER_SECTOR, d->file);
		fseek(d->file, d->buffer_offset2, SEEK_SET);
		fread(d->buffer + BYTES_PER_SECTOR, 1, BYTES_PER_SECTOR, d->file);
	}
	return 0;
}

int		disk_read_block_zero(disk d, long block)
{
	int		err;

	if (!(err = disk_read_block(d, block)))
	{
		memset(d->buffer, 0, BYTES_PER_BLOCK);
		disk_dirty(d);
	}
	return err;
}

int		disk_read_sector(disk d, int track, int sector)
{
	if (sector < 0 || sector >= SECTORS_PER_TRACK || track < 0 || track >= TRACKS_PER_DISK)
	{
		printf("disk_read_sector: bad track %d, sector %d\n", track, sector);
		return -1;
	}
	if (d->buffer_track != track || d->buffer_sector != sector)
	{
		long	s;

		disk_flush(d);
		d->buffer_track = track;
		d->buffer_sector = sector;
		d->buffer_block = -1;

		s = (long)track * SECTORS_PER_TRACK + (long)d->order[sector];
		d->buffer_offset1 = d->data_pos + (long)BYTES_PER_SECTOR * s;
		d->buffer_offset2 = -1;
		fseek(d->file, d->buffer_offset1, SEEK_SET);
		fread(d->buffer, 1, BYTES_PER_SECTOR, d->file);
	}
	return 0;
}

/*	disk_dirty: */
/*		Set the dirty flag for the current block or sector. */
int		disk_dirty(disk d)
{
	if (read_only)
		printf("Disk image is read only!\n");
	else
		d->buffer_dirty = 1;
	return d->buffer_dirty;
}

/*	disk_flush: */
/*		Write the buffered block to the disk if it is dirty. */
int		disk_flush(disk d)
{
	if (d->buffer_dirty)
	{
		d->buffer_dirty = 0;
		if (read_only)
			return 0;
		if (d->buffer_offset1 >= 0)
		{
			fseek(d->file, d->buffer_offset1, SEEK_SET);
			fwrite(d->buffer, 1, BYTES_PER_SECTOR, d->file);
		}
		if (d->buffer_offset2 >= 0)
		{
			fseek(d->file, d->buffer_offset2, SEEK_SET);
			fwrite(d->buffer + BYTES_PER_SECTOR, 1, BYTES_PER_SECTOR, d->file);
		}
		return 1;
	}
	return 0;
}

word	word_at(byte *data)
{
	return (word)data[0] | ((word)data[1] << 8);
}

void	word_set(byte *data, word value)
{
	data[0] = value & 0xff;
	data[1] = (value & 0xff00) >> 8;
}

byte	disk_byteat(disk d, int offset)
{
	return d->buffer[offset];
}

void	disk_byteset(disk d, int offset, byte value)
{
	d->buffer[offset] = value;
	disk_dirty(d);
}

word	disk_wordat(disk d, int offset)
{
	return word_at(d->buffer + offset);
}

void	disk_wordset(disk d, int offset, word value)
{
	word_set(d->buffer + offset, value);
	disk_dirty(d);
}

void 	disk_close(disk d)
{
	if (!d)
	{
		printf("aftp: Attempt to free NULL disk.\n");
		exit(1);
	}
	(*d->os->close)(d);
	if (d->os_data)
		free(d->os_data);
	disk_flush(d);
	fclose(d->file);
	free(d);
}

static	void	print_help(void)
{
	printf("Command:\n");
	printf("  aftp [options] {input_disk_files}\n");
	printf("Options:\n");
	printf("  -p  use a (P)rodos filesystem.\n");
	printf("  -o  use a prodos (O)rdered image.\n");
	printf("  -i  use an xgs image file.\n");
	printf("  -d  Load images as read-only.\n");
	printf("  -?  Display aftp help.\n");
	printf("Summary:\n");
	printf("  Ftp like interface to dsk, do, po, and img files.\n");
	printf("  Defaults to Dos 3.3 filesystem on a dsk disk.\n");
	exit(1);
}

int main (int argc, char **argv)
{
	disk 	d;
	int		arg;
	int		images = 0;


	if (argc == 1)
		print_help();

	for (arg = 1; arg < argc; arg++)
	{
		if (argv[arg][0] == '-')
		{
			switch (argv[arg][1])
			{
			case 'i': case 'I':
				order = XGS;
				break;
			case 'o': case 'O':
				order = SIMOS_PRODOS;
				break;
			case 'p': case 'P':
				filesystem = SIMOS_PRODOS;
				break;
			case 'r':
			case 'R':
				read_only = !read_only;
				break;
			case '?':
				print_help();
				break;
			default:
				printf("Unknown option: %s\n", argv[arg]);
			}
			continue;
		}

		images++;
		if (!(d = disk_open(argv[arg], "r+b")))
		{
			printf("Couldn't access %s\n", argv[arg]);
			continue;
		}

		printf("\n");
		(*d->os->dir)(d, "");

		for (; images == 1 && arg == argc - 1;)
		{
			char *	cmd;
			char	buf[1024];
			char	local[256];
			char	remot[256];
			char *	p;
			char *	arg;
			int		err;

			switch (text_mode)
			{
			case MODE_BINARY: printf("[binary]> "); break;
			case MODE_TEXT: printf("[text]> "); break;
			default: printf("[aftp]> "); break;
			}
			fflush(stdout);
			if (!fgets(buf, 1024, stdin))
				break;
			for (p = buf; *p && isspace(*p); p++)
				;
			cmd = p;
			if (*cmd == '!')
			{
				for (; *p && *p != '\n'; p++)
					;
				*p = '\0';
				system(cmd + 1);
				continue;
			}
			for (; *p && !isspace(*p); p++)
				;
			if (*p) *p++ = '\0';
			for (; *p && isspace(*p); p++)
				;
			arg = p;
			for (; *p && *p != '\n'; p++)
				;
			*p = '\0';
			if (!*cmd)
				continue;
			if (!strcmp(cmd, "dir"))
			{
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				err = (*d->os->dir)(d, remot);
			}
			else if (!strcmp(cmd, "quit"))
				break;
			else if (!strcmp(cmd, "text") || !strcmp(cmd, "txt"))
				text_mode = MODE_TEXT;
			else if  (!strcmp(cmd, "binary") || !strcmp(cmd, "bin"))
				text_mode = MODE_BINARY;
			else if (!strcmp(cmd, "get"))
			{
				FILE * 	fp;

				strcpy(local, arg);
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				printf("Getting %s from %s.\n", local, remot);
				if (fp = fopen(local, text_mode == MODE_TEXT ? "w" : "wb"))
				{
					err = (*d->os->get)(d, remot, fp);
					fclose(fp);
					if (err < 0)
					{
						unlink(local);
						printf("Deleted local file: %s\n", local);
					}
				}
				else
					printf("Unable to create local: %s\n", local);
			}
			else if (!strcmp(cmd, "put"))
			{
				FILE * 	fp;

				strcpy(local, arg);
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				printf("Putting %s to %s.\n", local, remot);
				if (fp = fopen(local, text_mode == MODE_TEXT ? "r" : "rb"))
				{
					err = (*d->os->put)(d, remot, fp);
					fclose(fp);
					if (err < 0)
						printf("Local file untouched: %s\n", local);
				}
				else
					printf("Unable to open local: %s\n", local);
			}
			else if (!strcmp(cmd, "del") || !strcmp(cmd, "delete") || !strcmp(cmd, "rm"))
			{
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				err = (*d->os->del)(d, remot);
				if (err < 0)
					printf("Remote file not deleted: %s\n", remot);
			}
			else if (!strcmp(cmd, "cd"))
			{
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				err = (*d->os->cd)(d, remot);
			}
		#ifdef DIRECTORY_COMMANDS
			else if (!strcmp(cmd, "rmdir") || !strcmp(cmd, "rd"))
			{
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				err = (*d->os->rmdir)(d, remot);
				if (err < 0)
					printf("* Remote directory not deleted: %s\n", remot);
			}
			else if (!strcmp(cmd, "mkdir") || !strcmp(cmd, "md"))
			{
				strcpy(remot, arg);
				(*d->os->name)(remot, sizeof(remot));
				err = (*d->os->mkdir)(d, remot);
				if (err < 0)
					printf("* Remote directory not created: %s\n", remot);
			}
		#endif
			else if (!strcmp(cmd, "dump"))
			{
				long	block;
				int		i;
				int		j;
				int		c;
				int		siz;

				block = atol(arg);
				printf("Block %ld:\n", block);
				disk_read_block(d, block);
				siz = 512;
				for (i = 0; i < siz; i += 16)
				{
					printf("  %03x: ", i);
					for (j = i; j < i + 16; j++)
						if (j < siz)
							printf("%02x ", d->buffer[j]);
						else
							printf("   ");
					for (j = i; j < i + 16; j++)
					{
						if (j < siz)
							c = d->buffer[j] & 0x7f;
						else
							c= ' ';
						printf("%c", (c == 0x7f || c < ' ') ? '.' : c);
					}
					printf("\n");
				}
			}
			else if (!strcmp(cmd, "help") || !strcmp(cmd, "?"))
			{
				printf("dir           Display disk image directory.\n");
				printf("get [file]    Get a file from disk image.\n");
				printf("put [file]    Put a file on disk image.\n");
				printf("del [file]    Delete a file from disk image.\n");
				printf("cd [dir]      Change disk image directory.\n");
			#ifdef DIRECTORY_COMMANDS
				printf("mkdir [dir]   Create a disk image directory.\n");
				printf("rmdir [dir]   Remove a disk image directory.\n");
			#endif
				printf("text          Set text mode.\n");
				printf("binary        Set binary mode.\n");
				printf("help          Display help.\n");
				printf("!dos command  Execute a dos command.\n");
				printf("quit          Quit Program.\n");
			}
			else
				printf("command %s not understood\n", cmd);
		}
		disk_close(d);
	}

	return 0;
}

void *alloc (int n)
{
	void* p;

	if (!(p = (void *)malloc(n)))
	{
		printf("aftp: out of memory\n");
		exit(1);
	}
	return p;
}

/* take an Apple II character code and convert it to ASCII */
/* the current implementation is workable but NOT right */
int apptoascii (int c)
{
	return c & 0x7f;
}

/* take an ASCII character code and convert it to Apple II code */
/* like apptoascii(), this is just a workable but NOT right implementation */
int asciitoapp (int c)
{
	return c | 0x80;
}

/*	End of File */
