/* */
/*	dos.c	 - Apple Dos 3.3 File System Functions */
/* */
/*	Written by Galen C. Hunt [email: gchunt@cc.dixie.edu] */
/*	Released into the public domain with no warranties. */
/* */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "aftp.h"

typedef struct _afile
{
	int 	filetype;
	disk	d;
	int 	ts_s, ts_t, ts_ent, dt_t, dt_s, dt_pos;
	int 	eof, dir_t, dir_s, dir_off;
	word	secgroup;
} *afile;

static	afile 	afile_open (disk d, char *filename, char *mode);
static	int 	afile_close (afile af);
static	int 	afile_putc (afile af, int ch);
static	int 	afile_getc (afile af);
static	long 	afile_getw (afile af);
static	int 	afile_putw (afile af, word w);
static	int 	isfree (disk d, int t, int s);
static	int 	allocate_sector (disk d, int *t, int *s, int pref_track);
static	void 	alloc_sector (disk d, int t, int s);
static	void 	free_sector (disk d, int t, int s);
static	void 	zero_sector (disk d, int t, int s);


/* afile_open -- open an Apple ][ file, return !0 if successful */
/* would be nice to share directory reading stuff with catalog() */
/* I assume that I cannot extend the directory and that the entire directory */
/* has been pre-allocated and pre-initialized by Apple DOS.  I also try */
/* to initialize things in the same way that Apple DOS would (e.g., every */
/* file will contain at least 2 sectors). */
static	afile afile_open(disk d, char *filename, char *mode)
{
	afile	af;
	int 	t, s, nt, ns, ft, fs, dir, i, found;
	int 	firstfree, fft, ffs, ffoff;
	char * 	p;

	if (!(af = alloc(sizeof(*af))))
		return NULL;
	memset(af, 0, sizeof(*af));
	t = 0x11;
	s = 0;
	firstfree = 0;
	for (found = 0; !found;)
	{
		disk_read_sector(d, t, s);
		nt = d->buffer[1];
		ns = d->buffer[2];

		if (!nt && !ns)
			break;

		t = nt;
		s = ns;
		disk_read_sector(d, t, s);
		for (dir = 0xb; dir + 35 <= BYTES_PER_SECTOR; dir += 35)
		{
			if (d->buffer[dir] == 0xff)
			{
				if (!firstfree)
				{
					firstfree = 1;
					fft = t;
					ffs = s;
					ffoff = dir;
				}
				continue;
			}
			if (d->buffer[dir] == 0)
			{
				if (!firstfree)
				{
					firstfree = 1;
					fft = t;
					ffs = s;
					ffoff = dir;
				}
				break;
			}
			found = 1;
			for (p = filename, i = 3; *p && i < 0x20; i++, p++)
			{
				if (*p != apptoascii(d->buffer[dir + i]))
				{
					found = 0;
					break;
				}
			}
			if (found)
			{
				for (; i < 0x20; i++)
				{
					if (apptoascii(d->buffer[dir + i]) != ' ')
						found = 0;
				}
			}
			if (found)
				break;
		}
		if (found)
			break;
	}

	if (!found)
	{
		int tt, ts, fs, ft;
		char* p;

		if (*mode == 'r' || !firstfree)
			return 0;
		if (!allocate_sector(d, &tt, &ts, -1))
			return 0;
		if (!allocate_sector(d, &ft, &fs, t))
		{
			free_sector(d, tt, ts);
			return 0;
		}
		/* fill in directory entry */
		disk_read_sector(d, fft, ffs);
		disk_dirty(d);
		d->buffer[ffoff] = tt;
		d->buffer[ffoff + 1] = ts;
		d->buffer[ffoff + 2] = text_mode ? 0 : 4;
		for (i = 3; i <= 0x20; i++)
			d->buffer[ffoff + i] = asciitoapp(' ');
		for (i = 3, p = filename; *p; i++, p++)
			d->buffer[ffoff + i] = asciitoapp(*p);
		d->buffer[ffoff + 0x21] = 2;
		d->buffer[ffoff + 0x22] = 0;

		/* fill in first track + sector list sector */
		disk_read_sector(d, tt, ts);
		disk_dirty(d);
		d->buffer[1] = 0;
		d->buffer[2] = 0;
		d->buffer[5] = 0;
		d->buffer[6] = 0;
		d->buffer[12] = ft;
		d->buffer[13] = fs;

		/* fudge some values so the fallthough will be right */
		t = fft;
		s = ffs;
		dir = ffoff;
	}

	/* enforce write protection on files */
	disk_read_sector(d, t, s);
	if (*mode == 'w' && (d->buffer[dir+2] & 0x80))
		return 0;

	switch (d->buffer[dir+2] & 0x7f)
	{
	case 0: af->filetype = 'T'; break;
	case 1: af->filetype = 'I'; break;
	case 2: af->filetype = 'A'; break;
	case 4: af->filetype = 'B'; break;
	default: af->filetype = '?'; break;
	}

	/* note which disk this file is on */
	af->d = d;

	/* keep track of where we are in the track/sector list */
	af->ts_t = d->buffer[dir];
	af->ts_s = d->buffer[dir + 1];

	/* initialize so that first read initializes other stuff */
	af->ts_ent = -1;
	af->dt_pos = BYTES_PER_SECTOR;

	/* these variables are only used by the writing routines */
	af->eof = 0;
	af->secgroup = 0;
	af->dir_t = t;
	af->dir_s = s;
	af->dir_off = dir;

	return af;
}

/* close the apple ][ file */
/* returns 0 on success, EOF on failure */
/* For now, there is nothing to do, but later versions may require this */
/* in order to fix up the disk (i.e., because of sector pre-allocation). */
/* Also, other versions may wish to truncate the file! */
static	int afile_close (afile af)
{
	return 0;
}

/* get a character from the open apple ][ file */
/* return the character or EOF on end of file */
static	int afile_getc (afile af)
{
	int	ch, t, s;

	if (af->ts_t == 0 && af->ts_s == 0)
		return EOF;

	if (af->dt_pos >= BYTES_PER_SECTOR)
	{
		af->dt_pos = 0;
		disk_read_sector(af->d, 0x11, 0);
		if (++af->ts_ent >= disk_byteat(af->d, 0x27))
		{
			af->ts_ent = 0;
			disk_read_sector(af->d, af->ts_t, af->ts_s);
			t = disk_byteat(af->d, 1);
			s = disk_byteat(af->d, 2);
			af->ts_t = t;
			af->ts_s = s;
			if (t == 0 && s == 0)
				return EOF;
		}
		disk_read_sector(af->d, af->ts_t, af->ts_s);
		af->dt_t = disk_byteat(af->d, 12 + af->ts_ent * 2);
		af->dt_s = disk_byteat(af->d, 12 + af->ts_ent * 2 + 1);
	}

	if (af->dt_t == 0 && af->dt_s == 0)
		return EOF;
	
	disk_read_sector(af->d, af->dt_t, af->dt_s);
	ch = disk_byteat(af->d, af->dt_pos);
	af->dt_pos++;
	return ch;
}

/* put a character into the open apple ][ file */
/* returns either the character written or EOF if some error occured */
/* I don't keep enough information around to give really good hints */
/* about where to allocate the next sector.  No big deal, really. */
static	int afile_putc (afile af, int ch)
{
	int 	t, s;
	disk	d;

	if (af->eof)
		return EOF;

	d = af->d;
	if (af->dt_pos >= BYTES_PER_SECTOR)
	{
		af->dt_pos = 0;
		disk_read_sector(d, 0x11, 0);
		if (++af->ts_ent >= d->buffer[0x27])
		{
			af->ts_ent = 0;
			disk_read_sector(d, af->ts_t, af->ts_s);
			t = d->buffer[1];
			s = d->buffer[2];

			disk_read_sector(d, 0x11, 0);
			af->secgroup += d->buffer[0x27];

			if (t == 0 && s == 0)
			{
				if (!allocate_sector(af->d, &t, &s, af->ts_t))
				{
					af->eof = 1;
					return EOF;
				}
				disk_read_sector(d, af->dir_t, af->dir_s);
				disk_dirty(d);
				d->buffer[af->dir_off + 0x21] += 1;

				disk_read_sector(d, af->ts_t, af->ts_s);
				disk_dirty(d);
				d->buffer[1] = t;
				d->buffer[2] = s;

				disk_read_sector(d, t, s);
				disk_dirty(d);
				d->buffer[5] = (af->secgroup & 255);
				d->buffer[6] = (af->secgroup & 0xff00) >> 8;
			}
			af->ts_t = t;
			af->ts_s = s;
		}
		disk_read_sector(d, af->ts_t, af->ts_s);
		af->dt_t = d->buffer[12 + af->ts_ent * 2];
		af->dt_s = d->buffer[12 + af->ts_ent * 2 + 1];
	}

	if (af->dt_t == 0 && af->dt_s == 0)
	{
		if (!allocate_sector(af->d, &af->dt_t, &af->dt_s, af->ts_t))
		{
			af->eof = 1;
			return EOF;
		}
		disk_read_sector(d, af->dir_t, af->dir_s);
		disk_dirty(d);
		d->buffer[af->dir_off + 0x21] += 1;

		disk_read_sector(d, af->ts_t, af->ts_s);
		disk_dirty(d);
		d->buffer[12 + af->ts_ent * 2] = af->dt_t;
		d->buffer[12 + af->ts_ent * 2 + 1] = af->dt_s;
	}
	disk_read_sector(d, af->dt_t, af->dt_s);
	disk_dirty(d);
	d->buffer[af->dt_pos++] = ch;
	return ch;
}

static	int afile_putw (afile af, word w)
{
	byte	b1, b2;

	b1 = (w & 0xff);
	b2 = (w & 0xff) >> 8;

	if (afile_putc(af, b1) == EOF)
		return EOF;
	if (afile_putc(af, b2) == EOF)
		return EOF;
	return 0;
}

/* get a 2 byte word (little endian, of course) from an apple ][ file */
static	long afile_getw (afile af)
{
	int b1, b2;

	if ((b1 = afile_getc(af)) == EOF || (b2 = afile_getc(af)) == EOF)
		return EOF;
	
	return b1 + b2 * 256;
}

/* sector allocation + de-allocation routines */

/* return non-zero if the given sector has not been allocated, 0 if free */
/* this routine is quite dependent on 16 sectors per track */
static	int isfree (disk d, int t, int s)
{
/*	This gives the sector mask which should be ANDed into the above, */
/*	but the sector masks on disks I've seen seem bogus (they should be 0xff) */
/*	but typically have only 1 or 2 bits set(!) */
/*	disk_read_sector(d, 0x11, 0); - Assume already read... */

	return d->buffer[0x38 + t * 4 + (s < 8)] & (1 << (s & 7));

}

/* allocate a sector for use.  If pref_track is valid, the sector will */
/* be allocated from the given track, if possible.  If pref_track is */
/* is invalid (e.g., < 0), then a search is made for an empty track */
/* and if none is found then we just look for the first sector we can */
/* find. */
/* */
/* As with the other sector routines, 16 sectors per track is hard-coded. */
static	int allocate_sector (disk d, int *t, int *s, int pref_track)
{
	int	i, j, found_track;

	disk_read_sector(d, 0x11, 0);

	if (pref_track < 0 || pref_track >= TRACKS_PER_DISK)
	{
		found_track = 0;
		for (i = 0; i < TRACKS_PER_DISK; i++)
		{
			if (d->buffer[0x38 + i * 4] == 0xff && d->buffer[0x38 + i * 4 + 1] == 0xff)
			{
				found_track = 1;
				break;
			}
		}
		if (found_track)
		{
			alloc_sector(d, *t = i, *s = 15);
			zero_sector(d, *t, *s);
			return 1;
		}
	}
	else
	{
		for (i = SECTORS_PER_TRACK - 1; i >= 0; i--)
		{
			if (isfree(d, pref_track, i))
			{
				alloc_sector(d, *t = pref_track, *s = i);
				zero_sector(d, *t, *s);
				return 1;
			}
		}
	}

	for (i = 0; i < TRACKS_PER_DISK; i++) {
		for (j = SECTORS_PER_TRACK - 1; j >= 0; j--) {
			if (isfree(d, i, j))
			{
				alloc_sector(d, *t = i, *s = j);
				zero_sector(d, *t, *s);
				return 1;
			}
		}
	}
	return 0;
}

/*	alloc_sector: */
/*		Mark the given sector as allocated. */
static	void alloc_sector (disk d, int t, int s)
{
	disk_read_sector(d, 0x11, 0);
	disk_dirty(d);
	d->buffer[0x38 + t * 4 + (s < 8)] &= ~(1 << (s & 7));
}

/*	free_sector: */
/*		Mark the given sector as free. */
static	void free_sector (disk d, int t, int s)
{
	disk_read_sector(d, 0x11, 0);
	disk_dirty(d);
	d->buffer[0x38 + t * 4 + (s < 8)] |= 1 << (s & 7);
}

/* clear an entire sector to zeros */
static	void zero_sector (disk d, int t, int s)
{
	int		i;

	disk_read_sector(d, t, s);
	disk_dirty(d);
	for (i = 0; i < 256; i++)
		d->buffer[i] = 0;
}

static int	i_dir(disk d, char *name)
{
	int 	t;
	int		s;
	int		nt;
	int		ns;
	int		dir;
	int		filetype;
	int		i;
	int		eod;
	int		total;
	int		used;

	disk_read_sector(d, 0x11, 0);
	printf("%s: disk volume %d\n", d->name, d->buffer[0x06]);

	total = TRACKS_PER_DISK * SECTORS_PER_TRACK;
	used = 0;
	for (s = 0; s < SECTORS_PER_TRACK; s++)
		for (t = 0; t < TRACKS_PER_DISK; t++)
			if (!isfree(d, t, s))
				used++;

	t = 0x11;
	s = 0;
	for (eod = 0; !eod;)
	{
		disk_read_sector(d, t, s);
		nt = d->buffer[1];
		ns = d->buffer[2];
		if (!nt && !ns)
			break;

		disk_read_sector(d, nt, ns);
		t = nt;
		s = ns;
		for (dir = 0xb; dir + 35 <= BYTES_PER_SECTOR; dir += 35)
		{
			if (d->buffer[dir] == 0xff)
				continue;
			if (d->buffer[dir] == 0)
				break;
			filetype = d->buffer[dir + 2];
			putchar((filetype & 0x80) ? '*' : ' ');
			switch (filetype & 0x7f) {
			case 0: putchar('T'); break;
			case 1: putchar('I'); break;
			case 2: putchar('A'); break;
			case 4: putchar('B'); break;
			default: putchar('?'); break;
			}
			printf(" %03d ", disk_wordat(d, dir + 0x21));
			for (i = 3; i < 0x20; i++)
				putchar(apptoascii(d->buffer[dir + i]));
			putchar('\n');
		}
	}

	printf("%d sectors, %d used, %d free.\n", total, used, total - used);
	return 0;
}

static int	i_get(disk d, char *name, FILE *fp)
{
	int		ch;
	long	len;
	long	siz;
	afile	af;

	if (!(af = afile_open(d, name, "r")))
	{
		printf("Couldn't open remote file %s\n", name);
		return -2;
	}

	if (text_mode == MODE_TEXT)
	{
		printf("Text mode file type: %c\n", af->filetype);
		for (len = 0; (ch = afile_getc(af)) != EOF; len++)
		{
			if (ch == '\0')
				break;
			ch = apptoascii(ch);
			if (ch == '\r')
				ch = '\n';
			fputc(ch, fp);
		}
	}
	else
	{
		printf("File type: %c\n", af->filetype);
		if (af->filetype == 'B')
		{
			long	start;

			start = afile_getw(af);
			siz = afile_getw(af);
			printf("%s, A$%lx, L$%lx\n", name, start, siz);
		}
		else
		{
			printf("%s is DOS %c file.\n", name, af->filetype);
			siz = 0xffffff;
		}
		for (len = 0; (ch = afile_getc(af)) != EOF && len < siz; len++)
			fputc(ch, fp);
	}
	printf("Read %ld bytes.\n", len);
	if (afile_close(af) == EOF)
		printf("Error closing remote file %s\n", name);
	return 0;
}

static int	i_put(disk d, char *name, FILE *fp)
{
	int 	ch;
	long	len;
	afile	af;

	if (!(af = afile_open(d, name, "w")))
	{
		printf("Couldn't open remote file: %s\n", name);
		return -2;
	}

	if (text_mode = MODE_TEXT)
	{
		for (len = 0; (ch = fgetc(fp)) != EOF; len++)
		{
			if (ch == '\n')
				ch = '\r';
			ch = asciitoapp(ch);
			if (afile_putc(af, ch) == EOF)
			{
				printf("Error writing to remote file: %s\n", name);
				break;
			}
		}
		afile_putc(af, '\0');
	}
	else
	{
		fseek(fp, 0, SEEK_END);
		len = ftell(fp);
		fseek(fp, 0, SEEK_SET);
		if (len > 0xffff)
			len = 0xffff;
		afile_putw(af, 0x2000);					/* A$2000 */
		afile_putw(af, len);						/* L$ */
		printf("Remote file: %s, A$2000, L$%lx\n", name, len);

		for (len = 0; (ch = fgetc(fp)) != EOF; len++)
			if (afile_putc(af, ch) == EOF)
			{
				printf("Error writing to remote file: %s\n", name);
				break;
			}
	}
	printf("Wrote %ld bytes.\n", len);
	if (afile_close(af) == EOF)
		printf("Error closing remote file: %s\n", name);
	return -1;
}

static int	i_init(disk d)
{
	disk_read_sector(d, 0x11, 0);

	printf("Vol:%02x, Trk:%02x,  Sec:%02x\n", d->buffer[6], d->buffer[1], d->buffer[2]);

	if (d->buffer[1] > 35 || d->buffer[2] > 15)
	{
		printf("This is not a Apple DOS 3.3 diskett.\n");
		return -1;
	}
	return 0;
}

static int	i_del(disk d, char *name)
{
	return -1;
}

static int	i_cd(disk d, char *name)
{
	return -1;
}

static int	i_mkdir(disk d, char *name)
{
	return -1;
}

static int	i_rmdir(disk d, char *name)
{
	return -1;
}

static int	i_name(char *name, int maxlen)
{
	for (; *name; name++)
		if (islower(*name))
			*name = toupper(*name);
		else if (*name == '\\')
			*name = '/';
	return 0;
}

static int	i_close(disk d)
{
	return 0;
}

struct _osys	os_dos = { 1, i_init, i_dir, i_get, i_put, i_del, i_cd, i_mkdir, i_rmdir, i_name, i_close };

/*	End of File */
