/*

    a2tools - utilities for transferring data between Unix and Apple II
              DOS 3.3 disk images.

    Copyright (C) 1998 Terry Kyriacopoulos

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Author's e-mail address: terryk@echo-on.net

*/

#include <stdio.h>
#include <sys/file.h>
#include <fcntl.h>
#include <unistd.h>
#include "a2tokens.h"

#define FILETYPE_T 0x00
#define FILETYPE_I 0x01
#define FILETYPE_A 0x02
#define FILETYPE_B 0x04
#define FILETYPE_S 0x08
#define FILETYPE_R 0x10
#define FILETYPE_X 0x20 /* "new A" */
#define FILETYPE_Y 0x40 /* "new B" */

#define MAX_HOPS  560

#define VTOC_CHK_NO 6
const unsigned char vtoc_chk_offset[VTOC_CHK_NO] =
	{ 0x03, 0x27, 0x34, 0x35, 0x36, 0x37};
const unsigned char vtoc_chk_value[VTOC_CHK_NO] =
	{ 0x03, 0x7a, 0x23, 0x10, 0x00, 0x01};

const unsigned char *Applesoft_tokens[] = APPLESOFT_BASIC_TOKENS;
const unsigned char *Integer_tokens[] = INTEGER_BASIC_TOKENS;

FILE *image_fp=0;
unsigned char padded_name[30], dir_entry_data[35];
unsigned char vtocbuffer[256];
unsigned int begun, baseaddress, rawmode, filetype, new_sectors;
unsigned long dir_entry_pos;

quit(exitcode, exitmsg) int exitcode; char *exitmsg; {
    fprintf(stderr,"%s\n",exitmsg);
    if (image_fp) fclose(image_fp);
    exit(exitcode);
    }

seek_sect (track, sector) int track, sector; {
    return fseek(image_fp, (track*16+sector)*256, SEEK_SET);
    }

read_sect (track, sector, buffer) int track, sector;
    unsigned char buffer[256]; {
    int i;
    seek_sect(track, sector);
    for (i=0; i<256; i++) buffer[i]=fgetc(image_fp);
    }

write_sect (track, sector, buffer) int track, sector;
    unsigned char buffer[256]; {
    int i;
    seek_sect(track, sector);
    for (i=0; i<256; i++) fputc(buffer[i],image_fp);
    }

dir_do (what_to_do) int (*what_to_do)(); {
    unsigned char buffer[256];
    unsigned int cur_trk, cur_sec, i, j, found, hop;
    hop=found=0;
    buffer[1]=vtocbuffer[1];
    buffer[2]=vtocbuffer[2];
    while(++hop < MAX_HOPS && !found && (buffer[1] || buffer[2])) {
	cur_trk=buffer[1];
	cur_sec=buffer[2];
	read_sect (buffer[1],buffer[2],buffer);
	i=0x0b;
	while(i<=0xdd && !(found=(*what_to_do)(&buffer[i]))) i+=35;
	if (found) dir_entry_pos=(cur_trk*16+cur_sec)*256+i;
	}
    if (hop >= MAX_HOPS) quit(1,"\n***Corrupted directory\n");
    return found;
    }

dir_find_name(buffer) unsigned char *buffer; {
    int j;
    j=0;
    if (buffer[0] == 0xff || buffer[3] == 0) return 0;
    while(j<30 && padded_name[j]==(buffer[j+3] & 0x7f)) j++;
    if (j != 30) return 0;
    for (j=0; j<35; j++) dir_entry_data[j]=buffer[j];
    return 1;
    }

dir_find_space(buffer) unsigned char *buffer; {
    return (buffer[0] == 0xff || buffer[3] ==0);
    }

dir_print_entry(buffer) unsigned char *buffer; {
    int ftype, j;
    if (buffer[0]!=0xff && buffer[3]!=0) {
	/* entry is present */
	printf(" ");
	if (buffer[2] & 0x80) printf("*"); else printf(" ");
	ftype = (buffer[2] & 0x7f);
	if      (ftype==FILETYPE_T) printf("T");
	else if (ftype==FILETYPE_I) printf("I");
	else if (ftype==FILETYPE_A) printf("A");
	else if (ftype==FILETYPE_B) printf("B");
	else if (ftype==FILETYPE_S) printf("S");
	else if (ftype==FILETYPE_R) printf("R");
	else if (ftype==FILETYPE_X) printf("X");
	else if (ftype==FILETYPE_Y) printf("Y");
	else printf("?");
	printf(" %03d ",buffer[33]+buffer[34]*256);
	for (j=3; j<33; j++)
	    printf("%c",(buffer[j] & 0x7f));
	printf("\n");
	}
    return 0;
    }

preproc (procmode) int procmode; {
    /* procmode: 0 - raw, 1 - text, 2 -binary */
    static unsigned long bytepos, lengthspec_pos;
    static int c;
    unsigned int sect_pos;
    sect_pos=0;
    if (!begun) {
	begun = 1;
	bytepos = 0;
	c=fgetc(stdin);
	if (procmode == 2) {
	    fputc((baseaddress & 0xff),image_fp);
	    fputc((baseaddress >> 8),image_fp);
	    /* we don't know the length now, so save the spot in the image */
	    lengthspec_pos=ftell(image_fp);
	    fputc(0xff,image_fp);
	    fputc(0xff,image_fp);
	    sect_pos = 4;
	    }
	}
    while (c != EOF && sect_pos < 256) {
	if (procmode == 1 && (c |= 0x80) == (0x80|'\n')) c = '\r'|0x80;
	fputc(c,image_fp);
	c=fgetc(stdin);
	sect_pos++;
	bytepos++;
	}
    while (sect_pos++ < 256) fputc(0,image_fp);
    if (c == EOF && procmode == 2) {
	/* now we know the length */
	fseek(image_fp, lengthspec_pos, SEEK_SET);
	fputc((bytepos & 0xff),image_fp);
	fputc((bytepos >> 8),image_fp);
	}
    return (c == EOF);
    }

new_sector(track, sector) int *track, *sector; {
    /* find a free sector, quit if no more */
    int lasttrack, direction, cur_track, cur_sector;
    int byteoffset, bitmask;
    direction=(signed char) vtocbuffer[0x31];
    cur_track=lasttrack=vtocbuffer[0x30];
    cur_sector=15;
    for (;;) {
        byteoffset=0x39+(cur_track<<2)-(cur_sector>>3&1);
	bitmask=(1 <<(cur_sector & 0x07));
	if (vtocbuffer[byteoffset] & bitmask) {
	    vtocbuffer[byteoffset]&=0xff^bitmask;
	    break;
	    }
	else if (!cur_sector--) {
	    cur_sector=15;
	    cur_track+=direction;
	    if (cur_track==35) {
		cur_track=17;
		direction=-1;
		}
	    else if (cur_track==-1) {
		cur_track=18;
		direction=1;
		}
	    if (cur_track==lasttrack) quit(1,"Disk Full.");
	    }
	}
    *track=vtocbuffer[0x30]=cur_track;
    *sector=cur_sector;
    vtocbuffer[0x31]=direction;
    new_sectors++;
    }

free_sector(track, sector) int track, sector; {
    vtocbuffer[0x39+(track<<2)-(sector>>3&1)]|=1<<(sector&0x07);
    }

postproc_B () {
    static unsigned long filelength, bytepos;
    unsigned int sect_pos;
    sect_pos=0;
    if (!begun) {
	begun = 1;
	bytepos = 0;
	fgetc(image_fp); /* Ignore 2 byte base address */
	fgetc(image_fp);
	filelength= fgetc(image_fp) + (fgetc(image_fp) << 8);
	sect_pos = 4;
	}
    while (bytepos < filelength && sect_pos < 256) {
	fputc(fgetc(image_fp),stdout);
	sect_pos++;
	bytepos++;
	}
    }

postproc_A () {
    static unsigned int bufstat, tokens_left, lastspot;
    static unsigned char lineheader[4];
    unsigned int sect_pos, thisspot, c;
    sect_pos=0;
    if (!begun) { /* first sector, initialize */
	begun = 1;
	fgetc(image_fp); /* ignore the length data, we use */
	fgetc(image_fp); /* null line pointer as EOF	   */
	sect_pos = 2;
	lastspot = 0x0801; /* normal absolute beginning address */
	tokens_left = bufstat = 0;
	}
    while(lastspot && sect_pos < 256) {
	if (!tokens_left && !bufstat) bufstat = 4;
	while (bufstat > 0 && sect_pos < 256) {
	    lineheader[4-bufstat]=fgetc(image_fp);
	    sect_pos++;
	    bufstat--;
	    }
	if (!bufstat && !tokens_left && lastspot) {
	    thisspot=lineheader[0]+(lineheader[1] << 8);
	    if (thisspot) {
		tokens_left = 1;
		printf("\n");
		printf(" %d ",lineheader[2]+(lineheader[3] << 8));
		}
	    lastspot=thisspot;
	    }
	while (tokens_left && lastspot && sect_pos < 256) {
	    if ((tokens_left=c=fgetc(image_fp)) & 0x80)
		printf("%s",Applesoft_tokens[(c & 0x7f)]);
	    else {
		if (c) printf("%c",c);
		}
	    sect_pos++;
	    }
	}
    if (!lastspot) printf("\n\n");
    }

postproc_I () {
    static unsigned long filelength, bytepos;
    static unsigned int bufstat, inputmode, quotemode, varmode;
    static unsigned char numbuf[3];
    unsigned int sect_pos, c;
    sect_pos=0;
    if (!begun) { /* first sector, initialize */
	begun = 1;
	filelength = fgetc(image_fp) + (fgetc(image_fp) << 8);
	sect_pos = 2;
	bytepos = inputmode = bufstat = quotemode = varmode = 0;
	}
    /* inputmode: 0 - header, 1 - integer, 2 - tokens */
    /* varmode: 1 means we are in the middle of an identifier */
    while(bytepos < filelength && sect_pos < 256) {
	if (inputmode < 2 && !bufstat) bufstat = 3 - inputmode;
	while (bufstat > 0 && bytepos < filelength && sect_pos < 256) {
	    numbuf[3-bufstat]=fgetc(image_fp);
	    sect_pos++;
	    bytepos++;
	    bufstat--;
	    }
	if (!bufstat && inputmode == 0) {
	    printf("\n");
	    printf("%5d ",numbuf[1]+(numbuf[2] << 8));
	    inputmode = 2;
	    }
	if (!bufstat && inputmode == 1) {
	    printf("%d",numbuf[1]+(numbuf[2] << 8));
	    inputmode = 2;
	    }
	while (inputmode == 2 && bytepos < filelength && sect_pos < 256) {
	    c=fgetc(image_fp);
	    sect_pos++;
	    bytepos++;
	    if (c == 0x28 || c == 0x5d) quotemode = 1;
	    if (c == 0x29) quotemode = 0;
	    if (!quotemode && !varmode && c >= 0xb0 && c <= 0xb9)
		inputmode = 1;
	    else if (c == 0x01) inputmode = quotemode = 0;
	    else {
		varmode = (c >= 0xc1 && c <= 0xda) ||
			  (c >= 0xb0 && c <= 0xb9) && varmode;
		if (c & 0x80) printf("%c",(c & 0x7f));
		else printf("%s",Integer_tokens[c]);
		}
	    }
	}
    if (bytepos >= filelength) printf("\n\n");
    }

postproc_T () {
    static unsigned int not_eof;
    unsigned int sect_pos, c;
    sect_pos=0;
    if (!begun) begun = not_eof = 1;
    while (not_eof && sect_pos < 256 &&
	(not_eof=c=fgetc(image_fp))) {
	c &= 0x7f;
	if (c == '\r') c='\n';
	fputc(c,stdout);
	sect_pos++;
	}
    }

postproc_raw () {
    unsigned int sect_pos;
    for (sect_pos=0; sect_pos < 256; sect_pos++)
	fputc(fgetc(image_fp),stdout);
    }

a2ls () {
    unsigned int trkmap, free_sect, i, j;
    free_sect = 0;

    /* count the free sectors */
    for (i=0x38; i<=0xc0; i+=4) {
	trkmap=(vtocbuffer[i] << 8)+vtocbuffer[i+1];
	for (j=0; j<16; j++) free_sect += ((trkmap & (1<<j))!=0);
	}
    printf("\nDisk Volume %d, Free Blocks: %d\n\n",
	vtocbuffer[0x06],free_sect);	
    dir_do(dir_print_entry);
    printf("\n");
    }

a2rm () {
    unsigned char listbuffer[256];
    unsigned int hop, next_trk, next_sec, i, j;
    if (!dir_do(dir_find_name)) quit(1,"File not found.");
    hop = begun = 0;
    next_trk=dir_entry_data[0];
    next_sec=dir_entry_data[1];
    fseek(image_fp,dir_entry_pos,SEEK_SET);
    fputc(0xff,image_fp); /* mark as deleted */
    while(++hop < MAX_HOPS && (next_trk || next_sec)) {
	free_sector(next_trk, next_sec);
	read_sect(next_trk, next_sec, listbuffer);
	next_trk=listbuffer[1];
	next_sec=listbuffer[2];
	for (i=0x0c; i <=0xfe; i+=2)
	    if (listbuffer[i] || listbuffer[i+1])
		free_sector(listbuffer[i],listbuffer[i+1]);
	}
    if (hop >= MAX_HOPS) quit(1,"Corrupted sector list");
    write_sect(0x11, 0, vtocbuffer);
    }

a2out () {
    unsigned char listbuffer[256];
    unsigned int hop, next_trk, next_sec, i, j;
    int (*postproc_function)();
    if (!dir_do(dir_find_name)) quit(1,"File not found.");
    hop = begun = 0;
    next_trk=dir_entry_data[0];
    next_sec=dir_entry_data[1];
    filetype=(dir_entry_data[2] & 0x7f);
    if (!rawmode && filetype != FILETYPE_T && filetype != FILETYPE_B
        && filetype != FILETYPE_A && filetype != FILETYPE_I)
	quit(1,"File type supported in raw mode only.");

    postproc_function= postproc_raw;
    if (filetype == FILETYPE_T) postproc_function= postproc_T;
    if (filetype == FILETYPE_B) postproc_function= postproc_B;
    if (filetype == FILETYPE_A) postproc_function= postproc_A;
    if (filetype == FILETYPE_I) postproc_function= postproc_I;
    if (rawmode) postproc_function= postproc_raw;

    while(++hop < MAX_HOPS && (next_trk || next_sec)) {
	read_sect(next_trk, next_sec, listbuffer);
	next_trk=listbuffer[1];
	next_sec=listbuffer[2];
	for (i=0x0c; i <= 0xfe; i+=2)
	    if (!listbuffer[i] && !listbuffer[i+1]) {
		if (filetype != FILETYPE_T || !rawmode) {
		    next_trk=next_sec=0;
		    break;
		    }
		else for (j=0; j<256; j++) fputc(0,stdout);
		}
	    else {
		++hop;
		seek_sect(listbuffer[i],listbuffer[i+1]);
		(*postproc_function) ();
		}
	}
    if (hop >= MAX_HOPS) quit(1,"Corrupted sector list");
    }

a2in () {
    unsigned char listbuffer[256], databuffer[256];
    int i, c, curlist_trk, curlist_sec, listentry_pos, list_no;
    int curdata_trk, curdata_sec, procmode;
    int newlist_trk, newlist_sec;
    new_sectors=list_no=procmode=0;
    if (!rawmode) {
	if (filetype==FILETYPE_T) procmode=1;
	else if (filetype==FILETYPE_B) procmode=2;
	else quit(1,"This type is supported only in raw mode.");
	}
    if (dir_do(dir_find_name)) quit(1,"File exists.");
    if (!dir_do(dir_find_space)) quit(1,"No space in directory.");
    for (i=0;i<30;i++) padded_name[i] &= 0x7f;
    if (padded_name[0] < 'A')
	quit(1,"Bad first filename character, must be >= 'A'.");
    for (i=0;i<30;i++)
	if (padded_name[i]==':'||padded_name[i]==',')
	    quit(1,"Filename must not contain a comma or colon.");
    for (i=0;i<30;i++) dir_entry_data[i+3]=padded_name[i]|0x80;
    dir_entry_data[2]=filetype;

    new_sector(&curlist_trk,&curlist_sec);
    dir_entry_data[0]=curlist_trk;
    dir_entry_data[1]=curlist_sec;
    for (i=0;i<256;i++) listbuffer[i]=0;
    listentry_pos=0;

    for (;;) {
        if (!rawmode || filetype!=FILETYPE_T) {
	    new_sector(&curdata_trk,&curdata_sec);
	    listbuffer[0x0c+(listentry_pos<<1)]=curdata_trk;
	    listbuffer[0x0d+(listentry_pos<<1)]=curdata_sec;
	    seek_sect(curdata_trk,curdata_sec);
	    if (preproc(procmode)) break;
	    }
	else {
	    for (i=0;i<256;i++) databuffer[i]=0;
	    i=0;
	    while((c=fgetc(stdin))!=EOF && i<256) databuffer[i++]=c;
	    while(i && !databuffer[i-1]) i--;
	    if (!i) {
		listbuffer[0x0c+(listentry_pos<<1)]=0;
		listbuffer[0x0d+(listentry_pos<<1)]=0;
		}
	    else {
		new_sector(&curdata_trk,&curdata_sec);
		listbuffer[0x0c+(listentry_pos<<1)]=curdata_trk;
		listbuffer[0x0d+(listentry_pos<<1)]=curdata_sec;
		write_sect(curdata_trk,curdata_sec,databuffer);
		}
	    if (c == EOF) break;
	    ungetc(c,stdin);
	    }
	if (++listentry_pos >= 0x7a) {
	    new_sector(&newlist_trk,&newlist_sec);
	    listbuffer[1]=newlist_trk;
	    listbuffer[2]=newlist_sec;
	    write_sect(curlist_trk,curlist_sec,listbuffer);
	    curlist_trk=newlist_trk;
	    curlist_sec=newlist_sec;
	    for (i=0;i<256;i++) listbuffer[i]=0;
	    listentry_pos=0;
	    listbuffer[5]=(++list_no*0x7a) & 0xff;
	    listbuffer[6]=(list_no*0x7a) >> 8;
	    }
	}

    listbuffer[1]=listbuffer[2]=0;
    write_sect(curlist_trk,curlist_sec,listbuffer);
    write_sect(0x11, 0, vtocbuffer);
    dir_entry_data[33]=new_sectors & 0xff;
    dir_entry_data[34]=new_sectors >> 8;
    fseek(image_fp,dir_entry_pos,SEEK_SET);
    /* writing ff first ensures directory is always in a safe state */
    fputc(0xff,image_fp);
    for (i=1;i<35;i++) fputc(dir_entry_data[i],image_fp);
    fseek(image_fp,dir_entry_pos,SEEK_SET);
    fputc(dir_entry_data[0],image_fp);
    }

main (argc, argv) int argc; char *argv[]; {
    unsigned char *image_name, *file_name, *filemode, *basename, *typestr;
    unsigned int found, bad_vtoc;
    unsigned int hop, next_trk, next_sec, i;
    int (*command)();

    baseaddress=0x2000; /* default, hi-res page 1 */
    rawmode = begun = 0;
    file_name = "";

    basename=argv[0];
    basename+=(i=strlen(basename));
    while(i-->0 && *--basename!='/');
    if (*basename=='/') basename++;

    if (!strcmp(basename,"a2ls")) {
	if (argc !=2)
	    quit(1,"Usage: a2ls <disk_image>");
	else {
	    image_name=argv[1];
	    filemode="r";
	    command= a2ls;
	    }
	}
    else if (!strcmp(basename,"a2out")) {
	if (argc > 1 && !strcmp(argv[1],"-r")) rawmode=1;
	if (argc != 3 + rawmode)
	    quit(1,"Usage: a2out [-r] <disk_image> <source_file>");
	else {
	    image_name=argv[1+rawmode];
	    filemode="r";
	    file_name=argv[2+rawmode];
	    command= a2out;
	    }
	}
    else if (!strcmp(basename,"a2in")) {
	if (argc > 1 && !strcmp(argv[1],"-r")) rawmode=1;
	if (argc != 4 + rawmode)
	    quit(1,
	    "Usage: a2in [-r] <type>[.<hex_addr>] <disk_image> <dest_file>");
	else {
	    image_name=argv[2+rawmode];
	    filemode="r+";
	    file_name=argv[3+rawmode];
	    typestr=argv[1+rawmode];
	    if      ((typestr[0]|0x20)=='t') filetype=FILETYPE_T;
	    else if ((typestr[0]|0x20)=='i') filetype=FILETYPE_I;
	    else if ((typestr[0]|0x20)=='a') filetype=FILETYPE_A;
	    else if ((typestr[0]|0x20)=='b') filetype=FILETYPE_B;
	    else if ((typestr[0]|0x20)=='s') filetype=FILETYPE_S;
	    else if ((typestr[0]|0x20)=='r') filetype=FILETYPE_R;
	    else if ((typestr[0]|0x20)=='x') filetype=FILETYPE_X;
	    else if ((typestr[0]|0x20)=='y') filetype=FILETYPE_Y;
	    else quit(1,"<type>: one of t,i,a,b,s,r,x,y without -");
	    if (typestr[1]=='.') {
		if (filetype==FILETYPE_B)
		    sscanf(&typestr[2],"%x",&baseaddress);
		else quit(1,"Base address applicable to type B only.");
		}
	    else if (typestr[1]!=0)
		quit(1,"The only modifier for <type> is .<hex_addr>");
	    command= a2in;
	    }
	}
    else if (!strcmp(basename,"a2rm")) {
	if (argc != 3) quit(1,"Usage: a2rm <disk_image> <file_name>");
	else {
	    image_name=argv[1];
	    filemode="r+";
	    file_name=argv[2];
	    command= a2rm;
	    }
	}
    else
	quit(1,"Invoke as a2ls, a2in, a2out, or a2rm.");

    if ((image_fp=fopen(image_name, filemode))==0 || seek_sect(0, 0)) {
	perror(image_name);
	return 1;
	}
    if (!strcmp(filemode,"r")) flock(fileno(image_fp),LOCK_SH);
    else flock(fileno(image_fp),LOCK_EX);

    /* prepare source filename by padding blanks */
    i=0;
    while(i<30 && file_name[i]) padded_name[i]=file_name[i++];
    while(i<30) padded_name[i++]=' ';

    /* get VTOC and check validity */
    read_sect(0x11, 0, vtocbuffer);
    bad_vtoc=0;
    for (i=0; i<VTOC_CHK_NO; i++)
	bad_vtoc |= (vtocbuffer[vtoc_chk_offset[i]]!=vtoc_chk_value[i]);
    if (bad_vtoc) 
	quit(1,"Not an Apple DOS 3.3 .dsk image.");

    (*command)();
    fclose(image_fp);
    return 0;
    }
