**********************************************************
*                                                        *
*                LOW-LEVEL PACKET FORMAT                 *
*                Revised ST Jun 27, 2005                 *
*                                                        *
* Start of packet:                                       *
*                                                        *
*  --//---+---//---+         +----+    +----+----+-//->  *
*  Locked |  ONE   |  ZERO   |ONE |ZERO|ONE |Bit7|       *
*  or Idle|  31cy  |  16cy   |8cy |8cy |8cy |8cy |       *
*  --//---+        +---------+    +----+    +----+-//->  *
*         |        |         |         |    |            *
*         |      Start     Coarse    Servo  |<- 8 -//->  *
*         |      sync       sync            |  data      *
*         |                                 |  bits      *
*         |                                 | (64cy)     *
*         |<---- Start sequence (71cy) ---->|            *
*                                                        *
*  (Note: data bits are transmitted inverted - 0-bit     *
*         in memory is ONE on wire and vice versa)       *
*                                                        *
* Interbyte separator:                                   *
*                                                        *
*  >-//-+----+----+               +----+----+----+-//->  *
*       |Bit1|Bit0|    ZERO       |ONE |Bit7|Bit6|       *
*       |8cy |8cy |   22-23cy     |8cy |8cy |8cy |       *
*  >-//-+----+----+---------------+    +----+----+-//->  *
*                 |               |    |                 *
*  >-//- 8 data ->|             Servo  |<- 8 data -//->  *
*         bits    |                    |    bits         *
*                 |<--- Interbyte ---->|                 *
*                       separator                        *
*                       (30-31cy)                        *
*                                                        *
* Packet end:                                            *
*                                                        *
*  >-//-+----+----+                                      *
*       |Bit1|Bit0|    ZERO (Idle)                       *
*       |8cy |8cy |                                      *
*  >-//-+----+----+--------------------------------//->  *
*                 |                                      *
*  >-//- End of ->|                                      *
*       checkbyte                                        *
*                                                        *
**********************************************************
         pag
         do    master+crate
**********************************************************
*                                                        *
*                    B C A S T A R B                     *
*                                                        *
*            Michael J. Mahon - Aug 20, 2008             *
*                                                        *
*                   Copyright (c) 2008                   *
*                                                        *
*  Broadcast Arbitrate is the precursor to any broadcast *
*  request.  Since there are no ACKs from receivers, it  *
*  takes steps to ensure that it controls the network    *
*  and all receivers are ready to receive data:          *
*                                                        *
*  1. Arbitrate for and lock the network                 *
*  2. Delay 20ms. for any collisions to resolve and for  *
*     any slow pollers to reach their RCVPKT holds       *
*  3. Set 'sbuf+dst' to 0 for broadcast                  *
*                                                        *
**********************************************************

BCASTARB jsr   ARBTRATE   ; Arbitrate and lock network.
         dlyms 20         ; Let collisions resolve.
         lda   #0         ; Set broadcast
         sta   sbuf+dst   ;  request.
         rts              ;  and return.
         pag
         fin   (master+crate)
         do    1-enhboot  ; (Only if 'not enhboot')
]end     align 256        ; Align to next page.
xmain    equ   *-]end     ; (Timing-critical code)

**********************************************************
*                                                        *
*                    A R B T R A T E                     *
*                                                        *
*             Michael J. Mahon - May 1, 1996             *
*                 Revised Nov 05, 2004                   *
*                                                        *
*                   Copyright (c) 1996                   *
*                                                        *
*  Waits until bus has been idle for 'arbtime' plus      *
*  machine id # * 22 cycles, then locks bus and sends    *
*  the request control packet.                           *
*                                                        *
**********************************************************

ARBTRATE ldx   arbxv      ; Set arbitration wait.
         cmp   zipslow    ; Zip Chip to 1MHz mode.
:waitidl bit   drecv      ; Wait for idle bus.
         bmi   ARBTRATE   ; Restart timing.
         dex
         bne   :waitidl   ; ...not yet.
         sta   dsend+1    ; Got it!  Lock the bus
         rts              ;  and return.
         pag
**********************************************************
*                                                        *
*                     S E N D P K T                      *
*                                                        *
*           Michael J. Mahon - April 15, 1996            *
*            Stephen Thomas - June 27, 2005              *
*                                                        *
*          Copyright (c) 1996, 2003, 2004, 2005          *
*                                                        *
*  Sends (X) bytes (1..256) starting at (A,Y) to the     *
*  currently selected machine(s).                        *
*                                                        *
*  SENDPKT does the following steps:                     *
*     1. Put Zip Chip in 'slow mode' for >38,000 cycles  *
*     2. Send start signal:  31 cyc ONE, 16 cyc ZERO,    *
*        8 cyc ONE, 8 cyc ZERO                           *
*     3. Send (X) data bytes (at 94-95 cyc/byte)         *
*     4. Send one check byte (95 cyc), leaves bus ZERO   *
*     5. Returns with Carry clear.                       *
*                                                        *
*  SENDCTL performs a SENDPKT on the control packet      *
*  send buffer 'sbuf'.                                   *
*                                                        *
*  SENDRSP builds a packet specified by A in 'sbuf'      *
*  for the request in 'rbuf', then sends it.             *
*                                                        *
*  SENDACK builds an ACK packet in 'sbuf' for the        *
*  request in 'rbuf', then sends it.                     *
*                                                        *
*  To obtain maximum sending speed (8 cycles/bit), the   *
*  inner loop of the actual sending code is unrolled     *
*  into a lattice, with two alternative straight-line    *
*  execution paths.  One of these sends an alternating   *
*  sequence of ones and zeroes; the other sends the      *
*  inverse alternating sequence.  Execution is bounced   *
*  from one path to the other depending on the data      *
*  being sent.  Branch-taken delays are compensated for  *
*  by the fact that branches are only necessary when no  *
*  change in bus state is required.                      *
*                                                        *
**********************************************************

SENDACK  lda   #rm_ACK    ; Build an ACK packet
SENDRSP  sta   ckbyte     ; Store modifier for ora
         lda   rbuf+rqmd  ; Get received request
         and   #reqmask   ;  isolate request
         ora   ckbyte     ;   and OR in modifier.
         sta   sbuf+rqmd  ; Set response code.
         lda   rbuf+frm
         sta   sbuf+dst   ; Destination (= requester)
SENDCTL  lda   #<sbuf     ; Control pkt send buffer
         ldy   #>sbuf
         ldx   #lenctl

         do    master     ; (Only if 'master')
SENDPKT  cmp   zipslow    ; Slow Zip Chip for packet.
         else
SENDPKT  cmp   ptrig      ; Trigger paddle timer
         fin              ; (end 'master')
         sta   dsend+1    ; Send start signal ONE
         jsr   :exit      ; Stretch it.
         stx   ckbyte     ; Seed ckbyte with length.
         sty   ptr+1      ; Y = start address hi
         ldy   #0         ; Index first data byte
         clc              ; Ensure C clear at exit
         php              ; Save interrupt state
         sei              ;  and disable interrupts.

* Time-critical region. Timings for :tnn labels and
* t= comments are relative to the preceding timing point
* (start sync or servo).

         sta   dsend+0    ; Send start sync ZERO
         bit   :exit      ; Set V to send ckbyte at end
         sta   ptr        ; A = start address lo
         lda   (ptr),y    ; Get first data (Y=0 so no px)
         sta   dsend+1    ; Send coarse sync at t=16
         nop
         sec              ; Ensure C set between bytes
         sta   dsend+0,y  ; Release coarse sync at t=24
         bcs   :servo     ; Go send servo at t=32

:t84v0   iny              ; Get next data byte and
         lda   (ptr),y    ;  send servo at t=94 or 95

:servo   sta   dsend+1    ; Servo ONE
         rol
         bcc   :t06b7v1
         sta   dsend+0    ; Bit 7 ZERO at t=8
         rol
         bcs   :t14b6v0
         sta   dsend+1    ; Bit 6 ONE at t=16
:t17b6v1 rol
         bcc   :t22b5v1
         sta   dsend+0    ; Bit 5 ZERO at t=24
:t25b5v0 rol
         bcs   :t30b4v0
         sta   dsend+1    ; Bit 4 ONE at t=32
:t33b4v1 rol
         bcc   :t38b3v1
         sta   dsend+0    ; Bit 3 ZERO at t=40
:t41b3v0 rol
         bcs   :t46b2v0
         sta   dsend+1    ; Bit 2 ONE at t=48
:t49b2v1 rol
         bcc   :t54b1v1
         sta   dsend+0    ; Bit 1 ZERO at t=56
:t57b1v0 rol
         bcs   :t62b0v0
         sta   dsend+1    ; Bit 0 ONE at t=64
:t65b0v1 rol              ; Restore data, set C
         nop
         sta   dsend+0    ; Idle/interbyte ZERO at t=72

:t73v0   eor   ckbyte     ; Compute checksum
         sta   ckbyte     ;  and save it.
         dex              ; Count bytes sent
         bne   :t84v0     ; Loop while more to send

:t83v0   bvc   :done      ; Quit if ckbyte already sent;
         inx              ;  else count ckbyte,
         clv              ;  clear send-ckbyte flag,
         bvc   :servo     ; Send ckbyte servo at t=95

:done    plp              ; Restore int state
:exit    rts              ;  and return with C clear.

:t06b7v1 bcc   :t09b7v1   ; These are all for timing
:t14b6v0 bcs   :t17b6v0   ;  equalization and all of
:t22b5v1 bcc   :t25b5v1   ;  them are always taken
:t30b4v0 bcs   :t33b4v0
:t38b3v1 bcc   :t41b3v1
:t46b2v0 bcs   :t49b2v0
:t54b1v1 bcc   :t57b1v1
:t62b0v0 bcs   :t65b0v0

:t14b6v1 bcc   :t17b6v1
:t22b5v0 bcs   :t25b5v0
:t30b4v1 bcc   :t33b4v1
:t38b3v0 bcs   :t41b3v0
:t46b2v1 bcc   :t49b2v1
:t54b1v0 bcs   :t57b1v0
:t62b0v1 bcc   :t65b0v1
:t70v0   bcs   :t73v0

:t09b7v1 rol
         bcc   :t14b6v1
         sta   dsend+0    ; Bit 6 ZERO at t=16
:t17b6v0 rol
         bcs   :t22b5v0
         sta   dsend+1    ; Bit 5 ONE at t=24
:t25b5v1 rol
         bcc   :t30b4v1
         sta   dsend+0    ; Bit 4 ZERO at t=32
:t33b4v0 rol
         bcs   :t38b3v0
         sta   dsend+1    ; Bit 3 ONE at t=40
:t41b3v1 rol
         bcc   :t46b2v1
         sta   dsend+0    ; Bit 2 ZERO at t=48
:t49b2v0 rol
         bcs   :t54b1v0
         sta   dsend+1    ; Bit 1 ONE at t=56
:t57b1v1 rol
         bcc   :t62b0v1
         sta   dsend+0    ; Bit 0 ZERO at t=64
:t65b0v0 rol              ; Restore data, set C
         bcs   :t70v0     ; Always taken
         pag
**********************************************************
*                                                        *
*                   L A S L = > A L                      *
*                                                        *
*                      L A = > A                         *
*                                                        *
**********************************************************

lasl=>al mov16 sbuf+len   ;length ; 'sbuf' length -> length
la=>a    mov16 locaddr    ;address ; Local address -> address
         rts
         pag
]end     align 256        ; Align to next page.
xsend    equ   *-]end     ; (Timing-critical code)
         fin              (end 'not enhboot')

**********************************************************
*                                                        *
*                      R C V P K T                       *
*                                                        *
*           Michael J. Mahon - April 15, 1996            *
*            Stephen Thomas - June 27, 2005              *
*                 Revised May 21, 2008                   *
*                                                        *
*       Copyright (c) 1996, 2003, 2004, 2005, 2008       *
*                                                        *
*  Receives (X) bytes (1..256) starting at (A,Y) from    *
*  the sending machine.                                  *
*                                                        *
*  If no packet is detected within the minimum arb time  *
*  plus 'tolim'-1 times 2.8ms, it returns with carry set *
*  and A = 0.                                            *
*                                                        *
*  If packet is received, but checksum doesn't compare,  *
*  it returns with carry set and A <> 0.                 *
*                                                        *
*  RCVPKT does the following steps:                      *
*     1. Detect 'start signal' ONE                       *
*     2. Put Zip Chip in 'slow mode' for >38,000 cycles  *
*     3. Sync to cycles 5-7 of 8-cycle data cells        *
*     3. Receive (X) bytes (at 93 +3/-0 cycles/byte)     *
*     4. Receive check byte and verify correctness,      *
*        keeping count of checksum errors.               *
*                                                        *
*  RCVCTL performs a RCVPKT to the control packet        *
*  receive buffer 'rbuf'.                                *
*                                                        *
*  RCVPTR performs a RCVPKT to the address in 'ptr' with *
*  length (X).                                           *
*                                                        *
**********************************************************
*                                                        *
*                 Implementation Note                    *
*                                                        *
*  RCVPKT maintains synchronization with the data stream *
*  by using a "digital PLL" technique.  The RCVPKT byte  *
*  loop is 93 cycles, which is 1 or 2 cycles shorter     *
*  than the send loop.  When RCVPKT samples the servo    *
*  transition and finds that it hasn't happened yet, it  *
*  adds a 3-cycle delay to make the total loop time 96   *
*  cycles and restore optimal sync.                      *
*                                                        *
*  The effect is to keep the data sampling window on the *
*  5th to 7th cycle of the 8-cycle data bitcell, in      *
*  spite of the send loop buffer crossing pages at some  *
*  point in a packet and clock frequency differences of  *
*  +/- 1%  between sending and receiving machines.       *
*                                                        *
*  A similar technique assures a well-controlled sample  *
*  position from the first byte of each received packet: *
*                                                        *
*  After the ONE marking the packet start, there's a 16  *
*  cycle ZERO.  Call the time the transmitter begins     *
*  that ZERO t=0.                                        *
*                                                        *
*  The receive loop waits for the ZERO, sampling the     *
*  bus in a tight loop with a 7-cycle period; call the   *
*  time its first ZERO sample occurs rt=0.  Allowing up  *
*  to 4 cycles for pulldown time on the worst network    *
*  bus we can possibly work with, rt=0 could be any time *
*  between t=0 and t=11.                                 *
*                                                        *
*  At t=16, the transmitter will actively drive the bus  *
*  to ONE (a hard-driven transition typically taking     *
*  much less than 1 cycle).  At rt=10, the receive code  *
*  samples the bus once again; if it sees ONE (which it  *
*  will only do if rt=0 occurred between t=6 and t=11)   *
*  it skips a 6-cycle time delay, arriving at rt=19 six  *
*  cycles early.  This makes the rest of the timing work *
*  as if rt=0 had actually fallen between t=0 and t=5    *
*  instead of t=6 and t=11.  Timings referred to rt=0    *
*  now have an uncertainty of only 6 cycles with respect *
*  to t=0 instead of the 11 cycle uncertainty they began *
*  with, and the receiver is in coarse sync.             *
*                                                        *
*  In the most-delayed case, with rt=0 at t=11, the      *
*  rt=10 sample will occur at t=21.  Since the trans-    *
*  mitter does not release the bus until t=24, this is   *
*  safe.                                                 *
*                                                        *
*  At t=32, the transmitter will drive the bus back to   *
*  ONE.  At rt=29, the receive code samples the bus and  *
*  if it sees ONE (which it will only do if rt=0 fell    *
*  between t=3 and t=6) it skips a 3-cycle time delay,   *
*  arriving at rt=36 three cycles early.  This makes the *
*  rest of the timing work as if rt=0 actually happened  *
*  between t=0 and t=3 instead of t=3 and t=6.  Timings  *
*  referred to rt=0 now have an uncertainty of only 3    *
*  cycles with respect to t=0, and the receiver is in    *
*  fine sync.                                            *
*                                                        *
*  The edge at t=32 is actually the servo edge for the   *
*  first byte.  Timings within a data byte are all taken *
*  relative to the servo edge, so t=32 is redefined as   *
*  t=0 and a corresponding adjustment is made to rt;     *
*  the point called rt=36 in the previous paragraph is   *
*  actually labelled :rt04 in the code.                  *
*                                                        *
*  The first data bitcell runs from t=8 to t=16.  The    *
*  receiver samples it at rt=12 - that is, some time     *
*  between t=12 and t=15.  This gives a 4-cycle margin   *
*  at the start of the bitcell and 1 cycle at the end,   *
*  which should be reliable even with truly woeful       *
*  pulldown times.                                       *
*                                                        *
*  Samples for the rest of the data bits are taken at    *
*  8-cycle intervals to match the transmit rate, and the *
*  3-cycle fine sync code is re-used to implement the    *
*  DPLL and make sure the receiver stays in sync for all *
*  subsequent data bytes.                                *
*                                                        *
**********************************************************

RCVCTL   lda   #<rbuf     ; Receive control pkt to 'rbuf'
         ldy   #>rbuf
         ldx   #lenctl
RCVPKT   sta   ptr        ; A = buf address lo
         sty   ptr+1      ; Y = buf address hi
RCVPTR   txa              ; Seed checksum with length
         dex              ; X = length 1..256 (0=>256);
         stx   lastidx    ;  convert to last buffer index

         ldy   tolim      ; Wait <= (tolim-1) * 2.8ms.
         ldx   #arbx      ;  plus minimum arb time.
         php              ; Save interrupt state
         sei              ;  and disable interrupts.
         bit   zipslow    ; Slow any Zip Chip to 1 MHz.

:waitstr bit   drecv      ; Wait for starting ONE.
         bmi   :gotstr
         dex              ; (inner loop is 11 cycles)
         bne   :waitstr   ; Keep waiting...
         dey              ; (outer loop is 2820 cycles)
         bne   :waitstr   ; Loop for 'timeout' ms.

         plp              ; Restore int state
         tya              ; Signal timeout (A=0, Z set)
         sec              ;  and return with C set.
:exit    rts

:gotstr  bit   zipslow    ; Slow Zip Chip for packet.

:waitsyn bit   drecv      ; Wait for 16-cycle sync ZERO;
         bmi   :waitsyn   ;  too bad if bus locks forever!

         ldy   #$FF       ; Index-1 of first data location
         ldx   #$7F       ; CPX #0-7F sets C, 80-FF clears
:synrt07 cpx   drecv      ; Check for coarse sync at rt=10
         bcs   :synrt14   ; Only do delay if still ZERO

:synrt19 bit   :exit      ; Set V (not-ckbyte flag)
         bvs   :servo     ; Do first servo check at rt=29

:synrt14 clc              ; 6-cycle coarse sync delay
         bcc   :synrt19   ; (1 extra to get here, 5 back)

:rt88    clv              ; Clear not-ckbyte flag

:servo   cpx   drecv      ; Check for servo transition
:rt01    bcc   :rt04      ; Delay 3 cyc if past servo,
         nop              ;  6 if not
         nop

:rt04    sta   ckbyte     ; Update checksum
         iny              ; Index next data location

:rt09    cpx   drecv      ; C <-- ~ bit 7 at rt=12
         rol              ; Shift bit 7 in.
         nop
         cpx   drecv      ; C <-- ~ bit 6 at rt=20
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 5 at rt=28
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 4 at rt=36
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 3 at rt=44
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 2 at rt=52
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 1 at rt=60
         rol
         nop
         cpx   drecv      ; C <-- ~ bit 0 at rt=68
:rt69    rol
:rt71    bvc   :rcvdone   ; quit after ckbyte

:rt73    sta   (ptr),y    ; Save data (always 6cy)
:rt79    eor   ckbyte     ; Compute checksum
:rt82    cpy   lastidx    ; Stored last byte?
:rt85    beq   :rt88      ; Go clear not-ckbyte flag if so
:rt87    bne   :servo     ; Do next servo sample at rt=93

:rcvdone eor   ckbyte     ; A = 0 if ckbyte = sum
         beq   :goodck    ; -No error.
         inc16 ckerr      ; Count checksum error.
:goodck  plp              ; Restore int state
         cmp   #1         ; Set C & NZ if checksum bad,
         tax              ;  clear C and set Z if good
         rts              ;   and return.
         pag
**********************************************************
*                                                        *
*                     P R O T E R R                      *
*                                                        *
**********************************************************

PROTERR  inc16 errprot    ; Count protocol error.
         rts


**********************************************************
*                                                        *
*                   R A R L = > A L                      *
*                                                        *
*                      R A = > A                         *
*                                                        *
**********************************************************

rarl=>al mov16 rbuf+len   ;length ; 'rbuf' length  -> length
ra=>a    mov16 rbuf+adr   ;address; 'rbuf' address -> address
         rts
         pag
         do    1-ROMboot  ; (Only if 'not ROMboot')
**********************************************************
*                                                        *
*                    S E N D L O N G                     *
*                                                        *
*             Michael J. Mahon - May 5, 1996             *
*                  Revised May 21, 2008                  *
*                                                        *
*                Copyright (c) 1996, 2008                *
*                                                        *
*  SENDLONG sends 'length' bytes from 'address' to the   *
*  currently selected machine(s).                        *
*                                                        *
*  DSENDLNG delays X*5-1 cycles and falls into SENDLONG. *
*                                                        *
*  It segments a "message" longer than 256 bytes into a  *
*  series of 256-byte packets, plus a final packet       *
*  with the remainder of the data.  Each message packet  *
*  is sent with 'SENDPKT'.                               *
*                                                        *
*  SENDLONG does not detect any errors.                  *
*                                                        *
**********************************************************

DSENDLNG dex              ; Delay 5 * X - 1 cycles
         bne   DSENDLNG   ;  and fall into SENDLONG.
SENDLONG lda   length+1   ; How many 256-byte pages?
         beq   :short     ; - None, just a short pkt.
:loop    ldx   #0         ; Set 256 byte packet.
         lda   address    ;  and point to
         ldy   address+1  ;   data buffer.
         jsr   SENDPKT    ; Send 256 bytes.
         inc   address+1  ; Advance to next page
         dec   length+1   ;  and decrement page
         bne   :loop      ;   count until done...
:short   ldx   length     ; Remaining data length.
         beq   :done      ; -All done.
         lda   address
         ldy   address+1
         jsr   SENDPKT    ; Send the final packet.
:done    rts
         pag
         fin              ; (end 'not ROMboot')
**********************************************************
*                                                        *
*                     R C V L O N G                      *
*                                                        *
*             Michael J. Mahon - May 5, 1996             *
*                  Revised May 21, 2008                  *
*                                                        *
*                Copyright (c) 1996, 2008                *
*                                                        *
*  RCVLONG receives 'length' bytes to 'address' from the *
*  currently sending machine.                            *
*                                                        *
*  It receives a series of packets if 'length' is        *
*  greater than 256 bytes.                               *
*                                                        *
*  RCVLONG detects checksum errors and timeouts, and     *
*  returns with Carry set and A=0 if timeout, and        *
*  Carry set and A>0 if a checksum error.  Timeouts in   *
*  this context are protocol errors.  Both kinds of      *
*  errors are tallyed in counters.                       *
*                                                        *
**********************************************************

RCVLONG  lda   length+1   ; How many 256-byte pages?
         beq   :short     ; - None, just a short pkt.
:loop    ldx   #0         ; Set 256 byte packet.
         lda   address    ;  and point to
         ldy   address+1  ;   data buffer.
         jsr   RCVPKT     ; Receive 256 bytes.
         bcs   :err       ; Receive error detected.
         inc   address+1  ; Advance to next page
         dec   length+1   ;  and decrement page
         bne   :loop      ;   count until done...
:short   ldx   length     ; Remaining data length.
         beq   :done      ; -All done.
         lda   address
         ldy   address+1
         jsr   RCVPKT     ; Receive final packet.
         bcs   :err       ; Keep track of any errors.
:done    rts

:err     bne   :ckerr     ; Checksum error.
         jsr   PROTERR    ; Count protocol error.
:ckerr   tay              ; Set Z flag from A.
         rts
         pag
