/*****************************************************************************
 * Copyright (C) 1998-2007 YAE
 * $Id$
 *
 * Author: Doug Kwan <chun_tak@yahoo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 * scanline32.c
 *
 * draw scanlines in text (40 & 80 columns), lowres, hires and
 * double hires video modes. The current implementation only works
 * with 32-bit pixels.
 *
 */
static char rcsid[]="$Id: scanline32.c,v 1.1 1998/11/29 11:10:44 ctkwan Exp ctkwan $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "memory.h"
#include "video.h"

extern scanline_t	Scanline[192];

static int	ColourMappingTable[16]=
	{0,2,4,6,8,10,12,14,1,3,5,7,9,11,13,15};

static unsigned long NormalTextFontPixel[130][8][7];
static unsigned long InverseTextFontPixel[130][8][7];

/* since alternate character set does not contain flash characters */
/* there is only one set for 40 and 80 column. There are two actual */
/* character sets for the primary one. One has the flash characters */
/* normal and the other the flash characters inverse */

/* 40 column character set, one set if split into two parts. One part */
/* for the even column and the other for the odd column */
static unsigned long *PrimaryCharSet0[256];
static unsigned long *PrimaryCharSet1[256];
static unsigned long *AltCharSet[256];

/* these point to character set currently being used */
static unsigned long **CharSet;

static unsigned long blankQuad;

static unsigned long PhaseShiftedQuad[2048][8];

/* another set of colours with 90 degree phase difference */
static unsigned long PhaseShiftedQuad2[2048][8];

static int BitShiftTable1[256];
static int BitShiftTable2[256];
static int BitShiftTable3[256];
static int BitShiftTable4[256];
 
static int ExpansionTable[256];
static int DLoresBitTable1[16];
static int DLoresBitTable2[16];
static int DLoresBitTable3[16];
static int DLoresBitTable4[16];
 
static int LoresBitTable1[16];
static int LoresBitTable2[16];

extern char *AppleFontBitmap[];
extern char *MouseTextFontBitmap[];

/*
 * Apple II Technical Notes Mouse#6
 *
 * Note:    The pictures of MouseText characters in these manuals differ 
 * from early implementations.  In early MouseText character sets, 
 * the icons mapped to the letters F and G combined to form a 
 * "running man."  In current production, these letters are different 
 * pictures (an inverse carriage return symbol and a window title bar 
 * pattern) which form no picture when placed next to each other.  
 * Programs should not attempt to use the running man MouseText 
 * characters.
 */
extern char *RunningManFontBitmap[];

void c32InitScanlineData( DisplayContext * );
void c32DrawBlankScanline( scanline_t* );
void c32DrawTextScanline( scanline_t* );
void c32DrawLoresScanline( scanline_t* );
void c32DrawHiresScanline( scanline_t* );
void c32DrawText80Scanline( scanline_t* );
void c32DrawDLoresScanline( scanline_t* );
void c32DrawDHiresScanline( scanline_t* );
void c32ChangeCharSet( int );

VideoModeDriver c32VideoModeDriver = {
	c32InitScanlineData,
	c32DrawBlankScanline,
	c32DrawTextScanline,
	c32DrawLoresScanline,
	c32DrawHiresScanline,
	c32DrawText80Scanline,
	c32DrawDLoresScanline,
	c32DrawDHiresScanline,
	c32ChangeCharSet,
};

void c32InitScanlineData( DisplayContext *dc )
{
	int i,j,k,data,bits;
	unsigned char *font_data;
	unsigned long c, c2;
	unsigned long PhaseShiftedColour[16][4];
	unsigned long PhaseShiftedColour2[16][4];

	/* text pixel data initialization */

	memset(NormalTextFontPixel,0,sizeof(NormalTextFontPixel));
	memset(InverseTextFontPixel,0,sizeof(InverseTextFontPixel));

	for(i=0;i<0x82;i++) {	/* create default font data */
	   /* generate 40 column font */
	   if (i<0x60)
	      font_data=AppleFontBitmap[i];
	   else if(i<0x80)
	      font_data=MouseTextFontBitmap[i-0x60];
	   else 
	      font_data=RunningManFontBitmap[i-0x80];

	   for(j=0;j<8;j++) {
	      data=font_data[j];
	      for(k=0;k<7;k++) {
	         if (data&0x01) {
	            c=dc->appleColourPixel[15];
	            c2=dc->appleColourPixel[0];
	         }
	         else {
	            c=dc->appleColourPixel[0];
	            c2=dc->appleColourPixel[15];
	         }
	         NormalTextFontPixel[i][j][k]=c;
	         InverseTextFontPixel[i][j][k]=c2;
	         data>>=1;
	      }
	   }
	}

	/* create primary and alternate character sets */
	for(i=0;i<256;i++){
	   if (i<0x40){ /* $00-$3F inverse characters */
	      PrimaryCharSet0[i]=(void*)InverseTextFontPixel[i];
	      PrimaryCharSet1[i]=(void*)InverseTextFontPixel[i];
	   }
	   else if (i<0x80) { /* $40-$7F flash characters */
	      PrimaryCharSet0[i]=(void*)NormalTextFontPixel[i&0x3f];
	      PrimaryCharSet1[i]=(void*)InverseTextFontPixel[i&0x3f];
	   }
	   else if (i<0xE0) { /* $80-$DF normal characters */
	      PrimaryCharSet0[i]=(void*)NormalTextFontPixel[i&0x3f];
	      PrimaryCharSet1[i]=(void*)NormalTextFontPixel[i&0x3f];
	   }
	   else { /* lower case characters */
	      PrimaryCharSet0[i]=(void*)NormalTextFontPixel[i-0xA0];
	      PrimaryCharSet1[i]=(void*)NormalTextFontPixel[i-0xA0];
	   }
	
	   /* alternate character set */
	   if (i>=0x40&&i<=0x5f) { /* MouseText characters */
	      if (RunningMan&&(i=='F'||i=='G')){ /* running man */
	         AltCharSet[i]=(void*)NormalTextFontPixel[i-'F'+0x80];
	      }
	      else {
	         AltCharSet[i]=(void*)NormalTextFontPixel[i+0x20];
	      }
	   }
	   else if (i>=0x60&&i<=0x7f) { /* inverse lower case */
	      AltCharSet[i]=(void*)InverseTextFontPixel[i-0x20];
	   }
	   else { /* same as primary character set */
	      AltCharSet[i]=PrimaryCharSet0[i];
	   }
	}

	c32ChangeCharSet(0);

	/* hires pixel data initialization */
	for(i=0;i<256;i++){ /* build expansion table */
	   for(k=j=0;j<8;j++)
	      if (i&(1<<j))
	         k|=3<<(j+j);
	   ExpansionTable[i]=k;
	}

	for(i=0;i<16;i++) /* build phase shift table */
	   for(j=0;j<4;j++){
	      k=((i<<j)|(i>>(4-j)))&0x0f;
	      PhaseShiftedColour[i][j]=dc->appleColourPixel[k];
	      PhaseShiftedColour2[i][j]=
	         dc->appleColourPixel[ColourMappingTable[k]];
	   }

	for(i=0;i<2048;i++){
	   bits=i;
	   for(j=0;j<8;j++){
	      PhaseShiftedQuad[i][j]=
	         PhaseShiftedColour[bits&0xf][(j+2)&0x3];
	      PhaseShiftedQuad2[i][j]=
	         PhaseShiftedColour2[bits&0xf][(j+2)&0x3];
	      bits>>=1;
	   }
	}

	for(i=0;i<256;i++){
	   BitShiftTable1[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 1:0);
	   BitShiftTable2[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 7:6);
	   BitShiftTable3[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 5:4);
	   BitShiftTable4[i]=
	      ExpansionTable[i&0x7f]<<((i&0x80)? 3:2);
	}

	/* Lores & Double lowres */
	for(i=0;i<16;i++){
	   j=i|i<<4;
	   j|=j<<8;
	   j|=j<<16;
	   LoresBitTable1[i]=j&0x3fff;
	   LoresBitTable2[i]=j&(0x3fff<<14);
	   DLoresBitTable1[i]=j&0x7f;
	   DLoresBitTable2[i]=j&(0x7f<<7);
	   DLoresBitTable3[i]=j&(0x7f<<14);
	   DLoresBitTable4[i]=j&(0x7f<<21);
	}

	/* blank scanline colour */
	blankQuad = dc->appleColourPixel[0];

}

void c32DrawBlankScanline(scanline_t *s)
{
	int i;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long quad;

	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	for(i=0;i<560;i++)
	   scanline_addr1[i]=scanline_addr2[i]=blankQuad;
}

void c32DrawTextScanline(scanline_t *s)
{
	int cline,col,offset;
	unsigned char *row_addr,c;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *char_pixel, quad;

	cline=s->y&0x07;
	offset=cline*7;
	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	for(col=0;col<40;col++){
	   c=row_addr[col];
	   char_pixel=(void*)(CharSet[c]+offset);
	   scanline_addr1[0]=scanline_addr2[0]=
	      scanline_addr1[1]=scanline_addr2[1]=char_pixel[0];
	   scanline_addr1[2]=scanline_addr2[2]=
	      scanline_addr1[3]=scanline_addr2[3]=char_pixel[1];
	   scanline_addr1[4]=scanline_addr2[4]=
	      scanline_addr1[5]=scanline_addr2[5]=char_pixel[2];
	   scanline_addr1[6]=scanline_addr2[6]=
	      scanline_addr1[7]=scanline_addr2[7]=char_pixel[3];
	   scanline_addr1[8]=scanline_addr2[8]=
	      scanline_addr1[9]=scanline_addr2[9]=char_pixel[4];
	   scanline_addr1[10]=scanline_addr2[10]=
	      scanline_addr1[11]=scanline_addr2[11]=char_pixel[5];
	   scanline_addr1[12]=scanline_addr2[12]=
	      scanline_addr1[13]=scanline_addr2[13]=char_pixel[6];
	   scanline_addr1 += 14;
	   scanline_addr2 += 14;
	}
}

void c32DrawLoresScanline(scanline_t *s)
{
	int col,i,bits1, bits2;
	unsigned char *row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2, *scanline_addr3;
	unsigned long *pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	/* optimization */
	if (s->y&3) {
	   i=s->y&~3;
	   if (Scanline[i].mode==s->mode && Scanline[i].addr==s->addr &&
	      Scanline[i].state!=SCANLINE_STATE_DIRTY){
	      scanline_addr3=Scanline[i].scanline_addr1;
	      for( i = 0; i < 560 ; i++ )
	         scanline_addr1[i]=scanline_addr2[i]=scanline_addr3[i];
	      return;
	   }
	}

	bp=buf;
	for(col=0;col<40;col+=4){
	   if (s->y&0x4){ /* higher nibble */
	      bits1=LoresBitTable1[row_addr[0]>>4]|
	         LoresBitTable2[row_addr[1]>>4];
	      bits2=LoresBitTable1[row_addr[2]>>4]|
	         LoresBitTable2[row_addr[3]>>4];
	   }
	   else {
	      bits1=LoresBitTable1[row_addr[0]&0xf]|
	         LoresBitTable2[row_addr[1]&0xf];
	      bits2=LoresBitTable1[row_addr[2]&0xf]|
	         LoresBitTable2[row_addr[3]&0xf];
	   }
	   bp[0]=bits1;
	   bp[1]=bits1>>8;
	   bp[2]=bits1>>16;
	   bits1=(bits2<<4)|(bits1>>24);
	   bp[3]=bits1;
	   bp[4]=bits1>>8;
	   bp[5]=bits1>>16;
	   bp[6]=bits1>>24;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits1=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits1|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits1&0x7ff];
	   scanline_addr1[0]=scanline_addr2[0]=pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=pixel[2];
	   scanline_addr1[3]=scanline_addr2[3]=pixel[3];
	   scanline_addr1[4]=scanline_addr2[4]=pixel[4];
	   scanline_addr1[5]=scanline_addr2[5]=pixel[5];
	   scanline_addr1[6]=scanline_addr2[6]=pixel[6];
	   scanline_addr1[7]=scanline_addr2[7]=pixel[7];
	   bits1>>=8;
	   scanline_addr1+=8;
	   scanline_addr2+=8;
	}
}

void c32DrawHiresScanline(scanline_t *s)
{
	int col,i,bits;
	unsigned char *row_addr,c1,buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	bits=0;
	for(col=0;col<40;col+=4){
	   c1=row_addr[col];
	   bits|=BitShiftTable1[c1];
	   bp[0]=bits;
	   bits>>=8;
	   c1=row_addr[col+1];
	   bits|=BitShiftTable2[c1];
	   bp[1]=bits;
	   bits>>=8;
	   bp[2]=bits;
	   bits>>=8;
	   c1=row_addr[col+2];
	   bits|=BitShiftTable3[c1];
	   bp[3]=bits;
	   bits>>=8;
	   bp[4]=bits;
	   bits>>=8;
	   c1=row_addr[col+3];
	   bits|=BitShiftTable4[c1];
	   bp[5]=bits;
	   bits>>=8;
	   bp[6]=bits;
	   bits>>=8;
	   bp+=7;
	}
	buf[70]=0;

	bits=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits&0x7ff];
	   scanline_addr1[0]=scanline_addr2[0]=pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=pixel[2];
	   scanline_addr1[3]=scanline_addr2[3]=pixel[3];
	   scanline_addr1[4]=scanline_addr2[4]=pixel[4];
	   scanline_addr1[5]=scanline_addr2[5]=pixel[5];
	   scanline_addr1[6]=scanline_addr2[6]=pixel[6];
	   scanline_addr1[7]=scanline_addr2[7]=pixel[7];
	   bits>>=8;
	   scanline_addr1+=8;
	   scanline_addr2+=8;
	}
}

void c32DrawText80Scanline(scanline_t *s)
{
	int cline,col,offset;
	unsigned char *row_addr, *aux_row_addr,c;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *char_pixel, quad;

	cline=s->y&0x07;
	offset=cline*7;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	for(col=0;col<40;col++){
	   c=aux_row_addr[col];
	   char_pixel=CharSet[c]+offset;
	   scanline_addr1[0]=scanline_addr2[0]=char_pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=char_pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=char_pixel[2];
	   scanline_addr1[3]=scanline_addr2[3]=char_pixel[3];
	   scanline_addr1[4]=scanline_addr2[4]=char_pixel[4];
	   scanline_addr1[5]=scanline_addr2[5]=char_pixel[5];
	   scanline_addr1[6]=scanline_addr2[6]=char_pixel[6];
	   c=row_addr[col];
	   char_pixel=CharSet[c]+offset;
	   scanline_addr1[7]=scanline_addr2[7]=char_pixel[0];
	   scanline_addr1[8]=scanline_addr2[8]=char_pixel[1];
	   scanline_addr1[9]=scanline_addr2[9]=char_pixel[2];
	   scanline_addr1[10]=scanline_addr2[10]=char_pixel[3];
	   scanline_addr1[11]=scanline_addr2[11]=char_pixel[4];
	   scanline_addr1[12]=scanline_addr2[12]=char_pixel[5];
	   scanline_addr1[13]=scanline_addr2[13]=char_pixel[6];
	   scanline_addr1 += 14;
	   scanline_addr2 += 14;
	}
}

void c32DrawDLoresScanline(scanline_t *s)
{
	int col,i,bits1,bits2;
	unsigned char *row_addr, *aux_row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2, *scanline_addr3;
	unsigned long *pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	/* optimization */
	if (s->y&3) {
	   i=s->y&~3;
	   if (Scanline[i].mode==s->mode && Scanline[i].addr==s->addr &&
	      Scanline[i].state!=SCANLINE_STATE_DIRTY){
	      scanline_addr3=Scanline[i].scanline_addr1;
	      for(i=0;i<560;i++)
	         scanline_addr1[i]=scanline_addr2[i]=scanline_addr3[i];
	      return;
	   }
	}

	bp=buf;
	for(col=0;col<40;col+=4){
	   if (s->y&0x4){ /* higher nibble */
	      bits1=DLoresBitTable1[aux_row_addr[0]>>4]|
	         DLoresBitTable2[row_addr[0]>>4]|
	         DLoresBitTable3[aux_row_addr[1]>>4]|
	         DLoresBitTable4[row_addr[1]>>4];
	      bits2=DLoresBitTable1[aux_row_addr[2]>>4]|
	         DLoresBitTable2[row_addr[2]>>4]|
	         DLoresBitTable3[aux_row_addr[3]>>4]|
	         DLoresBitTable4[row_addr[3]>>4];
	   }
	   else {
	      bits1=DLoresBitTable1[aux_row_addr[0]&0xf]|
	         DLoresBitTable2[row_addr[0]&0xf]|
	         DLoresBitTable3[aux_row_addr[1]&0xf]|
	         DLoresBitTable4[row_addr[1]&0xf];
	      bits2=DLoresBitTable1[aux_row_addr[2]&0xf]|
	         DLoresBitTable2[row_addr[2]>>4]|
	         DLoresBitTable3[aux_row_addr[3]>>4]|
	         DLoresBitTable4[row_addr[3]>>4];
	   }
	   bp[0]=bits1;
	   bp[1]=bits1>>8;
	   bp[2]=bits1>>16;
	   bits1=(bits2<<4)|(bits1>>24);
	   bp[3]=bits1;
	   bp[4]=bits1>>8;
	   bp[5]=bits1>>16;
	   bp[6]=bits1>>24;
	   aux_row_addr+=4;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits1=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits1|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad[bits1&0x7ff];
	   scanline_addr1[0]=scanline_addr2[0]=pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=pixel[2];
	   scanline_addr1[3]=scanline_addr2[3]=pixel[3];
	   scanline_addr1[4]=scanline_addr2[4]=pixel[4];
	   scanline_addr1[5]=scanline_addr2[5]=pixel[5];
	   scanline_addr1[6]=scanline_addr2[6]=pixel[6];
	   scanline_addr1[7]=scanline_addr2[7]=pixel[7];
	   bits1>>=8;
	   scanline_addr1+=8;
	   scanline_addr2+=8;
	}
}

void c32DrawDHiresScanline(scanline_t *s)
{
	int col,i,bits;
	unsigned char *row_addr, *aux_row_addr, buf[71], *bp;
	unsigned long *scanline_addr1, *scanline_addr2;
	unsigned long *pixel;

	row_addr=(BYTE*)MainMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	aux_row_addr=(BYTE*)AuxMemoryRAM[(s->addr)>>8]+(s->addr &0xff);
	scanline_addr1=s->scanline_addr1;
	scanline_addr2=s->scanline_addr2;

	bp=buf;
	for(col=0;col<40;col+=4){
	   bits=(aux_row_addr[0]&0x7f)|((row_addr[0]&0x7f)<<7);
	   bp[0]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[1]&0x7f)<<6)|((row_addr[1]&0x7f)<<13);
	   bp[1]=bits;
	   bits>>=8;
	   bp[2]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[2]&0x7f)<<4)|((row_addr[2]&0x7f)<<11);
	   bp[3]=bits;
	   bits>>=8;
	   bp[4]=bits;
	   bits>>=8;
	   bits|=((aux_row_addr[3]&0x7f)<<2)|((row_addr[3]&0x7f)<<9);
	   bp[5]=bits;
	   bits>>=8;
	   bp[6]=bits;
	   aux_row_addr+=4;
	   row_addr+=4;
	   bp+=7;
	}
	buf[70]=0;

	bits=buf[0]<<2;
	for(i=0;i<70;i++) {
	   bits|=(buf[i+1]<<10);
	   pixel=PhaseShiftedQuad2[bits&0x7ff];
	   scanline_addr1[0]=scanline_addr2[0]=pixel[0];
	   scanline_addr1[1]=scanline_addr2[1]=pixel[1];
	   scanline_addr1[2]=scanline_addr2[2]=pixel[2];
	   scanline_addr1[3]=scanline_addr2[3]=pixel[3];
	   scanline_addr1[4]=scanline_addr2[4]=pixel[4];
	   scanline_addr1[5]=scanline_addr2[5]=pixel[5];
	   scanline_addr1[6]=scanline_addr2[6]=pixel[6];
	   scanline_addr1[7]=scanline_addr2[7]=pixel[7];
	   bits>>=8;
	   scanline_addr1+=8;
	   scanline_addr2+=8;
	}
}

void c32ChangeCharSet(int charset)
{
	switch(charset){
	case 0: /* Primary Character Set, flash text normal */ 
	   CharSet=PrimaryCharSet0;
	   break;
	case 1: /* Primary Character Set, flash text inverse */ 
	   CharSet=PrimaryCharSet1;
	   break;
	case 2: /* Alt Character Set */ 
	   CharSet=AltCharSet;
	   break;
	}
}

