/*****************************************************************************
 * Copyright (C) 1997-2007 YAE
 * $Id$
 *
 * Author: Doug Kwan <chun_tak@yahoo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * paddle.c
 *
 * paddle support for apple II emulator
 * original idea due to Peter Koch 
 *
 */
#include "hardware.h"
#include "x_window.h"
#include "yae.h"

static char rcsid[]="$Id: paddle.c,v 1.6 1998/11/29 10:13:31 ctkwan Exp ctkwan $";

/*
 * paddle down time
 * when the paddles are triggered by accessing address $C07X
 * the paddles down time are calculated by adding current AppleClock
 * to a value depending on the pseudo paddle values, e.g. mouse coordinates.
 * Note that paddles can "triggered" wrongly because of wrapping of the
 * AppleClock. However, nobody should read the paddles with strobing $C07X 
 * right? :)
 */
static	APPLE_TICK	Paddle0Down=0;
static	APPLE_TICK	Paddle1Down=0;
static	APPLE_TICK	Paddle2Down=0;
static	APPLE_TICK	Paddle3Down=0;

/*
 * Apple Keys on a IIe keyboard
 */
static	BOOLEAN		LeftAppleKeyPressed=FALSE;
static	BOOLEAN		RightAppleKeyPressed=FALSE;

#define	LEFT_APPLE_KEY	1
#define	RIGHT_APPLE_KEY	2

void press_apple_key( int key )
{
	if ( key == LEFT_APPLE_KEY )
	   LeftAppleKeyPressed = TRUE;
	else if ( key == RIGHT_APPLE_KEY )
	   RightAppleKeyPressed = TRUE;
}

void release_apple_key( int key )
{
	if ( key == LEFT_APPLE_KEY )
	   LeftAppleKeyPressed = FALSE;
	else if ( key == RIGHT_APPLE_KEY )
	   RightAppleKeyPressed = FALSE;
}

/*
 * Paddle[01]DownTime are calculated when paddles are triggered. These
 * values depend on present mouse position relative to the top left
 * corner of the Apple II winow. Additional devices, e.g. a track ball
 * can be linked to paddles 3 & 4 in this way.
 */
void trigger_paddle(void)
{
	int		x, y;

	/*
	 * ask window system for mouse position relative to
	 * the Apple window
	 */
	if ( !getmouse( appleDisplayContext, &x , &y ) ) { 
	   /* mouse not on same screen */
	   x = 0;	/* arbitratry */
	   y = 0;
	}
	else {
	   /* restrict x & y in reasonable ranges */ 
	   x = x < 0 ? 0 : x > 511 ? 511 : x;
	   y = y < 0 ? 0 : y > 511 ? 511 : y;
	}

	/*
	 * additional devices can be linked to paddles 3 and 4 here.
	 */

	/*
	 * paddle value equals to mouse positions
	 * Magic formula, see $FB1E-$FB2E in monitor ROM
	 * We calculate the numbers of cycles after which
	 * the RC circuit of a triggered paddle will discharge.
	 */
	Paddle0Down = AppleClock + ( x * 11 ) / 2 + 8;
	Paddle1Down = AppleClock + ( y * 11 ) / 2 + 8;
	Paddle2Down = AppleClock;
	Paddle3Down = AppleClock;
}

/*
 * read_paddle: return $80 if a paddle is "up" or a button is pressed
 */
unsigned read_paddle( ADDR address )
{
	switch( address & 0x0F ) {
	case 0x00: /* Cassette input */
	   return 0x00;

	case 0x01:	/* paddle button 0 */
	   return ( getMouseButton(0) || LeftAppleKeyPressed ) ? 0x80 : 0x00;

	case 0x02:	/* paddle button 1 */
	   return ( getMouseButton(1) || RightAppleKeyPressed ) ? 0x80 : 0x00;

	case 0x03:	/* paddle button 2 */
	   return ( getMouseButton(2) ) ? 0x80 : 0x00;

	case 0x04: /* paddle 0 */
	   return ( TICK_LESS( AppleClock, Paddle0Down ) ) ? 0x80 : 0x00;

	case 0x05: /* paddle 1 */
	   return ( TICK_LESS( AppleClock, Paddle1Down ) ) ? 0x80 : 0x00;

	case 0x06: /* paddle 2 */
	   return ( TICK_LESS( AppleClock, Paddle2Down ) ) ? 0x80 : 0x00;

	case 0x07: /* paddle 3 */
	   return ( TICK_LESS( AppleClock, Paddle3Down ) ) ? 0x80 : 0x00;

	default:
	   return 0x00; /* arbitrary */
	}
}
