/*****************************************************************************
 * Copyright (C) 1997-2007 YAE
 * $Id$
 *
 * Author: Doug Kwan <chun_tak@yahoo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 * mouse.c
 *
 * Apple IIe mouse card emulation
 *
 * Apple II Mouse information based on:
 * "Accessing the Apple ][ Mouse" by David Empson
 * http://www.visi.com/~nathan/a2/faq/mouse.html
 */
static char rcsid[]="$Id: mouse.c,v 1.7 1997/08/12 13:40:29 ctkwan Exp ctkwan $";

#include "hardware.h"
#include "memory.h"
#include "yae.h"
#include "x_window.h"
#include "6502.h"

#define	MOUSE_ON		(1<<0)
#define	MOUSE_MOVE_INT		(1<<1)
#define	MOUSE_BTTN_INT		(1<<2)
#define	MOUSE_SCRN_INT		(1<<3)
/* bit 4 is reserved */
#define	MOUSE_XY_CHANGED	(1<<5)
#define	MOUSE_BTTN_WAS_DOWN	(1<<6)
#define	MOUSE_BTTN_DOWN		(1<<7)

#define	X_LOW		0x478
#define	Y_LOW		0x4F8
#define	X_HIGH		0x578
#define	Y_HIGH		0x5F8
#define	MOUSE_STATUS	0x778
#define	MOUSE_MODE	0x7F8

/* parameter for the ClampMouse */
#define	LOWER_LIMIT_LOW		0x478
#define	UPPER_LIMIT_LOW		0x4F8
#define	LOWER_LIMIT_HIGH	0x578
#define	UPPER_LIMIT_HIGH	0x5F8

#define DEFAULT_X_LOWER_LIMIT   0
#define DEFAULT_Y_LOWER_LIMIT   0
#define DEFAULT_X_UPPER_LIMIT   1023
#define DEFAULT_Y_UPPER_LIMIT   1023

/*
 * Official indentification: pos (val)
 * 0c (20) fb (d6)
 * According to David Empson
 * 05 (38) 07 (18) 0b (01) 0c (20) fb (db)
 * Dazzle draw recognized the card with the following
 * 02 (38) 05 (78) 06 (2c)
 * 
 */
static BYTE MouseCardROM[256]={
	0x60, 0x00, 0x38, 0x00, 0x00, 0x38, 0x2c, 0x18,
	0x00, 0x00, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x2a, 0x2e, 0x32, 0x36, 0x3a, 0x3e,
	0x42, 0x46, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xa0, 0x00, 0xf0, 0x1a, 0xa0, 0x01,
	0xd0, 0x16, 0xa0, 0x02, 0xd0, 0x12, 0xa0, 0x03,
	0xf0, 0x0e, 0xa0, 0x04, 0xd0, 0x0a, 0xa0, 0x05,
	0xd0, 0x06, 0xa0, 0x06, 0xd0, 0x02, 0xa0, 0x07,
	0x08, 0x78, 0x48, 0xa6, 0x00, 0xa9, 0x60, 0x85,
	0x00, 0x20, 0x00, 0x00, 0x86, 0x00, 0xba, 0xbd,
	0x00, 0x01, 0x0a, 0x0a, 0x0a, 0x0a, 0xaa, 0x68,
	0x28, 0x9d, 0x80, 0xc0, 0x60, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xD6, 0x00, 0x00, 0x00, 0x00 };

typedef struct {
	int	slot;
	int	x;
	int	y;
	int	real_x;
	int	real_y;
	int	origin_x;
	int	origin_y;
	int	mode;
	int	status;
	int	x_lower_limit;
	int	y_lower_limit;
	int	x_upper_limit;
	int	y_upper_limit;
	double	x_scale;
	double	y_scale;
} Mouse;

Mouse DefaultMouse;
Mouse *MouseList[8];

/* function prototypes */

static BYTE mouseCardRead(ADDR);
static void mouseCardWrite(ADDR,BYTE);
static void mouseCardEmuTrapHandler(AppleCPUContextStruct*, void*);
static void mouseVBLCallback(void);
static void doSetMouse(AppleCPUContextStruct*, Mouse*);
static void doServeMouse(AppleCPUContextStruct*, Mouse*);
static void doReadMouse(AppleCPUContextStruct*, Mouse*);
static void doClearMouse(AppleCPUContextStruct*, Mouse*);
static void doPosMouse(AppleCPUContextStruct*, Mouse*);
static void doClampMouse(AppleCPUContextStruct*, Mouse*);
static void doHomeMouse(AppleCPUContextStruct*, Mouse*);
static void doInitMouse(AppleCPUContextStruct*, Mouse*);

static void (*MouseFunctionTable[8])(AppleCPUContextStruct*,Mouse*)={
	doSetMouse, doServeMouse, doReadMouse, doClearMouse,
	doPosMouse, doClampMouse, doHomeMouse, doInitMouse	
};

static void EmuTrapHandler2(AppleCPUContextStruct *context, void *arg)
{
	printf("PC=%04X A=%02X X=%02X Y=%02X S=%02X P=%02X\n", context->PC,
	   context->A,context->X,context->Y,context->S,context->P);
}

void initMouseCard(void)
{
	/* II+ firmware cannot handle interrupt, hence no mouse */
	if (MachineType >= APPLE_IIE) {
	   expansion_slot_read[4]=mouseCardRead;
	   expansion_slot_write[4]=mouseCardWrite;
	   MouseList[4]=&DefaultMouse;
	   DefaultMouse.slot=4;
	   addVBLCallback(mouseVBLCallback);
	}
}

static BYTE mouseCardRead(ADDR address)
{
	if (address<0xC100)
	   return 0;
	else{
#if 0
	   RaiseEmulationException(EmuTrapHandler2,NULL);
#endif
	   return MouseCardROM[address&0xff];
	}
}

static void mouseCardWrite(ADDR address,BYTE data)
{
	int slot;
	Mouse *m;

#if 0 && defined(DEBUG)
	printf("mouseCardWrite(%04X, %02X)\n",address,data);
#endif
	if (address<0xC100){
	   slot=(address>>4)&7;
	   switch(address & 0xf){
	   case 0:
	      m=MouseList[slot];
	      if (m)
	         RaiseEmulationException(mouseCardEmuTrapHandler,m);
	   }
	}
}

static void mouseVBLCallback(void)
{
	int i, bit, old_status, x, y;

	for(i=1;i<8;i++){
	   if (MouseList[i]&& MouseList[i]->mode&MOUSE_ON){
	      old_status=MouseList[i]->status;
#if 0
	      x=Apple_memory_read(X_LOW+i)+(Apple_memory_read(X_HIGH+i)<<8);
	      y=Apple_memory_read(Y_LOW+i)+(Apple_memory_read(Y_HIGH+i)<<8);
	      printf("x=%d y=%d\n",x,y);
#endif
	      /* check movement interrupt */
	      if ( (MouseList[i]->mode & MOUSE_MOVE_INT) &&
	         !(MouseList[i]->status&MOUSE_MOVE_INT)){
	         getmouse( appleDisplayContext, &x,&y);

	         x=x*MouseList[i]->x_lower_limit+MouseList[i]->x_scale;
	         y=y*MouseList[i]->y_lower_limit+MouseList[i]->y_scale;

	         /* clip against limits */
	         if (x<MouseList[i]->x_lower_limit)
	            x=MouseList[i]->x_lower_limit;
	         if (y<MouseList[i]->y_lower_limit)
	            y=MouseList[i]->y_lower_limit;
	         if (x>MouseList[i]->x_upper_limit)
	            x=MouseList[i]->x_upper_limit;
	         if (y>MouseList[i]->y_upper_limit)
	            y=MouseList[i]->y_upper_limit;

	         if (x!=MouseList[i]->x||y!=MouseList[i]->y)
	            MouseList[i]->status |= MOUSE_MOVE_INT;
	      }

	      /* check button interrupt */
	      if ( (MouseList[i]->mode & MOUSE_BTTN_INT) && 
	         !(MouseList[i]->status&MOUSE_BTTN_INT)){
	         bit=getMouseButton(0)?MOUSE_BTTN_DOWN:0;
	         if ((MouseList[i]->mode^bit)&MOUSE_BTTN_DOWN)
	            MouseList[i]->status |= MOUSE_BTTN_INT;
	      }

	      /* check VBL interrupt */
	      if ((MouseList[i]->mode & MOUSE_SCRN_INT) &&
	         !(MouseList[i]->status&MOUSE_SCRN_INT)){
	         MouseList[i]->status |= MOUSE_SCRN_INT;
	      }

	      if (MouseList[i]->status != old_status) {
#ifdef DEBUG
	         printf("mouse interrupt at %u\n", AppleClock);
#endif
	         assert_IRQ();
	      }
#ifdef DEBUG
	      printf("mouseVBLCallBack():slot %d mode %02X status %02X\n",
	         i, MouseList[i]->mode,MouseList[i]->status);
#endif
	   }
	}
}

static void mouseCardEmuTrapHandler(AppleCPUContextStruct *context, void *arg)
{
	Mouse *m;
#if 0
	printf("PC=%04X A=%02X X=%02X Y=%02X S=%02X P=%02X\n", context->PC,
	   context->A,context->X,context->Y,context->S,context->P);
#endif

	m=arg;

#if 0
	/* do this trick only in the mouse card ROM */
	if (!arg || (context->PC>>8)!=((m->slot<<8)+0xC0));
	   return;
#endif
	
	/* dispatch function by Y */
	if (context->Y<8)
	   (*MouseFunctionTable[context->Y])(context,m);
	else
	   fprintf(stderr,"mouseCardEmuTrapHandler:unknown command %02X\n",
	      context->Y);
}

/*
 * doSetMouse
 *
 *        Sets mouse operation mode.
 *
 * Entry: A = mouse operation mode ($00 to $0F) - see mode byte above.
 *
 * Exit:  C = 1 if illegal mode entered.
 *        Screen hole mode byte is updated.
 */
static void doSetMouse(AppleCPUContextStruct *context, Mouse *m)
{

#ifdef DEBUG
	printf("SetMouse to mode %X\n",context->A);
#endif
	if (context->A<16){
	   m->mode=context->A;
	   Apple_memory_write(MOUSE_STATUS+m->slot,m->mode);
	   /* copy on/off to status */
	   if ((m->status^m->mode)&MOUSE_ON) {
	      if (m->mode&MOUSE_ON){
	         turnOffCursor( appleDisplayContext );
	         m->status|=MOUSE_ON;
	      }
	      else {
	         turnOnCursor( appleDisplayContext );
	         m->status&=~MOUSE_ON;
	      }
	   }
	   context->P &= ~C_BIT;
	}
	else /* illegal mode */
	   context->P |= C_BIT;
}

/*
 * doServeMouse
 *
 *        Tests for interrupt from mouse and resets mouse's interrupt
 *        line.
 *
 * Exit:  C = 0 if mouse interrupt occurred.
 *        Screen hole interrupt status bits are updated to show current
 *        status. 
 */
static void doServeMouse(AppleCPUContextStruct *context, Mouse *m)
{
	int irq;

#ifdef DEBUG
	printf("ServeMouse\n");
#endif

	irq=m->status & (MOUSE_MOVE_INT|MOUSE_BTTN_INT|MOUSE_SCRN_INT);
	m->status &= ~irq;	/* clear status */
	Apple_memory_write(MOUSE_STATUS+m->slot,m->status);

	/* set C bit according to interrupt status */
	if (irq)
	   context->P &= ~C_BIT;
	else
	   context->P |= C_BIT;
}

/*
 * doReadMouse
 *
 *        Reads delta (X/Y) positions, updates absolute X/Y positions,
 *        and reads button statuses from the mouse.
 *
 *  Exit:  C = 0 (always).
 *         Screen hole positions and button/movement status bits are
 *         updated, interrupt status bits are cleared.
 */
static void doReadMouse(AppleCPUContextStruct *context, Mouse *m)
{
	int	x, y, old_status;

	getmouse( appleDisplayContext, &m->real_x,&m->real_y);

	x=m->real_x;
	y=m->real_y;

	x=x*m->x_scale+m->x_lower_limit;
	y=y*m->y_scale+m->y_lower_limit;

	/* clip accorinding to limits */
	if (x<m->x_lower_limit)
	   x=m->x_lower_limit;
	if (y<m->y_lower_limit)
	   y=m->y_lower_limit;
	if (x>m->x_upper_limit)
	   x=m->x_upper_limit;
	if (y>m->y_upper_limit)
	   y=m->y_upper_limit;

#ifdef DEBUG
	printf("ReadMouse X=%d Y=%d\n",x,y);
#endif

	old_status=m->status;
	
	/* clear status bits */
	m->status &= ~(MOUSE_XY_CHANGED|MOUSE_BTTN_DOWN|MOUSE_BTTN_WAS_DOWN);
	if (x!=m->x||y!=m->y)
	   m->status |= MOUSE_XY_CHANGED;
	m->x=x;
	m->y=y;

	/* update button status */
	if (old_status&MOUSE_BTTN_DOWN)
	   m->status|=MOUSE_BTTN_WAS_DOWN;	
	if (getMouseButton(0))
	   m->status|=MOUSE_BTTN_DOWN;

	/* clear interrupt status */
	m->status &= ~(MOUSE_MOVE_INT|MOUSE_BTTN_INT|MOUSE_SCRN_INT);

	/* update position and status */
	Apple_memory_write(MOUSE_STATUS+m->slot,m->status);
	Apple_memory_write(X_LOW+m->slot,m->x);
	Apple_memory_write(X_HIGH+m->slot,m->x>>8);
	Apple_memory_write(Y_LOW+m->slot,m->y);
	Apple_memory_write(Y_HIGH+m->slot,m->y>>8);

	/* clear C flag */
	context->P &= ~C_BIT;
}

/* 
 * doClearMouse
 *
 *        Resets buttons, movement and interrupt status 0.  (This routine
 *        is intended to be used for delta mouse positioning instead of
 *        absolute positioning.)
 *
 * Exit:  C = 0 (always).
 *        Screen hole positions and button/movement status bits are
 *        updated, interrupt status bits are cleared.
 */

static void doClearMouse(AppleCPUContextStruct *context, Mouse *m)
{
#ifdef DEBUG
	printf("ClearMouse\n");
#endif

	/* clear interrupt and status bits */
	m->status &= ~(MOUSE_MOVE_INT|MOUSE_BTTN_INT|MOUSE_SCRN_INT|
	   MOUSE_XY_CHANGED|MOUSE_BTTN_DOWN|MOUSE_BTTN_WAS_DOWN);

	/* update position and status */
	Apple_memory_write(MOUSE_STATUS+m->slot,m->status);
	Apple_memory_write(X_LOW+m->slot,m->x);
	Apple_memory_write(X_HIGH+m->slot,m->x>>8);
	Apple_memory_write(Y_LOW+m->slot,m->y);
	Apple_memory_write(Y_HIGH+m->slot,m->y>>8);

	/* clear C flag */
	context->P &= ~C_BIT;
}

/*
 * doPosMouse
 *
 *        Allows caller to change current mouse position.
 *
 * Entry: Caller places new absolute X/Y positions directly in appropriate
 *        screen holes.
 *
 * Exit:  C = 0 (always).
 *        Screen hole positions may be updated if necessary (e.g. clamping).
 *
 */
static void doPosMouse(AppleCPUContextStruct *context, Mouse *m)
{
	int	x,y,real_x,real_y;

	x=Apple_memory_read(X_LOW+m->slot)+
	   (Apple_memory_read(X_HIGH+m->slot)<<8);
	y=Apple_memory_read(Y_LOW+m->slot)+
	   (Apple_memory_read(Y_HIGH+m->slot)<<8);

#ifdef DEBUG
	printf("PosMouse X=%d Y=%d\n",x, y);
#endif
	/* clip mouse against limits */
	if (x<m->x_lower_limit)
	   x=m->x_lower_limit;
	if (y<m->y_lower_limit)
	   y=m->y_lower_limit;
	if (x>m->x_upper_limit)
	   x=m->x_upper_limit;
	if (y>m->y_upper_limit)
	   y=m->y_upper_limit;

	if (m->x_lower_limit!=m->x_upper_limit)
	   real_x=(x-m->x_lower_limit)/(m->x_scale);
	else
	   real_x=0;
	if (m->y_lower_limit!=m->y_upper_limit)
	   real_y=(y-m->y_lower_limit)/(m->y_scale);
	else
	   real_y=0;
	setMousePosition( appleDisplayContext, real_x,real_y);

	/* update screen holes if position changed after clip */
	Apple_memory_write(X_LOW+m->slot,m->x);
	Apple_memory_write(X_HIGH+m->slot,m->x>>8);
	Apple_memory_write(Y_LOW+m->slot,m->y);
	Apple_memory_write(Y_HIGH+m->slot,m->y>>8);

	/* clear C flag */
	context->P &= ~C_BIT;
}

/*
 * doClampMouse
 *
 *        Sets up clamping window for mouse user.  Power up default values
 *        are 0 to 1023 ($0000 to $03FF).
 *
 * Entry: A = 0 if entering X clamps, 1 if entering Y clamps.
 *
 *        Clamps are entered in slot 0 screen holes as follows.
 *        NOTE: these are NOT indexed by the mouse slot number.
 *
 *        $0478 = low byte of low clamp.
 *        $04F8 = low byte of high clamp.
 *        $0578 = high byte of low clamp.
 *        $05F8 = high byte of high clamp.
 *
 * Exit:  C = 0 (always).
 *        Screen hole position is set to top left corner of clamping
 *        window for the IIgs.
 */
static void doClampMouse(AppleCPUContextStruct *context, Mouse *m)
{
	int upper, lower;

	/* get limits */
	lower=(Apple_memory_read(LOWER_LIMIT_HIGH)<<8)+
	   Apple_memory_read(LOWER_LIMIT_LOW);
	upper=(Apple_memory_read(UPPER_LIMIT_HIGH)<<8)+
	   Apple_memory_read(UPPER_LIMIT_LOW);

#ifdef DEBUG
	printf("ClampMouse %c axis to %d to %d\n",context->A?'Y':'X',
	   lower,upper);
#endif
	if (context->A==0){
	    m->x_lower_limit=lower;
	    m->x_upper_limit=upper;
	    /* recalcuate X scale factor */
	    m->x_scale=(double)(m->x_upper_limit-m->x_lower_limit+1)/
	      SCREEN_WIDTH;
	}
	else{
	    m->y_lower_limit=lower;
	    m->y_upper_limit=upper;
	    /* recalcuate Y scale factor */
	    m->y_scale=(double)(m->y_upper_limit-m->y_lower_limit+1)/
	       SCREEN_HEIGHT;
	}

	/* clear C flag */
	context->P &= ~C_BIT;
}

/*
 * HomeMouse
 *
 *       Sets the absolute position to upper-left corner of clamping
 *       window.
 *
 * Exit:  C = 0 (always).
 *        Screen hole positions are updated.
 */
static void doHomeMouse(AppleCPUContextStruct *context, Mouse *m)
{
#ifdef DEBUG
	printf("HomeMouse\n");
#endif
	m->x=m->x_lower_limit;
	m->y=m->y_lower_limit;

	/* update screen hole */
	Apple_memory_write(X_LOW+m->slot,m->x);
	Apple_memory_write(X_HIGH+m->slot,m->x);
	Apple_memory_write(Y_LOW+m->slot,m->y);
	Apple_memory_write(Y_HIGH+m->slot,m->y);

	setMousePosition( appleDisplayContext, 0,0);

	/* clear C flag */
	context->P &= ~C_BIT;
}

/*
 * doInitMouse
 *
 *        Sets screen holes to default values and sets clamping window
 *        to default value of 0 to 1023 in both X and Y directions,
 *        resets hardware.
 *
 * Exit:  C = 0 (always).
 *        Screen holes are updated.
 */
static void doInitMouse(AppleCPUContextStruct *context, Mouse *m)
{

#ifdef DEBUG
	printf("InitMouse\n");
#endif


	m->mode=0;
	m->status=0;
	m->x_lower_limit=DEFAULT_X_LOWER_LIMIT;
	m->y_lower_limit=DEFAULT_Y_LOWER_LIMIT;
	m->x_upper_limit=DEFAULT_X_UPPER_LIMIT;
	m->y_upper_limit=DEFAULT_Y_UPPER_LIMIT;
	m->x=m->x_lower_limit;
	m->y=m->y_lower_limit;
	setMousePosition( appleDisplayContext, 0,0);
	m->x_scale=(m->x_upper_limit-m->x_lower_limit+1)/SCREEN_WIDTH;
	m->y_scale=(m->y_upper_limit-m->y_lower_limit+1)/SCREEN_HEIGHT;

	/* update screen hole */
	Apple_memory_write(MOUSE_STATUS+m->slot,m->status);
	Apple_memory_write(MOUSE_MODE+m->slot,m->mode);
	Apple_memory_write(X_LOW+m->slot,m->x);
	Apple_memory_write(X_HIGH+m->slot,m->x>>8);
	Apple_memory_write(Y_LOW+m->slot,m->y);
	Apple_memory_write(Y_HIGH+m->slot,m->y>>8);

	/* clear C flag */
	context->P &= ~C_BIT;
}
