/*****************************************************************************
 * Copyright (C) 1994-2007 YAE
 * $Id$
 *
 * Author: Doug Kwan <chun_tak@yahoo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * keyboard.c
 *
 * Apple II Keyborad Emulation
 *
 * By Doug Kwan March 9 1994
 *
 * Right now only Apple II+ type keyboard emulation is supported. A FIFO
 * is implemented for buffering key strokes because the actual key strokes
 * are received asynchronously. The buffering prevents consecutive key
 * strokes from missing. Abilitiy for handling key releases, i.e. as in
 * a IIe keyboard will be added later.
 * 
 */
#define _KEYBOARD_C_
#include <stdio.h>
#include <sys/types.h>
#include "hardware.h"
#include "yae.h"

/*
 * since keystrokes may be recorded asynchronously
 * a FIFO is used as a buffer
 */
#define	KEY_QUEUE_SIZE	32	
static	struct	{
	BYTE		keycode;
	APPLE_TICK	time;
	BOOLEAN		keypress;
	} AppleKeyQueue[KEY_QUEUE_SIZE];

static	int	KQHead=0;	/* position of the first keystroke */
static	int	KQEnd=0;	/* position after the last keystroke */
static	int	KQLength=0;	/* length of FIFO */

static	BYTE	LastKey=0;	/* last key pressed, bit 7 clear if strobed */
static	int	KeyPressed=0;	/* counter, non zero if a key is held */


/*
 * we use a micro job of keyboard simluation. when we received
 * a key press/release from the window system, the event is entered
 * into AppleKeyQueue. The micro job dequeue events in queue and update
 * the keyboard buffer accordingly
 */
static	void processKeyEvent( void* );	/* micro job for keyboard */
static	BOOLEAN	KeyJobSubmitted = FALSE;	

/*
 * if a burst of key stroke is received, the sequence is dilated
 * so that we will not lose any key
 */
#define	MIN_KEY_EVENT_SEPARATION	100

void recordKeyEvent( BYTE keycode, APPLE_TICK time, BOOLEAN pressed )
{
	APPLE_TICK	job_time;

	/*
	 * append a keystroke at the end of FIFO if FIFO is not full
	 */
	if ( KQLength < KEY_QUEUE_SIZE - 1 ) {
	   AppleKeyQueue[KQEnd].keycode = keycode;
	   AppleKeyQueue[KQEnd].time = AppleClock;
	   AppleKeyQueue[KQEnd].keypress = pressed;
	   if ( ++KQEnd == KEY_QUEUE_SIZE )
	      KQEnd = 0;
	   KQLength++;

	   if ( !KeyJobSubmitted ) /* invoke keyboard micro job */ {
	      /*
	       * add delay so the micro job can be submitted
	       * a burst of key events are dilated
	       */
	      job_time = TICK_GREATER( time, AppleClock ) ? time :
	         AppleClock + MIN_KEY_EVENT_SEPARATION;
	      submit_micro_job( job_time, processKeyEvent, NULL );
	      KeyJobSubmitted = TRUE;
	   }
	}
	else
	   fprintf( stderr, "AppleKeyQueue overflow\n" );
}

static void processKeyEvent( void *data )
{
	APPLE_TICK	time, job_time;

	if ( KQLength ) {
	   if ( AppleKeyQueue[KQHead].keypress ) {
	      LastKey = AppleKeyQueue[KQHead].keycode;
	      KeyPressed++;
	   }
	   else {
	      KeyPressed--;
	   }

	   /*
	    * dequeue event
	    */
	   if ( ++KQHead == KEY_QUEUE_SIZE )
	      KQHead = 0;
	   KQLength--;

	   /*
	    * resubmit self if queue is not empty
	    */
	   if ( KQLength ) {
	      /*
	       * add delay so the micro job can be submitted
	       * a burst of key events is dilated
	       */
	      time = AppleKeyQueue[KQHead].time;
	      job_time = TICK_GREATER( time, AppleClock ) ? time :
	         AppleClock + MIN_KEY_EVENT_SEPARATION;
	      submit_micro_job( job_time, processKeyEvent, NULL );
	   }
	   else 
	      KeyJobSubmitted = FALSE;
	}
}

BYTE readKeyboard(void)
{
	return LastKey; /* return current value of keyboard buffer */
}

BYTE keyHeld(void)
{
	/* if any key is pressed, bit 7 is set */ 
	return KeyPressed? LastKey : LastKey & 0x7F;
}

void strobeKeyboard(void)
{
	LastKey &= 0x7F; /* clear bit 7 of keyboard buffer */
}
