/*****************************************************************************
 * Copyright (C) 1994-2007 YAE
 * $Id$
 *
 * Author: Doug Kwan <chun_tak@yahoo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 * 6502 emulator module
 *
 * by Doug Kwan, 23/05/94
 *
 * with references to the following 
 * 
 * 6502 emulation:
 * 6502.c -- Mostek/Rockwell/Synertek/Siliconix 6502 for Apple II Emulator
 * (C) 1989 Ben Koning [556498717 408/738-1763 ben@apple.com]
 *
 * 65C02 specific codes:
 * 6502.c -- Mostek/Rockwell/Synertek/Siliconix 6502 for Apple II Emulator
 * Modified 4/20/1990 Randy Frank randy@tessa.iaf.uiowa.edu
 *
 * Change log:
 *-----------------------------------------------------------------------------
 * 23/02/97	Doug Kwan	Implement IRQ and NMI semantics.
 */
#include <stdio.h>
#include "hardware.h"
#include "6502.h"
#include "memory.h"

/*
 * Apple_memory_read() & Apple_memory_write()
 *
 * These are the only interfaces between the 6502 module and the rest of
 * the system. There functions are required to handle memory read/write
 * operation. The emulator module only gurantees that the lower 16 bits of
 * addresses passed to these two function are valid. The upper bits may
 * contain garbage. 
 */
BYTE		Apple_memory_read();
void		Apple_memory_write();

/*
 * MEMORY_READ & MEMORY_WRITE MACRO
 */

#define	MEMORY_READ(addr)	(tmp_addr=(addr)&0xFFFF,\
				 tmp_ptr=Read_Page_Table[tmp_addr>>8],\
				 ((int)tmp_ptr & 0x03?\
				  Apple_memory_read(tmp_addr):\
				  tmp_ptr[tmp_addr&0xFF]))

#define MEMORY_WRITE(addr,data) tmp_addr = (addr) & 0xFFFF;\
				tmp_ptr = Write_Page_Table[tmp_addr>>8];\
				if ( (int)tmp_ptr & 0x03 )\
				   Apple_memory_write( tmp_addr, data );\
				else\
				   tmp_ptr[tmp_addr&0xFF] = data

#define	INSTRUCTION_FETCH()	(PC &= 0xFFFF,\
				 base=Read_Page_Table[PC>>8],\
				 (int)base & 0x03?\
				 Apple_memory_read( PC++ ): base[PC++&0xFF])

/*
 * note. the zero page and the stack page must be mapped to memory
 */

#if 1
#define	ZERO_PAGE_READ(addr)	(*(Read_Page_Table[0]+((addr) & 0xFF)))
#define	ZERO_PAGE_WRITE(addr,data) *(Write_Page_Table[0]+((addr) & 0xFF))=data

#define	STACK_READ(addr)	(*(Read_Page_Table[1]+((addr) & 0xFF)))
#define	STACK_WRITE(addr,data)	(*(Write_Page_Table[1]+((addr) & 0xFF)))=data
#else
#define	ZERO_PAGE_READ(addr)		Apple_memory_read(addr)
#define	ZERO_PAGE_WRITE(addr,data)	Apple_memory_write((addr),(data))
#define	STACK_READ(addr)		Apple_memory_read(addr)
#define	STACK_WRITE(addr,data)		Apple_memory_write((addr),(data))
#endif

/*
 * macros for the stack
 */
#define PUSH(x)	STACK_WRITE((S--&0xFF)+0x0100,x);
#define POP()	(STACK_READ((++S&0xFF)+0x0100))

/*
 * marcos for flags in P
 * SET_X(c): set flag X to 1 if c is not zero, clear flag otherwise
 * GET_X(c): return non-zero value if flag X is 1, 0 otherwise
 */
#define	SET_N(c)	if (c) P |= N_BIT; else P &= ~N_BIT
#define	SET_V(c)	if (c) P |= V_BIT; else P &= ~V_BIT
#define	SET_B(c)	if (c) P |= B_BIT; else P &= ~B_BIT
#define	SET_D(c)	if (c) P |= D_BIT; else P &= ~D_BIT
#define	SET_I(c)	if (c) P |= I_BIT; else P &= ~I_BIT
#define	SET_Z(c)	if (c) P |= Z_BIT; else P &= ~Z_BIT
#define	SET_C(c)	if (c) P |= C_BIT; else P &= ~C_BIT
#define GET_N()		(P & N_BIT)
#define GET_V()		(P & V_BIT)
#define GET_B()		(P & B_BIT)
#define GET_D()		(P & D_BIT)
#define GET_I()		(P & I_BIT)
#define GET_Z()		(P & Z_BIT)
#define GET_C()		(P & C_BIT)


/*
 * fast condition codes. Instead of using bits to encode condition codes,
 * recent ALU results are cached to that the condition codes can be
 * handled more easily by the emulator's native hardware.
 */
#define GET_FN()	(NZ_flags & 0x280)
#define GET_FZ()	(!(NZ_flags & 0xFF))
#define SET_FNZ(n,z)	NZ_flags = ((n)? 0x200: 0x00) | ((z)? 0x00 : 0x01 )
#define GET_FC()	(result>>8)
#define SET_FC(c)	result=((c)? 0x100: 0x00)

/*
 * macro for page crossing cycle regulation
 */
#define CheckCross(addr,offset)	(((addr+offset)^addr)&0xff00)? local_clock++:0

/*
 * macros for effective address calculation 
 * macros whose names end with NC do not check for page crossing
 *
 */
#define eaimm()		(MEMORY_READ(PC++))
#define eazp()		(MEMORY_READ(PC++))
#define eazpx()		((MEMORY_READ(PC++)+X)&0xFF)
#define eazpy()		((MEMORY_READ(PC++)+Y)&0xFF)
#define eaabs()		(easp1=MEMORY_READ(PC++),\
			 easp1+((ADDR)MEMORY_READ(PC++)<<8))
#define earel()		(easp1=(ADDR)MEMORY_READ(PC++),\
			 (easp1&0x80)?easp1-256:easp1)

#define eaabsx()	(easp1=eaabs(),CheckCross(easp1,X),easp1+X)
#define eaabsxNC()	(eaabs()+X)
#define eaabsy()	(easp1=eaabs(),CheckCross(easp1,Y),easp1+Y)
#define eaabsyNC()	(eaabs()+Y)

/*
 * indirect addressing
 */
#ifdef MC65C02
#define eaabsind()	(easp1=eaabs(),easp2=(ADDR)MEMORY_READ(easp1),\
			 easp2+((ADDR)MEMORY_READ(easp1+1)<<8))
#else
#define eaabsind()	(easp1=eaabs(),(ADDR)MEMORY_READ(easp1)+\
			  ((ADDR)MEMORY_READ(easp1&0xff?easp1+1:easp1-0xFF)<<8))
#endif

#define eazpxind()	(easp1=eazpx(),easp2=(ADDR)ZERO_PAGE_READ(easp1),\
			 easp2+((ADDR)ZERO_PAGE_READ((easp1+1)&0xff)<<8))
#define eazpindy()	(easp1=eaimm(),easp2=(ADDR)ZERO_PAGE_READ(easp1),\
			 easp2+=((ADDR)ZERO_PAGE_READ((easp1+1)&0xff)<<8),\
			   CheckCross(easp2,Y),easp2+Y)
#define eazpindyNC()	(easp1=eaimm(),easp2=(ADDR)ZERO_PAGE_READ(easp1),\
			 easp2+((ADDR)ZERO_PAGE_READ((easp1+1)&0xff)<<8)+Y)

#ifdef MC65C02

/*
 * new addressing mode in 65C02
 */
#define eazpind()	(easp1=eazp(),easp2=(ADDR)ZERO_PAGE_READ(easp1),\
			 easp2+((ADDR)ZERO_PAGE_READ((easp1+1)&0xff)<<8))
#define eaabsxind()	(easp1=eaabs(),easp2=(ADDR)MEMORY_READ(easp1),\
			 easp2+((ADDR)MEMORY_READ(easp1+1)<<8)+X)
#endif /* MC65C02 */

/*
 * macros for saving & restoring context
 * due to a bug in 6502, bit 5 of P is always 1
 */
#define SAVE_CONTEXT(c)		c->PC = PC;\
				c->A = A;\
				c->X = X;\
				c->Y = Y;\
	                        SET_C( GET_FC() );\
	                        SET_Z( GET_FZ() );\
	                        SET_N( GET_FN() );\
				c->P = P;\
				c->S = S

#define RESTORE_CONTEXT(c)	PC = c->PC;\
				A = c->A;\
				X = c->X;\
				Y = c->Y;\
				P = c->P | 0x20; /*bug!*/ \
				SET_FC( GET_C() );\
				SET_FNZ( GET_N(), GET_Z() );\
				S = c->S

/*
 * misc. macros
 */

#define	ADC_BCD_ADJUST	if ( GET_D() ) result = BCD_Table1[result]
#define	SBC_BCD_ADJUST	if ( GET_D() ) result = BCD_Table2[result]

#define	BRANCH( operand )\
	{CheckCross( PC, operand );\
	PC += operand;\
	local_clock++;}

/* 
 *	6502 registers
 *
 *	PC:
 *	only the lower 16 bits of PC is valid, upper bits may
 *	contain garbage.
 *
 *	A, X, Y, S, P
 *	these register must be unsigned char 
 *	
 */
/* power on reset */	
unsigned	Exception_Register = 0;
int		CPU_Initialized = 0;
APPLE_TICK	Next_Timer_Exception=0;	
int		Trace;

APPLE_TICK	enter, leave;

static	int	PendingIRQ;

/*
 * ALU look up tables
 */
static	unsigned	BCD_Table1[512];	/* addition correction */
static	unsigned	BCD_Table2[512];	/* subtraction correction */

int	Frequency[256];

void init_6502()
{
	int	i;

	for( i = 0; i < 512; i++ ) {
	   BCD_Table1[i] = ((i & 0x0F) <= 0x09) ? i : i + 0x06;
	   BCD_Table1[i] += ((BCD_Table1[i] & 0xF0) <= 0x90) ? 0 : 0x60;
	   if ( BCD_Table1[i] > 0x1FF )
	      BCD_Table1[i] -= 0x100;
	   BCD_Table2[i] = ((i & 0x0F) <= 0x09) ? i : i - 0x06;
	   BCD_Table2[i] -= ((BCD_Table2[i] & 0xF0) <= 0x90) ? 0 : 0x60;
	}

}

void assert_RES()
{
	Apple_memory_read( 0xc081 );
	Exception_Register |= SIG_6502_RES;

}

void assert_NMI()
{
	Exception_Register |= SIG_6502_NMI;

}

void assert_IRQ()
{
	Exception_Register |= SIG_6502_IRQ;

}

void shutdown_6502()
{
	Exception_Register |= SIG_6502_SHUTDOWN;
}

void set_next_timer_exception( when )
APPLE_TICK when;
{
	Next_Timer_Exception = when;

}

void set_trace()
{
	printf( "set Trace!\n" );
	Trace = 1;
}

void Signal6502( sig )
unsigned sig;
{
	Exception_Register |= sig;
}


int CPUExecute( context )
AppleCPUContextStruct *context;
{
	register unsigned	PC, A, X, Y, P, S;
	register APPLE_TICK	local_clock;
	register int		easp1, easp2;
	register unsigned 	operand_address;
	register unsigned 	opcode;
	register unsigned 	operand;
	register unsigned  	result;
	register unsigned  	NZ_flags;
	register unsigned	tmp_addr;
	register BYTE		*tmp_ptr, *base;

	/*
	 * load saved registers
	 */
	RESTORE_CONTEXT( context );
	local_clock = AppleClock;
	
resume_execution:
	while( !Exception_Register ) {

#ifdef	SLAVE_PROCESSOR
	   SAVE_CONTEXT( context );
#endif
	   /*
	    * fetch & decode instruction
	    */
	   opcode = INSTRUCTION_FETCH();

#ifdef	RUN_TIME_TRANSLATION
	   /* check if context switching to VM is possible */
	   if ( !GET_D() && COMPILED(((PC-1)>>8)&0xFF) && 
	       ENTRY_POINT((((PC-1)>>8)&0xFF),(PC-1)&0xFF)) {

	      /* perform context switching */
	      PC--;	/* roll back, restart instruction in VM */
	      SAVE_CONTEXT( context );
	      enter = AppleClock;
	      VMCPUExecute( context );
	      leave = AppleClock;
	      VMTime += leave - enter;

	      /* return from VM, reload local_clock */
	      RESTORE_CONTEXT( context );
	      local_clock = AppleClock;

	      /* ignore SIG_6502_CONTEXT, handle any other exception */	
	      Exception_Register &= ~SIG_6502_CONTEXT;
	      if ( Exception_Register )
	          break;

	      opcode = INSTRUCTION_FETCH();	/* restart */
	      /* continue in 6502 emulator */
	   }
#endif


#ifdef	SLAVE_PROCESSOR
	   SaveMasterCPUContext( context );	/* log CPU state */
#endif
	   switch( opcode ) {
	   case 0x69:	/* ADC #imm */
	      operand = eaimm();
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x6D:	/* ADC abs */
	      operand = MEMORY_READ( eaabs() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x65:	/* ADC zp */
	      operand = ZERO_PAGE_READ( eazp() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0x61:	/* ADC (zp,X) */
	      operand = MEMORY_READ( eazpxind() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0x71:	/* ADC (zp),Y */
	      operand_address = eazpindy();
	      operand = MEMORY_READ( operand_address );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0x75:	/* ADC zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x7D:	/* ADC abs,X */
	      operand = MEMORY_READ( eaabsx() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x79:	/* ADC abs,Y */
	      operand = MEMORY_READ( eaabsy() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x29:	/* AND #imm */
	      A &= eaimm();
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x2D:	/* AND abs */
	      A &= MEMORY_READ( eaabs() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x25:	/* AND zp */
	      A &= ZERO_PAGE_READ( eazp() );
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0x21:	/* AND (zp,X) */
	      A &= MEMORY_READ( eazpxind() );
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0x31:	/* AND (zp),Y */
	      A &= MEMORY_READ( eazpindy() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0x35:	/* AND zp,X */
	      A &= ZERO_PAGE_READ( eazpx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x3D:	/* AND abs,X */
	      A &= MEMORY_READ( eaabsx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x39:	/* AND abs,Y */
	      A &= MEMORY_READ( eaabsy() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x0E:	/* ASL abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      result = operand << 1;
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x06:	/* ASL zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = operand << 1;
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 5;
	      break;
	      
	   case 0x0A:	/* ASL acc */
	      result = A << 1;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x16:	/* ASL zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = operand << 1;
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x1E:	/* ASL abs,X */
	      operand_address = eaabsx();
	      operand = MEMORY_READ( operand_address );
	      result = operand << 1;
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 7;
	      break;
	      
	   case 0x90:	/* BCC rr */
	      operand = earel();
	      local_clock+=2;
	      if ( !GET_FC() ) 
	         BRANCH( operand );
	      break;
	      
	   case 0xB0:	/* BCS rr */
	      operand = earel();
	      local_clock+=2;
	      if ( GET_FC() ) 
	         BRANCH( operand );
	      break;
	     
	   case 0xF0:	/* BEQ rr */
	      operand = earel();
	      local_clock+=2;
	      if ( GET_FZ() ) 
	         BRANCH( operand );
	      break;
	   
	   case 0x2C:	/* BIT abs */
	      operand = MEMORY_READ( eaabs() );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      local_clock += 4;
	      break;

	   case 0x24:	/* BIT zp */
	      operand = ZERO_PAGE_READ( eazp() );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      local_clock += 3;
	      break;
	      	   	 
	   case 0x30:	/* BMI rr */
	      operand = earel();
	      local_clock+=2;
	      if ( GET_FN() ) 
	         BRANCH( operand );
	      break;
	     
	   case 0xD0:	/* BNE rr */
	      operand = earel();
	      local_clock+=2;
	      if ( !GET_FZ() ) 
	         BRANCH( operand );
	      break;
	   
	   case 0x10:	/* BPL rr */
	      operand = earel();
	      local_clock+=2;
	      if ( !GET_FN() ) 
	         BRANCH( operand );
	      break;
	     
	   case 0x00:	/* BRK */
	      PUSH(PC>>8);	/* save PCH, PCL & P */
	      PUSH(PC);
	      SET_N(GET_FN());
	      SET_Z(GET_FZ());
	      SET_C(GET_FC());
	      SET_B(1);
	      PUSH(P);
	      SET_I(1);
	      PC = MEMORY_READ(0xFFFE);
	      PC |= MEMORY_READ(0xFFFF)<<8;
	      local_clock += 7;
	      break;

	   case 0x50:	/* BVC rr */
	      operand = earel();
	      local_clock+=2;
	      if ( !GET_V() ) 
	         BRANCH( operand );
	      break;
	     
	   case 0x70:	/* BVS rr */
	      operand = earel();
	      local_clock+=2;
	      if ( GET_V() ) 
	         BRANCH( operand );
	      break;
	     
	   case 0x18:	/* CLC rr */
	      SET_FC(0);
	      local_clock += 2;
	      break;

	   case 0xD8:	/* CLD */
	      SET_D(0);
	      local_clock += 2;
	      break;

	   case 0x58:	/* CLI */
	      SET_I(0);
	      local_clock += 2;
	      if (PendingIRQ) {
	         PendingIRQ--;
	         assert_IRQ();
	      }
	      break;

	   case 0xB8:	/* CLV */
	      SET_V(0);
	      local_clock += 2;
	      break;

	   case 0xC9:	/* CMP #imm */
	      result = 0x100 + A - eaimm();
	      NZ_flags = result;
	      local_clock += 2;
	      break;
	      
	   case 0xCD:	/* CMP abs */
	      result = 0x100 + A - MEMORY_READ( eaabs() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xC5:	/* CMP zp */
	      result = 0x100 + A - ZERO_PAGE_READ( eazp() );
	      NZ_flags = result;
	      local_clock += 3;
	      break;
	      
	   case 0xC1:	/* CMP (zp,X) */
	      result = 0x100 + A - MEMORY_READ( eazpxind() );
	      NZ_flags = result;
	      local_clock += 6;
	      break;
	      
	   case 0xD1:	/* CMP (zp),Y */
	      result = 0x100 + A - MEMORY_READ( eazpindy() );
	      NZ_flags = result;
	      local_clock += 5;
	      break;
	      
	   case 0xD5:	/* CMP zp,X */
	      result = 0x100 + A - ZERO_PAGE_READ( eazpx() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xDD:	/* CMP abs,X */
	      result = 0x100 + A - MEMORY_READ( eaabsx() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xD9:	/* CMP abs,Y */
	      result = 0x100 + A - MEMORY_READ( eaabsy() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xE0:	/* CPX #imm */
	      result = 0x100 + X - eaimm();
	      NZ_flags = result;
	      local_clock += 2;
	      break;
	      
	   case 0xEC:	/* CPX abs */
	      result = 0x100 + X - MEMORY_READ( eaabs() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xE4:	/* CPX zp */
	      result = 0x100 + X - ZERO_PAGE_READ( eazp() );
	      NZ_flags = result;
	      local_clock += 3;
	      break;
	      
	   case 0xC0:	/* CPY #imm */
	      result = 0x100 + Y - eaimm();
	      NZ_flags = result;
	      local_clock += 2;
	      break;
	      
	   case 0xCC:	/* CPY abs */
	      result = 0x100 + Y - MEMORY_READ( eaabs() );
	      NZ_flags = result;
	      local_clock += 4;
	      break;
	      
	   case 0xC4:	/* CPY zp */
	      result = 0x100+ Y - ZERO_PAGE_READ( eazp() );
	      NZ_flags = result;
	      local_clock += 3;
	      break;
	      
	   case 0xCE:	/* DEC abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      NZ_flags = operand + 0xFF;
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0xC6:	/* DEC zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      NZ_flags = operand + 0xFF;
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 5;
	      break;
	      
	   case 0xD6:	/* DEC zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      NZ_flags = operand + 0xFF;
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0xDE:	/* DEC abs,X */
	      operand_address = eaabsx();
	      operand = MEMORY_READ( operand_address );
	      NZ_flags = operand + 0xFF;
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 7;
	      break;

	   case 0xCA:	/* DEX */
	      NZ_flags = X + 0xFF;
	      X = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;
	      
	   case 0x88:	/* DEY */
	      NZ_flags = Y + 0xFF;
	      Y = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;
	      
	   case 0x49:	/* EOR #imm */
	      A ^= eaimm();
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x4D:	/* EOR abs */
	      A ^= MEMORY_READ( eaabs() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x45:	/* EOR zp */
	      A ^= ZERO_PAGE_READ( eazp() );
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0x41:	/* EOR (zp,X) */
	      A ^= MEMORY_READ( eazpxind() );
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0x51:	/* EOR (zp),Y */
	      A ^= MEMORY_READ( eazpindy() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0x55:	/* EOR zp,X */
	      A ^= ZERO_PAGE_READ( eazpx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x5D:	/* EOR abs,X */
	      A ^= MEMORY_READ( eaabsx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x59:	/* EOR abs,Y */
	      A ^= MEMORY_READ( eaabsy() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xEE:	/* INC abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      NZ_flags = operand + 1;
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0xE6:	/* INC zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      NZ_flags = operand + 1;
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 5;
	      break;
	      
	   case 0xF6:	/* INC zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      NZ_flags = operand + 1;
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0xFE:	/* INC abs,X */
	      operand_address = eaabsxNC();
	      operand = MEMORY_READ( operand_address );
	      NZ_flags = operand + 1;
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 7;
	      break;

	   case 0xE8:	/* INX */
	      NZ_flags = X + 1;
	      X = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;
	      
	   case 0xC8:	/* INY */
	      NZ_flags = Y + 1;
	      Y = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;
	     
	   case 0x4C:	/* JMP abs */
	      PC = eaabs();
	      local_clock += 3;
	      break;
 
	   case 0x6C:	/* JMP (abs) */
	      PC = eaabsind();
	      local_clock += 5;
	      break;
 
	   case 0x20:	/* JSR abs */
	      operand_address = eaabs();
	      PUSH( (PC-1) >> 8 );
	      PUSH( PC-1 );
	      PC = operand_address;
	      local_clock += 6;
	      break;

	   case 0xA9:	/* LDA #imm */
	      A = eaimm();
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0xAD:	/* LDA abs */
	      A = MEMORY_READ( eaabs() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xA5:	/* LDA zp */
	      A = ZERO_PAGE_READ( eazp() );
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0xA1:	/* LDA (zp,X) */
	      A = MEMORY_READ( eazpxind() );
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0xB1:	/* LDA (zp),Y */
	      A = MEMORY_READ( eazpindy() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0xB5:	/* LDA zp,X */
	      A = ZERO_PAGE_READ( eazpx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xBD:	/* LDA abs,X */
	      A = MEMORY_READ( eaabsx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xB9:	/* LDA abs,Y */
	      A = MEMORY_READ( eaabsy() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xA2:	/* LDX #imm */
	      X = eaimm();
	      NZ_flags = X;
	      local_clock += 2;
	      break;
	      
	   case 0xAE:	/* LDX abs */
	      X = MEMORY_READ( eaabs() );
	      NZ_flags = X;
	      local_clock += 4;
	      break;
	      
	   case 0xA6:	/* LDX zp */
	      X = ZERO_PAGE_READ( eazp() );
	      NZ_flags = X;
	      local_clock += 3;
	      break;
	      
	   case 0xBE:	/* LDX abs,Y */
	      X = MEMORY_READ( eaabsy() );
	      NZ_flags = X;
	      local_clock += 4;
	      break;
	      
	   case 0xB6:	/* LDX zp,Y */
	      X = ZERO_PAGE_READ( eazpy() );
	      NZ_flags = X;
	      local_clock += 4;
	      break;
	      
	   case 0xA0:	/* LDY #imm */
	      Y = eaimm();
	      NZ_flags = Y;
	      local_clock += 2;
	      break;
	      
	   case 0xAC:	/* LDY abs */
	      Y = MEMORY_READ( eaabs() );
	      NZ_flags = Y;
	      local_clock += 4;
	      break;
	      
	   case 0xA4:	/* LDY zp */
	      Y = ZERO_PAGE_READ( eazp() );
	      NZ_flags = Y;
	      local_clock += 3;
	      break;
	      
	   case 0xB4:	/* LDY zp,X */
	      Y = ZERO_PAGE_READ( eazpx() );
	      NZ_flags = Y;
	      local_clock += 4;
	      break;
	      
	   case 0xBC:	/* LDY abs,X */
	      Y = MEMORY_READ( eaabsx() );
	      NZ_flags = Y;
	      local_clock += 4;
	      break;
	      
	   case 0x4E:	/* LSR abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      result = ( operand & 0x01 ) << 8;	/* just get the C bit */
	      NZ_flags = operand >> 1;		/* result in NZ_flags */
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0x46:	/* LSR zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = ( operand & 0x01 ) << 8;	/* just get the C bit */
	      NZ_flags = operand >> 1;		/* result in NZ_flags */
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 5;
	      break;
	      
	   case 0x4A:	/* LSR acc */
	      result = ( A & 0x01 ) << 8;	/* just get the C bit */
	      A >>= 1;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x56:	/* LSR zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = ( operand & 0x01 ) << 8;	/* just get the C bit */
	      NZ_flags = operand >> 1;		/* result in NZ_flags */
	      ZERO_PAGE_WRITE( operand_address, NZ_flags );
	      local_clock += 6;
	      break;
	      
	   case 0x5E:	/* LSR abs,X */
	      operand_address = eaabsx();
	      operand = MEMORY_READ( operand_address );
	      result = ( operand & 0x01 ) << 8;	/* just get the C bit */
	      NZ_flags = operand >> 1;		/* result in NZ_flags */
	      MEMORY_WRITE( operand_address, NZ_flags );
	      local_clock += 7;
	      break;

	   case 0xEA:	/* NOP */
	      local_clock += 2;
	      break;

	   case 0x09:	/* ORA #imm */
	      A |= eaimm();
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x0D:	/* ORA abs */
	      A |= MEMORY_READ( eaabs() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x05:	/* ORA zp */
	      A |= ZERO_PAGE_READ( eazp() );
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0x01:	/* ORA (zp,X) */
	      A |= MEMORY_READ( eazpxind() );
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0x11:	/* ORA (zp),Y */
	      A |= MEMORY_READ( eazpindy() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0x15:	/* ORA zp,X */
	      A |= ZERO_PAGE_READ( eazpx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x1D:	/* ORA abs,X */
	      A |= MEMORY_READ( eaabsx() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x19:	/* ORA abs,Y */
	      A |= MEMORY_READ( eaabsy() );
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x48:	/* PHA */
	      PUSH( A );
	      local_clock += 3;
	      break;

	   case 0x08:	/* PHP */
	      SET_N(GET_FN());
	      SET_Z(GET_FZ());
	      SET_C(GET_FC());
	      PUSH( P );
	      local_clock += 3;
	      break;

	   case 0x68:	/* PLA */
	      A = POP();
	      NZ_flags = A;
	      local_clock += 4;
	      break;

	   case 0x28:	/* PLP */
	      P = POP() | 0x20; /* fix bug in bit5 of P */
	      SET_FC(GET_C());
	      SET_FNZ( GET_N(), GET_Z() );
	      local_clock += 4;
	      if ( PendingIRQ && !GET_I()) {
	         PendingIRQ--;
	         assert_IRQ();
	      }
	      break;

	   case 0x2E:	/* ROL abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      result = (operand << 1) | GET_FC();
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x26:	/* ROL zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = (operand << 1) | GET_FC();
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 5;
	      break;
	      
	   case 0x2A:	/* ROL acc */
	      result = (A << 1) | GET_FC();
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x36:	/* ROL zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = (operand << 1) | GET_FC();
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x3E:	/* ROL abs,X */
	      operand_address = eaabsx();
	      operand = MEMORY_READ( operand_address );
	      result = (operand << 1) | GET_FC();
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 7;
	      break;

	   case 0x6E:	/* ROR abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      result = ( ( operand & 0x01 ) << 8 ) | ( GET_FC() << 7 ) |
	         ( operand >> 1 );
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x66:	/* ROR zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = ( ( operand & 0x01 ) << 8 ) | ( GET_FC() << 7 ) |
	         ( operand >> 1 );
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 5;
	      break;
	      
	   case 0x6A:	/* ROR acc */
	      result = ( ( A & 0x01 ) << 8 ) | ( GET_FC() << 7 ) | ( A >> 1 );
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0x76:	/* ROR zp,X */
	      operand_address = eazpx();
	      operand = ZERO_PAGE_READ( operand_address );
	      result = ( ( operand & 0x01 ) << 8 ) | ( GET_FC() << 7 ) |
	         ( operand >> 1 );
	      NZ_flags = result;
	      ZERO_PAGE_WRITE( operand_address, result );
	      local_clock += 6;
	      break;
	      
	   case 0x7E:	/* ROR abs,X */
	      operand_address = eaabsx();
	      operand = MEMORY_READ( operand_address );
	      result = ( ( operand & 0x01 ) << 8 ) | ( GET_FC() << 7 ) |
	         ( operand >> 1 );
	      NZ_flags = result;
	      MEMORY_WRITE( operand_address, result );
	      local_clock += 7;
	      break;

	   case 0x40:	/* RTI */
	      P = POP() | 0x20; /* bit 5 bug of 6502 */
	      SET_FC(GET_C());
	      SET_FNZ( GET_N(), GET_Z() );
	      PC = POP();	/* splitting is necessary */
	      PC += POP()<<8;	/* because of nested macros */
	      local_clock += 6;
	      break;

	   case 0x60:	/* RTS */
	      PC = POP();	/* splitting is necessary */
	      PC += POP()<<8;	/* because of nested macros */
	      PC++;
	      local_clock += 6;
	      break;

	   case 0xE9:	/* SBC #imm */
	      operand = 255 - eaimm();
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	      
	   case 0xED:	/* SBC abs */
	      operand = 255 - MEMORY_READ( eaabs() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xE5:	/* SBC zp */
	      operand = 255 - ZERO_PAGE_READ( eazp() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 3;
	      break;
	      
	   case 0xE1:	/* SBC (zp,X) */
	      operand = 255 - MEMORY_READ( eazpxind() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 6;
	      break;
	      
	   case 0xF1:	/* SBC (zp),Y */
	      operand = 255 - MEMORY_READ( eazpindy() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 5;
	      break;
	      
	   case 0xF5:	/* SBC zp,X */
	      operand = 255 - ZERO_PAGE_READ( eazpx() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xFD:	/* SBC abs,X */
	      operand = 255 - MEMORY_READ( eaabsx() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0xF9:	/* SBC abs,Y */
	      operand = 255 - MEMORY_READ( eaabsy() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 4;
	      break;
	      
	   case 0x38:	/* SEC */
	      SET_FC(1);
	      local_clock += 2;
	      break;
	   
	   case 0xF8:	/* SED */
	      SET_D( 1 );
	      local_clock += 2;
	      break;
	 
	   case 0x78:	/* SEI */
	      SET_I( 1 );
	      local_clock += 2;
	      break;
	   
	   case 0x8D:	/* STA abs */
	      MEMORY_WRITE( eaabs(), A );
	      local_clock += 4;
	      break;
	      
	   case 0x85:	/* STA zp */
	      ZERO_PAGE_WRITE( eazp(), A );
	      local_clock += 3;
	      break;
	      
	   case 0x81:	/* STA (zp,X) */
	      MEMORY_WRITE( eazpxind(), A );
	      local_clock += 6;
	      break;
	      
	   case 0x91:	/* STA (zp),Y */
	      MEMORY_WRITE( eazpindy(), A );
	      local_clock += 6;
	      break;
	      
	   case 0x95:	/* STA zp,X */
	      ZERO_PAGE_WRITE( eazpx(), A );
	      local_clock += 4;
	      break;
	      
	   case 0x9D:	/* STA abs,X */
	      MEMORY_WRITE( eaabsx(), A );
	      local_clock += 5;
	      break;
	      
	   case 0x99:	/* STA abs,Y */
	      MEMORY_WRITE( eaabsy(), A );
	      local_clock += 5;
	      break;
	      
	   case 0x8E:	/* STX abs */
	      MEMORY_WRITE( eaabs(), X );
	      local_clock += 4;
	      break;
	      
	   case 0x86:	/* STX zp */
	      ZERO_PAGE_WRITE( eazp(), X );
	      local_clock += 3;
	      break;
	      
	   case 0x96:	/* STX zp,Y */
	      ZERO_PAGE_WRITE( eazpy(), X );
	      local_clock += 4;
	      break;
	      
	   case 0x8C:	/* STY abs */
	      MEMORY_WRITE( eaabs(), Y );
	      local_clock += 4;
	      break;
	      
	   case 0x84:	/* STY zp */
	      ZERO_PAGE_WRITE( eazp(), Y );
	      local_clock += 3;
	      break;
	      
	   case 0x94:	/* STY zp,X */
	      ZERO_PAGE_WRITE( eazpx(), Y );
	      local_clock += 4;
	      break;
	      
	   case 0xAA:	/* TAX */
	      X = A;
	      NZ_flags = X;
	      local_clock += 2;
	      break;

	   case 0xA8:	/* TAY */
	      Y = A;
	      NZ_flags = Y;
	      local_clock += 2;
	      break;

	   case 0xBA:	/* TSX */
	      X = S;
	      NZ_flags = X;
	      local_clock += 2;
	      break;

	   case 0x8A:	/* TXA */
	      A = X;
	      NZ_flags = A;
	      local_clock += 2;
	      break;

	   case 0x9A:	/* TXS */
	      S = X;
	      local_clock += 2;
	      break;

	   case 0x98:	/* TYA */
	      A = Y;
	      NZ_flags = A;
	      local_clock += 2;
	      break;
	
	   /*
	    * 65C02 instructions
	    * note: timing is not correct
	    */

#ifdef MC65C02
	   case 0x72:	/* ADC (zp) */
	      operand = MEMORY_READ( eazpind() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      ADC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0x32:	/* AND (zp) */
	      A &= MEMORY_READ( eazpind() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0x34:	/* BIT zp,X */
	      operand = ZERO_PAGE_READ( eazpx() );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      local_clock += 3;
	      break;

	   case 0x89:	/* BIT #imm */
	      operand = eaimm();
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      local_clock += 2;
	      break;

	   case 0x3C:	/* BIT abs,X */
	      operand = eaabsx();
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      local_clock += 4;
	      break;

	   case 0x80:	/* BRA rr */
	      operand = earel();
	      local_clock+=2;
	      BRANCH( operand );
	      break;
	   
	   case 0xD2:	/* CMP (zp) */
	      result = 0x100 + A - MEMORY_READ( eazpind() );
	      NZ_flags = result;
	      local_clock += 5;
	      break;
	      
	   case 0x3A:	/* DEA acc */
	      NZ_flags = A + 0xFF;
	      A = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;

	   case 0x52:	/* EOR (zp) */
	      A ^= MEMORY_READ( eazpind() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0x1A:	/* INA acc */
	      NZ_flags = A + 1;
	      A = NZ_flags & 0xFF;
	      local_clock += 2;
	      break;

	   case 0x7C:	/* JMP (abs,X) */
	      PC = eaabsxind();
	      local_clock += 6;
	      break;

	   case 0xB2:	/* LDA (zp) */
	      A = MEMORY_READ( eazpind() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0x12:	/* ORA (zp) */
	      A |= MEMORY_READ( eazpind() );
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0xDA:	/* PHX */
	      PUSH( X );
	      local_clock += 3;
	      break;

	   case 0xFA:	/* PLX */
	      X = POP();
	      NZ_flags = X;
	      local_clock += 4;
	      break;

	   case 0x5A:	/* PHY */
	      PUSH( Y );
	      local_clock += 3;
	      break;

	   case 0x7A:	/* PLY */
	      Y = POP();
	      NZ_flags = Y;
	      local_clock += 4;
	      break;

	   case 0xF2:	/* SBC (zp) */
	      operand = 255 - MEMORY_READ( eazpind() );
	      result = operand + A + GET_FC();
	      SET_V( !((operand^A) & 0x80) && ((A^result) & 0x80));
	      SBC_BCD_ADJUST;
	      A = result & 0xFF;
	      NZ_flags = A;
	      local_clock += 5;
	      break;

	   case 0x92:	/* STA (zp) */
	      MEMORY_WRITE( eazpind(), A );
	      local_clock += 6;
	      break;

	   case 0x9C:	/* STZ abs */
	      MEMORY_WRITE( eaabs(), 0 );
	      local_clock += 4;
	      break;

	   case 0x64:	/* STZ zp */
	      ZERO_PAGE_WRITE( eazp(), 0 );
	      local_clock += 3;
	      break;

	   case 0x74:	/* STZ zp,X */
	      ZERO_PAGE_WRITE( eazpx(), 0 );
	      local_clock += 3;
	      break;

	   case 0x9E:	/* STZ abs,X */
	      MEMORY_WRITE( eaabsx(), 0 );
	      local_clock += 4;
	      break;

	   case 0x1C:	/* TRB abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      MEMORY_WRITE( operand_address, (operand & ~A) & 0xFF);
	      local_clock += 5;
	      break;

	   case 0x14:	/* TRB zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      ZERO_PAGE_WRITE( operand_address, (operand & ~A) & 0xFF);
	      local_clock += 5;
	      break;

	   case 0x0C:	/* TSB abs */
	      operand_address = eaabs();
	      operand = MEMORY_READ( operand_address );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      MEMORY_WRITE( operand_address, operand | A );
	      local_clock += 5;
	      break;

	   case 0x04:	/* TSB zp */
	      operand_address = eazp();
	      operand = ZERO_PAGE_READ( operand_address );
	      SET_V( operand & 0x40 ); 
	      NZ_flags = ( ( operand & 0x80 ) << 2 ) | ( A & operand  );
	      ZERO_PAGE_WRITE( operand_address, operand | A );
	      local_clock += 5;
	      break;

#endif /* MC65C02 */
	   default:	/* unknown instructions */
	      local_clock += 2;
	   }

	   /*
	    * sync AppleClock with local clock & check timer interrupt 
	    */
	   AppleClock = local_clock;
	   if (!TICK_LESS(local_clock, Next_Timer_Exception))
	      Exception_Register |= SIG_6502_TIMER;
	}

	SAVE_CONTEXT( context );

	if ( Exception_Register & SIG_6502_RES ) { /* reset */
	   if ( !CPU_Initialized ) {
	      init_6502();
	      CPU_Initialized = !CPU_Initialized;
	   }
	   A = X = Y = 0;
	   P = 0x20;
	   SET_FC(GET_C());
	   SET_FNZ(GET_N(),GET_Z());
	   S = 0xFF;
	   PC = (ADDR)MEMORY_READ(0xFFFC);
	   PC |= ((ADDR)MEMORY_READ(0xFFFD) << 8);
	   Exception_Register &= ~SIG_6502_RES;
	   SAVE_CONTEXT(context);
	   return SIG_6502_RES;
	}

	if ( Exception_Register & SIG_6502_NMI ) {
	   PUSH(PC>>8);	/* save PCH, PCL & P */
	   PUSH(PC);
	   SET_N(GET_FN());
	   SET_Z(GET_FZ());
	   SET_C(GET_FC());
	   PUSH(P);
	   PC = MEMORY_READ(0xFFFA);
	   PC |= MEMORY_READ(0xFFFB)<<8;
	   local_clock += 7;
	   Exception_Register ^= SIG_6502_NMI;
	   goto resume_execution;
	}

	if ( Exception_Register & SIG_6502_IRQ ) {
	   if ( GET_I() ) { /* pending interrupt */
	      PendingIRQ++;
	   }
	   else {
	      PUSH(PC>>8);	/* save PCH, PCL & P */
	      PUSH(PC);
	      SET_N(GET_FN());
	      SET_Z(GET_FZ());
	      SET_C(GET_FC());
	      SET_B(0);
	      PUSH(P);
	      SET_I(1);
	      PC = MEMORY_READ(0xFFFE);
	      PC |= MEMORY_READ(0xFFFF)<<8;
	      local_clock += 7;
	   }
	   Exception_Register ^= SIG_6502_IRQ;
	   goto resume_execution;
	}

#ifdef	RUN_TIME_TRANSLATION
	if ( Exception_Register & SIG_6502_CONTEXT ) {
	   Exception_Register &= ~SIG_6502_CONTEXT;
	   goto resume_execution;
	}
#endif

	if ( Exception_Register & SIG_6502_MEMORY ) {
	   Exception_Register &= ~SIG_6502_MEMORY;
	   goto resume_execution;
	}

	if ( Exception_Register & SIG_6502_TIMER ) {
	   Next_Timer_Exception = AppleClock + 0x80000000;
	   Exception_Register &= ~SIG_6502_TIMER;
	   return SIG_6502_TIMER;
	}

	for(easp1=0;easp1<32;easp1++) {
	   if ((1<<easp1)&Exception_Register) {
	      Exception_Register &= ~(1<<easp1);
	      return (1<<easp1);
	   }
	}
	return Exception_Register;
}
