/*
 * sound.c
 *
 * audio emulation module 
 *
 * $Log: sound.c,v $
 * Revision 1.6  1998/11/27  04:20:53  ctkwan
 * fix sgi defined constant for autoconf
 *
 */
static char rcsid[]="$Id: sound.c,v 1.6 1998/11/27 04:20:53 ctkwan Exp ctkwan $";

#include <stdio.h>
#include "hardware.h"
#include "yae.h"

/*
 * platform dependent information and globals
 */

#ifdef	sun 
/*
 * Sun audio library
 * for SunOS 4.x
 */
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/ioccom.h>
#include <sys/ioctl.h>
#include <sys/stropts.h>

#define SOUND_BUFFER_SIZE	4096
#define SOUND_SAMPLE_TICK	125	/* 8 kHz sampling */
#define SOUND_BUFFER_TICK	(SOUND_BUFFER_SIZE*SOUND_SAMPLE_TICK)
#define SOUND_OUTPUT_HIGH	255
#define SOUND_OUTPUT_LOW	0

#if defined(HAVE_SYS_AUDIOIO_H)
#include <sys/audioio.h>	/* Solaris 2.X */
#elif defined(HAVE_SUN_AUDIOIO_H)
#include <sun/audioio.h>	/* SunOS 4 or earlier */
#endif

static int		audio_fd;
static int		audioctl_fd;
static audio_info_t	Audio_Info;

#define	HAS_AUDIO
#endif	/* sun */

/*
 * Silicon Graphics AL audio library
 * for IRIS indigoes
 */

#if defined(sgi)
#include <audio.h>

#define SOUND_BUFFER_SIZE	1024
#define SOUND_OUTPUT_HIGH	127
#define SOUND_OUTPUT_LOW	-128
#define SOUND_SAMPLE_TICK	62	/* 16 kHz sampling */
#define SOUND_BUFFER_TICK	(SOUND_BUFFER_SIZE*SOUND_SAMPLE_TICK)

ALport		Apple_ALport;
ALconfig	Apple_ALconfig;

long		Apple_ALparams[] = {
		   AL_OUTPUT_RATE, 16000,
		};
#define	HAS_AUDIO
#endif

/*
 * linux and FreeBSD
 */
#if defined(linux) || defined(__FreeBSD__)
#include <fcntl.h>
#include <unistd.h>

static int		audio_fd;

#define SOUND_BUFFER_SIZE	4096
#define SOUND_SAMPLE_TICK	125	/* 8 kHz sampling */
#define SOUND_BUFFER_TICK	(SOUND_BUFFER_SIZE*SOUND_SAMPLE_TICK)
#define SOUND_OUTPUT_HIGH	255
#define SOUND_OUTPUT_LOW	0

#define	HAS_AUDIO
#endif /* linux and FreeBSD */

#ifdef	HAS_AUDIO
/*
 * platform independent globals
 */

/* Sound Queue: recording speaker toggle */
/* SOUND_QUEUE_MAX must be an integral power of 2 */
#define	SOUND_QUEUE_MAX	16384	
#define	MOD_SOUND_QUEUE_MAX(x)	((x)&(SOUND_QUEUE_MAX-1))
static int		Sound_Queue_Head=0;
static int		Sound_Queue_Size=0;
static APPLE_TICK	Apple_Sound_Queue[SOUND_QUEUE_MAX];

static int		Apple_Audio_Present=0;

/* Sound Buffer */
static BYTE		Apple_Sound_Buffer[SOUND_BUFFER_SIZE];
static BYTE		Last_Sound_Output = SOUND_OUTPUT_LOW;
static APPLE_TICK	Buffer_Start_Time;
static APPLE_TICK	Play_Start_Time;
static APPLE_TICK	Last_Click;

#endif

static void		FlushSoundQueue(void);

int init_sound(void)
{
#ifdef	HAS_AUDIO

#ifdef	sun
	audio_fd = open( "/dev/audio", O_WRONLY | O_NDELAY );
	if ( audio_fd == -1 ) {
	   fprintf( stderr, "Cannot open /dev/audio\n" );
	   return -1;
	}
	audioctl_fd = open( "/dev/audioctl", O_RDWR );
	if ( audioctl_fd == -1 ) {
	   fprintf( stderr, "Cannot open /dev/audioictl\n" );
	   return -1;
	}
#endif	/* sun */

#ifdef sgi
	Apple_ALconfig = ALnewconfig();
	ALsetchannels( Apple_ALconfig, AL_MONO );
	ALsetwidth( Apple_ALconfig, AL_SAMPLE_8 );
	Apple_ALport = ALopenport( "Apple][ Audio", "w", Apple_ALconfig );
	if ( ! Apple_ALport ) {
	   fprintf( stderr, "Cannot open ALport\n" );
	   return -1;
	}
	ALsetconfig( Apple_ALport, Apple_ALconfig );
	ALsetparams( AL_DEFAULT_DEVICE, Apple_ALparams,
	   sizeof(Apple_ALparams)/sizeof(long) );
#endif	/* sgi */

#if defined(linux) || defined(__FreeBSD__)
	audio_fd = open( "/dev/audio", O_WRONLY | O_NDELAY );
	if ( audio_fd == -1 ) {
	   fprintf( stderr, "Cannot open /dev/audio\n" );
	   return -1;
	}

#endif	/* linux */
	Apple_Audio_Present = 1;

#endif	/* HAS_AUDIO */
	return 0;
}


void toggleSpeaker(void)
{
#ifdef	HAS_AUDIO
	int		queue_end;

	/*
	 * record speaker toggle in queue
	 */
	if ( Apple_Audio_Present ) {
	   if ( !Sound_Queue_Size ) {
	      /*
	       * first click, submit audio micro job
	       */
	      Buffer_Start_Time = AppleClock;
	      Last_Click = AppleClock - SOUND_SAMPLE_TICK - 1;
	      submit_micro_job( AppleClock + 2 * SOUND_BUFFER_TICK,
	         FlushSoundQueue, 0 );
	   }

	   /*
	    * low pass filter
	    */
	   if ( (unsigned)(AppleClock-Last_Click) >= SOUND_SAMPLE_TICK ) {
	      /*
	       *	record speaker toggle
	       */
	      queue_end = MOD_SOUND_QUEUE_MAX(Sound_Queue_Head + 
	         Sound_Queue_Size);
	      Apple_Sound_Queue[queue_end] = AppleClock;
	      Sound_Queue_Size++;
	      Last_Click = AppleClock;
	   }
	}
#endif	/* HAS_AUDIO */
}

static void FlushSoundQueue(void)
{
#ifdef	HAS_AUDIO
	int		i;
	APPLE_TICK	t, head;

	if ( Apple_Audio_Present && Sound_Queue_Size ) {
	   head = Apple_Sound_Queue[Sound_Queue_Head];
	   t = Buffer_Start_Time;;

	   /*
	    * generate wave form from speaker toggle log
	    */
	   for( i = 0; i < SOUND_BUFFER_SIZE; i++ ) {  
	      if ( Sound_Queue_Size && !TICK_LESS( t, head ) ) {
	         if ( Last_Sound_Output == SOUND_OUTPUT_HIGH )
	            Last_Sound_Output = SOUND_OUTPUT_LOW;
	         else
	            Last_Sound_Output = SOUND_OUTPUT_HIGH;
	         Sound_Queue_Head = MOD_SOUND_QUEUE_MAX(Sound_Queue_Head + 1 );
	         Sound_Queue_Size--;
	         head = Apple_Sound_Queue[Sound_Queue_Head];
	      }
	      Apple_Sound_Buffer[i] = Last_Sound_Output;
	      t += SOUND_SAMPLE_TICK;
	   }

	   /*
	    * send generated wave form to audio device
	    */
#if defined(sun) || defined(linux) || defined(__FreeBSD__)
	   write( audio_fd, Apple_Sound_Buffer, SOUND_BUFFER_SIZE );
#endif	
#ifdef sgi
	   ALwritesamps( Apple_ALport, Apple_Sound_Buffer,
	      SOUND_BUFFER_SIZE );
#endif /* sgi */
	}

	/*
	 * re-submit micro job for next sound chunk
	 */
	Buffer_Start_Time += SOUND_BUFFER_TICK;
	if ( Sound_Queue_Size )
	   submit_micro_job( AppleClock + SOUND_BUFFER_TICK,
	      FlushSoundQueue, NULL );
#endif /* HAS_AUDIO */
}
