/*
 * shm.c
 *
 * X11 MIT/SHM Shared Memory Image extension module
 *
 * $Log$
 */
static char rcsid[]="$Id$";

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>

#include "hardware.h"
#include "x_window.h"

static shmError;

int errorHandler(Display *dpy, XErrorEvent *ev)
{
	shmError=1;
	return 0;
}

/*
 * createShmImage:
 * Ccreates a shared memory XImage of given width and height.
 * If MIT/Shm extension is not available in server or server
 * is not on local machine, create an ordinary XImage instead.
 * Returns a pointer to the XImage created if succeed.
 */
XImage *createShmImage( DisplayContext *dc, int width, int height,
	XShmSegmentInfo *info )
{
	XImage *image;
	int major,minor,pixmap;
	int (*handler)(Display*,XErrorEvent*);

	/* set values for undo in case of error */
	image=NULL;
	info->shmid=-1;
	info->shmaddr=(char*)-1;

	/* check if X server support XShm extension */
	if (!XShmQueryVersion( dc->display, &major, &minor, &pixmap)){
	   fprintf(stderr,
	      "XShm extension not supported. Use non-XShm XImage instead.\n");
	   goto no_Xshm;
	}

	if (UseXShm){
	   image = XShmCreateImage( dc->display, dc->visual,
	      dc->depth, ZPixmap, NULL, info, width, height );

	   if (!image) {
	      fprintf( stderr,
	         "XShmCreateImage failed. Fall back to non-XShm XImage.\n" );
	      goto no_Xshm;
	   }

	   info->shmid = shmget( IPC_PRIVATE, image->bytes_per_line *
	      image->height, IPC_CREAT|0777 );
	   if (info->shmid==-1) {
	      fprintf( stderr, "shmget failed\n" );
	      goto error;
	   }

	   info->shmaddr = image->data = (char*)shmat( info->shmid, NULL, 0 );
	   info->readOnly = False;
	 
	   /* trap X windows error */
	   shmError=0; 
	   XSynchronize( dc->display, True);
	   handler=XSetErrorHandler(errorHandler);

	   XShmAttach( dc->display, info );

	   /* restore default error handling */
	   XSetErrorHandler(handler);
	   XSynchronize( dc->display, False);

	   if ( shmError ) {
	      fprintf( stderr, "XShmAttach failed\n" );
	      XDestroyImage( image );
	      if (shmctl( info->shmid, IPC_RMID, NULL )==-1){
	         perror("destroyShmImage() cannot remove share memory");
	      }
	      goto no_Xshm;
	   }

	   return image;
	}

no_Xshm:
	UseXShm=FALSE;
	image = XCreateImage( dc->display, dc->visual, dc->depth,
	   ZPixmap, 0, NULL, width, height, 32, 0 );

	if (!image) {
	   fprintf( stderr, "XCreateImage failed\n" );
	   goto error;
	}
	image->data=malloc(image->height*image->bytes_per_line);
	return image;

error:
	/* undo what have done so far */
	if (UseXShm){
	   if (info->shmaddr!=(char*)-1)
	      shmdt( info->shmaddr );
	   if (info->shmid!=-1)
	      shmctl( info->shmid, IPC_RMID, NULL );
	}
	if(image!=NULL)
	   XDestroyImage(image);
	return NULL;
}

void destroyShmImage(DisplayContext *dc, XImage *image, XShmSegmentInfo *info)
{
	if (UseXShm) {
	   XShmDetach( dc->display, info );
	   XDestroyImage( image );
	   if (shmctl( info->shmid, IPC_RMID, NULL )==-1){
	      perror("destroyShmImage() cannot remove share memory");
	   }
	}
	else
	   XDestroyImage( image );
}
